PROGRAM ecmwfToBinary
   IMPLICIT NONE

!*******************************************************************
!
!   Date: 27 March 2013
!
!   Programmer: Frank Tilley (IMSG)
!
!   Platform: Linux server
!
!   Function: This program reads data extracted from ECMWF GRIB1/GRIB2
!             files and writes out a record for each 0.25-degree grid 
!             intersection to a file that can be read and displayed
!             by the Orbital Display System. The output file is also
!             intended to be used in NPROVS matchups.
!
!*****************************************************************

! Functions

   REAL(4) :: getlat, getlon, spfh2mr
   INTEGER(4) :: determineTemperatureQuality, determineMoistureQuality, determineOzoneQuality
      
! Variables

   INTEGER, PARAMETER :: short = selected_int_kind(3)

   INTEGER(4), PARAMETER :: numpoints = 1038240
   INTEGER(4), PARAMETER :: inreclength = 4152960
   INTEGER(4), PARAMETER :: infile = 11
   INTEGER(4), PARAMETER :: outfile = 20
   INTEGER(4), PARAMETER :: numwords = 382
   INTEGER(4), PARAMETER :: outreclength = numwords * 2

   INTEGER(4) :: inrec, point, ioerr, word, outrec, lev, top, bottom
   INTEGER(4) :: YYYY, MMDD, HH, lastrec, level, index

   INTEGER(KIND=short), DIMENSION(numwords) :: databuff
   INTEGER(KIND=short), PARAMETER :: missing = -32768

   REAL(4), DIMENSION(91) :: hybridpress, lpw, wvmr, profile

   REAL(4), DIMENSION(numpoints) :: lats, lons, surpress, tskin

   REAL(4), DIMENSION(91,numpoints) :: temp
   REAL(4), DIMENSION(91,numpoints) :: spfh
   REAL(4), DIMENSION(91,numpoints) :: o3mr

   REAL(4) :: sfcpress, tpw

   CHARACTER(LEN=200) :: erchar

!   Get the latitudes and longitudes for the grid points

   DO point = 1, numpoints
      lats(point) = getlat(point)
      lons(point) = getlon(point)
   END DO

!   Read in the date and time of the forecast

   READ(5,*) YYYY
   READ(5,*) MMDD
   READ(5,*) HH

!   Open the binary ECMWF file for direct access READ
     
   OPEN(infile,access="DIRECT",form="UNFORMATTED",status="OLD", &
        recl=inreclength,iomsg=erchar,iostat=ioerr)

   IF (ioerr /= 0) THEN
      PRINT *,'Error ',ioerr,' opening input file '
      PRINT *,'Iomsg ',erchar
      print *, 'checkpoint 1'
      GO TO 950
   END IF

!   Open the output binary file for direct access WRITE

   OPEN(outfile,access="DIRECT",form="UNFORMATTED",status="UNKNOWN", &
        recl=outreclength,iomsg=erchar,iostat=ioerr)

   IF (ioerr /= 0) THEN
      PRINT *,'Error ',ioerr,' opening output file '
      PRINT *,'Iomsg ',erchar
      CLOSE(infile)
      print *, 'checkpoint 2'
      GO TO 950
   END IF

!   Read the surface pressures from the first record of the file.

   inrec = 1

   READ(infile,rec=inrec,iomsg=erchar,iostat=ioerr) surpress

   IF (ioerr /= 0) THEN
      PRINT *,'Error ',ioerr,' reading record ',inrec
      PRINT *,'Iomsg ',erchar
      print *, 'checkpoint 3'
      GO TO 950
   END IF

!  Read the skin temperatures from the second record of the file

   inrec = inrec+1

   READ(infile,rec=inrec,iomsg=erchar,iostat=ioerr) tskin

   IF (ioerr /= 0) THEN
         PRINT *,'Error ',ioerr,' reading record ',inrec
         PRINT *,'Iomsg ',erchar
      print *, 'checkpoint 4'
         GO TO 950
   END IF

!   Read the temperatures at 91 hybrid levels

   DO level = 1, 91
      inrec = inrec+1
      READ(infile,rec=inrec,iomsg=erchar,iostat=ioerr) temp(level,1:numpoints)

      IF (ioerr /= 0) THEN
         PRINT *,'Error ',ioerr,' reading record ',inrec
         PRINT *,'Iomsg ',erchar
      print *, 'checkpoint 5'
         GO TO 950
      END IF
   END DO

!   Read the specific humidity at 91 hybrid levels

   DO level = 1, 91
      inrec = inrec+1
      READ(infile,rec=inrec,iomsg=erchar,iostat=ioerr) spfh(level,1:numpoints)

      IF (ioerr /= 0) THEN
         PRINT *,'Error ',ioerr,' reading record ',inrec
         PRINT *,'Iomsg ',erchar
      print *, 'checkpoint 6'
         GO TO 950
      END IF
   END DO

!   Read the ozone mixing ratio at 91 hybrid levels

   DO level = 1, 91
      inrec = inrec+1
      READ(infile,rec=inrec,iomsg=erchar,iostat=ioerr) o3mr(level,1:numpoints)

      IF (ioerr /= 0) THEN
         PRINT *,'Error ',ioerr,' reading record ',inrec
         PRINT *,'Iomsg ',erchar
      print *, 'checkpoint 7'
         GO TO 950
      END IF
   END DO

!  Write the data to the output file

   DO point = 1, numpoints
      DO word = 1, numwords
         databuff(word) = missing
      END DO

      databuff(1) = 0
      databuff(2) = 0
      databuff(3) = 0
      databuff(4) = NINT(lats(point) * 128.0)
      databuff(5) = NINT(lons(point) * 128.0)
      databuff(6) = YYYY
      databuff(7) = MMDD
      databuff(8) = HH
      databuff(9) = 0000

!   Convert surface pressure to hPa

      sfcpress = surpress(point) * 0.01

!   Calculate and store the hybrid level pressures for this point

      CALL press137(sfcpress,hybridpress)

      DO lev = 1, 36
         databuff(9+lev) = NINT(hybridpress(lev) * 100.0)
      END DO
      DO lev = 37, 91
         databuff(9+lev) = NINT(hybridpress(lev) * 10.0)
      END DO
          
!   Store the analysis temperature profile for this point

      index = 100

      DO level = 1, 91
         index = index + 1
         databuff(index) = NINT(temp(level,point) * 64.0)
      END DO

!  Convert specific humidity to mixing ratio and store the
!  water vapor mixing ratio profile.

      index = 191

      DO level = 1, 91
         index = index + 1
         wvmr(level) = spfh2mr(hybridpress(level),spfh(level,point))

!         if (wvmr(level) .lt. 0.0) then
!            write(6,*) 'NEG: ',hybridpress(level),'  ',wvmr(level),'  ',spfh(level,point)            
!         end if
         IF (NINT(wvmr(level) * 1000.0) .LE. 0) wvmr(level) = 0.001
         databuff(index) = NINT(wvmr(level) * 1000.0)
      END DO

      bottom = 91

      top = 0
      DO lev = 1, 91
         IF (top == 0) THEN
            IF (hybridpress(lev) >= 200.0) THEN
               top = lev
            END IF
         END IF
      END DO

!  Calculate and store Total Precipitable Water

      index = 283

      IF (top /= 0) THEN
         CALL prwater(wvmr,hybridpress,lpw,top,bottom,tpw)
         databuff(index) = NINT(tpw * 100.0)
      END IF

!  Store the skin temperature for this point

      index = index + 1
      databuff(index) = NINT(tskin(point) * 64.0)

!  Store the ozone mixing ratio profile. The values are tiny and
!  have to be scaled accordingly.

      DO level = 1, 91
         index = index + 1
         databuff(index) = NINT(ALOG(o3mr(level,point)*1.0e6) * 64.0)
      END DO

!  Set the NPROVS derived quality flags for the profiles

      DO level = 1, 91
         profile(level) = temp(level,point)
      END DO

      databuff(376) = determineTemperatureQuality(profile);

      DO level = 1, 91
         profile(level) = wvmr(level)
      END DO

      databuff(377) = determineMoistureQuality(profile);

      DO level = 1, 91
         profile(level) = o3mr(level,point)
      END DO

      databuff(378) = determineOzoneQuality(profile);

!      write(6,*) 'QC FLAGS: ',databuff(376),databuff(377),databuff(378)
        
!  Write the record out to the output file

      outrec = point
      WRITE(outfile,rec=outrec,iomsg=erchar,iostat=ioerr) databuff

      IF (ioerr /= 0) THEN
         PRINT *,'Error ',ioerr,' writing record ',outrec
         PRINT *,'Iomsg ',erchar
         GO TO 900
      END IF

   END DO

900 CONTINUE

   CLOSE(infile)

   CLOSE(outfile)

   GO TO 1000
 

950 CONTINUE

   PRINT *,"ecmwfToBinary.x : setting all databuff values to missing"

   DO point = 1, numpoints
      DO word = 1, numwords
         databuff(word) = missing
      END DO
      databuff(1) = 0
      databuff(2) = 0
      databuff(3) = 0
      databuff(4) = NINT(lats(point) * 128.0)
      databuff(5) = NINT(lons(point) * 128.0)
      databuff(6) = YYYY
      databuff(7) = MMDD
      databuff(8) = HH
      databuff(9) = 0000
      outrec = point
      WRITE(outfile,rec=outrec,iomsg=erchar,iostat=ioerr) databuff
   END DO
   CLOSE(infile)
   CLOSE(outfile)

1000 continue

   CALL EXIT(0)

END PROGRAM ecmwfToBinary

