import java.awt.Color;
import java.io.*;
import java.io.PrintWriter;
import java.io.BufferedWriter;
import java.util.Vector;
import java.util.Arrays;
import java.util.zip.*;

/**
 * <PRE>
 * GenHighchartsScripts.java 
 *
 * Version: 1.0   Date: 10/12/2016   Programmer: Charles Brown (IMSG)
 * Version: 2.0   Date: 10/03/2024   Programmer: Mike Pettey (IMSG)
 *
 * this is a version of StatGraph.java developed by Mike which generates
 * text files instead of an image file.  These files are actually a  
 * JavaScript file with the data points, titles and axis info.  Then these are
 * read by the Highcharts function.
 *
 * </PRE>
 */

public class GenHighchartsScripts
  {
  private boolean       data_available;
  private boolean       IsLongtermPlot;

  private String        baseline_name;
  private int           num_days, num_weeks, num_months, date_start_index, date_end_index;
  private String[]      date_strings, profile_names;
  private int[]         dates, profile_system_ids, profile_system_numbers, profile_types;
  private float[]       temp_level_pressures, temp_layer_pressures;
  private float[]       wvap_level_pressures, wvap_layer_pressures;
  
  private Vector        visible_profile_vector;
  private Vector        visible_sample_size_profile_vector, visible_statistic_profile_vector;
  private int           min_count, max_count;
  private float         min_stat, max_stat;

  private NarcsGlobals  globals2;
  private NarcsToHighchartsGlobals globals;

  private final int BIAS        = 0;
  private final int STDDEV      = 1;
  private final int RMS         = 2;
  private final int SAMPLE_SIZE = 3;

  private final int SYSTEM    = 0;
  private final int RETRIEVAL = 1;
  private final int GUESS     = 2;
  private final int FORECAST  = 3;
  private final int MICROWAVE = 4;
  private final int INFRARED  = 5;
  private final int RAW_DRY   = 6;

  private final int NUM_PIECES = 18;
  private PrintWriter pw;
  private PrintWriter out;
  private String file_name;


  public GenHighchartsScripts(boolean IsLongtermPlot)
    {
    this.IsLongtermPlot = IsLongtermPlot;
    //globals = new NarcsGlobals();
    globals = new NarcsToHighchartsGlobals();
    if (IsLongtermPlot) file_name = "NarcsHighcharts_longterm.js";
    else file_name = "NarcsHighcharts.js";

    visible_profile_vector = new Vector();
    visible_sample_size_profile_vector = new Vector();
    visible_statistic_profile_vector = new Vector();


    min_count = 1;
    max_count = 1000;

    min_stat  = -5F;
    max_stat  =  5F;

    data_available = false;
    }



  private void drawGraphWithLegend()
    {

    // Draw the title

    drawTitle();

    // Draw the legend

    drawLegend();

    // Draw the main graph
    

    // Setup colors for plot lines
    try
      {
       PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(file_name,true)));

    // Draw the y axis

    //drawYAxis();

    // Draw the x axis

    drawXAxis();

    out.println("        yAxis: {");

    if (globals.getLevelLayerDataType() == globals.TEMP_LEVEL)
      {
      out.println("          max: 4,");
      out.println("          min: 0,");
      }

    else if (globals.getLevelLayerDataType() == globals.TEMP_LAYER)
      {
      out.println("          max: 4,");
      out.println("          min: 0,");
      }
    else if (globals.getLevelLayerDataType() == globals.WVAP_LEVEL)
      {
      out.println("          max: 100,");
      out.println("          min: 5,");
      }
    else if (globals.getLevelLayerDataType() == globals.WVAP_LAYER)
      {
      out.println("          max: 100,");
      out.println("          min: 5,");
      }

    out.println("             title: {");
    out.println("                 text: 'RMS'");
    out.println("             },");
    out.println("             plotLines: [{");
    out.println("                 value: 0,");
    out.println("                 width: 1,");
    out.println("                 color: '#808080'");
    out.println("             }]");
    out.println("        },");
    out.println("        tooltip: {");
    out.println("            valueSuffix: ''");
    out.println("        },");
    out.println("        legend: {");
    out.println("            layout: 'horizontal',");
    out.println("            align: 'center',");
    out.println("            verticalAlign: 'bottom',");
    out.println("            borderWidth: 0");
    out.println("        },");
    out.flush();

    drawMainGraph();

    // Write ending lines to Highchart JavaScript file

    out.println("\n        }]");
    out.println("    })");
    out.flush();
    out.close();
    }
  catch (IOException ioe)
    {
    System.out.println("IOEception found for file name = "+file_name);
    ioe.printStackTrace();
    }
  finally
    {
    if (out != null)
      {
      out.close();
      }
    }

  // The ending bracket will be added via the run script
  //out.println("})");
  }



  private void drawTitle()
    {
    NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(0);
    String prof_name = (String)(profile.getLabel());
    int terrain_type = profile.getTerrainType();
    boolean[] show_piece = globals.getShowCombination();
    boolean landsea_option = show_piece[6];

    String container_string = "";
    String stat_type = profile.getLabel();
    int sat_type_index = 0;
    String title = "";
    String pressure_string = "" + globals.getLevelLayerPressure() + " mb";

    try
      {
       PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(file_name,true)));

       if (landsea_option)
         container_string = "land_";
       else
         container_string = "sea_";

       if ((sat_type_index = stat_type.indexOf("IR+MW")) > 0)
         {
         container_string = container_string + "irmw_";

         if ((sat_type_index = stat_type.indexOf("NU")) == 0)
           container_string = container_string + "sat1_";
         else
           container_string = container_string + "sat2_";
         }
      else if ((sat_type_index = stat_type.indexOf("MW")) > 0)
        {
        container_string = container_string + "mw_";

        if ((sat_type_index = stat_type.indexOf("NU")) == 0)
          container_string = container_string + "sat1_";
        else
          container_string = container_string + "sat2_";
        }
      else
       container_string = container_string + "non_sat_";

      if (globals.getLevelLayerDataType() == globals.TEMP_LEVEL)
        {
        title = "Temperature " + pressure_string + " Level Statistics";
        container_string = container_string + "temp_";
        container_string = container_string + "press"+pressure_string+"_";
        }
      else if (globals.getLevelLayerDataType() == globals.TEMP_LAYER)
        {
        title = "Temperature " + pressure_string + " Layer Statistics";
        container_string = container_string + "temp_";

        if (pressure_string.equals("827.124 mb")) container_string = container_string+"press850";
        else if (pressure_string.equals("729.638 mb")) container_string = container_string+"press700";
        else if (pressure_string.equals("506.009 mb")) container_string = container_string+"press500";
        else if (pressure_string.equals("293.017 mb")) container_string = container_string+"press300";
        else if (pressure_string.equals("206.407 mb")) container_string = container_string+"press200";
        else if (pressure_string.equals("113.926 mb")) container_string = container_string+"press100";
        else if (pressure_string.equals("51.488 mb")) container_string = container_string+"press50";
        else if (pressure_string.equals("10.237 mb")) container_string = container_string+"press10";
        }
      else if (globals.getLevelLayerDataType() == globals.WVAP_LEVEL)
        {
        title = "Water Vapor Percent Error " + pressure_string + " Level Statistics";
        container_string = container_string + "wvap_";
        }
      else if (globals.getLevelLayerDataType() == globals.WVAP_LAYER)
        {
        title = "Water Vapor Percent Error " + pressure_string + " Layer Statistics";
        container_string = container_string + "wvap_";

        if (pressure_string.equals("852.523 mb")) container_string = container_string+"press850";
        else if (pressure_string.equals("753.385 mb")) container_string = container_string+"press700";
        else if (pressure_string.equals("555.025 mb")) container_string = container_string+"press500";
        else if (pressure_string.equals("328.57 mb")) container_string = container_string+"press300";
        }

      if (!IsLongtermPlot)
        {
        String month = date_strings[0].substring(0,2);
        if (month.equals(" 1")) month = "01";
        if (month.equals(" 2")) month = "02";
        if (month.equals(" 3")) month = "03";
        if (month.equals(" 4")) month = "04";
        if (month.equals(" 5")) month = "05";
        if (month.equals(" 6")) month = "06";
        if (month.equals(" 7")) month = "07";
        if (month.equals(" 8")) month = "08";
        if (month.equals(" 9")) month = "09";

        // For year, the first two years, 2008 and 2009, are singl numbers
        // so have to test special for those two years.

        String year_single_digit = date_strings[0].substring(6,7);
        String year = " ";

        if (year_single_digit.equals("8") || year_single_digit.equals("9"))
	        year = "200" + year_single_digit;
        else
    	    year = "20" + date_strings[0].substring(6,8);

        container_string = container_string + "_" + year + month;
        }

      System.out.println("container_string = "+container_string);

      out.println("    $('#"+container_string+"').highcharts({");

      if (IsLongtermPlot)
        {
        out.println("        chart: {");
        out.println("           zoomType: 'x'");
        out.println("        },");
        }

      out.println("        title: {");
      out.println("            text: 'NOAA/NESDIS/STAR NPROVS Collocation Summary Statistics (NARCS)',");
      //out.println("            x: -20 //center");
      out.println("        },");
      String subtitle = "            text: '"+title+", ";

      if (landsea_option)
        subtitle = subtitle + "Land, ";
      else
        subtitle = subtitle + "Sea, ";

      out.println("        subtitle: {");

      if ((sat_type_index = prof_name.indexOf("IR")) > 0)
        out.println(subtitle + "Infrared plus Microwave'," );
      else
        out.println(subtitle + "Microwave',");

      out.println("            style: {");
      out.println("            color: '#FF0000',");
      out.println("            fontWeight: 'bold'");
      out.println("            },");
      out.println("            x: -20 //center");
      out.println("        },");
      out.flush();
      out.close();
      }
    catch (IOException ioe)
      {
      System.out.println("IOEception found for file name = "+file_name);
      ioe.printStackTrace();
      }
    finally
      {
      if (out != null)
        {
        out.close();
        }
      }
    }



  private void drawXAxis()
    {
    if (data_available)
      {
      try
        {
        PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(file_name,true)));
        out.println("        xAxis: {");
        out.print("            categories: [");
        System.out.println("Number of labels="+date_strings.length);

        for (int i=0; i<date_strings.length; i++)
          {
          out.print("'" + date_strings[i]+"'");
          if (i == date_strings.length-1) out.print("],");
          if ((i != date_strings.length) && (i != date_strings.length-1)) out.print(", ");
          if (i % 6 == 0) out.println(" ");
          }

        out.println("            labels: {");
        out.println("                rotation: 90");
        out.println("            },");
        out.println("            minorGridLineDashStyle: 'longdash',");
        out.println("            minorTickInterval: 'auto',");
        out.println("            minorTickWidth: 0,");
        out.println("            plotLines: [{");
        out.println("                dashStyle: 'longdash',");
        out.println("                value: 0,");
        out.println("                width: 1,");
        out.println("                color: '#808080'");
        out.println("            }]");
        out.println("        },");
        out.flush();
        out.close();
        }
      catch (IOException ioe)
        {
        System.out.println("IOEception found for file name = "+file_name);
        ioe.printStackTrace();
        }
      finally
        {
        if (out != null)
          {
          out.close();
          }
        }
      }  // if (data_available...
    }



  private void drawMainGraph()
    {
    if (! data_available)
      {
      //drawNoFileOpenMessage(g, x_offset, y_offset, image_width, image_height);
      }
    else
      {
      int number_of_dates = date_strings.length;

      // Draw each of the visible profiles

      try
        {
        PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(file_name,true)));
        out.println("        series: [{");

        for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
          {
          int sat_type_index = 0;
          NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);

          String prof_name = (String)(profile.getLabel());
          sat_type_index = prof_name.indexOf(" MW");

          if ((sat_type_index = prof_name.indexOf("IR+MW")) > 0 || (sat_type_index = prof_name.indexOf(" MW")) > 0)
            System.out.println("Profile name: "+prof_name+" sat_type_index="+sat_type_index);

          if ((sat_type_index = prof_name.indexOf("IR+MW")) > 0 )
            {
            out.println("            name: '"+prof_name.substring(0,sat_type_index-1)+"'," );
            }
	        else
            {
            sat_type_index = prof_name.indexOf(" MW");
            if (sat_type_index > 0) 
              out.println("            name: '"+prof_name.substring(0,sat_type_index)+"'," );
            else
              out.println("            name: '"+prof_name+"'," );
            }

          out.println("data: [");
          float[] values = profile.getValues();

          for (int ival=0; ival<values.length;ival++)
            {
            System.out.println("Data Value["+ival+"]="+values[ival]);

            if (ival != values.length-1)
		          if (values[ival] != -32768)
                out.print(values[ival]+", ");
		          else
                out.print(null+", ");
            else if (values[ival] != -32768)
              out.print(values[ival]+"] ");
		        else
              out.print(null+"] ");
            }

          if (profnum != visible_profile_vector.size()-1)
            out.println("}, { ");

          boolean previous_data_found = false;
          }  // for (profnum=0...

        out.flush();
        out.close();
        }
      catch (IOException ioe)
        {
        System.out.println("IOEception found for file name = "+file_name);
        ioe.printStackTrace();
        }
      finally
        {
        if (out != null)
          {
          out.close();
          }
        }
      }  // if (data_available...
    }



  public void drawLegend()
    {
    if (data_available)
      {

      // Calculate the number of profiles being plotted and save
      // the unique profile names. Also determine the number of
      // statistic types being displayed.
      
      Vector code_vector = new Vector();
      Vector profile_name_vector = new Vector();
      Vector unique_profile_vector = new Vector();
      
      boolean bias_used        = false;
      boolean stddev_used      = false;
      boolean rms_used         = false;
      boolean sample_size_used = false;
      
      for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
        {
        NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);

        int system_id    = profile.getSystemID();
        int profile_type = profile.getProfileType();

        
        int code = (system_id * 100) + profile_type;
        Integer icode = new Integer(code);
        
        if (! code_vector.contains(icode))
          {
          code_vector.addElement(icode);
          profile_name_vector.addElement(profile.getLabel());
          unique_profile_vector.addElement(profile);
          }
        
        int stat_type = profile.getStatType();
        
        if (stat_type == BIAS)
          bias_used = true;
        else if (stat_type == STDDEV)
          stddev_used = true;
        else if (stat_type == RMS)
          rms_used = true;
        else if (stat_type == SAMPLE_SIZE)
          sample_size_used = true;
        }  // for (profnum=0...


      int num_stats_used = 0;
      
      if (bias_used)
        num_stats_used++;
      
      if (stddev_used)
        num_stats_used++;
      
      if (rms_used)
        num_stats_used++;
      
      if (sample_size_used)
        num_stats_used++;
      
      int num_profiles = code_vector.size();
      int total_lines = num_profiles + num_stats_used + 1;

      // Draw labels for each unique profile

      int sat_type_index = 0;

      String prof_name = (String)(profile_name_vector.elementAt(0));
      boolean[] show_piece = globals.getShowCombination();
      boolean landsea_option = show_piece[6];

      try
        {
        PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(file_name,true)));

        out.print("        colors: ['#");

       for (int prof=0; prof<code_vector.size(); prof++)
         {
         int code = ((Integer)(code_vector.elementAt(prof))).intValue();
         Color color = globals.getProfileColor(code);
         out.print(String.format("%02x", color.getRed())+
         String.format("%02x", color.getGreen())+
		     String.format("%02x", color.getBlue()));
    	   if (prof != (code_vector.size()-1))
           out.print("', '#");
         }

       out.println("'],");
//       out.println("        colors: ['#0088ff', '#ff8800', '#90ed7d', '#f7a35c', '#8085e9',");
//       out.println("        '#f15c80', '#e4d354', '#2b908f', '#f45b5b', '#91e8e1'],");
       out.flush();
       out.close();
       }
    catch (IOException ioe)
      {
      System.out.println("IOEception found for file name = "+file_name);
      ioe.printStackTrace();
      }
    finally
      {
      if (out != null)
        {
        out.close();
        }
      }
    }  // if (data_available...
  }



//  private void drawNoFileOpenMessage(Graphics2D g, int x_offset, int y_offset, int image_width, int image_height)
//    {
//    g.setColor(Color.gray);
//    g.drawRect(x_offset, y_offset, (image_width-1), (image_height-1));
//    
//    g.setClip(x_offset, y_offset, image_width, image_height);
//    
//    // Find an optimal font size
//    
//    String line1 = "No file opened";
//    String line2 = "Click the \"Options\" button and then open a file";
//    String line3 = "in the dialog that pops up";
//    
//    // Starting with a font size of 24, loop while looking for a size that 
//    // allows all of the title to be displayed within the available space
//
//    FontMetrics fm = g.getFontMetrics();
//    int font_size = 36;
//    boolean fit = false;
//
//    while ((fit == false) && (font_size > 5))
//      {
//      font_size--;
//
//      g.setFont(new Font("Dialog", Font.PLAIN, font_size));
//      fm = g.getFontMetrics();
//
//      if (((fm.getHeight()*3) < (image_height-2)) && 
//          (fm.stringWidth(line1) < (image_width-4)) &&
//          (fm.stringWidth(line2) < (image_width-4)) &&
//          (fm.stringWidth(line3) < (image_width-4)))
//        {
//        fit = true;
//        }
//      }
//
//    // Draw the message
//    
//    int line2_y = y_offset + (image_height / 2) - (fm.getHeight() / 2);
//    int line1_y = line2_y - fm.getHeight();
//    int line3_y = line2_y + fm.getHeight();
//    
//    int x = x_offset + ((image_width / 2) - (fm.stringWidth(line1) / 2));
//    g.drawString(line1, x, line1_y);
//    
//    x = x_offset + ((image_width / 2) - (fm.stringWidth(line2) / 2));
//    g.drawString(line2, x, line2_y);
//    
//    x = x_offset + ((image_width / 2) - (fm.stringWidth(line3) / 2));
//    g.drawString(line3, x, line3_y);
//
//    g.setClip(null);
//    }



  public void addFileInformation(String baseline_name, int num_days, int num_weeks, int num_months,
                                 String[] profile_names, int[] profile_system_ids,
                                 int[] profile_system_numbers, int[] profile_types,
                                 int start_index, int end_index, String[] date_strings, 
                                 int[] dates, float[] temp_level_pressures, float[] temp_layer_pressures, 
                                 float[] wvap_level_pressures, float[] wvap_layer_pressures)
    {
    this.baseline_name = baseline_name;
    this.num_days      = num_days;
    this.num_weeks     = num_weeks;
    this.num_months    = num_months;

    this.profile_names          = profile_names;
    this.profile_system_ids     = profile_system_ids;
    this.profile_system_numbers = profile_system_numbers;
    this.profile_types          = profile_types;

    date_start_index  = start_index;
    date_end_index    = end_index;
    this.date_strings = date_strings;
    this.dates        = dates;
    
    this.temp_level_pressures = temp_level_pressures;
    this.temp_layer_pressures = temp_layer_pressures;
    this.wvap_level_pressures = wvap_level_pressures;
    this.wvap_layer_pressures = wvap_layer_pressures;

    data_available = true;
    
    //updateGraph();
    }



  public void setTimeWindow(int start_index, int end_index, String[] date_strings, int[] dates)
    {
    date_start_index  = start_index;
    date_end_index    = end_index;
    this.date_strings = date_strings;
    this.dates        = dates;
    }



  public void updateGraph()
    {
    if (data_available)
      {
      visible_profile_vector.removeAllElements();

      // Build the profiles that are visible

      buildProfiles();

      // Find the mininum and maximum values
      
      findStatisticMinMax();
      findSampleSizeMinMax();

      // Remove any unwanted data from each profile
      
      removeUnwantedData();

      // Draw the graph

      //if (this.isVisible() && this.isValid())
      //  {
      //  offscreen_image = createImage(width, height);
      //  Graphics2D g = (Graphics2D)(offscreen_image.getGraphics());

      //  drawGraph(g, 0, 0, width, height, true);

      //  g.dispose();

      //  paintComponent(this.getGraphics());
      //  }
      }  // if (data_available...
    }



  public void updateGraphWithLegend()
    {
    if (data_available)
      {
      visible_profile_vector.removeAllElements();

      // Build the profiles that are visible

      buildProfiles();

      // Find the mininum and maximum values
      
      findStatisticMinMax();
      findSampleSizeMinMax();

      // Remove any unwanted data from each profile
      
      removeUnwantedData();
      }

    // Print beginning of JS file
  
  //  System.setOut(new PrintStream(new BufferedOutputStream(new FileOutputStream("test.js"))));
      //try
        //{

		//File file = new File("NarcsHighcharts.js",true);
		//FileOutputStream fos = new FileOutputStream(file);
		//PrintStream ps = new PrintStream(fos);
    //System.setOut(ps);
     //    PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter("NarcsHighcharts.js",true)));
    //out.println("$(function () {");
    //out.println("    $('#container').highcharts({");
    //out.println("        title: {");
    /*    }
        catch (FileNotFoundException e)
        {
            e.printStackTrace();
        }
        finally
        {
            if ( pw != null )
            {
                pw.close();
            }
        }
    */
 // pw.write("$(function () {\n");
    // Draw the graph

//    drawGraphWithLegend(g, x_offset, y_offset, image_width, image_height);
    drawGraphWithLegend();
    }



  private void buildProfiles()
    {
    Vector visible_profiles = globals.getVisibleProfiles();

    boolean show_bias        = globals.getShowBias();
    boolean show_stddev      = globals.getShowStdDev();
    boolean show_rms         = globals.getShowRMS();
    boolean show_sample_size = globals.getShowSampleSize();

    // Loop through each system and build retrieval profiles for those that
    // are selected
    
    for (int sys=0; sys<visible_profiles.size(); sys++)
      {
      
      // Determine the system number (0 to N)

      int system_val = ((Integer)(visible_profiles.elementAt(sys))).intValue();
      int system_id    = system_val / 100;
      int system_type  = system_val % 100;

      int system_number = -32768;

      for (int n=0; n<profile_system_ids.length; n++)
        {
        if (system_id == profile_system_ids[n])
          system_number = profile_system_numbers[n];
        }

      // If the system number was found, create profiles for the visible profiles

      if (system_number != -32768)
        {
        boolean[] show_piece = globals.getShowCombination();

        for (int piece=0; piece<NUM_PIECES; piece++)
          {
          if (show_piece[piece] == true)
            {
            NarcsDataProfile bias_profile = null;
            NarcsDataProfile stddev_profile = null;
            NarcsDataProfile rms_profile = null;
            NarcsDataProfile sample_size_profile = null;

            // Create the profiles for each type of statistic

            sample_size_profile = buildProfile(system_id, system_number, system_type, SAMPLE_SIZE, piece,
                                               null);

            if (show_bias)
              {
              bias_profile = buildProfile(system_id, system_number, system_type, BIAS, piece,
                                          sample_size_profile);
              }

            if (show_stddev)
              {
              stddev_profile = buildProfile(system_id, system_number, system_type, STDDEV, piece,
                                            sample_size_profile);
              }

            if (show_rms)
              {
              rms_profile = buildProfile(system_id, system_number, system_type, RMS, piece,
                                         sample_size_profile);
              }


            // Go through the data and remove any data that may be out of range or
            // below the sample size cutoff. Then add the profile to the visible profile
            // vector

            if (bias_profile != null)
              {
              //bias_profile = removeUnwantedStatisticData(bias_profile, sample_size_profile);
              visible_profile_vector.add(bias_profile);
              }

            if (stddev_profile != null)
              {
              //stddev_profile = removeUnwantedStatisticData(stddev_profile, sample_size_profile);
              visible_profile_vector.add(stddev_profile);
              }

            if (rms_profile != null)
              {
              //rms_profile = removeUnwantedStatisticData(rms_profile, sample_size_profile);
              visible_profile_vector.add(rms_profile);
              }

            if (show_sample_size)
              {
              //sample_size_profile = removeUnwantedSampleSizeData(sample_size_profile);
              visible_profile_vector.add(sample_size_profile);
              }
            }  // if (show_piece[piece]...
          }  // for (piece=0...
        }  // if (system_number...
      }  // for (sys=0...
    }



  private NarcsDataProfile buildProfile(int system_id, int system_number, int profile_type, 
                                        int stat_type, int piece, NarcsDataProfile sample_size_profile)
    {
    NarcsDataProfile profile = null;

    int time_window = globals.getTimeWindow();

    // Build the file name

    String end_of_file_name = "/";
      
    if (time_window == globals.DAILY)
      end_of_file_name += "daily";
    else if (time_window == globals.WEEKLY)
      end_of_file_name += "weekly";
    else
      end_of_file_name += "monthly";
      
    end_of_file_name += "_system" + system_number;

    if (profile_type < 10)
      end_of_file_name += "_profile0" + profile_type + "_";
    else
      end_of_file_name += "_profile" + profile_type + "_";

    int pressure_index = 0;

    if (globals.getLevelLayerDataType() == globals.TEMP_LEVEL)
      {
      end_of_file_name += "templevel";
      
      for (int i=0; i<temp_level_pressures.length; i++)
        {
        if (temp_level_pressures[i] == globals.getLevelLayerPressure())
          pressure_index = i;
        }
      }
    else if (globals.getLevelLayerDataType() == globals.TEMP_LAYER)
      {
      //end_of_file_name += "templayer";
      end_of_file_name += "templevel";
      
      for (int i=0; i<temp_layer_pressures.length; i++)
        {
        if (temp_layer_pressures[i] == globals.getLevelLayerPressure())
          pressure_index = i;
        }
      }
    else if (globals.getLevelLayerDataType() == globals.WVAP_LEVEL)
      {
      end_of_file_name += "wvaplevel";
      
      for (int i=0; i<wvap_level_pressures.length; i++)
        {
        if (wvap_level_pressures[i] == globals.getLevelLayerPressure())
          pressure_index = i;
        }
      }
    else if (globals.getLevelLayerDataType() == globals.WVAP_LAYER)
      {
      //end_of_file_name += "wvaplayer";
      end_of_file_name += "wvaplevel";
      
      for (int i=0; i<wvap_layer_pressures.length; i++)
        {
        if (wvap_layer_pressures[i] == globals.getLevelLayerPressure())
          pressure_index = i;
        }
      }

    end_of_file_name += pressure_index;
    
    if (stat_type == BIAS)
      end_of_file_name += "_bias.dat";
    else if (stat_type == STDDEV)
      end_of_file_name += "_stddev.dat";
    else if (stat_type == RMS)
      end_of_file_name += "_rms.dat";
    else if (stat_type == SAMPLE_SIZE)
      end_of_file_name += "_counts.dat";

    // Open the file for the system

    //String file_name = globals.getCurrentFileName();
    String file_name = "input.file";

    try
      {
      DataInputStream in = null;

      //if (globals.getCurrentFileType() == globals.USING_ZIP_FILE)
      //  {
        if (file_name != null)
          {
          File file = new File(file_name);
        
          if (file.exists())
            {
            try
              {
              ZipFile zf = new ZipFile(file_name);
              ZipEntry ze = zf.getEntry("narcsdir" + end_of_file_name);

              in = new DataInputStream(zf.getInputStream(ze));
              }
            catch (ZipException ze)
              {
              ze.printStackTrace();
              }
            }
          }
      //  }

      // If the DataInputStream is not null then loop through the pieces
      // and build a NarcsDataProfile for each visible one

      if (in != null)
        {
        
        // Create a buffer into which the values will be read

        int num_values = num_days;
        
        if (time_window == globals.WEEKLY)
          num_values = num_weeks;
        else if (time_window == globals.MONTHLY)
          num_values = num_months;

        byte[] buffer = new byte[num_values*4];

        // Skip to the start of the desired piece and read in the buffer for the piece
        
        int bytes_to_skip = piece * buffer.length;
        in.skipBytes(bytes_to_skip);

        in.readFully(buffer);

        // Read the data values from the buffer into the temporary array

        float[] temp_array = new float[num_values];
            
        DataInputStream data_in = new DataInputStream(new ByteArrayInputStream(buffer));

        for (int n=0; n<num_values; n++)
          {
          int value = data_in.readInt();
              
          if (value == -32768)
            {
            temp_array[n] = -32768F;
            }
          else
            {
            if (stat_type == SAMPLE_SIZE)
              temp_array[n] = (float)value;
            else
              temp_array[n] = (float)value / 1000F;
            }
          }

        data_in.close();

        // Extract the values within the date range from the temporary array
        // to data_array

        int num_values_in_date_range = (date_end_index - date_start_index) + 1;
        float[] data_array = new float[num_values_in_date_range];

        for (int n=date_start_index; n<=date_end_index; n++)
          data_array[n-date_start_index] = temp_array[n];

        // Create the new NarcsDataProfile
        
        String name = "system name";
        
        for (int n=0; n<profile_names.length; n++)
          {
          if ((system_id == profile_system_ids[n]) && (profile_type == profile_types[n]))
            {
            name = profile_names[n];
            }
          }

        profile = new NarcsDataProfile(data_array, Color.black, system_id, system_number, profile_type,
                                       stat_type, piece, 0, 0, name, sample_size_profile);
        in.close();
        }  // if (in != null...
      }
    catch (IOException ioe)
      {
      ioe.printStackTrace();
      }
    
    return profile;
    }



  private void removeUnwantedData()
    {
    for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
      {
      NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);
        
      if (profile.getStatType() == SAMPLE_SIZE)
        {
        profile = removeUnwantedSampleSizeData(profile);
        }
      else
        {
        NarcsDataProfile sample_size_profile = profile.getSampleSizeProfile();
        profile = removeUnwantedStatisticData(profile, sample_size_profile);
        }
        
      visible_profile_vector.setElementAt(profile, profnum);
      }
    }



  private NarcsDataProfile removeUnwantedStatisticData(NarcsDataProfile in_profile, 
                                                       NarcsDataProfile sample_size_profile)
    {
    NarcsDataProfile profile = in_profile;

    boolean ignore_extreme_values = globals.getLevelLayerStatsIgnoreExtremes();

    // Unpack the values and sample sizes

    float[] values = profile.getValues();
    float[] sample_sizes = sample_size_profile.getValues();

    // Calculate the sample size cutoff
      
    int sample_size_sum = 0;
    int sample_size_num = 0;
      
    for (int n=0; n<sample_sizes.length; n++)
      {
      if (sample_sizes[n] != -32768)
        {
        sample_size_sum += sample_sizes[n];
        sample_size_num++;
        }
      }

    float mean_sample_size = (float)sample_size_sum / (float)sample_size_num;
    float sample_size_cutoff = mean_sample_size * globals.getSampleSizeCutoff();


    // Loop through the values and remove unwanted data

    System.out.println("In NarcsDataProfile: Number values = "+values.length);
    for (int n=0; n<values.length; n++)
      {
    System.out.println("In NarcsDataProfile:  ["+n+"]= "+values[n]);
      
      // Remove any extreme values

      if (ignore_extreme_values)
        {
        if ((values[n] < min_stat) || (values[n] > max_stat))
          values[n] = -32768F;
        }

      // Remove any values when the sample size falls below the cutoff

      if (globals.getUseSampleSizeCutoff() == true)
        {
        if (sample_sizes[n] < sample_size_cutoff)
          {
          values[n] = -32768F;
          }
        }
      }  // for (n=0...
    
    profile.setValues(values);

    return profile;
    }



  private NarcsDataProfile removeUnwantedSampleSizeData(NarcsDataProfile sample_size_profile)
    {
    NarcsDataProfile profile = sample_size_profile;

    boolean ignore_extreme_values = globals.getLevelLayerCountsIgnoreExtremes();

    // Unpack the sample sizes

    float[] sample_sizes = sample_size_profile.getValues();

    // Loop through the values and remove unwanted data

    for (int n=0; n<sample_sizes.length; n++)
      {
      if (ignore_extreme_values)
        {
        if ((sample_sizes[n] < min_count) || (sample_sizes[n] > max_count))
          sample_sizes[n] = -32768F;
        }
      }  // for (n=0...
    
    profile.setValues(sample_sizes);

    return profile;
    }



  private void findSampleSizeMinMax()
    {
    if (globals.getLevelLayerCountsMinMaxOption() == 1)
      {
      int time_window = globals.getTimeWindow();
      
      if (time_window == globals.DAILY)
        {
        min_count = globals.getLevelLayerCountsDailyDefaultMin();
        max_count = globals.getLevelLayerCountsDailyDefaultMax();
        }
       else if (time_window == globals.WEEKLY)
        {
        min_count = globals.getLevelLayerCountsWeeklyDefaultMin();
        max_count = globals.getLevelLayerCountsWeeklyDefaultMax();
        }
      else
        {
        min_count = globals.getLevelLayerCountsMonthlyDefaultMin();
        max_count = globals.getLevelLayerCountsMonthlyDefaultMax();
        }
      }
    else if (globals.getLevelLayerCountsMinMaxOption() == 2)
      {
      min_count = globals.getLevelLayerCountsMin();
      max_count = globals.getLevelLayerCountsMax();
      }
    else
      {
      min_count =  9999999;
      max_count = -9999999;

      // Loop through the visible profiles
      
      for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
        {
        NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);

        if (profile.getStatType() == SAMPLE_SIZE)
          {
          float min = profile.getMinimumValue();
          float max = profile.getMaximumValue();

          if (min != -32768F)
            min_count = Math.min(min_count, (int)min);
          
          if (max != -32768F)
            max_count = Math.max(max_count, (int)max);
          }  // for (profnum...
      
        if (min_count == 9999999)
          min_count = 0;
      
        if (max_count == -9999999)
          max_count = 100;
        }
      }
    }



  private void findAutomaticStatisticMinMax()
    {
    if (globals.getLevelLayerStatsMinMaxOption() == 1)  // default
      {
      if ((globals.getLevelLayerDataType() == globals.TEMP_LEVEL) ||
          (globals.getLevelLayerDataType() == globals.TEMP_LAYER))
        {
        min_stat = globals.getLevelLayerTemperatureDefaultMin();
        max_stat = globals.getLevelLayerTemperatureDefaultMax();
        }
      else
        {
        min_stat = globals.getLevelLayerWaterVaporDefaultMin();
        max_stat = globals.getLevelLayerWaterVaporDefaultMax();
        }
      }
    else if (globals.getLevelLayerStatsMinMaxOption() == 2)   // manual
      {
      min_stat = globals.getLevelLayerStatsMin();
      max_stat = globals.getLevelLayerStatsMax();
      }
    else    // automatic
      {
      min_stat =  9999999F;
      max_stat = -9999999F;

      // Loop through the visible profiles
      
      for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
        {
        NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);

        if (profile.getStatType() != SAMPLE_SIZE)
          {
          float min = profile.getMinimumValue();
          float max = profile.getMaximumValue();

          if (min != -32768F)
            min_stat = Math.min(min_stat, min);
          
          if (max != -32768F)
            max_stat = Math.max(max_stat, max);
          }
        } // for (profnum=0...

      if ((min_stat == 9999999F) || (min_stat == -9999999F) ||
          (max_stat == 9999999F) || (max_stat == -9999999F))
        {
        min_stat = -5F;
        max_stat =  5F;
        }

      if ((min_stat == 0F) && (max_stat == 0F))
        {
        min_stat = -10F;
        max_stat =  10F;
        }
      }
    }



  private void findStatisticMinMax()
    {
    if (globals.getLevelLayerStatsMinMaxOption() == 1)  // default
      {
      if ((globals.getLevelLayerDataType() == globals.TEMP_LEVEL) ||
          (globals.getLevelLayerDataType() == globals.TEMP_LAYER))
        {
        min_stat = globals.getLevelLayerTemperatureDefaultMin();
        max_stat = globals.getLevelLayerTemperatureDefaultMax();
        }
      else
        {
        min_stat = globals.getLevelLayerWaterVaporDefaultMin();
        max_stat = globals.getLevelLayerWaterVaporDefaultMax();
        }
      }
    else if (globals.getLevelLayerStatsMinMaxOption() == 2)   // manual
      {
      min_stat = globals.getLevelLayerStatsMin();
      max_stat = globals.getLevelLayerStatsMax();
      }
    else    // automatic
      {
      min_stat =  9999999F;
      max_stat = -9999999F;

      // Loop through the visible profiles
      
      for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
        {
        NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);

        if (profile.getStatType() != SAMPLE_SIZE)
          {
          float min = profile.getMinimumValue();
          float max = profile.getMaximumValue();

          if (min != -32768F)
            min_stat = Math.min(min_stat, min);
          
          if (max != -32768F)
            max_stat = Math.max(max_stat, max);
          }
        } // for (profnum=0...

      if ((min_stat == 9999999F) || (min_stat == -9999999F) ||
          (max_stat == 9999999F) || (max_stat == -9999999F))
        {
        min_stat = -5F;
        max_stat =  5F;
        }

      if ((min_stat == 0F) && (max_stat == 0F))
        {
        min_stat = -10F;
        max_stat =  10F;
        }
      }
    }

  }

// end of file
