import java.awt.Color;
import java.io.*;
import java.io.PrintWriter;
import java.io.BufferedWriter;
import java.util.Vector;
import java.util.Arrays;
import java.util.zip.*;

/**
 * <PRE>
 * HighchartScriptGenerator.java
 *
 * Version: 1.0   Date: 10/12/2016   Programmer: Charles Brown (IMSG)
 * Version: 2.0   Date: 10/03/2024   Programmer: Mike Pettey (IMSG)
 *
 * this is a version of StatGraph.java developed by Mike which generates
 * text files instead of an image file.  These files are actually a  
 * JavaScript file with the data points, titles and axis info.  Then these are
 * read by the Highcharts function.
 *
 * </PRE>
 */

public class HighchartScriptGenerator
  {
  private String        grouping_name;
  private int           terrain, stat_type, data_type;
  private float         pressure;

  private boolean       is_longterm_plot;

  private String        baseline_name;
  private int           num_days, num_weeks, num_months, date_start_index, date_end_index;
  private String[]      date_strings, profile_names;
  private int[]         dates, profile_system_ids, profile_system_numbers, profile_types;
  private float[]       pressures;

  private Vector        visible_profile_vector;
  private int           min_count, max_count;
  private float         min_stat, max_stat;

      //private NarcsGlobals  globals2;
  private NarcsToHighchartsGlobals globals;

  private final int BIAS        = 0;
  private final int STDDEV      = 1;
  private final int RMS         = 2;
  private final int SAMPLE_SIZE = 3;

  private final int SYSTEM    = 0;
  private final int RETRIEVAL = 1;
  private final int GUESS     = 2;
  private final int FORECAST  = 3;
  private final int MICROWAVE = 4;
  private final int INFRARED  = 5;
  private final int RAW_DRY   = 6;

  private final int NUM_PIECES = 18;
  private PrintWriter pw;
  private PrintWriter out;
  private String file_name;


  public HighchartScriptGenerator(PrintWriter out, String grouping_name, float pressure, int terrain,
                                  int stat_type, int data_type, boolean is_longterm_plot, String baseline_name,
                                  int num_days, int num_weeks, int num_months,
                                  String[] profile_names, int[] profile_system_ids,
                                  int[] profile_system_numbers, int[] profile_types,
                                  int start_index, int end_index, String[] date_strings,
                                  int[] dates, float[] pressures)
    {
    this.out = out;

    this.grouping_name = grouping_name;

    this.pressure  = pressure;
    this.terrain   = terrain;
    this.stat_type = stat_type;
    this.data_type = data_type;

    this.is_longterm_plot = is_longterm_plot;

    this.pressures = pressures;

    this.baseline_name = baseline_name;
    this.num_days      = num_days;
    this.num_weeks     = num_weeks;
    this.num_months    = num_months;

    this.profile_names          = profile_names;
    this.profile_system_ids     = profile_system_ids;
    this.profile_system_numbers = profile_system_numbers;
    this.profile_types          = profile_types;

    date_start_index  = start_index;
    date_end_index    = end_index;
    this.date_strings = date_strings;
    this.dates        = dates;

    globals = new NarcsToHighchartsGlobals();

    //if (is_longterm_plot) file_name = "NarcsHighcharts_longterm.js";
    //else file_name = "NarcsHighcharts.js";

    visible_profile_vector = new Vector();
    }



  public void generateGraph()
    {
    visible_profile_vector.removeAllElements();

    // Build the profiles that are visible

    buildProfiles();

    // Find the mininum and maximum values

    findStatisticMinMax();

    // Remove any unwanted data from each profile

    removeUnwantedData();

    // Print beginning of JS file

  //  System.setOut(new PrintStream(new BufferedOutputStream(new FileOutputStream("test.js"))));
      //try
        //{

		//File file = new File("NarcsHighcharts.js",true);
		//FileOutputStream fos = new FileOutputStream(file);
		//PrintStream ps = new PrintStream(fos);
    //System.setOut(ps);
     //    PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter("NarcsHighcharts.js",true)));
    //out.println("$(function () {");
    //out.println("    $('#container').highcharts({");
    //out.println("        title: {");
    /*    }
        catch (FileNotFoundException e)
        {
            e.printStackTrace();
        }
        finally
        {
            if ( pw != null )
            {
                pw.close();
            }
        }
    */

    // Draw the graph beginning with the title

    drawTitle();

    // Draw the legend

    drawLegend();

    // Draw the y axis

    //drawYAxis();

    // Draw the x axis

    drawXAxis();

    out.println("        yAxis: {");

    out.println("          max: " + max_stat + ",");
    out.println("          min: " + min_stat + ",");

    out.println("             title: {");

    if (stat_type == globals.BIAS)
      out.println("                 text: 'BIAS'");
    else if (stat_type == globals.STDDEV)
      out.println("                 text: 'STDDEV'");
    else if (stat_type == globals.RMS)
      out.println("                 text: 'RMS'");

    out.println("             },");
    out.println("             plotLines: [{");
    out.println("                 value: 0,");
    out.println("                 width: 1,");
    out.println("                 color: '#808080'");
    out.println("             }]");
    out.println("        },");
    out.println("        tooltip: {");
    out.println("            valueSuffix: ''");
    out.println("        },");
    out.println("        legend: {");
    out.println("            layout: 'horizontal',");
    out.println("            align: 'center',");
    out.println("            verticalAlign: 'bottom',");
    out.println("            borderWidth: 0");
    out.println("        },");
    out.flush();

    // Draw the main graph

    drawMainGraph();

    // Write ending lines to Highchart JavaScript file

    out.println("\n        }]");
    out.println("    })");
    out.flush();

    // The ending bracket will be added via the run script

    //out.println("})");
    //out.flush();
    }



  private void drawTitle()
    {
    NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(0);
    //String prof_name = (String)(profile.getLabel());
    //int terrain_type = profile.getTerrainType();
    //boolean[] show_piece = globals.getShowCombination();
    //boolean landsea_option = show_piece[6];

    //String container_string = "";
    //String stat_type = profile.getLabel();
    //int sat_type_index = 0;
    String title = "";
    //String pressure_string = "" + pressure + " mb";


    String container_string = grouping_name;

    if (terrain == globals.SEA)
      container_string += "_Sea";
    else if (terrain == globals.LAND)
      container_string += "_Land";
    else if (terrain == globals.LAND_AND_SEA)
      container_string += "_LandAndSea";

    if (data_type == globals.TEMPERATURE)
      {
      container_string += "_Temperature";
      title = "Temperature " + (int)pressure + " hPa Level Statistics";
      }
    else if (data_type == globals.WATER_VAPOR)
      {
      container_string += "_WaterVapor";
      title = "Water Vapor Percent Error " + (int)pressure + " hPa Level Statistics";
      }

    if (stat_type == globals.BIAS)
      container_string += "_Bias";
    else if (stat_type == globals.STDDEV)
      container_string += "_StdDev";
    else if (stat_type == globals.RMS)
      container_string += "_RMS";

    container_string += "_" + (int)pressure + "hPa";

//    if (! is_longterm_plot)
//      {
//      String month = date_strings[0].substring(0,2);
//      if (month.equals(" 1")) month = "01";
//      if (month.equals(" 2")) month = "02";
//      if (month.equals(" 3")) month = "03";
//      if (month.equals(" 4")) month = "04";
//      if (month.equals(" 5")) month = "05";
//      if (month.equals(" 6")) month = "06";
//      if (month.equals(" 7")) month = "07";
//      if (month.equals(" 8")) month = "08";
//      if (month.equals(" 9")) month = "09";
//
//      // For year, the first two years, 2008 and 2009, are single numbers
//      // so have to test special for those two years.
//
//      String year_single_digit = date_strings[0].substring(6,7);
//      String year = " ";
//
//      if (year_single_digit.equals("8") || year_single_digit.equals("9"))
//        year = "200" + year_single_digit;
//      else
//        year = "20" + date_strings[0].substring(6,8);
//
//      container_string = container_string + "_" + year + month;


      //System.out.println("container_string = "+container_string);

      out.println("    $('#"+container_string+"').highcharts({");

is_longterm_plot = true;
      if (is_longterm_plot)
        {
        out.println("        chart: {");
        out.println("           zoomType: 'x'");
        out.println("        },");
        }

      out.println("        title: {");
      out.println("            text: 'NOAA/NESDIS/STAR NPROVS Collocation Summary Statistics (NARCS)',");
      //out.println("            x: -20 //center");
      out.println("        },");
      String subtitle = "            text: '"+title+", ";

      //if (terrain == globals.SEA)
      //  subtitle += "Sea, ";
      //else if (terrain == globals.LAND)
      //  subtitle += "Land, ";
      //else if (terrain == globals.LAND_AND_SEA)
      //  subtitle += "Land and Sea, ";

      ////if (landsea_option)
      ////  subtitle = subtitle + "Land, ";
      ////else
      ////  subtitle = subtitle + "Sea, ";

      //if (data_type == globals.TEMPERATURE)
      //  subtitle += "Temperature";
      //else if (data_type == globals.WATER_VAPOR)
      //  subtitle += "Water Vapor Percent Error";

      //subtitle += " " + (int)pressure + " hPa Statistics, ";

      if (terrain == globals.SEA)
        subtitle += "Sea,";
      else if (terrain == globals.LAND)
        subtitle += "Land,";
      else if (terrain == globals.LAND_AND_SEA)
        subtitle += "Land and Sea,";

      if (stat_type == globals.BIAS)
        subtitle += " Bias\'";
      else if (stat_type == globals.STDDEV)
        subtitle += " Std Dev\'";
      else if (stat_type == globals.RMS)
        subtitle += " RMS\'";


      out.println("        subtitle: {");
      out.println(subtitle + ",");

      //System.out.println(subtitle + ",");

      //if ((sat_type_index = prof_name.indexOf("IR")) > 0)
      //  out.println(subtitle + "Infrared plus Microwave'," );
      //else
      //  out.println(subtitle + "Microwave',");

      out.println("            style: {");
      out.println("            color: '#FF0000',");
      out.println("            fontWeight: 'bold'");
      out.println("            },");
      out.println("            x: -20 //center");
      out.println("        },");
      out.flush();
      //out.close();
    //  }
    //catch (IOException ioe)
    //  {
      //System.out.println("IOEception found for file name = "+file_name);
    //  ioe.printStackTrace();
    //  }
    //finally
    //  {
      //if (out != null)
      //  {
      //  out.close();
      //  }
    //  }
    }



  private void drawXAxis()
    {
    out.println("        xAxis: {");
    out.print("            categories: [");

    for (int i=0; i<date_strings.length; i++)
      {
      out.print("'" + date_strings[i]+"'");

      if (i == date_strings.length-1)
        out.print("],");

      if ((i != date_strings.length) && (i != date_strings.length-1))
        out.print(", ");

      if (i % 6 == 0)
        out.println(" ");
      }

    out.println("            labels: {");
    out.println("                rotation: 90");
    out.println("            },");
    out.println("            minorGridLineDashStyle: 'longdash',");
    out.println("            minorTickInterval: 'auto',");
    out.println("            minorTickWidth: 0,");
    out.println("            plotLines: [{");
    out.println("                dashStyle: 'longdash',");
    out.println("                value: 0,");
    out.println("                width: 1,");
    out.println("                color: '#808080'");
    out.println("            }]");
    out.println("        },");
    out.flush();
    }



  private void drawMainGraph()
    {
    int number_of_dates = date_strings.length;

    // Draw each of the visible profiles

    out.println("        series: [{");

    for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
      {
      int sat_type_index = 0;
      NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);

      String prof_name = (String)(profile.getLabel());
      //sat_type_index = prof_name.indexOf(" MW");

      //if ((sat_type_index = prof_name.indexOf("IR+MW")) > 0 || (sat_type_index = prof_name.indexOf(" MW")) > 0)
      //  System.out.println("Profile name: "+prof_name+" sat_type_index="+sat_type_index);

      //if ((sat_type_index = prof_name.indexOf("IR+MW")) > 0 )
      //  {
      //  out.println("            name: '"+prof_name.substring(0,sat_type_index-1)+"'," );
      //  }
      //else
      //  {
      //  sat_type_index = prof_name.indexOf(" MW");
      //  if (sat_type_index > 0)
      //    out.println("            name: '"+prof_name.substring(0,sat_type_index)+"'," );
      //  else
      //    out.println("            name: '"+prof_name+"'," );
      //  }

      out.println("            name: '"+prof_name+"'," );

      out.println("data: [");
      float[] values = profile.getValues();

      for (int ival=0; ival<values.length;ival++)
        {
//        System.out.println("Data Value["+ival+"]="+values[ival]);

        if (ival != values.length-1)
          {
          if (values[ival] != -32768)
            out.print(values[ival]+", ");
          else
            out.print(null+", ");
          }
        else if (values[ival] != -32768)
          {
          out.print(values[ival]+"] ");
          }
        else
          {
          out.print(null+"] ");
          }
        }

      if (profnum != visible_profile_vector.size()-1)
        out.println("}, { ");

      //boolean previous_data_found = false;
      }  // for (profnum=0...

    out.flush();
    }



  public void drawLegend()
    {

    // Calculate the number of profiles being plotted and save
    // the unique profile names. Also determine the number of
    // statistic types being displayed.

    Vector code_vector = new Vector();
    Vector profile_name_vector = new Vector();
    Vector unique_profile_vector = new Vector();

    for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
      {
      NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);

      int system_id    = profile.getSystemID();
      int profile_type = profile.getProfileType();

      int code = (system_id * 100) + profile_type;
      Integer icode = new Integer(code);

      if (! code_vector.contains(icode))
        {
        code_vector.addElement(icode);
        profile_name_vector.addElement(profile.getLabel());
        unique_profile_vector.addElement(profile);
        }
      }  // for (profnum=0...

    int num_stats_used = 1;

    int num_profiles = code_vector.size();
    int total_lines = num_profiles + num_stats_used + 1;

    // Draw labels for each unique profile

    int sat_type_index = 0;

    String prof_name = (String)(profile_name_vector.elementAt(0));
    boolean[] show_piece = globals.getShowCombination();
    boolean landsea_option = show_piece[6];

    out.print("        colors: ['#");

    for (int prof=0; prof<code_vector.size(); prof++)
      {
      int code = ((Integer)(code_vector.elementAt(prof))).intValue();
      Color color = globals.getProfileColor(code);
      out.print(String.format("%02x", color.getRed())+
      String.format("%02x", color.getGreen())+
      String.format("%02x", color.getBlue()));
      if (prof != (code_vector.size()-1))
        out.print("', '#");
      }

    out.println("'],");
//       out.println("        colors: ['#0088ff', '#ff8800', '#90ed7d', '#f7a35c', '#8085e9',");
//       out.println("        '#f15c80', '#e4d354', '#2b908f', '#f45b5b', '#91e8e1'],");
    out.flush();
    }



  private void buildProfiles()
    {
    Vector visible_profiles = globals.getVisibleProfiles();

    // Loop through each system and build retrieval profiles for those that
    // are selected

    for (int sys=0; sys<visible_profiles.size(); sys++)
      {

      // Determine the system number (0 to N)

      int system_val  = ((Integer)(visible_profiles.elementAt(sys))).intValue();
      int system_id   = system_val / 100;
      int system_type = system_val % 100;

      int system_number = -32768;

      for (int n=0; n<profile_system_ids.length; n++)
        {
        if (system_id == profile_system_ids[n])
          system_number = profile_system_numbers[n];
        }

      // If the system number was found, create profiles for the visible profiles

      if (system_number != -32768)
        {
        boolean[] show_piece = globals.getShowCombination();

        for (int piece=0; piece<NUM_PIECES; piece++)
          {
          if (show_piece[piece] == true)
            {
            NarcsDataProfile sample_size_profile = null;
            NarcsDataProfile statistic_profile   = null;

            // Create the profiles for each type of statistic

            sample_size_profile = buildProfile(system_id, system_number, system_type, globals.SAMPLE_SIZE, piece, null);

            if (stat_type == globals.BIAS)
              {
              statistic_profile = buildProfile(system_id, system_number, system_type, globals.BIAS, piece,
                                               sample_size_profile);
              }

            if (stat_type == globals.STDDEV)
              {
              statistic_profile = buildProfile(system_id, system_number, system_type, globals.STDDEV, piece,
                                               sample_size_profile);
              }

            if (stat_type == globals.RMS)
              {
              statistic_profile = buildProfile(system_id, system_number, system_type, globals.RMS, piece,
                                               sample_size_profile);
              }

            // Go through the data and remove any data that may be out of range or
            // below the sample size cutoff. Then add the profile to the visible profile
            // vector

            statistic_profile = removeUnwantedStatisticData(statistic_profile, sample_size_profile);
            visible_profile_vector.add(statistic_profile);
            }  // if (show_piece[piece]...
          }  // for (piece=0...
        }  // if (system_number...
      }  // for (sys=0...
    }



  private NarcsDataProfile buildProfile(int system_id, int system_number, int profile_type, 
                                        int stat_type, int piece, NarcsDataProfile sample_size_profile)
    {
    NarcsDataProfile profile = null;

    int time_window = globals.getTimeWindow();

    // Build the file name

    String end_of_file_name = "/";

    if (time_window == globals.DAILY)
      end_of_file_name += "daily";
    else if (time_window == globals.WEEKLY)
      end_of_file_name += "weekly";
    else
      end_of_file_name += "monthly";

    end_of_file_name += "_system" + system_number;

    if (profile_type < 10)
      end_of_file_name += "_profile0" + profile_type + "_";
    else
      end_of_file_name += "_profile" + profile_type + "_";

    int pressure_index = 0;

    if (data_type == globals.TEMPERATURE)
      {
      end_of_file_name += "templevel";

      for (int i=0; i<pressures.length; i++)
        {
        if (pressures[i] == pressure)
          pressure_index = i;
        }
      }
    else if (data_type == globals.WATER_VAPOR)
      {
      end_of_file_name += "wvaplevel";

      for (int i=0; i<pressures.length; i++)
        {
        if (pressures[i] == pressure)
          pressure_index = i;
        }
      }

    end_of_file_name += pressure_index;

    if (stat_type == globals.BIAS)
      end_of_file_name += "_bias.dat";
    else if (stat_type == globals.STDDEV)
      end_of_file_name += "_stddev.dat";
    else if (stat_type == globals.RMS)
      end_of_file_name += "_rms.dat";
    else if (stat_type == globals.SAMPLE_SIZE)
      end_of_file_name += "_counts.dat";

    // Open the file for the system

    //String file_name = globals.getCurrentFileName();
    String file_name = "input.file";

    try
      {
      DataInputStream in = null;

      if (file_name != null)
        {
        File file = new File(file_name);

        if (file.exists())
          {
          try
            {
            ZipFile zf = new ZipFile(file_name);
            ZipEntry ze = zf.getEntry("narcsdir" + end_of_file_name);

            in = new DataInputStream(zf.getInputStream(ze));
            }
          catch (ZipException ze)
            {
            ze.printStackTrace();
            }
          }
        }

      // If the DataInputStream is not null then loop through the pieces
      // and build a NarcsDataProfile for each visible one

      if (in != null)
        {

        // Create a buffer into which the values will be read

        int num_values = num_days;

        if (globals.getTimeWindow() == globals.WEEKLY)
          num_values = num_weeks;
        else if (globals.getTimeWindow() == globals.MONTHLY)
          num_values = num_months;

        byte[] buffer = new byte[num_values*4];

        // Skip to the start of the desired piece and read in the buffer for the piece

        int bytes_to_skip = piece * buffer.length;
        in.skipBytes(bytes_to_skip);

        in.readFully(buffer);

        // Read the data values from the buffer into the temporary array

        float[] temp_array = new float[num_values];

        DataInputStream data_in = new DataInputStream(new ByteArrayInputStream(buffer));

        for (int n=0; n<num_values; n++)
          {
          int value = data_in.readInt();

          if (value == -32768)
            {
            temp_array[n] = -32768F;
            }
          else
            {
            if (stat_type == SAMPLE_SIZE)
              temp_array[n] = (float)value;
            else
              temp_array[n] = (float)value / 1000F;
            }
          }

        data_in.close();

        // Extract the values within the date range from the temporary array
        // to data_array

        int num_values_in_date_range = (date_end_index - date_start_index) + 1;
        float[] data_array = new float[num_values_in_date_range];

        for (int n=date_start_index; n<=date_end_index; n++)
          data_array[n-date_start_index] = temp_array[n];

        // Create the new NarcsDataProfile

        String name = "system name";

        for (int n=0; n<profile_names.length; n++)
          {
          if ((system_id == profile_system_ids[n]) && (profile_type == profile_types[n]))
            {
            name = profile_names[n];
            }
          }

        profile = new NarcsDataProfile(data_array, Color.black, system_id, system_number, profile_type,
                                       stat_type, piece, 0, 0, name, sample_size_profile);
        in.close();
        }  // if (in != null...
      }
    catch (IOException ioe)
      {
      ioe.printStackTrace();
      }
    
    return profile;
    }



  private void removeUnwantedData()
    {
    for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
      {
      NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);

      NarcsDataProfile sample_size_profile = profile.getSampleSizeProfile();
      profile = removeUnwantedStatisticData(profile, sample_size_profile);

      visible_profile_vector.setElementAt(profile, profnum);
      }
    }



  private NarcsDataProfile removeUnwantedStatisticData(NarcsDataProfile in_profile, 
                                                       NarcsDataProfile sample_size_profile)
    {
    NarcsDataProfile profile = in_profile;

    boolean ignore_extreme_values = globals.getIgnoreExtremeValues();

    // Unpack the values and sample sizes

    float[] values = profile.getValues();
    float[] sample_sizes = sample_size_profile.getValues();

    // Calculate the sample size cutoff

    int sample_size_sum = 0;
    int sample_size_num = 0;
      
    for (int n=0; n<sample_sizes.length; n++)
      {
      if (sample_sizes[n] != -32768)
        {
        sample_size_sum += sample_sizes[n];
        sample_size_num++;
        }
      }

    float mean_sample_size = (float)sample_size_sum / (float)sample_size_num;
    float sample_size_cutoff = mean_sample_size * globals.getSampleSizeCutoff();

    // Loop through the values and remove unwanted data

    for (int n=0; n<values.length; n++)
      {

      // Remove any extreme values

      if (ignore_extreme_values)
        {
        if ((values[n] < min_stat) || (values[n] > max_stat))
          values[n] = -32768F;
        }

      // Remove any values when the sample size falls below the cutoff

      if (globals.getUseSampleSizeCutoff() == true)
        {
        if (sample_sizes[n] < sample_size_cutoff)
          {
          values[n] = -32768F;
          }
        }
      }  // for (n=0...

    profile.setValues(values);

    return profile;
    }



  private void findStatisticMinMax()
    {
    min_stat = -5F;
    max_stat =  5F;

    if (globals.getMinMaxOption() == globals.DEFAULT)  // default
      {
      min_stat = globals.getStatDefaultMinimum(stat_type, data_type);
      max_stat = globals.getStatDefaultMaximum(stat_type, data_type);
      }
    else if (globals.getMinMaxOption() == globals.MANUAL)   // manual
      {
      min_stat = globals.getStatManualMinimum(stat_type, data_type);
      max_stat = globals.getStatManualMaximum(stat_type, data_type);
      }
    else    // automatic
      {
      min_stat =  9999999F;
      max_stat = -9999999F;

      // Loop through the visible profiles

      for (int profnum=0; profnum<visible_profile_vector.size(); profnum++)
        {
        NarcsDataProfile profile = (NarcsDataProfile)visible_profile_vector.elementAt(profnum);

        if (profile.getStatType() != SAMPLE_SIZE)
          {
          float min = profile.getMinimumValue();
          float max = profile.getMaximumValue();

          if (min != -32768F)
            min_stat = Math.min(min_stat, min);

          if (max != -32768F)
            max_stat = Math.max(max_stat, max);
          }
        } // for (profnum=0...

      if ((min_stat == 9999999F) || (min_stat == -9999999F) ||
          (max_stat == 9999999F) || (max_stat == -9999999F))
        {
        min_stat = -5F;
        max_stat =  5F;
        }

      if ((min_stat == 0F) && (max_stat == 0F))
        {
        min_stat = -10F;
        max_stat =  10F;
        }
      }
    }

  }

// end of file
