import java.awt.*;
import java.io.*;
import java.util.zip.*;
import java.util.Arrays;
import java.util.Vector;
import java.util.Calendar;

/**
 * <PRE>
 * NarcsToHighcharts.java
 *
 * Version: 1.0   Date: 10/03/2024   Programmer: Mike Pettey (IMSG)
 *
 * This class is the main class for the NarcsToHighcharts program. This
 * program creates an Edge image and saves it to a local file.
 * The name of a parameter file is supplied as a command line
 * argument. This file is read to get information about what data
 * to display and how to display it.
 * 
 * </PRE>
 */

public class NarcsToHighcharts
  {
  private String       version = "1.0";
  private String       version_date = "October 3, 2024";

  //private boolean  verbose;

  //private NarcsGlobals globals2;
  private NarcsToHighchartsGlobals globals;

  private String       input_file_name, output_file_name;

  private int          date_range_start, date_range_end;
  private int          file_start_date, file_end_date;

  private int          global_start_date, global_end_date;

  private Vector       grouping_vector;
  private Vector       pressure_vector;

  private String       baseline_name;
  private int          num_days, num_weeks, num_months;
  private int          num_temp_levels, num_temp_layers, num_wvap_levels, num_wvap_layers;
  private int[]        days, week_starts, week_ends, months;
  private float[]      temp_level_pressures, temp_layer_pressures;
  private float[]      wvap_level_pressures, wvap_layer_pressures;
  private String[]     day_strings, week_strings, month_strings;

  private int          num_systems_in_file, num_profiles_in_file;
  private int[]        system_ids_in_file, profile_system_ids_in_file;
  private int[]        profile_system_numbers_in_file, profile_profile_types_in_file;
  private String[]     system_names_in_file, profile_system_names_in_file,
                       profile_profile_names_in_file;

  private boolean      show_baseline_sea, show_baseline_nonsea, show_baseline_both;
  private boolean      show_system_sea, show_system_nonsea, show_system_both;
  private boolean      show_pass, show_fail;
  
  private boolean      show_samplesize, show_bias, show_stddev, show_rms;

  private boolean      use_sample_size_cutoff;
  private boolean      is_longterm_plot;           // which means zoomable in Highcharts
  private float        sample_size_cutoff;

  private boolean      show_data_points;

  private int          min_max_option, counts_min, counts_max;
  private float        data_min, data_max;
  private boolean      ignore_extreme_values;

  private int          time_option, start_date, end_date;
  private int          data_type;
  private float        data_pressure;

  private final int    NUM_PIECES = 18;

  private final int    TEMPERATURE       = 0;
  private final int    WATER_VAPOR       = 1;
  private final int    RELATIVE_HUMIDITY = 2;

  private final int    SAMPLE_SIZE       = 0;
  private final int    TEMPERATURE_LEVEL = 1;
  private final int    TEMPERATURE_LAYER = 2;
  private final int    WATER_VAPOR_LEVEL = 3;
  private final int    WATER_VAPOR_LAYER = 4;



  public static void main(String args[]) 
    {
    NarcsToHighcharts nthc = new NarcsToHighcharts(args);
    System.exit(0);
    }



  public NarcsToHighcharts(String[] args)
    {
    System.out.println("====================================================================");
    System.out.print("Starting NarcsToHighcharts version " + version + " at ");
    printTime();
    System.out.println("");

    globals = new NarcsToHighchartsGlobals();
    globals.readDefaultColors();

    //input_file_name  = args[0];
    //output_file_name = args[1];
    input_file_name = "input.file";
    output_file_name = "output.file";

    System.out.println("In:   " + input_file_name);
    System.out.println("Out:  " + output_file_name);

    if (args.length > 1)
      readGlobalDateRange(args);

    is_longterm_plot = false;
    if (args.length == 3) is_longterm_plot = true;

    // Parse the input XML file to obtain the information
    // about what image to create

    //parseSettingsFile();
//System.exit(0);

    try
      {
      XMLParser parser = new XMLParser();
      //XMLTree xml_tree = parser.parseFile("parameter.file");
      XMLTree xml_tree = parser.parseFile("settings.file");
      //xml_tree.printTree();

      if (xml_tree != null)
        {
        processGroupingSettings(xml_tree);
        processPressureSettings(xml_tree);
        processGraphSettings(xml_tree);

        // Read the file header

        readNarcsFileHeader();

        // Open the output file for writing

        try
          {
          PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(output_file_name,true)));

          // Create the header of the output file

          out.println("$(function () {");

          // Process each grouping, pressure, stat type and terrain

          for (int group=0; group<grouping_vector.size(); group++)
            {
            Grouping grouping = (Grouping)grouping_vector.elementAt(group);

            // Pressure loop

            for (int level=0; level<pressure_vector.size(); level++)
              {
              float pressure = ((Float)pressure_vector.elementAt(level)).floatValue();

              // Terrain loop

              for (int terrain=0; terrain<3; terrain++)
                {

                // Stat type loop

                for (int stat=0; stat<3; stat++)
                  {
                  processStats(out, grouping, pressure, terrain, stat, TEMPERATURE);
                  processStats(out, grouping, pressure, terrain, stat, WATER_VAPOR);
                  }  // for (stat=0...
                }  // for (terrain=0...
              }  // for (level=0...
            }  // for (group=0...


          out.println("})");

          // Close the output file

          out.flush();
          out.close();
          }
        catch (IOException ioe)
          {
          ioe.printStackTrace();
          }
        }
      else
        {
        System.out.println("\nThe input parameter file could not be parsed.");
        System.out.println("The structure of the XML data is probably not correct.");
        System.out.println("Please check the file and try again.\n");
        System.exit(1);
        }
      }
    catch (XMLParserException xpe)
      {
      System.out.println("\nThe input parameter file could not be parsed.");
      System.out.println("The structure of the XML data is probably not correct.");
      System.out.println("Please check the file and try again.\n");
      System.exit(1);
      }
    }



  private void processGroupingSettings(XMLTree xml_tree)
    {
    grouping_vector = new Vector();

    // Pull the grouping settings from the XML tree

    Object object = xml_tree.getValue("platform_groupings");

    if ((object != null) && (object instanceof Vector))
      {
      Vector vector = (Vector)object;

      for (int n=0; n<vector.size(); n++)
        {
        object = vector.elementAt(n);

        if (object instanceof XMLTree)
          {
          XMLTree tree = (XMLTree)object;

          // Create a new Grouping object

          Grouping grouping = new Grouping();

          // Unpack the grouping name

          object = tree.getValue("grouping_name");

          if ((object != null) && (object instanceof String))
            {
            String name = (String)object;
            grouping.setName(name);
            }

          // Unpack the profiles in the grouping

          object = tree.getValue("profiles");

          if ((object != null) && (object instanceof Vector))
            {
            Vector profile_vector = (Vector)object;

            for (int v=0; v<profile_vector.size(); v++)
              {
              ProfileSettings profile_settings = new ProfileSettings();

              object = profile_vector.elementAt(v);

              if ((object != null) && (object instanceof XMLTree))
                {
                XMLTree prof_tree = (XMLTree)object;

                // Profile platform name

                object = prof_tree.getValue("platform_name");

                if ((object != null) && (object instanceof String))
                  {
                  String name = (String)object;
                  profile_settings.setPlatformName(name);
                  }

                // Profile platform ID

                object = prof_tree.getValue("platform_id");

                if ((object != null) && (object instanceof String))
                  {
                  String id = (String)object;
                  profile_settings.setPlatformID(Integer.parseInt(id));
                  }

                // Profile name

                object = prof_tree.getValue("profile_name");

                if ((object != null) && (object instanceof String))
                  {
                  String name = (String)object;
                  profile_settings.setProfileName(name);
                  }

                // Profile ID

                object = prof_tree.getValue("profile_id");

                if ((object != null) && (object instanceof String))
                  {
                  String id = (String)object;
                  profile_settings.setProfileID(Integer.parseInt(id));
                  }

                // Line color red

                object = prof_tree.getValue("line_color_red");

                if ((object != null) && (object instanceof String))
                  {
                  String color = (String)object;
                  profile_settings.setLineColorRed(Integer.parseInt(color));
                  }

                // Line color green

                object = prof_tree.getValue("line_color_green");

                if ((object != null) && (object instanceof String))
                  {
                  String color = (String)object;
                  profile_settings.setLineColorGreen(Integer.parseInt(color));
                  }

                // Line color blue

                object = prof_tree.getValue("line_color_blue");

                if ((object != null) && (object instanceof String))
                  {
                  String color = (String)object;
                  profile_settings.setLineColorBlue(Integer.parseInt(color));
                  }
                }  // if (object != null...

              grouping.addProfile(profile_settings);
              }  // for (int v=0...
            }  // if (object != null...

          grouping_vector.add(grouping);
          }  // if (object instanceof XMLTree...
        }  // for (int n=0...
      }  // if (object != null...
    }



  private void processPressureSettings(XMLTree xml_tree)
    {
    pressure_vector = new Vector();

    // Pull the terrain settings from the XML tree

    Object object = xml_tree.getValue("pressures");

    if ((object != null) && (object instanceof Vector))
      {
      Vector vector = (Vector)object;

      // Loop through each pressure

      for (int level=0; level<vector.size(); level++)
        {
        object = vector.elementAt(level);

        if ((object != null) && (object instanceof XMLTree))
          {
          XMLTree pressure_tree = (XMLTree)object;

          object = pressure_tree.getValue("pressure");

          if ((object != null) && (object instanceof String))
            {
            String string = (String)object;
            float pressure = Float.parseFloat(string);
            pressure_vector.add(new Float(pressure));
            }
          }  //if (object != null...
        }  // for (level=0...
      }  // if (object != null...

    // If no pressures were successfully read in, then add some default values
    // to the pressure vector

    if (pressure_vector.size() == 0)
      {
      pressure_vector.add(new Float(10.2));
      pressure_vector.add(new Float(51.5));
      pressure_vector.add(new Float(113.9));
      pressure_vector.add(new Float(206.4));
      pressure_vector.add(new Float(293.1));
      pressure_vector.add(new Float(506.1));
      pressure_vector.add(new Float(729.6));
      pressure_vector.add(new Float(827.1));
      }
    }



  private void processGraphSettings(XMLTree xml_tree)
    {

    // Pull the graph settings from the XML tree

    Object object = xml_tree.getValue("graph_settings");

    if ((object != null) && (object instanceof Vector))
      {
      Vector vector = (Vector)object;

      // Time window option

      globals.setTimeWindow(globals.MONTHLY);

      object = xml_tree.getValue("time_window");

      if ((object != null) && (object instanceof String))
        {
        String string = (String)object;
        string = string.toLowerCase();

        if (string.equals("daily"))
          globals.setTimeWindow(globals.DAILY);
        else if (string.equals("weekly"))
          globals.setTimeWindow(globals.WEEKLY);
        }

      // Min/max options

      globals.setMinMaxOption(globals.AUTOMATIC);

      object = xml_tree.getValue("min_max_option");

      if ((object != null) && (object instanceof String))
        {
        String string = (String)object;
        string = string.toLowerCase();

        if (string.equals("default"))
          globals.setMinMaxOption(globals.DEFAULT);
        else if (string.equals("manual"))
          globals.setMinMaxOption(globals.MANUAL);
        }

      // Temperature bias min and max

      object = xml_tree.getValue("temp_bias_stats_min");

      if ((object != null) && (object instanceof String))
        {
        globals.setTemperatureBiasStatsMin(Float.parseFloat((String)object));
        String string = (String)object;
        }

      object = xml_tree.getValue("temp_bias_stats_max");

      if ((object != null) && (object instanceof String))
        {
        globals.setTemperatureBiasStatsMax(Float.parseFloat((String)object));
        String string = (String)object;
        }

      // Temperature stddev min and max

      object = xml_tree.getValue("temp_stddev_stats_min");

      if ((object != null) && (object instanceof String))
        {
        globals.setTemperatureStddevStatsMin(Float.parseFloat((String)object));
        String string = (String)object;
        }

      object = xml_tree.getValue("temp_stddev_stats_max");

      if ((object != null) && (object instanceof String))
        {
        globals.setTemperatureStddevStatsMax(Float.parseFloat((String)object));
        String string = (String)object;
        }

      // Temperature RMS min and max

      object = xml_tree.getValue("temp_rms_stats_min");

      if ((object != null) && (object instanceof String))
        {
        globals.setTemperatureRMSStatsMin(Float.parseFloat((String)object));
        String string = (String)object;
        }

      object = xml_tree.getValue("temp_rms_stats_max");

      if ((object != null) && (object instanceof String))
        {
        globals.setTemperatureRMSStatsMax(Float.parseFloat((String)object));
        String string = (String)object;
        }

      // Water vapor bias min and max

      object = xml_tree.getValue("wvap_bias_stats_min");

      if ((object != null) && (object instanceof String))
        {
        globals.setWaterVaporBiasStatsMin(Float.parseFloat((String)object));
        String string = (String)object;
        }

      object = xml_tree.getValue("wvaop_bias_stats_max");

      if ((object != null) && (object instanceof String))
        {
        globals.setWaterVaporBiasStatsMax(Float.parseFloat((String)object));
        String string = (String)object;
        }

      // Water vapor stddev min and max

      object = xml_tree.getValue("wvap_stddev_stats_min");

      if ((object != null) && (object instanceof String))
        {
        globals.setWaterVaporStddevStatsMin(Float.parseFloat((String)object));
        String string = (String)object;
        }

      object = xml_tree.getValue("wvap_stddev_stats_max");

      if ((object != null) && (object instanceof String))
        {
        globals.setWaterVaporStddevStatsMax(Float.parseFloat((String)object));
        String string = (String)object;
        }

      // Water vapor RMS min and max

      object = xml_tree.getValue("wvap_rms_stats_min");

      if ((object != null) && (object instanceof String))
        {
        globals.setWaterVaporRMSStatsMin(Float.parseFloat((String)object));
        String string = (String)object;
        }

      object = xml_tree.getValue("wvap_rms_stats_max");

      if ((object != null) && (object instanceof String))
        {
        globals.setWaterVaporRMSStatsMax(Float.parseFloat((String)object));
        String string = (String)object;
        }

      // Ignore extreme values option

      globals.setIgnoreExtremeValues(false);

      object = xml_tree.getValue("ignore_extreme_values");

      if ((object != null) && (object instanceof String))
        {
        String string = (String)object;
        string = string.toLowerCase();

        if (string.equals("yes"))
          globals.setIgnoreExtremeValues(true);
        }

      // Sample size cutoff options

      globals.setUseSampleSizeCutoff(false);

      object = xml_tree.getValue("use_sample_size_cutoff");

      if ((object != null) && (object instanceof String))
        {
        String string = (String)object;
        string = string.toLowerCase();

        if (string.equals("yes"))
          globals.setUseSampleSizeCutoff(true);
        }

      object = xml_tree.getValue("sample_size_cutoff");

      if ((object != null) && (object instanceof String))
        {
        globals.setSampleSizeCutoff(Float.parseFloat((String)object));
        String string = (String)object;
        }

      }  // if (object != null...
    }



  private void readGlobalDateRange(String[] args)
    {
    date_range_start = -32768;
    date_range_end   = -32768;

    try
      {
      date_range_start = Integer.parseInt(args[0]);
      date_range_end   = Integer.parseInt(args[1]);
      }
    catch (NumberFormatException nfe) {}
    }



  private void processStats(PrintWriter out, Grouping grouping, float pressure, int terrain, int stat, int data_type)
    {
    System.out.println("-------------------------------------------");
    System.out.println("Processing Stats...");
    System.out.println("   Grouping:  " + grouping.getName());
    System.out.println("   Pressure:  " + pressure);
    System.out.println("   Terrain:   " + terrain);
    System.out.println("   Stat Type: " + stat);
    System.out.println("   Data Type: " + data_type);
    System.out.println("");

    // Read the baseline information

    readBaselineInformation();

    // Replace any system and profile information read from the file with
    // values read from the XML

    replaceSystemAndProfileData(grouping);

    // Obtain the current date indices and date strings

    int[] date_indices = findDateRangeIndices();
    int start_index = date_indices[0];
    int end_index   = date_indices[1];

    String[] date_strings = getDateStrings(start_index, end_index);
    int[] dates = getDates(start_index, end_index);

    // Obtain the profile names

    String[] profile_names = new String[profile_system_names_in_file.length];

    for (int n=0; n<profile_names.length; n++)
      {
      profile_names[n] = profile_system_names_in_file[n].trim() + " " +
                         profile_profile_names_in_file[n].trim();
      }

    // Set the pressures based on the type of data

    float[] pressures = new float[temp_level_pressures.length];
    Arrays.fill(pressures, -32768F);

    if (data_type == TEMPERATURE)
      {
      for (int level=0; level<temp_level_pressures.length; level++)
        pressures[level] = temp_level_pressures[level];
      }
    else
      {
      pressures = new float[wvap_level_pressures.length];

      for (int level=0; level<wvap_level_pressures.length; level++)
        pressures[level] = wvap_level_pressures[level];
      }

    // Determine what data to display

    updateDataToDisplay(grouping, terrain, stat);

    // Create the graph

    HighchartScriptGenerator hs_gen = new HighchartScriptGenerator(out, grouping.getName(), pressure, terrain,
                                                                   stat, data_type, is_longterm_plot, baseline_name,
                                                                   num_days, num_weeks, num_months,
                                                                   profile_names, profile_system_ids_in_file,
                                                                   profile_system_numbers_in_file,
                                                                   profile_profile_types_in_file,
                                                                   start_index, end_index, date_strings, dates,
                                                                   pressures);

    hs_gen.generateGraph();
    }



  private int[] findDateRangeIndices()
    {
    int[] indices = new int[2];
    indices[0] = -32768;
    indices[1] = -32768;

    int start_date = globals.getDateRangeStart();
    int end_date   = globals.getDateRangeEnd();

    if (globals.getTimeWindow() == globals.DAILY)
      {
      for (int n=0; n<days.length; n++)
        {
        if (start_date == days[n])
          indices[0] = n;

        if (end_date == days[n])
          indices[1] = n;
        }
      }

    else if (globals.getTimeWindow() == globals.WEEKLY)
      {
      for (int n=0; n<week_starts.length; n++)
        {
        if ((start_date >= week_starts[n]) && (start_date <= week_ends[n]))
          indices[0] = n;

        if ((end_date >= week_starts[n]) && (end_date <= week_ends[n]))
          indices[1] = n;
        }
      }

    else if (globals.getTimeWindow() == globals.MONTHLY)
      {
      int start_year  = start_date / 10000;
      int start_month = (start_date % 10000) / 100;
      int end_year    = end_date / 10000;
      int end_month   = (end_date % 10000) / 100;

      for (int n=0; n<months.length; n++)
        {
        int year  = months[n] / 100;
        int month = months[n] % 100;

        if ((start_year == year) && (start_month == month))
          indices[0] = n;

        if ((end_year == year) && (end_month == month))
          indices[1] = n;
        }
      }

    return indices;
    }



  private int[] getDates(int start_index, int end_index)
    {
    int number_of_dates = (end_index - start_index) + 1;
    
    int[] dates = new int[number_of_dates];
    
    if (globals.getTimeWindow() == globals.DAILY)
      {
      for (int n=start_index; n<=end_index; n++)
        dates[n-start_index] = days[n];
      }

    else if (globals.getTimeWindow() == globals.WEEKLY)
      {
      for (int n=start_index; n<=end_index; n++)
        dates[n-start_index] = week_starts[n];
      }

    else if (globals.getTimeWindow() == globals.MONTHLY)
      {
      for (int n=start_index; n<=end_index; n++)
        {
        int year  = months[n] / 100;
        int month = months[n] % 100;

        dates[n-start_index] = (year * 10000) + (month * 100) + 1;
        }
      }

    return dates;
    }



  private String[] getDateStrings(int start_index, int end_index)
    {
    int number_of_dates = (end_index - start_index) + 1;
    
    String[] date_strings = new String[number_of_dates];
    
    if (globals.getTimeWindow() == globals.DAILY)
      {
      for (int n=start_index; n<=end_index; n++)
        date_strings[n-start_index] = day_strings[n];
      }

    else if (globals.getTimeWindow() == globals.WEEKLY)
      {
      for (int n=start_index; n<=end_index; n++)
        date_strings[n-start_index] = week_strings[n];
      }

    else if (globals.getTimeWindow() == globals.MONTHLY)
      {
      for (int n=start_index; n<=end_index; n++)
        date_strings[n-start_index] = month_strings[n];
      }

    return date_strings;
    }



  private void createDateStrings()
    {
    
    // Days

    day_strings = new String[days.length];
    
    for (int n=0; n<day_strings.length; n++)
      {
      int year  = days[n] / 10000;
      int month = (days[n] % 10000) / 100;
      int day   = days[n] % 100;
      
      if (month < 10)
        day_strings[n] = " " + month + "/";
      else
        day_strings[n] = "" + month + "/";

      if (day < 10)
        day_strings[n] += "0";
      
      day_strings[n] += day + "/" + (year%100);
      }

    // Weeks

    week_strings = new String[week_starts.length];

    for (int n=0; n<week_strings.length; n++)
      {
      int year  = week_starts[n] / 10000;
      int month = (week_starts[n] % 10000) / 100;
      int day   = week_starts[n] % 100;
      
      if (month < 10)
        week_strings[n] = " " + month + "/";
      else
        week_strings[n] = "" + month + "/";

      if (day < 10)
        week_strings[n] += "0";
      
      week_strings[n] += day + "/" + (year%100);
      }

    // Months

    month_strings = new String[months.length];

    for (int n=0; n<month_strings.length; n++)
      {
      int year  = months[n] / 100;
      int month = months[n] % 100;

      //month_strings[n] = EdgeUtils.getShortMonthString(month) + " " + year;
      month_strings[n] = getShortMonthString(month) + " " + year;
      }
    }



  public static String getShortMonthString(int month)
    {
    String s = "";
    
    switch (month)
      {
      case 1:  s = "Jan";
               break;

      case 2:  s = "Feb";
               break;

      case 3:  s = "Mar";
               break;

      case 4:  s = "Apr";
               break;

      case 5:  s = "May";
               break;

      case 6:  s = "Jun";
               break;

      case 7:  s = "Jul";
               break;

      case 8:  s = "Aug";
               break;

      case 9:  s = "Sep";
               break;

      case 10: s = "Oct";
               break;

      case 11: s = "Nov";
               break;

      case 12: s = "Dec";
               break;
      }

    return s;
    }



  private void readNarcsFileHeader()
    {
    try
      {
      ZipFile zf = new ZipFile(input_file_name);
      ZipEntry ze = zf.getEntry("narcsdir/header.dat");

      DataInputStream in = new DataInputStream(zf.getInputStream(ze));

      // Read the header record

      int file_type = in.readInt();
      float file_version = (float)in.readInt() / 10F;

      if ((file_type != 78638165) || (file_version != 3F))
        {
        System.out.println("The file type and version of the input file are not correct.");
        System.out.println("File name:  " + input_file_name);
        System.out.println("The <image_request> will be skipped.\n\n");
        }
      else
        {
        int dummy = in.readInt();
        dummy = in.readInt();

        // Read the level and layer pressures

        int num_temp_levels = in.readInt();
        int num_wvap_levels = in.readInt();
        int num_temp_layers = in.readInt();
        int num_wvap_layers = in.readInt();

        temp_level_pressures = new float[num_temp_levels];
    
        for (int n=0; n<num_temp_levels; n++)
          temp_level_pressures[n] = in.readInt() / 1000F;

        wvap_level_pressures = new float[num_wvap_levels];
    
        for (int n=0; n<num_wvap_levels; n++)
          wvap_level_pressures[n] = in.readInt() / 1000F;

        temp_layer_pressures = new float[num_temp_layers];
    
        for (int n=0; n<num_temp_layers; n++)
          temp_layer_pressures[n] = in.readInt() / 1000F;

        wvap_layer_pressures = new float[num_wvap_layers];
    
        for (int n=0; n<num_wvap_layers; n++)
          wvap_layer_pressures[n] = in.readInt() / 1000F;

        // Read the dates
          
        num_days   = in.readInt();
        num_weeks  = in.readInt();
        num_months = in.readInt();
          
        file_start_date = in.readInt();
        file_end_date   = in.readInt();

        // Set the usable date range

        if (date_range_start == -32768)
          date_range_start = file_start_date;
        else
          date_range_start = Math.max(file_start_date, date_range_start);

        if (date_range_end == -32768)
          date_range_end = file_end_date;
        else
          date_range_end = Math.min(file_end_date, date_range_end);

        // Set the date range in the globals. If the most recent selected
        // start and end dates are outside of the file date range then
        // adjust them accordingly

        globals.setFileStartDate(file_start_date);
        globals.setFileEndDate(file_end_date);

        globals.setDateRangeStart(date_range_start);
        globals.setDateRangeEnd(date_range_end);

        days = new int[num_days];
          
        for (int n=0; n<num_days; n++)
          days[n] = in.readInt();


        week_starts = new int[num_weeks];
        week_ends   = new int[num_weeks];

        for (int n=0; n<num_weeks; n++)
          {
          week_starts[n] = in.readInt();
          week_ends[n]   = in.readInt();
          }


        months = new int[num_months];
          
        for (int n=0; n<num_months; n++)
          months[n] = in.readInt();
          
          
        // Create the strings for each date
          
        createDateStrings();

        // Read the system information

        num_systems_in_file = in.readInt();

        system_names_in_file = new String[num_systems_in_file];
        system_ids_in_file   = new int[num_systems_in_file];

        //profile_types       = new Vector[num_systems_in_file];
        //input_profile_names = new Vector[num_systems_in_file];

        Vector temp_id_vector           = new Vector();
        Vector temp_sysnum_vector       = new Vector();
        Vector temp_ptype_vector        = new Vector();
        Vector temp_system_name_vector  = new Vector();
        Vector temp_profile_name_vector = new Vector();
        
        num_profiles_in_file = 0;

        for (int system=0; system<num_systems_in_file; system++)
          {
          system_ids_in_file[system] = in.readInt();

          int name_length = in.readInt();
            
          system_names_in_file[system] = "";
            
          for (int i=0; i<name_length; i++)
            system_names_in_file[system] += (char)(in.readInt());

          int num_profiles = in.readInt();

          // Set up the profile Vectors

          for (int prof=0; prof<num_profiles; prof++)
            {
            num_profiles_in_file++;

            int prof_type = in.readInt();

            temp_id_vector.addElement(new Integer(system_ids_in_file[system]));
            temp_sysnum_vector.addElement(new Integer(system));
            temp_ptype_vector.addElement(new Integer(prof_type));
              
            name_length = in.readInt();
              
            String prof_name = "";
              
            for (int i=0; i<name_length; i++)
              prof_name += (char)(in.readInt());

            temp_system_name_vector.addElement(system_names_in_file[system]);
            temp_profile_name_vector.addElement(prof_name);
            }  // for (prof=0...
          }  // for (system=0...


        // Determine the system ids and system numbers for each profile

        profile_system_ids_in_file     = new int[num_profiles_in_file];
        profile_system_numbers_in_file = new int[num_profiles_in_file];
        profile_profile_types_in_file  = new int[num_profiles_in_file];
        profile_system_names_in_file   = new String[num_profiles_in_file];
        profile_profile_names_in_file   = new String[num_profiles_in_file];

        for (int n=0; n<num_profiles_in_file; n++)
          {
          profile_system_ids_in_file[n]     = ((Integer)(temp_id_vector.elementAt(n))).intValue();
          profile_system_numbers_in_file[n] = ((Integer)(temp_sysnum_vector.elementAt(n))).intValue();
          profile_profile_types_in_file[n]  = ((Integer)(temp_ptype_vector.elementAt(n))).intValue();
          profile_system_names_in_file[n]   = (String)(temp_system_name_vector.elementAt(n));
          profile_profile_names_in_file[n]  = (String)(temp_profile_name_vector.elementAt(n));
          }
        }  // else (file_type !=...

      in.close();
      }
    catch (ZipException ze)
      {
      System.out.println("*** Error reading the NARCS file ***");
      System.out.println("The image will be skipped.\n");
      ze.printStackTrace();
      }
    catch (IOException ioe)
      {
      System.out.println("*** Error reading the NARCS file ***");
      System.out.println("The image will be skipped.\n");
      ioe.printStackTrace();
      }
    }



  private void readBaselineInformation()
    {
    try
      {
      ZipFile zf = new ZipFile(input_file_name);
      ZipEntry ze = zf.getEntry("narcsdir/baseline_info.dat");

      DataInputStream in = new DataInputStream(zf.getInputStream(ze));

      // Read the baseline information
      
      int baseline_system_id = in.readInt();
      int baseline_prof_num  = in.readInt();
      
      baseline_name = "";

      int length = in.readInt();
      
      for (int i=0; i<length; i++)
        {
        baseline_name += (char)(in.readInt());
        }

      in.close();
      }
    catch (ZipException ze)
      {
      System.out.println("*** Error reading the NARCS file ***");
      System.out.println("The image will be skipped.\n");
      ze.printStackTrace();
      }
    catch (IOException ioe)
      {
      System.out.println("*** Error reading the NARCS file ***");
      System.out.println("The image will be skipped.\n");
      ioe.printStackTrace();
      }
    }



  private void initializeImageParameters()
    {
    start_date = -32768;
    end_date   = -32768;
      
    data_type = SAMPLE_SIZE;
    data_pressure = 1000F;

    min_max_option = globals.DEFAULT;
    counts_min = 0;
    counts_max = 10000;
    data_min   = -5F;
    data_max   = 5F;
      
    ignore_extreme_values = false;
      
    time_option = globals.DAILY;

    show_data_points = false;

    show_bias = true;
    show_stddev = true;
    show_rms = false;
    show_samplesize = false;
    
    use_sample_size_cutoff = true;
    sample_size_cutoff = 0.25F;

    show_baseline_both   = true;
    show_baseline_sea    = false;
    show_baseline_nonsea = false;

    show_system_both   = true;
    show_system_sea    = false;
    show_system_nonsea = false;

    show_pass = true;
    show_fail = false;
    }



  private void replaceSystemAndProfileData(Grouping grouping)
    {

    // Replace the profile data
    
    for (int prof=0; prof<num_profiles_in_file; prof++)
      {
      for (int xmlprof=0; xmlprof<grouping.getNumberOfProfiles(); xmlprof++)
        {
        ProfileSettings prof_settings = grouping.getProfile(xmlprof);

        // If the system ids and profile types match then replace non-missing values

        if ((prof_settings.getPlatformID() == profile_system_ids_in_file[prof]) &&
            (prof_settings.getProfileID() == profile_profile_types_in_file[prof]))
          {
          if (prof_settings.getPlatformName() != null)
            profile_system_names_in_file[prof] = prof_settings.getPlatformName();

          if (prof_settings.getProfileName() != null)
            profile_profile_names_in_file[prof] = prof_settings.getProfileName();

          int code = (prof_settings.getPlatformID() * 100) + prof_settings.getProfileID();

          Color color = globals.getProfileColor(code);
          int red   = color.getRed();
          int green = color.getGreen();
          int blue  = color.getBlue();

          if ((prof_settings.getLineColorRed() != -32768) && (prof_settings.getLineColorGreen() != -32768) &&
              (prof_settings.getLineColorBlue() != -32768))
            {
            red   = prof_settings.getLineColorRed();
            green = prof_settings.getLineColorGreen();
            blue  = prof_settings.getLineColorBlue();

            color = new Color(red,green,blue);

            globals.setProfileColor(code, color);
            }
          }
        }  // for (xmlsys=0...
      }  // for (system=0...
    }



  private void updateDataToDisplay(Grouping grouping, int terrain, int stat_type)
    {

    // Combinations

    boolean[] baseline_options = new boolean[3];
    Arrays.fill(baseline_options, false);

    boolean[] system_options = new boolean[3];
    Arrays.fill(system_options, false);

    if (terrain == globals.LAND_AND_SEA)
      {
      baseline_options[0] = true;
      system_options[0]   = true;
      }
    else if (terrain == globals.SEA)
      {
      baseline_options[1] = true;
      system_options[1]   = true;
      }
    else if (terrain == globals.LAND)
      {
      baseline_options[2] = true;
      system_options[2]   = true;
      }

    boolean[] qc_options = new boolean[2];
    qc_options[0] = true;    // passed
    qc_options[1] = false;   // failed

    boolean[] show_piece = new boolean[NUM_PIECES];
    Arrays.fill(show_piece, false);

    for (int bterr=0; bterr<baseline_options.length; bterr++)
      {
      for (int sterr=0; sterr<system_options.length; sterr++)
        {
        for (int qc=0; qc<qc_options.length; qc++)
          {
          int piece_num = (qc * 9) + (bterr * 3) + sterr;
          show_piece[piece_num] = (baseline_options[bterr] && system_options[sterr] && qc_options[qc]);
          }
        }
      }

    globals.setShowCombination(show_piece);


    // Set the visible profiles

    int num_profiles = grouping.getNumberOfProfiles();

    Vector visible_profiles = new Vector(num_profiles);

    for (int prof=0; prof<num_profiles; prof++)
      {
      ProfileSettings prof_settings = grouping.getProfile(prof);
      int code = (prof_settings.getPlatformID() * 100) + prof_settings.getProfileID();

      visible_profiles.addElement(new Integer(code));
      }

    globals.setVisibleProfiles(visible_profiles);
    }



  /**
   * Prints the current system time.
   */

  private void printTime()
    {
    Calendar now = Calendar.getInstance();
    int hour   = now.get(Calendar.HOUR);
    int minute = now.get(Calendar.MINUTE);
    int second = now.get(Calendar.SECOND);

    String time = "" + hour + ":";

    if (minute < 10)
      time = time + "0";

    time = time + minute + ":";

    if (second < 10)
      time = time + "0";

    time = time + second + "\t";

    System.out.println(time);
    }
  }

// end of file
