import java.awt.Color;
import java.io.*;
import java.io.PrintWriter;
import java.io.BufferedWriter;
import java.util.Vector;
import java.util.Arrays;
import java.util.zip.*;

/**
 * <PRE>
 * HeatmapScriptGenerator.java
 *
 * Version: 1.0   Date: 10/24/2024   Programmer: Mike Pettey (IMSG)
 *
 * this is a version of StatGraph.java developed by Mike which generates
 * text files instead of an image file.  These files are actually a  
 * JavaScript file with the data points, titles and axis info.  Then these are
 * read by the Highcharts function.
 *
 * </PRE>
 */

public class HeatmapScriptGenerator
  {
  private String        clean_profile_name, profile_name;
  private int           terrain, stat_type, data_type;

  private boolean       is_longterm_plot;

  private String        baseline_name;
  private int           num_days, num_weeks, num_months, date_start_index, date_end_index;
  private String[]      date_strings, profile_names;
  private int[]         dates, profile_system_ids, profile_system_numbers, profile_types;
  private float[]       pressures;
  private int[]         pressure_indices;

  private Vector        visible_profile_vector;  //***** get rid of eventually
  private int           min_count, max_count;
  private float         min_stat, max_stat;

  private float[][]     data_array;
  private int[][]       sample_size_array;

  private NarcsToHighchartsGlobals globals;

  private final int SYSTEM    = 0;
  private final int RETRIEVAL = 1;
  private final int GUESS     = 2;
  private final int FORECAST  = 3;
  private final int MICROWAVE = 4;
  private final int INFRARED  = 5;
  private final int RAW_DRY   = 6;

  private final int NUM_PIECES = 18;
  private PrintWriter pw;
  private PrintWriter out;
  private String file_name;


  public HeatmapScriptGenerator(PrintWriter out, String clean_profile_name, String profile_name, int terrain,
                                int stat_type, int data_type, boolean is_longterm_plot, String baseline_name,
                                int num_days, int num_weeks, int num_months,
                                String[] profile_names, int[] profile_system_ids,
                                int[] profile_system_numbers, int[] profile_types,
                                int start_index, int end_index, String[] date_strings,
                                int[] dates, float[] pressures, int[] pressure_indices)
    {
    this.out = out;

    this.clean_profile_name = clean_profile_name;
    this.profile_name = profile_name;

    this.terrain   = terrain;
    this.stat_type = stat_type;
    this.data_type = data_type;

    this.is_longterm_plot = is_longterm_plot;

    this.pressures = pressures;
    this.pressure_indices = pressure_indices;

    this.baseline_name = baseline_name;
    this.num_days      = num_days;
    this.num_weeks     = num_weeks;
    this.num_months    = num_months;

    this.profile_names          = profile_names;
    this.profile_system_ids     = profile_system_ids;
    this.profile_system_numbers = profile_system_numbers;
    this.profile_types          = profile_types;

    date_start_index  = start_index;
    date_end_index    = end_index;
    this.date_strings = date_strings;
    this.dates        = dates;

    globals = new NarcsToHighchartsGlobals();

    //if (is_longterm_plot) file_name = "NarcsHighcharts_longterm.js";
    //else file_name = "NarcsHighcharts.js";

    visible_profile_vector = new Vector();
    }



  public void generateGraph()
    {
    visible_profile_vector.removeAllElements();

    // Build the profiles that are visible

    buildDataArray();

    // Remove any unwanted data

    removeUnwantedData();

    // Find the mininum and maximum values

    findStatisticMinMax();

    // Draw the graph beginning with the title

    drawTitle();

    // Draw the x axis

    drawXAxis();

    // Draw the y axis

    drawYAxis();

    // Draw the color scale

    drawColorScale();

    // Draw the main graph

    drawMainGraph();

    // Write ending lines to Highchart JavaScript file

    //out.println("\n        }]");
    out.println("  })");
    out.flush();
    }



  private void drawTitle()
    {

    // Start printing the beginning of the container

    String container_name = clean_profile_name + "_";

    if (terrain == globals.SEA)
      container_name += "Sea_";
    else if (terrain == globals.LAND)
      container_name += "Land_";
    else
      container_name += "LandAndSea_";

    if (data_type == globals.TEMPERATURE)
      container_name += "Temperature_";
    else if (data_type == globals.WATER_VAPOR)
      container_name += "WaterVapor_";

    if (stat_type == globals.BIAS)
      container_name += "Bias";
    else if (stat_type == globals.STDDEV)
      container_name += "StdDev";
    else if (stat_type == globals.RMS)
      container_name += "RMS";
    else
      container_name += "SampleSize";

    out.println("");
    out.println("$('#" + container_name + "').highcharts(");
    out.println("  {");

    out.println("");
    out.println("  chart:");
    out.println("    {");
    out.println("    type: \'heatmap\',");
    out.println("    plotBackgroundColor: \'#000000\',");
    out.println("    plotBorderWidth: 2,");
    out.println("    plotBorderColor:  \'#000000\'");
    out.println("    },");
    out.println("");
    out.println("  boost:");
    out.println("    {");
    out.println("    useGPUTranslations: true");
    out.println("    },");
    out.println("");
    out.println("  title:");
    out.println("    {");
    out.println("    text: \'NOAA/NESDIS/STAR NPROVS Collocation Summary Statistics (NARCS)\',");
    out.println("    },");
    out.println("");

    String subtitle = profile_name + ", ";

    if (terrain == globals.SEA)
      subtitle += "Sea, ";
    else if (terrain == globals.LAND)
      subtitle += "Land, ";
    else
      subtitle += "Land and Sea, ";

    if (data_type == globals.TEMPERATURE)
      subtitle += "Temperature, ";
    else if (data_type == globals.WATER_VAPOR)
      subtitle += "Water Vapor, ";

    if (stat_type == globals.BIAS)
      subtitle += "Bias";
    else if (stat_type == globals.STDDEV)
      subtitle += "StdDev";
    else if (stat_type == globals.RMS)
      subtitle += "RMS";
    else
      subtitle += "Sample Size";

    out.println("  subtitle:");
    out.println("    {");
    out.println("    text: \'" + subtitle + "\',");
    out.println("    style:");
    out.println("      {");
    out.println("      color: \'#555555\',");
    out.println("      fontWeight: \'bold\'");
    out.println("      },");
    out.println("");
    out.println("    x: -20 //center");
    out.println("    },");
    out.println("");

    out.flush();
    }



  private void drawXAxis()
    {
    out.println("  xAxis:");
    out.println("    {");
    out.println("    categories:");
    out.println("      [");

    int tab = 0;

    for (int date=0; date<date_strings.length; date++)
      {
      tab++;

      if (tab > 6)
        {
        out.println("");
        tab = 1;
        }

      if (tab == 1)
        out.print("      ");

      out.print("\'" + date_strings[date] + "\'");

      if (date != (date_strings.length - 1))
        out.print(", ");
      }  // for (date=0...

    out.println("\n      ],");
    out.println("");

    out.println("    labels:");
    out.println("      {");
    out.println("      rotation: 90");
    out.println("      },");
    out.println("");
    out.println("    minorGridLineDashStyle: \'longdash\',");
    out.println("    minorTickInterval: \'auto\',");
    out.println("    minorTickWidth: 0,");
    out.println("");
    out.println("    plotLines:");
    out.println("      [{");
    out.println("      dashStyle: \'longdash\',");
    out.println("      value: 0,");
    out.println("      width: 1,");
    out.println("      color: \'#808080\'");
    out.println("      }]");
    out.println("    },");
    out.println("");

    out.flush();
    }



  private void drawYAxis()
    {
    out.println("  yAxis:");
    out.println("    {");
    out.println("    categories:");
    out.println("      [");

    int tab = 0;
    String gap = "      ";

    for (int press=0; press<pressures.length; press++)
      {
      tab++;

      out.print(gap);

      if (pressures[press] >= 10F)
        out.print("\'" + (int)pressures[press] + "\'");
      else
        out.print("\'" + pressures[press] + "\'");

      if (tab <= 9)
        {
        gap = ", ";
        }
      else
        {
        tab = 0;
        gap = ",\n      ";
        }
      }  // for (press=0...

//Add extra blank pressures to account for a strange bug where Highcharts
//appears to expect more pressure levels than are actually available
out.println(", \' \', \' \', \' \', \' \', \' \', \' \'");

    out.println("\n      ],");

//    out.println("      ],");
    out.println("");
    out.println("    title:");
    out.println("      {");
    out.println("      text: \"Pressure (hPa)\"");
    out.println("      },");
    out.println("");
    out.println("    labels:");
    out.println("      {");
    out.println("      rotation: 0");
    out.println("      },");
    out.println("");
    out.println("    minorGridLineDashStyle: \'longdash\',");
    out.println("    minorTickInterval: \'auto\',");
    out.println("    minorTickWidth: 0,");
    out.println("");
    out.println("    plotLines:");
    out.println("      [{");
    out.println("      dashStyle: \'longdash\',");
    out.println("      value: 0,");
    out.println("      width: 1,");
    out.println("      color: \'#808080\'");
    out.println("      }],");
    out.println("");
    out.println("    reversed: true");
    out.println("    },");
    out.println("");

    out.flush();
    }



  private void drawColorScale()
    {
    out.println("  colorAxis:");
    out.println("    {");
    out.println("    stops:");
    out.println("      [");

    if (stat_type == globals.BIAS)
      {
      out.println("      [0, \'#00ff00\'],");
      out.println("      [0.01, \'#00ff00\'],");
      out.println("      [0.01, \'#000088\'],");
      out.println("      [0.5, \'#ffffff\'],");
      out.println("      [0.99, \'#880000\'],");
      out.println("      [0.99, \'#ffff00\'],");
      out.println("      [1, \'#ffff00\']");
      }
    else
      {
      out.println("      [0, '#8888ff'],");
      out.println("      [0.01, '#8888ff'],");
      out.println("      [0.01, '#ff00ff'],");
      out.println("      [0.20, '#0000ff'],");
      out.println("      [0.40, '#00ff00'],");
      out.println("      [0.60, '#ffff00'],");
      out.println("      [0.80, '#ff8800'],");
      out.println("      [0.99, '#880000'],");
      out.println("      [0.99, '#ff8888'],");
      out.println("      [1, '#ff8888']");
      }

    out.println("      ],");
    out.println("");

    float adj_min = min_stat - ((max_stat - min_stat) * 0.01F);
    float adj_max = max_stat + ((max_stat - min_stat) * 0.01F);
    out.println("    min: " + adj_min + ",");
    out.println("    max: " + adj_max + ",");

    out.println("    width: \'50%\',");
    out.println("    startOnTick: false,");
    out.println("    endOnTick: false,");
    out.println("");
    out.println("    labels:");
    out.println("      {");

    if (stat_type == globals.SAMPLE_SIZE)
      {
      out.println("      format: \'{value}\'");
      }
    else
      {
      if (data_type == globals.TEMPERATURE)
        out.println("      format: \'{value}°K\'");
      else
        out.println("      format: \'{value}%\'");
      }

    out.println("      }");
    out.println("    },");
    out.println("");

    out.flush();
    }



  private void drawMainGraph()
    {
    out.println("  series:");
    out.println("    [{");
    out.println("    boostThreshold: 100,");
    out.println("    borderWidth: 0,");
    out.println("    nullColor: \'#EFEFEF\',");
    out.println("");
    out.println("    tooltip:");
    out.println("      {");

    if (stat_type == globals.BIAS)
      out.println("      headerFormat: \'Bias<br/>\',");
    else if (stat_type == globals.STDDEV)
      out.println("      headerFormat: \'Std Dev<br/>\',");
    else if (stat_type == globals.RMS)
      out.println("      headerFormat: \'RMS<br/>\',");
    else
      out.println("      headerFormat: \'Sample Size<br/>\',");

    out.println("      pointFormatter: function()");
    out.println("        {");
    out.println("        return `${getPointCategoryName(this, \'x\')}, ${getPointCategoryName(this, \'y\')} hPa: <b>${this.value}</b>`;");
    out.println("        },");
    out.println("      nullFormat: \"Value not availible\"");
    out.println("      },");
    out.println("");
    out.println("    turboThreshold: Number.MAX_VALUE, // #3404, remove after 4.0.5 release");
    out.println("");
    out.println("    data:");
    out.println("      [");

    int tab = 0;
    String gap = "      ";

    for (int press=0; press<pressures.length; press++)
     {
      for (int date=0; date<date_strings.length; date++)
        {
        tab++;

        out.print(gap);

        if (data_array[press][date] != -32768F)
          out.print("[" + date + ", " + press + ", " + data_array[press][date] + "]");
        else
          out.print("[" + date + ", " + press + ", null]");

        if (tab <= 4)
          {
          gap = ", ";
          }
        else
          {
          tab = 0;
          gap = ",\n      ";
          }
        }  // for (date=0...
      }  // for (press=0...

    out.println("\n      ],");
    out.println("");
    out.println("    //colsize: stepSize, //setting the column width");
    out.println("    //rowsize: 2 //setting the row height");
    out.println("    nullColor: '#000000'");
    out.println("    }]");
    out.println("");

    out.flush();
    }


  private void buildDataArray()
    {

    // Set up and initialize the data array

    int num_pressures = pressures.length;
    int num_dates     = dates.length;

    data_array = new float[num_pressures][num_dates];
    sample_size_array = new int[num_pressures][num_dates];
    //Arrays.fill(data_array, -32768F);

    for (int press=0; press<num_pressures; press++)
      {
      for (int date=0; date<num_dates; date++)
        {
        data_array[press][date]        = -32768F;
        sample_size_array[press][date] = -32768;
        }
      }

    // Determine the system number (0 to N)

    Vector visible_profiles = globals.getVisibleProfiles();

    int system_val  = ((Integer)(visible_profiles.elementAt(0))).intValue();
    int system_id   = system_val / 100;
    int system_type = system_val % 100;

    int platform_number = -32768;

    for (int n=0; n<profile_system_ids.length; n++)
      {
      if (system_id == profile_system_ids[n])
        platform_number = profile_system_numbers[n];
      }

    // If the platform number was found, create profiles for the visible profiles

    if (platform_number != -32768)
      {
      boolean[] show_piece = globals.getShowCombination();

      for (int piece=0; piece<NUM_PIECES; piece++)
        {
        if (show_piece[piece] == true)
          {
          NarcsDataProfile sample_size_profile = null;
          NarcsDataProfile statistic_profile   = null;

          // Loop for each pressure level

          for (int press=0; press<pressures.length; press++)
            {

            // Create the profiles for each type of statistic

            sample_size_profile = buildProfile(system_id, platform_number, system_type, globals.SAMPLE_SIZE,
                                               pressure_indices[press], piece, null);

            statistic_profile = buildProfile(system_id, platform_number, system_type, stat_type,
                                             pressure_indices[press], piece, sample_size_profile);

            // Add the sample sizes for this pressure level to the sample size array

            if (sample_size_profile != null)
              {
              for (int date=0; date<num_dates; date++)
                sample_size_array[press][date] = (int)sample_size_profile.getValue(date);
              }

            // Add the data for this pressure level to the data array

            if (statistic_profile != null)
              {
              for (int date=0; date<num_dates; date++)
                data_array[press][date] = statistic_profile.getValue(date);
              }
            }  // for (press=0...
          }  // if (show_piece[piece]...
        }  // for (piece=0...
      }  // if (platform_number...
    }



  private NarcsDataProfile buildProfile(int system_id, int system_number, int profile_type,
                                        int stat_type, int pressure_index, int piece, NarcsDataProfile sample_size_profile)
    {
    NarcsDataProfile profile = null;

    int time_window = globals.getTimeWindow();

    // Build the file name

    String end_of_file_name = "/";

    if (time_window == globals.DAILY)
      end_of_file_name += "daily";
    else if (time_window == globals.WEEKLY)
      end_of_file_name += "weekly";
    else
      end_of_file_name += "monthly";

    end_of_file_name += "_system" + system_number;

    if (profile_type < 10)
      end_of_file_name += "_profile0" + profile_type + "_";
    else
      end_of_file_name += "_profile" + profile_type + "_";

    if (data_type == globals.TEMPERATURE)
      {
      end_of_file_name += "templevel";
      }
    else if (data_type == globals.WATER_VAPOR)
      {
      end_of_file_name += "wvaplevel";
      }

    end_of_file_name += pressure_index;

    if (stat_type == globals.BIAS)
      end_of_file_name += "_bias.dat";
    else if (stat_type == globals.STDDEV)
      end_of_file_name += "_stddev.dat";
    else if (stat_type == globals.RMS)
      end_of_file_name += "_rms.dat";
    else if (stat_type == globals.SAMPLE_SIZE)
      end_of_file_name += "_counts.dat";

    // Open the file for the system

    //String file_name = globals.getCurrentFileName();
    String file_name = "input.file";

    try
      {
      DataInputStream in = null;

      if (file_name != null)
        {
        File file = new File(file_name);

        if (file.exists())
          {
          try
            {
            ZipFile zf = new ZipFile(file_name);
            ZipEntry ze = zf.getEntry("narcsdir" + end_of_file_name);

            if (ze != null)
              in = new DataInputStream(zf.getInputStream(ze));
            }
          catch (ZipException ze)
            {
            ze.printStackTrace();
            }
          }
        }

      // If the DataInputStream is not null then loop through the pieces
      // and build a NarcsDataProfile for each visible one

      if (in != null)
        {

        // Create a buffer into which the values will be read

        int num_values = num_days;

        if (globals.getTimeWindow() == globals.WEEKLY)
          num_values = num_weeks;
        else if (globals.getTimeWindow() == globals.MONTHLY)
          num_values = num_months;

        byte[] buffer = new byte[num_values*4];

        // Skip to the start of the desired piece and read in the buffer for the piece

        int bytes_to_skip = piece * buffer.length;
        in.skipBytes(bytes_to_skip);

        in.readFully(buffer);

        // Read the data values from the buffer into the temporary array

        float[] temp_array = new float[num_values];

        DataInputStream data_in = new DataInputStream(new ByteArrayInputStream(buffer));

        for (int n=0; n<num_values; n++)
          {
          int value = data_in.readInt();

          if (value == -32768)
            {
            temp_array[n] = -32768F;
            }
          else
            {
            if (stat_type == globals.SAMPLE_SIZE)
              temp_array[n] = (float)value;
            else
              temp_array[n] = (float)value / 1000F;
            }
          }

        data_in.close();

        // Extract the values within the date range from the temporary array
        // to data_array

        int num_values_in_date_range = (date_end_index - date_start_index) + 1;
        float[] data_array = new float[num_values_in_date_range];

        for (int n=date_start_index; n<=date_end_index; n++)
          data_array[n-date_start_index] = temp_array[n];

        // Create the new NarcsDataProfile

        String name = "system name";

        for (int n=0; n<profile_names.length; n++)
          {
          if ((system_id == profile_system_ids[n]) && (profile_type == profile_types[n]))
            {
            name = profile_names[n];
            }
          }

        profile = new NarcsDataProfile(data_array, Color.black, system_id, system_number, profile_type,
                                       stat_type, piece, 0, 0, name, sample_size_profile);
        in.close();
        }  // if (in != null...
      }
    catch (IOException ioe)
      {
      ioe.printStackTrace();
      }
    
    return profile;
    }



  private void removeUnwantedData()
    {
    int num_pressures = pressures.length;
    int num_dates     = dates.length;

    // If the sample size cutoff is being used, remove values that have a sample size
    // below the cutoff value

    if (globals.getUseSampleSizeCutoff() == true)
      {

      // Calculate the cutoff

      int sample_size_sum = 0;
      int count = 0;

      for (int press=0; press<num_pressures; press++)
        {
        for (int date=0; date<num_dates; date++)
          {
          if ((sample_size_array[press][date] != -32768F) && (data_array[press][date] != -32768F))
            {
            sample_size_sum += (int)sample_size_array[press][date];
            count++;
            }
          }
        }

      float mean_sample_size = (float)sample_size_sum / (float)count;
      float cutoff_value = mean_sample_size * (1F / globals.getSampleSizeCutoff());

      // Remove data for any cell that has a sample size below the cutoff value

      for (int press=0; press<num_pressures; press++)
        {
        for (int date=0; date<num_dates; date++)
          {
          if (sample_size_array[press][date] < cutoff_value)
            data_array[press][date] = -32768F;
          }
        }
      }  // if (globals.getUseSampleSizeCutoff...


    // If the ignore extreme value flag is set, remove data that is outside
    // of the min/max range

    if (globals.getIgnoreExtremeValues() == true)
      {
      findStatisticMinMax();

      for (int press=0; press<num_pressures; press++)
        {
        for (int date=0; date<num_dates; date++)
          {
          if ((data_array[press][date] < min_stat) || (data_array[press][date] > max_stat))
            data_array[press][date] = -32768F;
          }
        }
      }  // if (globals.getIgnoreExtremeValues...
    }



  private void findStatisticMinMax()
    {
    min_stat = -5F;
    max_stat =  5F;

    if (globals.getMinMaxOption() == globals.DEFAULT)  // default
      {
      min_stat = globals.getStatDefaultMinimum(stat_type, data_type);
      max_stat = globals.getStatDefaultMaximum(stat_type, data_type);
      }
    else if (globals.getMinMaxOption() == globals.MANUAL)   // manual
      {
      min_stat = globals.getStatManualMinimum(stat_type, data_type);
      max_stat = globals.getStatManualMaximum(stat_type, data_type);
      }
    else    // automatic
      {
      min_stat =  9999999F;
      max_stat = -9999999F;

      // Loop through the visible profiles

      int num_pressures = pressures.length;
      int num_dates     = dates.length;

      for (int press=0; press<num_pressures; press++)
        {
        for (int date=0; date<num_dates; date++)
          {
          if (data_array[press][date] != -32768F)
            {
            min_stat = Math.min(min_stat, data_array[press][date]);
            max_stat = Math.max(max_stat, data_array[press][date]);
            }  // if (data_array[press][date]...
          }  // for (date=0...
        }  // for (press=0...

      if ((min_stat == 9999999F) || (min_stat == -9999999F) ||
          (max_stat == 9999999F) || (max_stat == -9999999F))
        {
        min_stat = -5F;
        max_stat =  5F;
        }

      if ((min_stat == 0F) && (max_stat == 0F))
        {
        min_stat = -10F;
        max_stat =  10F;
        }
      }
    }

  }

// end of file
