import java.awt.Color;
import java.util.HashMap;
import java.io.*;

/**
 * <PRE>
 * NarcsDataProfile.java
 *
 * Version: 1.0   Date: 12/02/2009   Programmer: Mike Pettey (IMSG)
 * Version: 2.0   Date:  6/01/2010   Programmer: Mike Pettey (IMSG)
 *
 * This class defines a vertical profile, which is basically a collection
 * of line segments. Contained within this class are x and y values that
 * indicate where each segment is plotted. This class also contains values
 * that indicate how the profile is to be plotted. These values include
 * line color, line thickness and line style.
 *
 * </PRE>
 */

public class NarcsDataProfile implements Externalizable
  {
  static final long serialVersionUID = 1L;
  private int class_version = 1;

  private boolean visible;
  private Color   default_line_color, line_color;
  private int     default_line_thickness, line_thickness;
  private int     default_line_style, line_style;
  private int     system_id, group_num, profile_type;
  private int     stat_type, piece, terrain_type, cloud_type;
  private String  label;
  private float[] values;
  private NarcsDataProfile sample_size_profile;



  public NarcsDataProfile()
    {
    }



  public NarcsDataProfile(float[] values, Color line_color, 
                          int system_id, int group_num, int profile_type, 
                          int stat_type, int piece, int terrain_type, int cloud_type, String label)
    {
    this.values       = values.clone();
    this.line_color   = line_color;
    this.system_id    = system_id;
    this.group_num    = group_num;
    this.profile_type = profile_type;
    this.stat_type    = stat_type;
    this.piece        = piece;
    this.terrain_type = terrain_type;
    this.cloud_type   = cloud_type;
    this.label        = label;
    
    sample_size_profile = null;

    default_line_color     = line_color;
    //default_line_thickness = line_thickness;
    //default_line_style     = line_style;
    }



  public NarcsDataProfile(float[] values, Color line_color, 
                          int system_id, int group_num, int profile_type, 
                          int stat_type, int piece, int terrain_type, int cloud_type, String label,
                          NarcsDataProfile sample_size_profile)
    {
    this.values       = values.clone();
    this.line_color   = line_color;
    this.system_id    = system_id;
    this.group_num    = group_num;
    this.profile_type = profile_type;
    this.stat_type    = stat_type;
    this.piece        = piece;
    this.terrain_type = terrain_type;
    this.cloud_type   = cloud_type;
    this.label        = label;
    
    this.sample_size_profile = sample_size_profile;

    default_line_color     = line_color;
    //default_line_thickness = line_thickness;
    //default_line_style     = line_style;
    }



  public void setValues(float[] values)
    {
    this.values = values.clone();
    }



  /**
   * Returns the desired x-axis value.
   *
   * @param   index  the index of the desired value. The index is based on
   *                 the order in which each point is added to the profile
   *                 (i.e., the 5th point added is at index 4).
   */

  public float getValue(int index)
    {
		float val = -32768F;

    if (index < values.length)
      val = values[index];

    return val;
    }

  
  public float[] getValues()
    {
    return values.clone();
    }



  /**
   * Set the visibilty of the profile. A true value will result in the
   * profile being displayed.  A false value will cause it to be hidden.
   *
   * @param  val  the visibilty flag (true=visible, false=hidden)
   */

  public void setVisible(boolean val)
    {
    visible = val;
    }
    

  /**
   * Returns the visiblity value of the profile.
   */

  public boolean isVisible()
    {
    return visible;
    }


  /**
   * Returns the number of points in the profile.
   */

  public int getNumberOfPoints()
    {
    return values.length;
    }



  public float getMinimumValue()
    {
    float min = 999999F;
    
    for (int x=0; x<values.length; x++)
      {
      if (values[x] != -32768F)
        {
        min = Math.min(min, values[x]);
        }
      }

    return min;
    }



  public float getMaximumValue()
    {
    float max = -999999F;
    
    for (int x=0; x<values.length; x++)
      {
      if (values[x] != -32768F)
        {
        max = Math.max(max, values[x]);
        }
      }

    return max;
    }



  public int getSystemID()
    {
    return system_id;
    }



  public int getGroupNumber()
    {
    return group_num;
    }
  
  
  public int getProfileType()
    {
    return profile_type;
    }


  public int getStatType()
    {
    return stat_type;
    }


  public int getPieceNumber()
    {
    return piece;
    }
  
  
  public int getTerrainType()
    {
    return terrain_type;
    }
  
  
  public int getCloudType()
    {
    return cloud_type;
    }


  /**
   * Returns the description of the profile.
   */

  public String getLabel()
    {
    return label;
    }



  public void setLabel(String name)
    {
    label = name;
    }



  /**
   * Sets or changes the line color of the profile.
   *
   * @param   color  the color that should be used when drawing the profile.
   */

  public void setLineColor(Color color)
    {
    line_color = color;
    }


  /**
   * Returns the line color of the profile.
   */

  public Color getLineColor()
    {
    return line_color;
    }

  public Color getDefaultLineColor()
    {
    return default_line_color;
    }


  /**
   * Sets or changes the line thickness of the profile.
   *
   * @param   thickness  the thickness that should be used when
   *                     drawing the profile.
   */

  public void setLineThickness(int thickness)
    {
    line_thickness = thickness;
    }


  /**
   * Returns the line thickness of the profile.
   */

  public int getLineThickness()
    {
    return line_thickness;
    }

  public int getDefaultLineThickness()
    {
    return default_line_thickness;
    }


  /**
   * Sets or changes the line style of the profile.
   *
   * @param   style  the style that should be used when drawing
   *                 the profile (0=solid, 1=dashed).
   */

  public void setLineStyle(int style)
    {
    line_style = style;
    }
  

  /**
   * Returns the line style of the profile.
   */

  public int getLineStyle()
    {
    return line_style;
    }

  public int getDefaultLineStyle()
    {
    return default_line_style;
    }



  public NarcsDataProfile getSampleSizeProfile()
    {
    return sample_size_profile;
    }



  /**
   * Overrides the system-level method to save the class and its
	 * necessary variables to an output stream using externalization.
   *
   * @param  out  The ObjectOutput stream that the class will
	 *              be saved to.
   */

  public void writeExternal(ObjectOutput out) throws java.io.IOException
    {

    // Create a HashMap and put all of the variables into it.
    
    HashMap<String, Object> hm = new HashMap<String, Object>();

    hm.put("visible", Boolean.valueOf(visible));
    //hm.put("default_line_color", default_line_color);
    //hm.put("line_color", line_color);
    //hm.put("default_line_thickness", new Integer(default_line_thickness));
    //hm.put("line_thickness", new Integer(line_thickness));
    //hm.put("default_line_style", new Integer(default_line_style));
    //hm.put("line_style", new Integer(line_style));
    hm.put("group_num", Integer.valueOf(group_num));
    hm.put("profile_type", Integer.valueOf(profile_type));
    hm.put("stat_type", Integer.valueOf(stat_type));
    hm.put("piece", Integer.valueOf(piece));
    hm.put("terrain_type", Integer.valueOf(terrain_type));
    hm.put("cloud_type", Integer.valueOf(cloud_type));
    hm.put("label", label);
    hm.put("values", values);
    hm.put("sample_size_profile", sample_size_profile);
    
    out.writeObject(hm);
    }


  /**
   * Overrides the system-level method to read the class and its
	 * necessary variables from an input stream using externalization.
	 * Versioning is handled by reading the first integer to determine
	 * the version number used when the class was saved. A specialized
	 * method will be called based on the version number.
   *
   * @param  in  The ObjectInput stream from which the class
	 *             will be restored.
   */

  public void readExternal(ObjectInput in) throws java.io.IOException,
																									java.lang.ClassNotFoundException
    {
    @SuppressWarnings("unchecked")
    HashMap<String, Object> hm = (HashMap<String, Object>)in.readObject();

    if (hm.containsKey("visible"))
      visible = ((Boolean)hm.get("visible")).booleanValue();

    //if (hm.containsKey("default_line_color"))
    //  default_line_color = (Color)hm.get("default_line_color");

    //if (hm.containsKey("line_color"))
    //  line_color = (Color)hm.get("line_color");

    //if (hm.containsKey("default_line_thickness"))
    //  default_line_thickness = ((Integer)hm.get("default_line_thickness")).intValue();

    //if (hm.containsKey("line_thickness"))
    //  line_thickness = ((Integer)hm.get("line_thickness")).intValue();

    //if (hm.containsKey("default_line_style"))
    //  default_line_style = ((Integer)hm.get("default_line_style")).intValue();

    //if (hm.containsKey("line_style"))
    //  line_style = ((Integer)hm.get("line_style")).intValue();

    if (hm.containsKey("group_num"))
      group_num = ((Integer)hm.get("group_num")).intValue();

    if (hm.containsKey("profile_type"))
      profile_type = ((Integer)hm.get("profile_type")).intValue();

    if (hm.containsKey("stat_type"))
      stat_type = ((Integer)hm.get("stat_type")).intValue();

    if (hm.containsKey("piece"))
      piece = ((Integer)hm.get("piece")).intValue();

    if (hm.containsKey("terrain_type"))
      terrain_type = ((Integer)hm.get("terrain_type")).intValue();

    if (hm.containsKey("cloud_type"))
      cloud_type = ((Integer)hm.get("cloud_type")).intValue();

    if (hm.containsKey("label"))
      label = (String)hm.get("label");

    if (hm.containsKey("values"))
      values = (float[])hm.get("values");
    
    if (hm.containsKey("sample_size_profile"))
      sample_size_profile = (NarcsDataProfile)hm.get("sample_size_profile");
    }

  }

// end of file
