import java.awt.*;
import java.util.Arrays;
import java.util.Vector;
import java.util.HashMap;
import java.io.*;

/**
 * <PRE>
 * NarcsToHighchartsGlobals.java
 *
 * Version: 1.0   Date: 10/10/2024   Programmer: Mike Pettey (IMSG)
 *
 * This class defines global variables that are used in various
 * places throughout EDGEIS. These variables are defined as static
 * to allow a change made to them in one location to be visible at
 * other locations.
 *
 * </PRE>
 */

public class NarcsToHighchartsGlobals
  {
  private static boolean      variables_initialized;

  private static int          file_start_date, file_end_date;
  private static int          date_range_start, date_range_end;

  private static Vector       visible_profiles;

  private static boolean      show_bias, show_stddev, show_rms, show_samplesize;
  private static boolean[]    show_combination;

  private static int          time_window;

  private static HashMap      profile_color_map;

  private static int          minmax_option;
  private static float        temperature_bias_stats_min, temperature_bias_stats_max;
  private static float        temperature_stddev_stats_min, temperature_stddev_stats_max;
  private static float        temperature_rms_stats_min, temperature_rms_stats_max;
  private static float        temperature_samplesize_stats_min, temperature_samplesize_stats_max;
  private static float        water_vapor_bias_stats_min, water_vapor_bias_stats_max;
  private static float        water_vapor_stddev_stats_min, water_vapor_stddev_stats_max;
  private static float        water_vapor_rms_stats_min, water_vapor_rms_stats_max;
  private static float        water_vapor_samplesize_stats_min, water_vapor_samplesize_stats_max;
  private static float        default_temperature_bias_stats_min, default_temperature_bias_stats_max;
  private static float        default_temperature_stddev_stats_min, default_temperature_stddev_stats_max;
  private static float        default_temperature_rms_stats_min, default_temperature_rms_stats_max;
  private static float        default_temperature_samplesize_stats_min, default_temperature_samplesize_stats_max;
  private static float        default_water_vapor_bias_stats_min, default_water_vapor_bias_stats_max;
  private static float        default_water_vapor_stddev_stats_min, default_water_vapor_stddev_stats_max;
  private static float        default_water_vapor_rms_stats_min, default_water_vapor_rms_stats_max;
  private static float        default_water_vapor_samplesize_stats_min, default_water_vapor_samplesize_stats_max;
  private static boolean      ignore_extreme_values;
  private static boolean      use_sample_size_display_cutoff;
  private static float        sample_size_display_cutoff;

  private final int NUM_PIECES = 18;

  public static final int DAILY   = 0;
  public static final int WEEKLY  = 1;
  public static final int MONTHLY = 2;

  public static final int AUTOMATIC = 0;
  public static final int DEFAULT   = 1;
  public static final int MANUAL    = 2;

  public static final int TEMP_LEVEL = 0;
  public static final int TEMP_LAYER = 1;
  public static final int WVAP_LEVEL = 2;
  public static final int WVAP_LAYER = 3;

  public static final int TEMPERATURE       = 0;
  public static final int WATER_VAPOR       = 1;
  public static final int RELATIVE_HUMIDITY = 2;

  public static final int BIAS        = 0;
  public static final int STDDEV      = 1;
  public static final int RMS         = 2;
  public static final int SAMPLE_SIZE = 3;

  public static final int SEA          = 0;
  public static final int LAND         = 1;
  public static final int LAND_AND_SEA = 2;



  public NarcsToHighchartsGlobals()
    {
    if (! variables_initialized)
      initializeVariables();
    }



  private void initializeVariables()
    {
    variables_initialized = true;

    file_start_date = -32768;
    file_end_date   = -32768;

    date_range_start = -32768;
    date_range_end   = -32768;

    visible_profiles = new Vector();

    show_combination = new boolean[NUM_PIECES];

    for (int j=0; j<NUM_PIECES; j++)
      show_combination[j] = false;

    show_combination[0] = true;

    show_bias       = true;
    show_stddev     = true;
    show_rms        = false;
    show_samplesize = false;

    time_window = DAILY;

    profile_color_map = new HashMap();

    minmax_option = AUTOMATIC;

    default_temperature_bias_stats_min       =    -5.0F;
    default_temperature_bias_stats_max       =     5.0F;
    default_temperature_stddev_stats_min     =     0.0F;
    default_temperature_stddev_stats_max     =     5.0F;
    default_temperature_rms_stats_min        =     0.0F;
    default_temperature_rms_stats_max        =     5.0F;
    default_temperature_samplesize_stats_min =     0.0F;
    default_temperature_samplesize_stats_max = 40000.0F;

    default_water_vapor_bias_stats_min       =   -50.0F;
    default_water_vapor_bias_stats_max       =   100.0F;
    default_water_vapor_stddev_stats_min     =     0.0F;
    default_water_vapor_stddev_stats_max     =   100.0F;
    default_water_vapor_rms_stats_min        =     0.0F;
    default_water_vapor_rms_stats_max        =   100.0F;
    default_water_vapor_samplesize_stats_min =     0.0F;
    default_water_vapor_samplesize_stats_max = 40000.0F;

    temperature_bias_stats_min       = default_temperature_bias_stats_min;
    temperature_bias_stats_max       = default_temperature_bias_stats_max;
    temperature_stddev_stats_min     = default_temperature_stddev_stats_min;
    temperature_stddev_stats_max     = default_temperature_stddev_stats_max;
    temperature_rms_stats_min        = default_temperature_rms_stats_min;
    temperature_rms_stats_max        = default_temperature_rms_stats_max;
    temperature_samplesize_stats_min = default_temperature_samplesize_stats_min;
    temperature_samplesize_stats_max = default_temperature_samplesize_stats_max;
    water_vapor_bias_stats_min       = default_water_vapor_bias_stats_min;
    water_vapor_bias_stats_max       = default_water_vapor_bias_stats_max;
    water_vapor_stddev_stats_min     = default_water_vapor_stddev_stats_min;
    water_vapor_stddev_stats_max     = default_water_vapor_stddev_stats_max;
    water_vapor_rms_stats_min        = default_water_vapor_rms_stats_min;
    water_vapor_rms_stats_max        = default_water_vapor_rms_stats_max;
    water_vapor_samplesize_stats_min = default_water_vapor_samplesize_stats_min;
    water_vapor_samplesize_stats_max = default_water_vapor_samplesize_stats_max;

    ignore_extreme_values = false;

    use_sample_size_display_cutoff = false;
    sample_size_display_cutoff = 0.25F;

    // Read in the default profile colors

    //readDefaultColors();
    }



  public void readDefaultColors()
    {
    try
      {
      XMLParser parser = new XMLParser();
      XMLTree xml_tree = parser.parseFile("default_colors.xml");
      //xml_tree.printTree();

      Object platform_object = xml_tree.getValue("nprovs_color_ramps");

      if ((platform_object != null) && (platform_object instanceof Vector))
        {
        Vector platform_vector = (Vector)platform_object;

        for (int plat=0; plat<platform_vector.size(); plat++)
          {
          Object object = platform_vector.elementAt(plat);

          if ((object != null) && (object instanceof XMLTree))
            {
            XMLTree plat_tree = (XMLTree)object;

            // Extract the platform id

            int platform_id = -32768;

            object = plat_tree.getValue("id");

            if ((object != null) && (object instanceof String))
              {
              String id = (String)object;
              platform_id = Integer.parseInt(id);
              }

            // Pull out each profile

            object = plat_tree.getValue("profiles");

            if ((object != null) && (object instanceof Vector))
              {
              Vector profile_vector = (Vector)object;

              for (int prof=0; prof<profile_vector.size(); prof++)
                {
                object = profile_vector.elementAt(prof);

                if ((object != null) && (object instanceof XMLTree))
                  {
                  XMLTree prof_tree = (XMLTree)object;

                  // Extract the profile type

                  int profile_type = -32768;

                  object = prof_tree.getValue("type");

                  if ((object != null) && (object instanceof String))
                    {
                    String id = (String)object;
                    profile_type = Integer.parseInt(id);
                    }

                  // Extract the red value

                  int red = -32768;

                  object = prof_tree.getValue("red");

                  if ((object != null) && (object instanceof String))
                    {
                    String id = (String)object;
                    red = Integer.parseInt(id);
                    }

                  // Extract the green value

                  int green = -32768;

                  object = prof_tree.getValue("green");

                  if ((object != null) && (object instanceof String))
                    {
                    String id = (String)object;
                    green = Integer.parseInt(id);
                    }

                  // Extract the blue value

                  int blue = -32768;

                  object = prof_tree.getValue("blue");

                  if ((object != null) && (object instanceof String))
                    {
                    String id = (String)object;
                    blue = Integer.parseInt(id);
                    }

                  // Set the default color

                  if ((platform_id != -32768) && (profile_type != -32768) &&
                      (red != -32768) && (green != -32768) && (blue != -32768))
                    {
                    int code = (platform_id * 100) + profile_type;
                    setProfileColor(code, new Color(red,green,blue));
                    }
                  }  // if (object != null...
                }  // for (prof=0...
              }  // if (object != null...
            }  // if (object != null...
          }  // for (plat=0...
        }  // if (platform_object instanceof Vector...
      }
    catch (XMLParserException xpe)
      {
      System.out.println("\nThe default colors file could not be parsed.");
      System.out.println("The structure of the XML data is probably not correct.");
      System.out.println("Please check the file and try again.\n");
      System.exit(1);
      }
    }



  public void setFileStartDate(int date)
    {
    file_start_date = date;
    }


  public int getFileStartDate()
    {
    return file_start_date;
    }



  public void setFileEndDate(int date)
    {
    file_end_date = date;
    }


  public int getFileEndDate()
    {
    return file_end_date;
    }



  public void setDateRangeStart(int date)
    {
    date_range_start = date;
    }


  public int getDateRangeStart()
    {
    return date_range_start;
    }



  public void setDateRangeEnd(int date)
    {
    date_range_end = date;
    }


  public int getDateRangeEnd()
    {
    return date_range_end;
    }



  public void setProfileColor(int profile_code, Color color)
    {
    profile_color_map.put(profile_code, color);
    }


  public Color getProfileColor(int profile_code)
    {
    Color color = (Color)(profile_color_map.get(profile_code));

    if (color == null)
      color = Color.black;

    return color;
    }



  public void setMinMaxOption(int option)
    {
    minmax_option = option;
    }


  public int getMinMaxOption()
    {
    return minmax_option;
    }



  public float getStatDefaultMinimum(int stat_type, int data_type)
    {
    float val = 0F;

    if (data_type == TEMPERATURE)
      {
      if (stat_type == BIAS)
        val = default_temperature_bias_stats_min;
      else if (stat_type == STDDEV)
        val = default_temperature_stddev_stats_min;
      else if (stat_type == RMS)
        val = default_temperature_rms_stats_min;
      else
        val = default_temperature_samplesize_stats_min;
      }
    else if (data_type == WATER_VAPOR)
      {
      if (stat_type == BIAS)
        val = default_water_vapor_bias_stats_min;
      else if (stat_type == STDDEV)
        val = default_water_vapor_stddev_stats_min;
      else if (stat_type == RMS)
        val = default_water_vapor_rms_stats_min;
      else
        val = default_water_vapor_samplesize_stats_min;
      }

    return val;
    }


  public float getStatDefaultMaximum(int stat_type, int data_type)
    {
    float val = 0F;

    if (data_type == TEMPERATURE)
      {
      if (stat_type == BIAS)
        val = default_temperature_bias_stats_max;
      else if (stat_type == STDDEV)
        val = default_temperature_stddev_stats_max;
      else if (stat_type == RMS)
        val = default_temperature_rms_stats_max;
      else
        val = default_temperature_samplesize_stats_max;
      }
    else if (data_type == WATER_VAPOR)
      {
      if (stat_type == BIAS)
        val = default_water_vapor_bias_stats_max;
      else if (stat_type == STDDEV)
        val = default_water_vapor_stddev_stats_max;
      else if (stat_type == RMS)
        val = default_water_vapor_rms_stats_max;
      else
        val = default_water_vapor_samplesize_stats_max;
      }

    return val;
    }



  public float getStatManualMinimum(int stat_type, int data_type)
    {
    float val = 0F;

    if (data_type == TEMPERATURE)
      {
      if (stat_type == BIAS)
        val = temperature_bias_stats_min;
      else if (stat_type == STDDEV)
        val = temperature_stddev_stats_min;
      else if (stat_type == RMS)
        val = temperature_rms_stats_min;
      else
        val = temperature_samplesize_stats_min;
      }
    else if (data_type == WATER_VAPOR)
      {
      if (stat_type == BIAS)
        val = water_vapor_bias_stats_min;
      else if (stat_type == STDDEV)
        val = water_vapor_stddev_stats_min;
      else if (stat_type == RMS)
        val = water_vapor_rms_stats_min;
      else
        val = water_vapor_samplesize_stats_min;
      }

    return val;
    }


  public float getStatManualMaximum(int stat_type, int data_type)
    {
    float val = 0F;

    if (data_type == TEMPERATURE)
      {
      if (stat_type == BIAS)
        val = temperature_bias_stats_max;
      else if (stat_type == STDDEV)
        val = temperature_stddev_stats_max;
      else
        val = temperature_rms_stats_max;
      }
    else if (data_type == WATER_VAPOR)
      {
      if (stat_type == BIAS)
        val = water_vapor_bias_stats_max;
      else if (stat_type == STDDEV)
        val = water_vapor_stddev_stats_max;
      else
        val = water_vapor_rms_stats_max;
      }

     return val;
     }



  public void setTemperatureBiasStatsMin(float min)
    {
    temperature_bias_stats_min = min;
    }


  public float getTemperatureBiasStatsMin()
    {
    return temperature_bias_stats_min;
    }



  public void setTemperatureBiasStatsMax(float max)
    {
    temperature_bias_stats_max = max;
    }


  public float getTemperatureBiasStatsMax()
    {
    return temperature_bias_stats_max;
    }



  public void setTemperatureStddevStatsMin(float min)
    {
    temperature_stddev_stats_min = min;
    }


  public float getTemperatureStddevStatsMin()
    {
    return temperature_stddev_stats_min;
    }



  public void setTemperatureStddevStatsMax(float max)
    {
    temperature_stddev_stats_max = max;
    }


  public float getTemperatureStddevStatsMax()
    {
    return temperature_stddev_stats_max;
    }



  public void setTemperatureRMSStatsMin(float min)
    {
    temperature_rms_stats_min = min;
    }


  public float getTemperatureRMSStatsMin()
    {
    return temperature_rms_stats_min;
    }



  public void setTemperatureRMSStatsMax(float max)
    {
    temperature_rms_stats_max = max;
    }


  public float getTemperatureRMSStatsMax()
    {
    return temperature_rms_stats_max;
    }



  public void setTemperatureSampleSizeStatsMin(float min)
    {
    temperature_samplesize_stats_min = min;
    }


  public float getTemperatureSampleSizeStatsMin()
    {
    return temperature_samplesize_stats_min;
    }



  public void setTemperatureSampleSizeStatsMax(float max)
    {
    temperature_samplesize_stats_max = max;
    }


  public float getTemperatureSampleSizeStatsMax()
    {
    return temperature_samplesize_stats_max;
    }



  public void setWaterVaporBiasStatsMin(float min)
    {
    water_vapor_bias_stats_min = min;
    }


  public float getWaterVaporBiasStatsMin()
    {
    return water_vapor_bias_stats_min;
    }



  public void setWaterVaporBiasStatsMax(float max)
    {
    water_vapor_bias_stats_max = max;
    }


  public float getWaterVaporBiasStatsMax()
    {
    return water_vapor_bias_stats_max;
    }



  public void setWaterVaporStddevStatsMin(float min)
    {
    water_vapor_stddev_stats_min = min;
    }


  public float getWaterVaporStddevStatsMin()
    {
    return water_vapor_stddev_stats_min;
    }



  public void setWaterVaporStddevStatsMax(float max)
    {
    water_vapor_stddev_stats_max = max;
    }


  public float getWaterVaporStddevStatsMax()
    {
    return water_vapor_stddev_stats_max;
    }



  public void setWaterVaporRMSStatsMin(float min)
    {
    water_vapor_rms_stats_min = min;
    }


  public float getWaterVaporRMSStatsMin()
    {
    return water_vapor_rms_stats_min;
    }



  public void setWaterVaporRMSStatsMax(float max)
    {
    water_vapor_rms_stats_max = max;
    }


  public float getWaterVaporRMSStatsMax()
    {
    return water_vapor_rms_stats_max;
    }



  public void setWaterVaporSampleSizeStatsMin(float min)
    {
    water_vapor_samplesize_stats_min = min;
    }


  public float getWaterVaporSampleSizeStatsMin()
    {
    return water_vapor_samplesize_stats_min;
    }



  public void setWaterVaporSampleSizeStatsMax(float max)
    {
    water_vapor_samplesize_stats_max = max;
    }


  public float getWaterVaporSampleSizeStatsMax()
    {
    return water_vapor_samplesize_stats_max;
    }



  public void setIgnoreExtremeValues(boolean ignore)
    {
    ignore_extreme_values = ignore;
    }


  public boolean getIgnoreExtremeValues()
    {
    return ignore_extreme_values;
    }



  public void setUseSampleSizeCutoff(boolean use)
    {
    use_sample_size_display_cutoff = use;
    }

  public boolean getUseSampleSizeCutoff()
    {
    return use_sample_size_display_cutoff;
    }


  public void setSampleSizeCutoff(float cutoff)
    {
    sample_size_display_cutoff = cutoff;
    }

  public float getSampleSizeCutoff()
    {
    return sample_size_display_cutoff;
    }


  
  // Display variables

  public void setTimeWindow(int option)
    {
    time_window = option;
    }

//***
  public int getTimeWindow()
    {
    return time_window;
    }



  public void setShowBias(boolean show)
    {
    show_bias = show;
    }
  
//***
  public boolean getShowBias()
    {
    return show_bias;
    }



  public void setShowStdDev(boolean show)
    {
    show_stddev = show;
    }
  
//***
  public boolean getShowStdDev()
    {
    return show_stddev;
    }



  public void setShowRMS(boolean show)
    {
    show_rms = show;
    }
  
//***
  public boolean getShowRMS()
    {
    return show_rms;
    }



  public void setShowSampleSize(boolean show)
    {
    show_samplesize = show;
    }
  
//***
  public boolean getShowSampleSize()
    {
    return show_samplesize;
    }



  public void setShowCombination(boolean[] show)
    {
    for (int n=0; n<NUM_PIECES; n++)
      show_combination[n] = show[n];
    }
  
//***
  public boolean[] getShowCombination()
    {
    boolean[] show = new boolean[NUM_PIECES];
    
    for (int n=0; n<NUM_PIECES; n++)
      show[n] = show_combination[n];
    
    return show;
    }



  public void setVisibleProfiles(Vector show)
    {
    visible_profiles = show;
    }
  

  public Vector getVisibleProfiles()
    {
    return visible_profiles;
    }

  }

// end of file
