import java.io.*;
import java.util.*;

/**
 * <PRE>
 * XMLParser.java
 *
 * Version: 1.0   Date:  9/01/2001   Programmer: Mike Pettey (Raytheon)
 * Version: 1.1   Date:  9/06/2001   Programmer: Mike Pettey (Raytheon)
 *
 * This class reads an XML file and processes the XML tags that
 * are contained within it. The XML tags and values are stored in
 * an XMLTree.
 *
 *   Version 1.1: This class was changed to give it the ability to
 *                read data from any input stream.
 * 
 * </PRE>
 */

public class XMLParser
  {
  public XMLParser()
    {
    }


  /**
   * Processes the XML file. As each tag or value is read in,
   * it is pushed onto a stack. When a closing tag is read from
   * the file, the value and/or tag are popped off the stack
   * and stored in the XMLTree. Subtrees are created by using
   * recursive instances of XMLTree.
   *
   * @parm  file_name  the name of the XML file.
   */

  public XMLTree parseFile(String file_name) throws XMLParserException
    {
    XMLTree xml_tree = null;

    try
      {
      File file = new File(file_name);

      if (! file.exists())
        {
        String message = "The parameter file " + file_name +
                         " could not be found. Please check the name of " +
			                   "the file and try running the program again.";

        throw new XMLParserException(message);
        }

      DataInputStream in = new DataInputStream(new FileInputStream(file));

      xml_tree = parseTree(in,null);

      in.close();
      }
    catch (IOException ioe)
      {
      String message = "" + ioe;
      throw new XMLParserException(message);
      }

    return xml_tree;
    }


  /**
   * Processes the XML file. As each tag or value is read in,
   * it is pushed onto a stack. When a closing tag is read from
   * the file, the value and/or tag are popped off the stack
   * and stored in the XMLTree. Subtrees are created by using
   * recursive instances of XMLTree.
   *
   * @parm  istream  the input stream.
   */

  public XMLTree parseFile(DataInputStream istream) throws XMLParserException
    {
    XMLTree xml_tree = parseTree(istream,null);

    return xml_tree;
    }



  private XMLTree parseTree(DataInputStream in, String prev_tag) throws XMLParserException
    {
    XMLTree xml_tree = null;

    try
      {
      String tag = prev_tag;

      if (tag == null)
        tag = getNextTag(in,' ');

      //System.out.println("Tag = " + tag);

      // Create the XMLTree

      String stripped_tag = tag.substring(1,(tag.length()-1));
      xml_tree = new XMLTree(stripped_tag);

      // If the tag contains binary data, read the data and add
      // it to the tree

      String next_tag = "";

      if (tag.startsWith("<bin_"))
	      {
	      byte[] binary_data = readBinaryData(in);
	      xml_tree.addValue(binary_data);

	      next_tag = getNextTag(in,' ');
	      }

      // If the tag does not contain binary data, process the
      // input until the closing tag is encountered

      else
	      {
	      Vector tree_vector = new Vector();
	      String value_string = "";

	      while (! next_tag.startsWith("</"))
	        {
	        char c = ' ';

          while (c != '<')
	          {
	          int n = in.read();

	          if (n == -1)
	            throwEOFException();

	          c = (char)n;

	          if ((c != '\n') && (c != '\r') && 
		            (c != '\t') && (c != '<'))
	            value_string = value_string + c;
	          }

	        next_tag = "<";

          while (c != '>')
	          {
	          int n = in.read();

	          if (n == -1)
	            throwEOFException();

	          c = (char)n;

	          if ((c != '\n') && (c != '\r') && (c != '\t'))
	            next_tag = next_tag + c;
	          }

	        if (! next_tag.startsWith("</"))
	          {
	          if ((! next_tag.startsWith("<?")) && (! next_tag.startsWith("<!")))
	            {
	            XMLTree subtree = parseTree(in,next_tag);
	            tree_vector.add(subtree);
	            }
	          }
	        }

	      if (tree_vector.size() == 0)
	        xml_tree.addValue(value_string.trim());
	      else
	        xml_tree.addValue(tree_vector);
	      }

      // Compare the opening and closing tags. If they do not
      // match then throw an exception

      String t1 = tag.substring(1);
      String t2 = next_tag.substring(2);

      if (! t1.equals(t2))
	      {
	      String message = "The tag " + tag + " either does not have a " +
	                       "matching tag or the tags are not properly " +
                         "nested.";

	      throw new XMLParserException(message);
	      }
      }
    catch (IOException ioe)
      {
      throwEOFException();
      }

    return xml_tree;
    }



  /**
   * Reads characters from the input stream and combines
   * them into a string which will either be a tag or a value.
   *
   * @param  in  the input stream.
   */

  private String getNextTag(DataInputStream in, char prev) throws XMLParserException
    {
    String tag = "";

    try
      {

      // Search for the start of the next tag

      char c = prev;

      // Read characters from the input stream until an opening bracket
      // is found

      while (c != '<')
	      {
	      int n = in.read();

	      if (n == -1)
	        throwEOFException();

	      // If the character is anything other than whitespace, then the 
	      // XML code is not well-formed

	      c = (char)n;

	      if ((c != '<') && (c != ' ') && (c != '\n') && (c != '\r') && (c != '\t'))
	        {
	        String message = "A bad character was read from the XML data. " +
                           "The character \"" + c + "\" was read while the " +
                           "program was looking for an XML tag. Please check " +
	                         "the XML code and try again.";
	        throw new XMLParserException(message);
	        }
	      }

      // Now that the start of the tag has been found, read the next 
      // character. If this character is a question mark or exclamation
      // point, then the tag will be skipped.

      int n = in.read();

      if (n == -1)
	      throwEOFException();

      c = (char)n;

      if ((c == '!') || (c == '?'))
	      {
	      while (c != '>')
	        {
	        n = in.read();

          if (n == -1)
	          throwEOFException();

	        c = (char)n;
	        }

	      tag = getNextTag(in,' ');
	      }
      else
        {
	      tag = "<" + c;

	      while (c != '>')
	        {
	        n = in.read();

          if (n == -1)
	          throwEOFException();

	        c = (char)n;

          if ((c != '\n') && (c != '\r'))
	          tag = tag + c;
	        }
	      }
      }
    catch (IOException ioe)
      {
      throwEOFException();
      }

    return tag;
    }



  private byte[] readBinaryData(DataInputStream in)
    {
    byte[] binary_data = null;

    try
      {
      int b1 = in.read();
      int b2 = in.read();
      int b3 = in.read();
      int b4 = in.read();

      int length = (b1*16777216) + (b2*65536) + (b3*256) + b4;

      binary_data = new byte[length];
      in.readFully(binary_data);
      }
    catch (IOException ioe)
      {
      System.out.println(ioe);
      }

    return binary_data;
    }



  private void throwEOFException() throws XMLParserException
    {
    String message = "The end of the XML data was reached prematurely. " +
                     "It is likely that the XML code is not properly " +
	             "formed. Please check the XML code and try again.";
    throw new XMLParserException(message);
    }
  }

// end of file
