/*
   Name- HRRRtoHDDF.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  12/01/2021   Programmer- Mike Pettey (IMSG)

   Function- This program extracts data from an HRRR analysis file
             and copies the data to an HDDF file.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>

#define  TRUE      0
#define  FALSE     1

#define  SWAP_IN   FALSE
#define  SWAP_OUT  FALSE

#define  HDDF_RECORD_LENGTH  360
#define  HDDF_HEADER_VALUES  HDDF_RECORD_LENGTH / 4
#define  HDDF_DATA_VALUES    HDDF_RECORD_LENGTH / 2

#define  BIN_SIZE   5.0

void createAndInitializeOutputFile();
void processNetCDF(int date_to_process, int time_to_process, FILE *out);
void getVariableIDS(int ncid);
int pointIsWithinBin(float lat, float lon, float bin_lat, float bin_lon);

int findBinNumber(double lat, double lon, int num_bins, float *bin_top_lats, float *bin_bottom_lats,
                  float *bin_left_lons, float *bin_right_lons);

float convertToWVMR(float pressure, float dwpt);
float spfhToWVMR(float pressure, float spfh);


int number_of_data_frames, number_of_output_records;

int lat_id, lon_id, terrain_id, surf_press_id, surf_temp_id, tpw_id;
int cape_surf_id, cape_255_id, cape_180_id, cape_90_id, cape_3000_id;
int temp_2m_id, spfh_2m_id, rh_2m_id;
int press_ids[50], temp_ids[50], spfh_ids[50];


int main(int argc, char *argv[])
  {
  int     i;
  int     date_to_process, time_to_process;
  int     year, month, day, yyyymmdd;
  int     data_date, earliest_date, latest_date;
  int     output_header[HDDF_DATA_VALUES];
  FILE    *out;
  time_t  date_time;
  struct  tm *ts;


  // Print an initial message

  printf("\nCopying HRRR data to an HDDF\n\n");


  // Unpack the date of the data to be processed

  date_to_process = atoi(argv[1]);
  time_to_process = atoi(argv[2]);

  printf("   Date:  %d\n", date_to_process);
  printf("   Time:  %02dz\n\n", time_to_process);


  // If the output file does not exist, create and initialize it

  if ((out = fopen("out.file", "r")) == NULL)
    createAndInitializeOutputFile();
  else
    fclose(out);


  // Open the output file and read the header

  if ((out=fopen("out.file", "r+")) != NULL)
    {
    fread(&output_header, HDDF_HEADER_VALUES, 1, out);

    if (SWAP_IN == TRUE)
      {
      for (i=0; i<HDDF_HEADER_VALUES; i++)
        output_header[i] = ntohl(output_header[i]);
      }

    number_of_output_records = output_header[2];
    number_of_data_frames    = output_header[6];
    earliest_date            = output_header[7];
    latest_date              = output_header[8];

    // Process the input file

    processNetCDF(date_to_process, time_to_process, out);


    // Get the current date

    time(&date_time);
    ts = localtime(&date_time);

    month  = ts->tm_mon + 1;
    day    = ts->tm_mday;
    year   = ts->tm_year;

    if (year > 56)
      year = 1900 + year;
    else
      year = 2000 + year;

    yyyymmdd = (year * 10000) + (month * 100) + day;

    // Update the date range values

    data_date = (date_to_process * 100) + time_to_process;

    if ((earliest_date == -32768) || (data_date < earliest_date))
      earliest_date = data_date;

    if ((latest_date == -32768) || (data_date > latest_date))
      latest_date = data_date;

    // Update the output file header

    output_header[2] = number_of_output_records + 1;
    output_header[5] = yyyymmdd;
    output_header[6] = number_of_data_frames;
    output_header[7] = earliest_date;
    output_header[8] = latest_date;

    if (SWAP_OUT == TRUE)
      {
      for (i=0; i<HDDF_HEADER_VALUES; i++)
        output_header[i] = htonl(output_header[i]);
      }

    fseek(out, 0, SEEK_SET);
    fwrite(&output_header, HDDF_RECORD_LENGTH, 1, out);

    fclose(out);
    }  // if ((out=fopen("out.file...

  printf("\nProcessing completed.\n\n");
  }



// ===============================================================================
void createAndInitializeOutputFile()
  {
  int  n;
  int  output_header[HDDF_DATA_VALUES];
  FILE *out;

  // Create a new output file

  printf("Creating and initializing the output file\n\n");

  if ((out=fopen("out.file", "w")) != NULL)
    {
    for (n=0; n<HDDF_HEADER_VALUES; n++)
      output_header[n] = -32768;

    output_header[0] = 72686870;
    output_header[1] = 10;
    output_header[2] = 1;
    output_header[3] = HDDF_RECORD_LENGTH;
    output_header[4] = HDDF_RECORD_LENGTH;
    output_header[5] = -32768;
    output_header[6] = 0;
    output_header[7] = -32768;
    output_header[8] = -32768;

    if (SWAP_OUT == TRUE)
      {
      for (n=0; n<HDDF_HEADER_VALUES; n++)
        output_header[n] = htonl(output_header[n]);
      }

    fseek(out, 0, SEEK_SET);
    fwrite(&output_header, HDDF_RECORD_LENGTH, 1, out);
    }

  fclose(out);
  }



// ===============================================================================
void processNetCDF(int date_to_process, int time_to_process, FILE *out)
  {
  int     n, x, y, status, level;
  int     bin_array_size_x, bin_array_size_y, total_num_bins;
  int     grid_size_x, grid_size_y;
  //int     num_footprints_processed;
  int     ncid, var_id;
  int     num_dims, num_attributes, dim_ids[MAX_VAR_DIMS];
  int     num_footprints, footprint;
  //int     num_in_bin, bin_number;
  int     *num_in_bin, *bin_number, *last_record_number_in_bin;
  int     bin_num, next_record;
  //int     *num_in_row;
  long    offset;
  short   buffer[HDDF_DATA_VALUES];
  //short   i2val, buffer[HDDF_DATA_VALUES];
  //float   bin_lat, bin_lon, fval, dwpt;
  float   bin_lat, bin_lon;
  float   *bin_left_lon, *bin_right_lon, *bin_top_lat, *bin_bottom_lat;
  float   wvmr;

  size_t start_2d[2], count_2d[2];
  size_t start_3d[3], count_3d[3];
  size_t  dim_len;
  nc_type var_type;

  double *lats, *lons;
  float *terrain, *surf_pressure, *surf_temperature, *tpw;
  float *cape_surf, *cape_255, *cape_180, *cape_90, *cape_3000;
  float *temp_2m, *spfh_2m, *rh_2m;

  float *press_1, *press_2, *press_3, *press_4, *press_5;
  float *press_6, *press_7, *press_8, *press_9, *press_10;
  float *press_11, *press_12, *press_13, *press_14, *press_15;
  float *press_16, *press_17, *press_18, *press_19, *press_20;
  float *press_21, *press_22, *press_23, *press_24, *press_25;
  float *press_26, *press_27, *press_28, *press_29, *press_30;
  float *press_31, *press_32, *press_33, *press_34, *press_35;
  float *press_36, *press_37, *press_38, *press_39, *press_40;
  float *press_41, *press_42, *press_43, *press_44, *press_45;
  float *press_46, *press_47, *press_48, *press_49, *press_50;

  float *temp_1, *temp_2, *temp_3, *temp_4, *temp_5;
  float *temp_6, *temp_7, *temp_8, *temp_9, *temp_10;
  float *temp_11, *temp_12, *temp_13, *temp_14, *temp_15;
  float *temp_16, *temp_17, *temp_18, *temp_19, *temp_20;
  float *temp_21, *temp_22, *temp_23, *temp_24, *temp_25;
  float *temp_26, *temp_27, *temp_28, *temp_29, *temp_30;
  float *temp_31, *temp_32, *temp_33, *temp_34, *temp_35;
  float *temp_36, *temp_37, *temp_38, *temp_39, *temp_40;
  float *temp_41, *temp_42, *temp_43, *temp_44, *temp_45;
  float *temp_46, *temp_47, *temp_48, *temp_49, *temp_50;

  float *spfh_1, *spfh_2, *spfh_3, *spfh_4, *spfh_5;
  float *spfh_6, *spfh_7, *spfh_8, *spfh_9, *spfh_10;
  float *spfh_11, *spfh_12, *spfh_13, *spfh_14, *spfh_15;
  float *spfh_16, *spfh_17, *spfh_18, *spfh_19, *spfh_20;
  float *spfh_21, *spfh_22, *spfh_23, *spfh_24, *spfh_25;
  float *spfh_26, *spfh_27, *spfh_28, *spfh_29, *spfh_30;
  float *spfh_31, *spfh_32, *spfh_33, *spfh_34, *spfh_35;
  float *spfh_36, *spfh_37, *spfh_38, *spfh_39, *spfh_40;
  float *spfh_41, *spfh_42, *spfh_43, *spfh_44, *spfh_45;
  float *spfh_46, *spfh_47, *spfh_48, *spfh_49, *spfh_50;

  // Find the size of the bin array

  bin_array_size_x = (int)(360.0 / BIN_SIZE);
  bin_array_size_y = (int)(180.0 / BIN_SIZE);

  total_num_bins = bin_array_size_x * bin_array_size_y;

  // Allocate memory for arrays that will contain bin array data

  num_in_bin = (int*)malloc(total_num_bins*sizeof(int));
  bin_number = (int*)malloc(total_num_bins*sizeof(int));
  last_record_number_in_bin = (int*)malloc(total_num_bins*sizeof(int));

  bin_left_lon   = (float*)malloc(total_num_bins*sizeof(float));
  bin_right_lon  = (float*)malloc(total_num_bins*sizeof(float));
  bin_top_lat    = (float*)malloc(total_num_bins*sizeof(float));
  bin_bottom_lat = (float*)malloc(total_num_bins*sizeof(float));

  // Set the bin boundary values

  x = -1;

  for (bin_lat=90.0; bin_lat>-90.0; bin_lat-=BIN_SIZE)
    {
    for (bin_lon=-180.0; bin_lon<180.0; bin_lon+=BIN_SIZE)
      {
      x++;

      num_in_bin[x] = 0;
      bin_number[x] = 0;
      last_record_number_in_bin[x] = 0;

      bin_left_lon[x]   = bin_lon;
      bin_right_lon[x]  = bin_lon + BIN_SIZE;
      bin_top_lat[x]    = bin_lat;
      bin_bottom_lat[x] = bin_lat - BIN_SIZE;
      }  // for (bin_lon=-180.0...
    }  // for (bin_lat=90.0...


  // Open the input file

  status = nc_open("in.file", 0, &ncid);

  if (status == NC_NOERR)
    {

    // Get the variable ids for all of the variables

    getVariableIDS(ncid);

    // Find the size of the grid

    status = ncvarinq(ncid, lat_id, 0, &var_type, &num_dims, dim_ids, &num_attributes);

    num_footprints = 1;

    for (n=0; n<num_dims; n++)
      {
      status = nc_inq_dimlen(ncid, dim_ids[n], &dim_len);
      num_footprints = num_footprints * (int)dim_len;

      if (n == 0)
        grid_size_y = (int)dim_len;
      else
        grid_size_x = (int)dim_len;
      }

    printf("\nGrid Size:   x = %d     y = %d\n\n", grid_size_x, grid_size_y);

    // Loop through all of the footprints to find out how many are in
    // each bin

    lats = (double*)malloc(grid_size_x*sizeof(double));
    lons = (double*)malloc(grid_size_x*sizeof(double));


    for (y=0; y<grid_size_y; y++)
      {
      start_2d[0] = y;
      start_2d[1] = 0;

      count_2d[0] = 1;
      count_2d[1] = grid_size_x;

      status = nc_get_vara_double(ncid, lat_id, start_2d, count_2d, lats);
      status = nc_get_vara_double(ncid, lon_id, start_2d, count_2d, lons);

      for (x=0; x<grid_size_x; x++)
        {

        // Adjust the longitude to -180/+180

        if (lons[x] > 180.0)
          lons[x] = -180.0 + (lons[x] - 180.0);

        // Find the bin number of the footprint

        bin_num = findBinNumber(lats[x], lons[x], total_num_bins, bin_top_lat,
                                bin_bottom_lat, bin_left_lon, bin_right_lon);

        if (bin_num != -32768)
          num_in_bin[bin_num] = num_in_bin[bin_num] + 1;
        }  // for (x=0...
      }  // for (y=0...


    // Create data frames in the output file for each bin that contains
    // at least one footprint

    next_record = number_of_output_records;

    for (bin_num=0; bin_num<total_num_bins; bin_num++)
      {
      if (num_in_bin[bin_num] > 0)
        {
        printf("Bin:    %d      %d\n", bin_num, num_in_bin[bin_num]);

        number_of_data_frames++;

        bin_number[bin_num] = number_of_data_frames;

        for (n=0; n<HDDF_DATA_VALUES; n++)
          buffer[n] = -32768;

        buffer[0] = 3;
        buffer[1] = number_of_data_frames;
        buffer[2] = num_in_bin[bin_num];

        buffer[4] = (short)(bin_bottom_lat[bin_num] * 128.0);
        buffer[5] = (short)(bin_bottom_lat[bin_num] * 128.0);
        buffer[6] = (short)(bin_top_lat[bin_num] * 128.0);
        buffer[7] = (short)(bin_top_lat[bin_num] * 128.0);

        buffer[8]  = (short)(bin_left_lon[bin_num] * 128.0);
        buffer[9]  = (short)(bin_right_lon[bin_num] * 128.0);
        buffer[10] = (short)(bin_right_lon[bin_num] * 128.0);
        buffer[11] = (short)(bin_left_lon[bin_num] * 128.0);

        buffer[12] = date_to_process / 10000;
        buffer[13] = date_to_process % 10000;
        buffer[14] = time_to_process * 100;

        next_record++;

//printf("Data Frame:  %d   %d    %d\n", bin_number[bin_num], next_record, (next_record+num_in_bin[bin_num]+1));

        offset = (long)(next_record - 1) * (long)HDDF_RECORD_LENGTH;
        fseek(out, offset, SEEK_SET);

        fwrite(&buffer, HDDF_RECORD_LENGTH, 1, out);

        last_record_number_in_bin[bin_num] = next_record;

        next_record = next_record + num_in_bin[bin_num];
        number_of_output_records = next_record - 1;

        }  // if (num_in_bin[bin_num]...
      }  // for (bin_num=0...


    // Allocate memory for each parameter

    lats = (double*)malloc(grid_size_x*sizeof(double));
    lons = (double*)malloc(grid_size_x*sizeof(double));
    terrain = (float*)malloc(grid_size_x*sizeof(float));
    surf_pressure = (float*)malloc(grid_size_x*sizeof(float));
    surf_temperature = (float*)malloc(grid_size_x*sizeof(float));
    tpw       = (float*)malloc(grid_size_x*sizeof(float));
    cape_surf = (float*)malloc(grid_size_x*sizeof(float));
    cape_255  = (float*)malloc(grid_size_x*sizeof(float));
    cape_180  = (float*)malloc(grid_size_x*sizeof(float));
    cape_90   = (float*)malloc(grid_size_x*sizeof(float));
    cape_3000 = (float*)malloc(grid_size_x*sizeof(float));
    temp_2m   = (float*)malloc(grid_size_x*sizeof(float));
    spfh_2m   = (float*)malloc(grid_size_x*sizeof(float));
    rh_2m     = (float*)malloc(grid_size_x*sizeof(float));

    press_1  = (float*)malloc(grid_size_x*sizeof(float));
    press_2  = (float*)malloc(grid_size_x*sizeof(float));
    press_3  = (float*)malloc(grid_size_x*sizeof(float));
    press_4  = (float*)malloc(grid_size_x*sizeof(float));
    press_5  = (float*)malloc(grid_size_x*sizeof(float));
    press_6  = (float*)malloc(grid_size_x*sizeof(float));
    press_7  = (float*)malloc(grid_size_x*sizeof(float));
    press_8  = (float*)malloc(grid_size_x*sizeof(float));
    press_9  = (float*)malloc(grid_size_x*sizeof(float));
    press_10 = (float*)malloc(grid_size_x*sizeof(float));
    press_11 = (float*)malloc(grid_size_x*sizeof(float));
    press_12 = (float*)malloc(grid_size_x*sizeof(float));
    press_13 = (float*)malloc(grid_size_x*sizeof(float));
    press_14 = (float*)malloc(grid_size_x*sizeof(float));
    press_15 = (float*)malloc(grid_size_x*sizeof(float));
    press_16 = (float*)malloc(grid_size_x*sizeof(float));
    press_17 = (float*)malloc(grid_size_x*sizeof(float));
    press_18 = (float*)malloc(grid_size_x*sizeof(float));
    press_19 = (float*)malloc(grid_size_x*sizeof(float));
    press_20 = (float*)malloc(grid_size_x*sizeof(float));
    press_21 = (float*)malloc(grid_size_x*sizeof(float));
    press_22 = (float*)malloc(grid_size_x*sizeof(float));
    press_23 = (float*)malloc(grid_size_x*sizeof(float));
    press_24 = (float*)malloc(grid_size_x*sizeof(float));
    press_25 = (float*)malloc(grid_size_x*sizeof(float));
    press_26 = (float*)malloc(grid_size_x*sizeof(float));
    press_27 = (float*)malloc(grid_size_x*sizeof(float));
    press_28 = (float*)malloc(grid_size_x*sizeof(float));
    press_29 = (float*)malloc(grid_size_x*sizeof(float));
    press_30 = (float*)malloc(grid_size_x*sizeof(float));
    press_31 = (float*)malloc(grid_size_x*sizeof(float));
    press_32 = (float*)malloc(grid_size_x*sizeof(float));
    press_33 = (float*)malloc(grid_size_x*sizeof(float));
    press_34 = (float*)malloc(grid_size_x*sizeof(float));
    press_35 = (float*)malloc(grid_size_x*sizeof(float));
    press_36 = (float*)malloc(grid_size_x*sizeof(float));
    press_37 = (float*)malloc(grid_size_x*sizeof(float));
    press_38 = (float*)malloc(grid_size_x*sizeof(float));
    press_39 = (float*)malloc(grid_size_x*sizeof(float));
    press_40 = (float*)malloc(grid_size_x*sizeof(float));
    press_41 = (float*)malloc(grid_size_x*sizeof(float));
    press_42 = (float*)malloc(grid_size_x*sizeof(float));
    press_43 = (float*)malloc(grid_size_x*sizeof(float));
    press_44 = (float*)malloc(grid_size_x*sizeof(float));
    press_45 = (float*)malloc(grid_size_x*sizeof(float));
    press_46 = (float*)malloc(grid_size_x*sizeof(float));
    press_47 = (float*)malloc(grid_size_x*sizeof(float));
    press_48 = (float*)malloc(grid_size_x*sizeof(float));
    press_49 = (float*)malloc(grid_size_x*sizeof(float));
    press_50 = (float*)malloc(grid_size_x*sizeof(float));

    temp_1  = (float*)malloc(grid_size_x*sizeof(float));
    temp_2  = (float*)malloc(grid_size_x*sizeof(float));
    temp_3  = (float*)malloc(grid_size_x*sizeof(float));
    temp_4  = (float*)malloc(grid_size_x*sizeof(float));
    temp_5  = (float*)malloc(grid_size_x*sizeof(float));
    temp_6  = (float*)malloc(grid_size_x*sizeof(float));
    temp_7  = (float*)malloc(grid_size_x*sizeof(float));
    temp_8  = (float*)malloc(grid_size_x*sizeof(float));
    temp_9  = (float*)malloc(grid_size_x*sizeof(float));
    temp_10 = (float*)malloc(grid_size_x*sizeof(float));
    temp_11 = (float*)malloc(grid_size_x*sizeof(float));
    temp_12 = (float*)malloc(grid_size_x*sizeof(float));
    temp_13 = (float*)malloc(grid_size_x*sizeof(float));
    temp_14 = (float*)malloc(grid_size_x*sizeof(float));
    temp_15 = (float*)malloc(grid_size_x*sizeof(float));
    temp_16 = (float*)malloc(grid_size_x*sizeof(float));
    temp_17 = (float*)malloc(grid_size_x*sizeof(float));
    temp_18 = (float*)malloc(grid_size_x*sizeof(float));
    temp_19 = (float*)malloc(grid_size_x*sizeof(float));
    temp_20 = (float*)malloc(grid_size_x*sizeof(float));
    temp_21 = (float*)malloc(grid_size_x*sizeof(float));
    temp_22 = (float*)malloc(grid_size_x*sizeof(float));
    temp_23 = (float*)malloc(grid_size_x*sizeof(float));
    temp_24 = (float*)malloc(grid_size_x*sizeof(float));
    temp_25 = (float*)malloc(grid_size_x*sizeof(float));
    temp_26 = (float*)malloc(grid_size_x*sizeof(float));
    temp_27 = (float*)malloc(grid_size_x*sizeof(float));
    temp_28 = (float*)malloc(grid_size_x*sizeof(float));
    temp_29 = (float*)malloc(grid_size_x*sizeof(float));
    temp_30 = (float*)malloc(grid_size_x*sizeof(float));
    temp_31 = (float*)malloc(grid_size_x*sizeof(float));
    temp_32 = (float*)malloc(grid_size_x*sizeof(float));
    temp_33 = (float*)malloc(grid_size_x*sizeof(float));
    temp_34 = (float*)malloc(grid_size_x*sizeof(float));
    temp_35 = (float*)malloc(grid_size_x*sizeof(float));
    temp_36 = (float*)malloc(grid_size_x*sizeof(float));
    temp_37 = (float*)malloc(grid_size_x*sizeof(float));
    temp_38 = (float*)malloc(grid_size_x*sizeof(float));
    temp_39 = (float*)malloc(grid_size_x*sizeof(float));
    temp_40 = (float*)malloc(grid_size_x*sizeof(float));
    temp_41 = (float*)malloc(grid_size_x*sizeof(float));
    temp_42 = (float*)malloc(grid_size_x*sizeof(float));
    temp_43 = (float*)malloc(grid_size_x*sizeof(float));
    temp_44 = (float*)malloc(grid_size_x*sizeof(float));
    temp_45 = (float*)malloc(grid_size_x*sizeof(float));
    temp_46 = (float*)malloc(grid_size_x*sizeof(float));
    temp_47 = (float*)malloc(grid_size_x*sizeof(float));
    temp_48 = (float*)malloc(grid_size_x*sizeof(float));
    temp_49 = (float*)malloc(grid_size_x*sizeof(float));
    temp_50 = (float*)malloc(grid_size_x*sizeof(float));

    spfh_1  = (float*)malloc(grid_size_x*sizeof(float));
    spfh_2  = (float*)malloc(grid_size_x*sizeof(float));
    spfh_3  = (float*)malloc(grid_size_x*sizeof(float));
    spfh_4  = (float*)malloc(grid_size_x*sizeof(float));
    spfh_5  = (float*)malloc(grid_size_x*sizeof(float));
    spfh_6  = (float*)malloc(grid_size_x*sizeof(float));
    spfh_7  = (float*)malloc(grid_size_x*sizeof(float));
    spfh_8  = (float*)malloc(grid_size_x*sizeof(float));
    spfh_9  = (float*)malloc(grid_size_x*sizeof(float));
    spfh_10 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_11 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_12 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_13 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_14 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_15 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_16 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_17 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_18 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_19 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_20 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_21 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_22 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_23 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_24 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_25 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_26 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_27 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_28 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_29 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_30 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_31 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_32 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_33 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_34 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_35 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_36 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_37 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_38 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_39 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_40 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_41 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_42 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_43 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_44 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_45 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_46 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_47 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_48 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_49 = (float*)malloc(grid_size_x*sizeof(float));
    spfh_50 = (float*)malloc(grid_size_x*sizeof(float));

    // Loop through all of the footprints one row at a time

    for (y=0; y<grid_size_y; y++)
      {
      printf("Row %d of %d\n", y, grid_size_y);

      start_2d[0] = y;
      start_2d[1] = 0;

      count_2d[0] = 1;
      count_2d[1] = grid_size_x;

      start_3d[0] = 0;
      start_3d[1] = y;
      start_3d[2] = 0;  // x

      count_3d[0] = 1;
      count_3d[1] = 1;
      count_3d[2] = grid_size_x;

      // Read the data

      status = nc_get_vara_double(ncid, lat_id, start_2d, count_2d, lats);
      status = nc_get_vara_double(ncid, lon_id, start_2d, count_2d, lons);
      status = nc_get_vara_float(ncid, terrain_id, start_3d, count_3d, terrain);
      status = nc_get_vara_float(ncid, surf_press_id, start_3d, count_3d, surf_pressure);
      status = nc_get_vara_float(ncid, surf_temp_id, start_3d, count_3d, surf_temperature);
      status = nc_get_vara_float(ncid, tpw_id, start_3d, count_3d, tpw);
      status = nc_get_vara_float(ncid, cape_surf_id, start_3d, count_3d, cape_surf);
      status = nc_get_vara_float(ncid, cape_255_id, start_3d, count_3d, cape_255);
      status = nc_get_vara_float(ncid, cape_180_id, start_3d, count_3d, cape_180);
      status = nc_get_vara_float(ncid, cape_90_id, start_3d, count_3d, cape_90);
      status = nc_get_vara_float(ncid, cape_3000_id, start_3d, count_3d, cape_3000);
      status = nc_get_vara_float(ncid, temp_2m_id, start_3d, count_3d, temp_2m);
      status = nc_get_vara_float(ncid, spfh_2m_id, start_3d, count_3d, spfh_2m);
      status = nc_get_vara_float(ncid, rh_2m_id, start_3d, count_3d, rh_2m);

      status = nc_get_vara_float(ncid, press_ids[0], start_3d, count_3d, press_1);
      status = nc_get_vara_float(ncid, press_ids[1], start_3d, count_3d, press_2);
      status = nc_get_vara_float(ncid, press_ids[2], start_3d, count_3d, press_3);
      status = nc_get_vara_float(ncid, press_ids[3], start_3d, count_3d, press_4);
      status = nc_get_vara_float(ncid, press_ids[4], start_3d, count_3d, press_5);
      status = nc_get_vara_float(ncid, press_ids[5], start_3d, count_3d, press_6);
      status = nc_get_vara_float(ncid, press_ids[6], start_3d, count_3d, press_7);
      status = nc_get_vara_float(ncid, press_ids[7], start_3d, count_3d, press_8);
      status = nc_get_vara_float(ncid, press_ids[8], start_3d, count_3d, press_9);
      status = nc_get_vara_float(ncid, press_ids[9], start_3d, count_3d, press_10);
      status = nc_get_vara_float(ncid, press_ids[10], start_3d, count_3d, press_11);
      status = nc_get_vara_float(ncid, press_ids[11], start_3d, count_3d, press_12);
      status = nc_get_vara_float(ncid, press_ids[12], start_3d, count_3d, press_13);
      status = nc_get_vara_float(ncid, press_ids[13], start_3d, count_3d, press_14);
      status = nc_get_vara_float(ncid, press_ids[14], start_3d, count_3d, press_15);
      status = nc_get_vara_float(ncid, press_ids[15], start_3d, count_3d, press_16);
      status = nc_get_vara_float(ncid, press_ids[16], start_3d, count_3d, press_17);
      status = nc_get_vara_float(ncid, press_ids[17], start_3d, count_3d, press_18);
      status = nc_get_vara_float(ncid, press_ids[18], start_3d, count_3d, press_19);
      status = nc_get_vara_float(ncid, press_ids[19], start_3d, count_3d, press_20);
      status = nc_get_vara_float(ncid, press_ids[20], start_3d, count_3d, press_21);
      status = nc_get_vara_float(ncid, press_ids[21], start_3d, count_3d, press_22);
      status = nc_get_vara_float(ncid, press_ids[22], start_3d, count_3d, press_23);
      status = nc_get_vara_float(ncid, press_ids[23], start_3d, count_3d, press_24);
      status = nc_get_vara_float(ncid, press_ids[24], start_3d, count_3d, press_25);
      status = nc_get_vara_float(ncid, press_ids[25], start_3d, count_3d, press_26);
      status = nc_get_vara_float(ncid, press_ids[26], start_3d, count_3d, press_27);
      status = nc_get_vara_float(ncid, press_ids[27], start_3d, count_3d, press_28);
      status = nc_get_vara_float(ncid, press_ids[28], start_3d, count_3d, press_29);
      status = nc_get_vara_float(ncid, press_ids[29], start_3d, count_3d, press_30);
      status = nc_get_vara_float(ncid, press_ids[30], start_3d, count_3d, press_31);
      status = nc_get_vara_float(ncid, press_ids[31], start_3d, count_3d, press_32);
      status = nc_get_vara_float(ncid, press_ids[32], start_3d, count_3d, press_33);
      status = nc_get_vara_float(ncid, press_ids[33], start_3d, count_3d, press_34);
      status = nc_get_vara_float(ncid, press_ids[34], start_3d, count_3d, press_35);
      status = nc_get_vara_float(ncid, press_ids[35], start_3d, count_3d, press_36);
      status = nc_get_vara_float(ncid, press_ids[36], start_3d, count_3d, press_37);
      status = nc_get_vara_float(ncid, press_ids[37], start_3d, count_3d, press_38);
      status = nc_get_vara_float(ncid, press_ids[38], start_3d, count_3d, press_39);
      status = nc_get_vara_float(ncid, press_ids[39], start_3d, count_3d, press_40);
      status = nc_get_vara_float(ncid, press_ids[40], start_3d, count_3d, press_41);
      status = nc_get_vara_float(ncid, press_ids[41], start_3d, count_3d, press_42);
      status = nc_get_vara_float(ncid, press_ids[42], start_3d, count_3d, press_43);
      status = nc_get_vara_float(ncid, press_ids[43], start_3d, count_3d, press_44);
      status = nc_get_vara_float(ncid, press_ids[44], start_3d, count_3d, press_45);
      status = nc_get_vara_float(ncid, press_ids[45], start_3d, count_3d, press_46);
      status = nc_get_vara_float(ncid, press_ids[46], start_3d, count_3d, press_47);
      status = nc_get_vara_float(ncid, press_ids[47], start_3d, count_3d, press_48);
      status = nc_get_vara_float(ncid, press_ids[48], start_3d, count_3d, press_49);
      status = nc_get_vara_float(ncid, press_ids[49], start_3d, count_3d, press_50);

      status = nc_get_vara_float(ncid, temp_ids[0], start_3d, count_3d, temp_1);
      status = nc_get_vara_float(ncid, temp_ids[1], start_3d, count_3d, temp_2);
      status = nc_get_vara_float(ncid, temp_ids[2], start_3d, count_3d, temp_3);
      status = nc_get_vara_float(ncid, temp_ids[3], start_3d, count_3d, temp_4);
      status = nc_get_vara_float(ncid, temp_ids[4], start_3d, count_3d, temp_5);
      status = nc_get_vara_float(ncid, temp_ids[5], start_3d, count_3d, temp_6);
      status = nc_get_vara_float(ncid, temp_ids[6], start_3d, count_3d, temp_7);
      status = nc_get_vara_float(ncid, temp_ids[7], start_3d, count_3d, temp_8);
      status = nc_get_vara_float(ncid, temp_ids[8], start_3d, count_3d, temp_9);
      status = nc_get_vara_float(ncid, temp_ids[9], start_3d, count_3d, temp_10);
      status = nc_get_vara_float(ncid, temp_ids[10], start_3d, count_3d, temp_11);
      status = nc_get_vara_float(ncid, temp_ids[11], start_3d, count_3d, temp_12);
      status = nc_get_vara_float(ncid, temp_ids[12], start_3d, count_3d, temp_13);
      status = nc_get_vara_float(ncid, temp_ids[13], start_3d, count_3d, temp_14);
      status = nc_get_vara_float(ncid, temp_ids[14], start_3d, count_3d, temp_15);
      status = nc_get_vara_float(ncid, temp_ids[15], start_3d, count_3d, temp_16);
      status = nc_get_vara_float(ncid, temp_ids[16], start_3d, count_3d, temp_17);
      status = nc_get_vara_float(ncid, temp_ids[17], start_3d, count_3d, temp_18);
      status = nc_get_vara_float(ncid, temp_ids[18], start_3d, count_3d, temp_19);
      status = nc_get_vara_float(ncid, temp_ids[19], start_3d, count_3d, temp_20);
      status = nc_get_vara_float(ncid, temp_ids[20], start_3d, count_3d, temp_21);
      status = nc_get_vara_float(ncid, temp_ids[21], start_3d, count_3d, temp_22);
      status = nc_get_vara_float(ncid, temp_ids[22], start_3d, count_3d, temp_23);
      status = nc_get_vara_float(ncid, temp_ids[23], start_3d, count_3d, temp_24);
      status = nc_get_vara_float(ncid, temp_ids[24], start_3d, count_3d, temp_25);
      status = nc_get_vara_float(ncid, temp_ids[25], start_3d, count_3d, temp_26);
      status = nc_get_vara_float(ncid, temp_ids[26], start_3d, count_3d, temp_27);
      status = nc_get_vara_float(ncid, temp_ids[27], start_3d, count_3d, temp_28);
      status = nc_get_vara_float(ncid, temp_ids[28], start_3d, count_3d, temp_29);
      status = nc_get_vara_float(ncid, temp_ids[29], start_3d, count_3d, temp_30);
      status = nc_get_vara_float(ncid, temp_ids[30], start_3d, count_3d, temp_31);
      status = nc_get_vara_float(ncid, temp_ids[31], start_3d, count_3d, temp_32);
      status = nc_get_vara_float(ncid, temp_ids[32], start_3d, count_3d, temp_33);
      status = nc_get_vara_float(ncid, temp_ids[33], start_3d, count_3d, temp_34);
      status = nc_get_vara_float(ncid, temp_ids[34], start_3d, count_3d, temp_35);
      status = nc_get_vara_float(ncid, temp_ids[35], start_3d, count_3d, temp_36);
      status = nc_get_vara_float(ncid, temp_ids[36], start_3d, count_3d, temp_37);
      status = nc_get_vara_float(ncid, temp_ids[37], start_3d, count_3d, temp_38);
      status = nc_get_vara_float(ncid, temp_ids[38], start_3d, count_3d, temp_39);
      status = nc_get_vara_float(ncid, temp_ids[39], start_3d, count_3d, temp_40);
      status = nc_get_vara_float(ncid, temp_ids[40], start_3d, count_3d, temp_41);
      status = nc_get_vara_float(ncid, temp_ids[41], start_3d, count_3d, temp_42);
      status = nc_get_vara_float(ncid, temp_ids[42], start_3d, count_3d, temp_43);
      status = nc_get_vara_float(ncid, temp_ids[43], start_3d, count_3d, temp_44);
      status = nc_get_vara_float(ncid, temp_ids[44], start_3d, count_3d, temp_45);
      status = nc_get_vara_float(ncid, temp_ids[45], start_3d, count_3d, temp_46);
      status = nc_get_vara_float(ncid, temp_ids[46], start_3d, count_3d, temp_47);
      status = nc_get_vara_float(ncid, temp_ids[47], start_3d, count_3d, temp_48);
      status = nc_get_vara_float(ncid, temp_ids[48], start_3d, count_3d, temp_49);
      status = nc_get_vara_float(ncid, temp_ids[49], start_3d, count_3d, temp_50);

      status = nc_get_vara_float(ncid, spfh_ids[0], start_3d, count_3d, spfh_1);
      status = nc_get_vara_float(ncid, spfh_ids[1], start_3d, count_3d, spfh_2);
      status = nc_get_vara_float(ncid, spfh_ids[2], start_3d, count_3d, spfh_3);
      status = nc_get_vara_float(ncid, spfh_ids[3], start_3d, count_3d, spfh_4);
      status = nc_get_vara_float(ncid, spfh_ids[4], start_3d, count_3d, spfh_5);
      status = nc_get_vara_float(ncid, spfh_ids[5], start_3d, count_3d, spfh_6);
      status = nc_get_vara_float(ncid, spfh_ids[6], start_3d, count_3d, spfh_7);
      status = nc_get_vara_float(ncid, spfh_ids[7], start_3d, count_3d, spfh_8);
      status = nc_get_vara_float(ncid, spfh_ids[8], start_3d, count_3d, spfh_9);
      status = nc_get_vara_float(ncid, spfh_ids[9], start_3d, count_3d, spfh_10);
      status = nc_get_vara_float(ncid, spfh_ids[10], start_3d, count_3d, spfh_11);
      status = nc_get_vara_float(ncid, spfh_ids[11], start_3d, count_3d, spfh_12);
      status = nc_get_vara_float(ncid, spfh_ids[12], start_3d, count_3d, spfh_13);
      status = nc_get_vara_float(ncid, spfh_ids[13], start_3d, count_3d, spfh_14);
      status = nc_get_vara_float(ncid, spfh_ids[14], start_3d, count_3d, spfh_15);
      status = nc_get_vara_float(ncid, spfh_ids[15], start_3d, count_3d, spfh_16);
      status = nc_get_vara_float(ncid, spfh_ids[16], start_3d, count_3d, spfh_17);
      status = nc_get_vara_float(ncid, spfh_ids[17], start_3d, count_3d, spfh_18);
      status = nc_get_vara_float(ncid, spfh_ids[18], start_3d, count_3d, spfh_19);
      status = nc_get_vara_float(ncid, spfh_ids[19], start_3d, count_3d, spfh_20);
      status = nc_get_vara_float(ncid, spfh_ids[20], start_3d, count_3d, spfh_21);
      status = nc_get_vara_float(ncid, spfh_ids[21], start_3d, count_3d, spfh_22);
      status = nc_get_vara_float(ncid, spfh_ids[22], start_3d, count_3d, spfh_23);
      status = nc_get_vara_float(ncid, spfh_ids[23], start_3d, count_3d, spfh_24);
      status = nc_get_vara_float(ncid, spfh_ids[24], start_3d, count_3d, spfh_25);
      status = nc_get_vara_float(ncid, spfh_ids[25], start_3d, count_3d, spfh_26);
      status = nc_get_vara_float(ncid, spfh_ids[26], start_3d, count_3d, spfh_27);
      status = nc_get_vara_float(ncid, spfh_ids[27], start_3d, count_3d, spfh_28);
      status = nc_get_vara_float(ncid, spfh_ids[28], start_3d, count_3d, spfh_29);
      status = nc_get_vara_float(ncid, spfh_ids[29], start_3d, count_3d, spfh_30);
      status = nc_get_vara_float(ncid, spfh_ids[30], start_3d, count_3d, spfh_31);
      status = nc_get_vara_float(ncid, spfh_ids[31], start_3d, count_3d, spfh_32);
      status = nc_get_vara_float(ncid, spfh_ids[32], start_3d, count_3d, spfh_33);
      status = nc_get_vara_float(ncid, spfh_ids[33], start_3d, count_3d, spfh_34);
      status = nc_get_vara_float(ncid, spfh_ids[34], start_3d, count_3d, spfh_35);
      status = nc_get_vara_float(ncid, spfh_ids[35], start_3d, count_3d, spfh_36);
      status = nc_get_vara_float(ncid, spfh_ids[36], start_3d, count_3d, spfh_37);
      status = nc_get_vara_float(ncid, spfh_ids[37], start_3d, count_3d, spfh_38);
      status = nc_get_vara_float(ncid, spfh_ids[38], start_3d, count_3d, spfh_39);
      status = nc_get_vara_float(ncid, spfh_ids[39], start_3d, count_3d, spfh_40);
      status = nc_get_vara_float(ncid, spfh_ids[40], start_3d, count_3d, spfh_41);
      status = nc_get_vara_float(ncid, spfh_ids[41], start_3d, count_3d, spfh_42);
      status = nc_get_vara_float(ncid, spfh_ids[42], start_3d, count_3d, spfh_43);
      status = nc_get_vara_float(ncid, spfh_ids[43], start_3d, count_3d, spfh_44);
      status = nc_get_vara_float(ncid, spfh_ids[44], start_3d, count_3d, spfh_45);
      status = nc_get_vara_float(ncid, spfh_ids[45], start_3d, count_3d, spfh_46);
      status = nc_get_vara_float(ncid, spfh_ids[46], start_3d, count_3d, spfh_47);
      status = nc_get_vara_float(ncid, spfh_ids[47], start_3d, count_3d, spfh_48);
      status = nc_get_vara_float(ncid, spfh_ids[48], start_3d, count_3d, spfh_49);
      status = nc_get_vara_float(ncid, spfh_ids[49], start_3d, count_3d, spfh_50);


      // Loop through the footprints in the row

      for (x=0; x<grid_size_x; x++)
        {
        if (lons[x] > 180.0)
          lons[x] = -180.0 + (lons[x] - 180.0);

        // Get the bin number for the footprint

        bin_num = findBinNumber(lats[x], lons[x], total_num_bins, bin_top_lat,
                                bin_bottom_lat, bin_left_lon, bin_right_lon);

        if (bin_num != -32768)
          {

//if (y < 5)
//printf("Footprint:  %d   %d    %f  %f    %d\n", x, y, lats[x], lons[x], bin_num);

          // Fill the buffer with data

          for (n=0; n<HDDF_DATA_VALUES; n++)
            buffer[n] = -32768;

          buffer[0] = 0;

          // Latitude and longitude

          buffer[1] = (short)(lats[x] / 1);
          buffer[2] = (short)((lats[x] - (int)lats[x]) * 10000.0);

          buffer[3] = (short)(lons[x] / 1);
          buffer[4] = (short)((lons[x] - (int)lons[x]) * 10000.0);


          // Date and time

          buffer[5]  = date_to_process / 10000;
          buffer[6]  = date_to_process % 10000;
          buffer[7]  = time_to_process;
          buffer[8] = 0;


buffer[9]  = (short)bin_number[bin_num];
buffer[10] = (short)((bin_number[bin_num] % 5) * 1000);

          // Terrain

          buffer[11] = (short)terrain[x];

          // TPW

          buffer[12] = (short)(tpw[x] * 100.0);

          // CAPE: surface

          buffer[13] = (short)(cape_surf[x] * 1.0);

          // CAPE: 255mb

          buffer[14] = (short)(cape_255[x] * 1.0);

          // CAPE: 180mb

          buffer[15] = (short)(cape_180[x] * 1.0);

          // CAPE: 90mb

          buffer[16] = (short)(cape_90[x] * 1.0);

          // CAPE: 3000m

          buffer[17] = (short)(cape_3000[x] * 1.0);

          // Surface pressure

          buffer[18] = (short)(surf_pressure[x] / 100.0 * 10.0);

          // Surface temperature

          buffer[19] = (short)(surf_temperature[x] * 64.0);

          // Temperature 2m above surface

          buffer[170] = (short)(temp_2m[x] * 64.0);

          // Specific humidity 2m above surface
//printf("2M DATA:  %d    %f     %f    %f    %f\n", x, temp_2m[x], spfh_2m[x], rh_2m[x], dwpt_2m[x]);
          buffer[171] = (short)(spfh_2m[x] * 100000.0);

          // Relative humidity 2m above surface

          buffer[172] = (short)(rh_2m[x] * 100.0);

          // Water vapor 2m above surface

          buffer[173] = (short)(log(spfhToWVMR(surf_pressure[x], spfh_2m[x])) * 1024.0);

          // Pressures

          buffer[20] = (short)(press_1[x] / 100.0 * 10.0);
          buffer[21] = (short)(press_2[x] / 100.0 * 10.0);
          buffer[22] = (short)(press_3[x] / 100.0 * 10.0);
          buffer[23] = (short)(press_4[x] / 100.0 * 10.0);
          buffer[24] = (short)(press_5[x] / 100.0 * 10.0);
          buffer[25] = (short)(press_6[x] / 100.0 * 10.0);
          buffer[26] = (short)(press_7[x] / 100.0 * 10.0);
          buffer[27] = (short)(press_8[x] / 100.0 * 10.0);
          buffer[28] = (short)(press_9[x] / 100.0 * 10.0);
          buffer[29] = (short)(press_10[x] / 100.0 * 10.0);
          buffer[30] = (short)(press_11[x] / 100.0 * 10.0);
          buffer[31] = (short)(press_12[x] / 100.0 * 10.0);
          buffer[32] = (short)(press_13[x] / 100.0 * 10.0);
          buffer[33] = (short)(press_14[x] / 100.0 * 10.0);
          buffer[34] = (short)(press_15[x] / 100.0 * 10.0);
          buffer[35] = (short)(press_16[x] / 100.0 * 10.0);
          buffer[36] = (short)(press_17[x] / 100.0 * 10.0);
          buffer[37] = (short)(press_18[x] / 100.0 * 10.0);
          buffer[38] = (short)(press_19[x] / 100.0 * 10.0);
          buffer[39] = (short)(press_20[x] / 100.0 * 10.0);
          buffer[40] = (short)(press_21[x] / 100.0 * 10.0);
          buffer[41] = (short)(press_22[x] / 100.0 * 10.0);
          buffer[42] = (short)(press_23[x] / 100.0 * 10.0);
          buffer[43] = (short)(press_24[x] / 100.0 * 10.0);
          buffer[44] = (short)(press_25[x] / 100.0 * 10.0);
          buffer[45] = (short)(press_26[x] / 100.0 * 10.0);
          buffer[46] = (short)(press_27[x] / 100.0 * 10.0);
          buffer[47] = (short)(press_28[x] / 100.0 * 10.0);
          buffer[48] = (short)(press_29[x] / 100.0 * 10.0);
          buffer[49] = (short)(press_30[x] / 100.0 * 10.0);
          buffer[50] = (short)(press_31[x] / 100.0 * 10.0);
          buffer[51] = (short)(press_32[x] / 100.0 * 10.0);
          buffer[52] = (short)(press_33[x] / 100.0 * 10.0);
          buffer[53] = (short)(press_34[x] / 100.0 * 10.0);
          buffer[54] = (short)(press_35[x] / 100.0 * 10.0);
          buffer[55] = (short)(press_36[x] / 100.0 * 10.0);
          buffer[56] = (short)(press_37[x] / 100.0 * 10.0);
          buffer[57] = (short)(press_38[x] / 100.0 * 10.0);
          buffer[58] = (short)(press_39[x] / 100.0 * 10.0);
          buffer[59] = (short)(press_40[x] / 100.0 * 10.0);
          buffer[60] = (short)(press_41[x] / 100.0 * 10.0);
          buffer[61] = (short)(press_42[x] / 100.0 * 10.0);
          buffer[62] = (short)(press_43[x] / 100.0 * 10.0);
          buffer[63] = (short)(press_44[x] / 100.0 * 10.0);
          buffer[64] = (short)(press_45[x] / 100.0 * 10.0);
          buffer[65] = (short)(press_46[x] / 100.0 * 10.0);
          buffer[66] = (short)(press_47[x] / 100.0 * 10.0);
          buffer[67] = (short)(press_48[x] / 100.0 * 10.0);
          buffer[68] = (short)(press_49[x] / 100.0 * 10.0);
          buffer[69] = (short)(press_50[x] / 100.0 * 10.0);

          // Temperatures

          buffer[70] = (short)(temp_1[x] * 64.0);
          buffer[71] = (short)(temp_2[x] * 64.0);
          buffer[72] = (short)(temp_3[x] * 64.0);
          buffer[73] = (short)(temp_4[x] * 64.0);
          buffer[74] = (short)(temp_5[x] * 64.0);
          buffer[75] = (short)(temp_6[x] * 64.0);
          buffer[76] = (short)(temp_7[x] * 64.0);
          buffer[77] = (short)(temp_8[x] * 64.0);
          buffer[78] = (short)(temp_9[x] * 64.0);
          buffer[79] = (short)(temp_10[x] * 64.0);
          buffer[80] = (short)(temp_11[x] * 64.0);
          buffer[81] = (short)(temp_12[x] * 64.0);
          buffer[82] = (short)(temp_13[x] * 64.0);
          buffer[83] = (short)(temp_14[x] * 64.0);
          buffer[84] = (short)(temp_15[x] * 64.0);
          buffer[85] = (short)(temp_16[x] * 64.0);
          buffer[86] = (short)(temp_17[x] * 64.0);
          buffer[87] = (short)(temp_18[x] * 64.0);
          buffer[88] = (short)(temp_19[x] * 64.0);
          buffer[89] = (short)(temp_20[x] * 64.0);
          buffer[90] = (short)(temp_21[x] * 64.0);
          buffer[91] = (short)(temp_22[x] * 64.0);
          buffer[92] = (short)(temp_23[x] * 64.0);
          buffer[93] = (short)(temp_24[x] * 64.0);
          buffer[94] = (short)(temp_25[x] * 64.0);
          buffer[95] = (short)(temp_26[x] * 64.0);
          buffer[96] = (short)(temp_27[x] * 64.0);
          buffer[97] = (short)(temp_28[x] * 64.0);
          buffer[98] = (short)(temp_29[x] * 64.0);
          buffer[99] = (short)(temp_30[x] * 64.0);
          buffer[100] = (short)(temp_31[x] * 64.0);
          buffer[101] = (short)(temp_32[x] * 64.0);
          buffer[102] = (short)(temp_33[x] * 64.0);
          buffer[103] = (short)(temp_34[x] * 64.0);
          buffer[104] = (short)(temp_35[x] * 64.0);
          buffer[105] = (short)(temp_36[x] * 64.0);
          buffer[106] = (short)(temp_37[x] * 64.0);
          buffer[107] = (short)(temp_38[x] * 64.0);
          buffer[108] = (short)(temp_39[x] * 64.0);
          buffer[109] = (short)(temp_40[x] * 64.0);
          buffer[110] = (short)(temp_41[x] * 64.0);
          buffer[111] = (short)(temp_42[x] * 64.0);
          buffer[112] = (short)(temp_43[x] * 64.0);
          buffer[113] = (short)(temp_44[x] * 64.0);
          buffer[114] = (short)(temp_45[x] * 64.0);
          buffer[115] = (short)(temp_46[x] * 64.0);
          buffer[116] = (short)(temp_47[x] * 64.0);
          buffer[117] = (short)(temp_48[x] * 64.0);
          buffer[118] = (short)(temp_49[x] * 64.0);
          buffer[119] = (short)(temp_50[x] * 64.0);

          // Water vapors

          buffer[120] = (short)(log(spfhToWVMR(press_1[x], spfh_1[x])) * 1024.0);
          buffer[121] = (short)(log(spfhToWVMR(press_2[x], spfh_2[x])) * 1024.0);
          buffer[122] = (short)(log(spfhToWVMR(press_3[x], spfh_3[x])) * 1024.0);
          buffer[123] = (short)(log(spfhToWVMR(press_4[x], spfh_4[x])) * 1024.0);
          buffer[124] = (short)(log(spfhToWVMR(press_5[x], spfh_5[x])) * 1024.0);
          buffer[125] = (short)(log(spfhToWVMR(press_6[x], spfh_6[x])) * 1024.0);
          buffer[126] = (short)(log(spfhToWVMR(press_7[x], spfh_7[x])) * 1024.0);
          buffer[127] = (short)(log(spfhToWVMR(press_8[x], spfh_8[x])) * 1024.0);
          buffer[128] = (short)(log(spfhToWVMR(press_9[x], spfh_9[x])) * 1024.0);
          buffer[129] = (short)(log(spfhToWVMR(press_10[x], spfh_10[x])) * 1024.0);
          buffer[130] = (short)(log(spfhToWVMR(press_11[x], spfh_11[x])) * 1024.0);
          buffer[131] = (short)(log(spfhToWVMR(press_12[x], spfh_12[x])) * 1024.0);
          buffer[132] = (short)(log(spfhToWVMR(press_13[x], spfh_13[x])) * 1024.0);
          buffer[133] = (short)(log(spfhToWVMR(press_14[x], spfh_14[x])) * 1024.0);
          buffer[134] = (short)(log(spfhToWVMR(press_15[x], spfh_15[x])) * 1024.0);
          buffer[135] = (short)(log(spfhToWVMR(press_16[x], spfh_16[x])) * 1024.0);
          buffer[136] = (short)(log(spfhToWVMR(press_17[x], spfh_17[x])) * 1024.0);
          buffer[137] = (short)(log(spfhToWVMR(press_18[x], spfh_18[x])) * 1024.0);
          buffer[138] = (short)(log(spfhToWVMR(press_19[x], spfh_19[x])) * 1024.0);
          buffer[139] = (short)(log(spfhToWVMR(press_20[x], spfh_20[x])) * 1024.0);
          buffer[140] = (short)(log(spfhToWVMR(press_21[x], spfh_21[x])) * 1024.0);
          buffer[141] = (short)(log(spfhToWVMR(press_22[x], spfh_22[x])) * 1024.0);
          buffer[142] = (short)(log(spfhToWVMR(press_23[x], spfh_23[x])) * 1024.0);
          buffer[143] = (short)(log(spfhToWVMR(press_24[x], spfh_24[x])) * 1024.0);
          buffer[144] = (short)(log(spfhToWVMR(press_25[x], spfh_25[x])) * 1024.0);
          buffer[145] = (short)(log(spfhToWVMR(press_26[x], spfh_26[x])) * 1024.0);
          buffer[146] = (short)(log(spfhToWVMR(press_27[x], spfh_27[x])) * 1024.0);
          buffer[147] = (short)(log(spfhToWVMR(press_28[x], spfh_28[x])) * 1024.0);
          buffer[148] = (short)(log(spfhToWVMR(press_29[x], spfh_29[x])) * 1024.0);
          buffer[149] = (short)(log(spfhToWVMR(press_30[x], spfh_30[x])) * 1024.0);
          buffer[150] = (short)(log(spfhToWVMR(press_31[x], spfh_31[x])) * 1024.0);
          buffer[151] = (short)(log(spfhToWVMR(press_32[x], spfh_32[x])) * 1024.0);
          buffer[152] = (short)(log(spfhToWVMR(press_33[x], spfh_33[x])) * 1024.0);
          buffer[153] = (short)(log(spfhToWVMR(press_34[x], spfh_34[x])) * 1024.0);
          buffer[154] = (short)(log(spfhToWVMR(press_35[x], spfh_35[x])) * 1024.0);
          buffer[155] = (short)(log(spfhToWVMR(press_36[x], spfh_36[x])) * 1024.0);
          buffer[156] = (short)(log(spfhToWVMR(press_37[x], spfh_37[x])) * 1024.0);
          buffer[157] = (short)(log(spfhToWVMR(press_38[x], spfh_38[x])) * 1024.0);
          buffer[158] = (short)(log(spfhToWVMR(press_39[x], spfh_39[x])) * 1024.0);
          buffer[159] = (short)(log(spfhToWVMR(press_40[x], spfh_40[x])) * 1024.0);
          buffer[160] = (short)(log(spfhToWVMR(press_41[x], spfh_41[x])) * 1024.0);
          buffer[161] = (short)(log(spfhToWVMR(press_42[x], spfh_42[x])) * 1024.0);
          buffer[162] = (short)(log(spfhToWVMR(press_43[x], spfh_43[x])) * 1024.0);
          buffer[163] = (short)(log(spfhToWVMR(press_44[x], spfh_44[x])) * 1024.0);
          buffer[164] = (short)(log(spfhToWVMR(press_45[x], spfh_45[x])) * 1024.0);
          buffer[165] = (short)(log(spfhToWVMR(press_46[x], spfh_46[x])) * 1024.0);
          buffer[166] = (short)(log(spfhToWVMR(press_47[x], spfh_47[x])) * 1024.0);
          buffer[167] = (short)(log(spfhToWVMR(press_48[x], spfh_48[x])) * 1024.0);
          buffer[168] = (short)(log(spfhToWVMR(press_49[x], spfh_49[x])) * 1024.0);
          buffer[169] = (short)(log(spfhToWVMR(press_50[x], spfh_50[x])) * 1024.0);

          // Write the record to the output file

          last_record_number_in_bin[bin_num] = last_record_number_in_bin[bin_num] + 1;

          offset = (long)(last_record_number_in_bin[bin_num] - 1) * (long)HDDF_RECORD_LENGTH;
          fseek(out, offset, SEEK_SET);
          fwrite(&buffer, HDDF_RECORD_LENGTH, 1, out);
          }  // if (bin_num != -32768...
        }  // for (x=0...
      }  // for (y=0...


    // Free the memory that was allocated for the parameters

    free(lats);
    free(lons);
    free(terrain);
    free(surf_pressure);
    free(surf_temperature);
    free(tpw);
    free(cape_surf);
    free(cape_255);
    free(cape_180);
    free(cape_90);
    free(cape_3000);
    free(temp_2m);
    free(spfh_2m);
    free(rh_2m);

    free(press_1);
    free(press_2);
    free(press_3);
    free(press_4);
    free(press_5);
    free(press_6);
    free(press_7);
    free(press_8);
    free(press_9);
    free(press_10);
    free(press_11);
    free(press_12);
    free(press_13);
    free(press_14);
    free(press_15);
    free(press_16);
    free(press_17);
    free(press_18);
    free(press_19);
    free(press_20);
    free(press_21);
    free(press_22);
    free(press_23);
    free(press_24);
    free(press_25);
    free(press_26);
    free(press_27);
    free(press_28);
    free(press_29);
    free(press_30);
    free(press_31);
    free(press_32);
    free(press_33);
    free(press_34);
    free(press_35);
    free(press_36);
    free(press_37);
    free(press_38);
    free(press_39);
    free(press_40);
    free(press_41);
    free(press_42);
    free(press_43);
    free(press_44);
    free(press_45);
    free(press_46);
    free(press_47);
    free(press_48);
    free(press_49);
    free(press_50);

    free(temp_1);
    free(temp_2);
    free(temp_3);
    free(temp_4);
    free(temp_5);
    free(temp_6);
    free(temp_7);
    free(temp_8);
    free(temp_9);
    free(temp_10);
    free(temp_11);
    free(temp_12);
    free(temp_13);
    free(temp_14);
    free(temp_15);
    free(temp_16);
    free(temp_17);
    free(temp_18);
    free(temp_19);
    free(temp_20);
    free(temp_21);
    free(temp_22);
    free(temp_23);
    free(temp_24);
    free(temp_25);
    free(temp_26);
    free(temp_27);
    free(temp_28);
    free(temp_29);
    free(temp_30);
    free(temp_31);
    free(temp_32);
    free(temp_33);
    free(temp_34);
    free(temp_35);
    free(temp_36);
    free(temp_37);
    free(temp_38);
    free(temp_39);
    free(temp_40);
    free(temp_41);
    free(temp_42);
    free(temp_43);
    free(temp_44);
    free(temp_45);
    free(temp_46);
    free(temp_47);
    free(temp_48);
    free(temp_49);
    free(temp_50);

    free(spfh_1);
    free(spfh_2);
    free(spfh_3);
    free(spfh_4);
    free(spfh_5);
    free(spfh_6);
    free(spfh_7);
    free(spfh_8);
    free(spfh_9);
    free(spfh_10);
    free(spfh_11);
    free(spfh_12);
    free(spfh_13);
    free(spfh_14);
    free(spfh_15);
    free(spfh_16);
    free(spfh_17);
    free(spfh_18);
    free(spfh_19);
    free(spfh_20);
    free(spfh_21);
    free(spfh_22);
    free(spfh_23);
    free(spfh_24);
    free(spfh_25);
    free(spfh_26);
    free(spfh_27);
    free(spfh_28);
    free(spfh_29);
    free(spfh_30);
    free(spfh_31);
    free(spfh_32);
    free(spfh_33);
    free(spfh_34);
    free(spfh_35);
    free(spfh_36);
    free(spfh_37);
    free(spfh_38);
    free(spfh_39);
    free(spfh_40);
    free(spfh_41);
    free(spfh_42);
    free(spfh_43);
    free(spfh_44);
    free(spfh_45);
    free(spfh_46);
    free(spfh_47);
    free(spfh_48);
    free(spfh_49);
    free(spfh_50);

    status = nc_close(ncid);
    }  // if (status==NC_NOERR -- nc_open...


  // Free the memory used by the bin arrays

  free(bin_bottom_lat);
  free(bin_top_lat);
  free(bin_right_lon);
  free(bin_left_lon);
  free(last_record_number_in_bin);
  free(bin_number);
  free(num_in_bin);
  }



// ===============================================================================
float convertToWVMR(float pressure, float dwpt)
  {
  double saturation_pressure;
  float wvmr = -32768.0;

  if ((pressure != -32768.0) && (dwpt != -32768.0))
    {
    saturation_pressure = 6.1078 * exp((dwpt-273.15) / ((dwpt-273.15)+238.3) * 17.2694);
    wvmr = (float)((622.0 * saturation_pressure) / (pressure - saturation_pressure));
    }

  return wvmr;
  }



float spfhToWVMR(float pressure, float spfh)
  {
  float wvmr = -32768.0;
  float ewv, tempr;

  if ((pressure != -32768.0) && (spfh != -32768.0))
    {
    ewv   = (spfh * pressure) / 0.622;
    tempr = (0.622 * ewv) / (pressure - ewv);
    wvmr  = tempr * 1000.0;
    }

  return wvmr;
  }



// ===============================================================================
void getVariableIDS(int ncid)
  {
  int  var_id, status, n;
  char var_name[100];

  status = nc_inq_varid(ncid, "latitude", &lat_id);
  status = nc_inq_varid(ncid, "longitude", &lon_id);
  status = nc_inq_varid(ncid, "terrain", &terrain_id);
  status = nc_inq_varid(ncid, "surface_pressure", &surf_press_id);
  status = nc_inq_varid(ncid, "surface_temperature", &surf_temp_id);
  status = nc_inq_varid(ncid, "tpw", &tpw_id);
  status = nc_inq_varid(ncid, "cape_surface", &cape_surf_id);
  status = nc_inq_varid(ncid, "cape_255-0mb", &cape_255_id);
  status = nc_inq_varid(ncid, "cape_180-0mb", &cape_180_id);
  status = nc_inq_varid(ncid, "cape_90-0mb", &cape_90_id);
  status = nc_inq_varid(ncid, "cape_0-3000mb", &cape_3000_id);

  status = nc_inq_varid(ncid, "temperature_2m_above_ground", &temp_2m_id);
  status = nc_inq_varid(ncid, "specific_humidity_2m_above_ground", &spfh_2m_id);
  status = nc_inq_varid(ncid, "relative_humidity_2m_above_ground", &rh_2m_id);

  for (n=0; n<50; n++)
    {
    sprintf(var_name, "pressure_level_%02d", (n+1));
    status = nc_inq_varid(ncid, var_name, &var_id);
    press_ids[n] = var_id;

    sprintf(var_name, "temperature_level_%02d", (n+1));
    status = nc_inq_varid(ncid, var_name, &var_id);
    temp_ids[n] = var_id;

    sprintf(var_name, "specific_humidity_level_%02d", (n+1));
    status = nc_inq_varid(ncid, var_name, &var_id);
    spfh_ids[n] = var_id;
    }
  }



// ===============================================================================
//int pointIsWithinBin(float lat, float lon, float bin_lat, float bin_lon)
//  {
//  int is_in_bin = FALSE;
//
//  if ((lat <= bin_lat) && (lat > (bin_lat-BIN_SIZE)) &&
//      (lon >= bin_lon) && (lon < (bin_lon+BIN_SIZE)))
//    {
//    is_in_bin = TRUE;
//    }
//
//  return is_in_bin;
//  }


int findBinNumber(double lat, double lon, int num_bins, float *bin_top_lats,
                  float *bin_bottom_lats, float *bin_left_lons, float *bin_right_lons)
  {
  int n;
  int bin_number = -32768;

  n = -1;

  while ((bin_number == -32768) && (n < num_bins))
    {
    n++;

    if ((lat <= bin_top_lats[n]) && (lat > bin_bottom_lats[n]) &&
        (lon >= bin_left_lons[n]) && (lon < bin_right_lons[n]))
      {
      bin_number = n;
      }
    }

  return bin_number;
  }



// ===============================================================================
void convertTime(double fov_time, int *yyyy, int *mmdd, int *hh, int *mmss)
  {
  time_t     epoch_time;
  struct tm *date;

  epoch_time = (time_t)fov_time;
  date = localtime(&epoch_time);

  // Return the values

  *yyyy = 1900 + date->tm_year;;
  *mmdd = ((date->tm_mon + 1) * 100) + date->tm_mday;
  *hh   = date->tm_hour;
  *mmss = (date->tm_min * 100) + date->tm_sec;
  }

// end of file
