/*
   Name- HRRRtoHDDF.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  12/01/2021   Programmer- Mike Pettey (IMSG)

   Function- This program extracts data from an HRRR analysis file
             and copies the data to an HDDF file.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>

#define  TRUE      0
#define  FALSE     1

#define  SWAP_IN   FALSE
#define  SWAP_OUT  FALSE

#define  HDDF_RECORD_LENGTH  400
#define  HDDF_HEADER_VALUES  HDDF_RECORD_LENGTH / 4
#define  HDDF_DATA_VALUES    HDDF_RECORD_LENGTH / 2

#define  BIN_SIZE   5.0
#define  BUFFER_SIZE  1000

void createAndInitializeOutputFile();
void processNetCDF(int date_to_process, int time_to_process, FILE *out);
void getVariableIDS(int ncid);
int pointIsWithinBin(float lat, float lon, float bin_lat, float bin_lon);
float convertToWVMR(float pressure, float dwpt);


int number_of_data_frames, number_of_output_records;

int lat_id, lon_id, surf_press_id, surf_temp_id, prate_id, total_precip_id, tpw_id;
int cape_surf_id, cape_255_id, cape_180_id, cape_90_id, cape_3000_id;
int cin_surf_id, cin_255_id, cin_180_id, cin_90_id;
int li_500_id, li_4layer_id;
int low_cloud_id, mid_cloud_id, high_cloud_id, total_cloud_id, cloud_base_id, cloud_top_id;
int terrain_id, ice_id, snow_id;
int temp_ids[40], rh_ids[40], dwpt_ids[40];


int main(int argc, char *argv[])
  {
  int     i;
  int     date_to_process, time_to_process;
  int     year, month, day, yyyymmdd;
  int     data_date, earliest_date, latest_date;
  int     output_header[HDDF_DATA_VALUES];
  FILE    *out;
  time_t  date_time;
  struct  tm *ts;


  // Print an initial message

  printf("\nCopying HRRR data to an HDDF\n\n");


  // Unpack the date of the data to be processed

  date_to_process = atoi(argv[1]);
  time_to_process = atoi(argv[2]);

  printf("   Date:  %d\n", date_to_process);
  printf("   Time:  %02dz\n\n", time_to_process);

  // If the output file does not exist, create and initialize it

  if ((out = fopen("out.file", "r")) == NULL)
    createAndInitializeOutputFile();
  else
    fclose(out);


  // Open the output file and read the header

  if ((out=fopen("out.file", "r+")) != NULL)
    {
    fread(&output_header, HDDF_HEADER_VALUES, 1, out);

    if (SWAP_IN == TRUE)
      {
      for (i=0; i<HDDF_HEADER_VALUES; i++)
        output_header[i] = ntohl(output_header[i]);
      }

    number_of_output_records = output_header[2];
    number_of_data_frames    = output_header[6];
    earliest_date            = output_header[7];
    latest_date              = output_header[8];


    // Process the input file

    processNetCDF(date_to_process, time_to_process, out);


    // Get the current date

    time(&date_time);
    ts = localtime(&date_time);

    month  = ts->tm_mon + 1;
    day    = ts->tm_mday;
    year   = ts->tm_year;

    if (year > 56)
      year = 1900 + year;
    else
      year = 2000 + year;

    yyyymmdd = (year * 10000) + (month * 100) + day;

    // Update the date range values

    data_date = (date_to_process * 100) + time_to_process;

    if ((earliest_date == -32768) || (data_date < earliest_date))
      earliest_date = data_date;

    if ((latest_date == -32768) || (data_date > latest_date))
      latest_date = data_date;

    // Update the output file header

    output_header[2] = number_of_output_records;
    output_header[5] = yyyymmdd;
    output_header[6] = number_of_data_frames;
    output_header[7] = earliest_date;
    output_header[8] = latest_date;

    if (SWAP_OUT == TRUE)
      {
      for (i=0; i<HDDF_HEADER_VALUES; i++)
        output_header[i] = htonl(output_header[i]);
      }

    fseek(out, 0, SEEK_SET);
    fwrite(&output_header, HDDF_RECORD_LENGTH, 1, out);

    fclose(out);
    }  // if ((out=fopen("out.file...

  printf("\nProcessing completed.\n\n");
  }



// ===============================================================================
void createAndInitializeOutputFile()
  {
  int  n;
  int  output_header[HDDF_DATA_VALUES];
  FILE *out;

  // Create a new output file

  printf("Creating and initializing the output file\n\n");

  if ((out=fopen("out.file", "w")) != NULL)
    {
    for (n=0; n<HDDF_HEADER_VALUES; n++)
      output_header[n] = -32768;

    output_header[0] = 72686870;
    output_header[1] = 10;
    output_header[2] = 1;
    output_header[3] = HDDF_RECORD_LENGTH;
    output_header[4] = HDDF_RECORD_LENGTH;
    output_header[5] = -32768;
    output_header[6] = 0;
    output_header[7] = -32768;
    output_header[8] = -32768;

    if (SWAP_OUT == TRUE)
      {
      for (n=0; n<HDDF_HEADER_VALUES; n++)
        output_header[n] = htonl(output_header[n]);
      }

    fseek(out, 0, SEEK_SET);
    fwrite(&output_header, HDDF_RECORD_LENGTH, 1, out);
    }

  fclose(out);
  }



// ===============================================================================
void processNetCDF(int date_to_process, int time_to_process, FILE *out)
  {
  int     n, x, y, status, level;
  int     grid_size_x, grid_size_y;
  int     buffer_size, num_footprints_processed;
  int     ncid, var_id;
  int     num_dims, num_attributes, dim_ids[MAX_VAR_DIMS];
  int     num_footprints, footprint;
  int     num_in_bin, bin_number;
  int     *num_in_row;
  long    offset;
  short   i2val, buffer[HDDF_DATA_VALUES];
  float   bin_lat, bin_lon, fval, dwpt;

  size_t start_2d[2], count_2d[2];
  size_t start_3d[3], count_3d[3];
  size_t  dim_len;
  nc_type var_type;

  double  *lats, *lons;
  float *surf_pressure, *surf_temperature, *precip_rate, *total_precip, *tpw;
  float *cape_surf, *cape_255, *cape_180, *cape_90, *cape_3000;
  float *cin_surf, *cin_255, *cin_180, *cin_90, *li_500, *li_4layer;
  float *low_cloud, *mid_cloud, *high_cloud, *total_cloud, *cloud_base, *cloud_top;
  float *terrain, *ice, *snow;
  float *temp_50, *temp_75, *temp_100, *temp_125, *temp_150, *temp_175, *temp_200, *temp_225;
  float *temp_250, *temp_275, *temp_300, *temp_325, *temp_350, *temp_375, *temp_400;
  float *temp_425, *temp_450, *temp_475, *temp_500, *temp_525, *temp_550, *temp_575;
  float *temp_600, *temp_625, *temp_650, *temp_675, *temp_700, *temp_725, *temp_750;
  float *temp_775, *temp_800, *temp_825, *temp_850, *temp_875, *temp_900, *temp_925;
  float *temp_950, *temp_975, *temp_1000, *temp_1013;
  float *rh_50, *rh_75, *rh_100, *rh_125, *rh_150, *rh_175, *rh_200, *rh_225;
  float *rh_250, *rh_275, *rh_300, *rh_325, *rh_350, *rh_375, *rh_400;
  float *rh_425, *rh_450, *rh_475, *rh_500, *rh_525, *rh_550, *rh_575;
  float *rh_600, *rh_625, *rh_650, *rh_675, *rh_700, *rh_725, *rh_750;
  float *rh_775, *rh_800, *rh_825, *rh_850, *rh_875, *rh_900, *rh_925;
  float *rh_950, *rh_975, *rh_1000, *rh_1013;
  float *dwpt_50, *dwpt_75, *dwpt_100, *dwpt_125, *dwpt_150, *dwpt_175, *dwpt_200, *dwpt_225;
  float *dwpt_250, *dwpt_275, *dwpt_300, *dwpt_325, *dwpt_350, *dwpt_375, *dwpt_400;
  float *dwpt_425, *dwpt_450, *dwpt_475, *dwpt_500, *dwpt_525, *dwpt_550, *dwpt_575;
  float *dwpt_600, *dwpt_625, *dwpt_650, *dwpt_675, *dwpt_700, *dwpt_725, *dwpt_750;
  float *dwpt_775, *dwpt_800, *dwpt_825, *dwpt_850, *dwpt_875, *dwpt_900, *dwpt_925;
  float *dwpt_950, *dwpt_975, *dwpt_1000, *dwpt_1013;


  float pressures[] = { 50.0, 75.0, 100.0, 125.0, 150.0, 175.0, 200.0, 225.0,
                        250.0, 275.0, 300.0, 325.0, 350.0, 375.0, 400.0, 425.0,
                        450.0, 475.0, 500.0, 525.0, 550.0, 575.0, 600.0, 625.0,
                        650.0, 675.0, 700.0, 725.0, 750.0, 775.0, 800.0, 825.0,
                        850.0, 875.0, 900.0, 925.0, 950.0, 975.0, 1000.0, 1013.2 };


float min, max;
min =  9999999999.9;
max = -9999999999.9;

  // Open the input file

  status = nc_open("in.file", 0, &ncid);

  if (status == NC_NOERR)
    {

    // Get the variable ids for all of the variables

    getVariableIDS(ncid);

    // Find the size of the grid

    //status = nc_inq_varid(ncid, "latitude", &var_id);
    status = ncvarinq(ncid, lat_id, 0, &var_type, &num_dims, dim_ids, &num_attributes);

    num_footprints = 1;

    for (n=0; n<num_dims; n++)
      {
      status = nc_inq_dimlen(ncid, dim_ids[n], &dim_len);
      num_footprints = num_footprints * (int)dim_len;

      if (n == 0)
        grid_size_y = (int)dim_len;
      else
        grid_size_x = (int)dim_len;
      }

    num_in_row = (int*)malloc(grid_size_y*sizeof(int));

    for (y=0; y<grid_size_y; y++)
      num_in_row[y] = 0;

    printf("\nGrid Size:   x = %d     y = %d\n", grid_size_x, grid_size_y);

    // Allocate memory for each parameter

    lats = (double*)malloc(grid_size_x*sizeof(double));
    lons = (double*)malloc(grid_size_x*sizeof(double));
    surf_pressure = (float*)malloc(grid_size_x*sizeof(float));
    surf_temperature = (float*)malloc(grid_size_x*sizeof(float));
    precip_rate = (float*)malloc(grid_size_x*sizeof(float));
    total_precip = (float*)malloc(grid_size_x*sizeof(float));
    tpw = (float*)malloc(grid_size_x*sizeof(float));
    cape_surf = (float*)malloc(grid_size_x*sizeof(float));
    cape_255 = (float*)malloc(grid_size_x*sizeof(float));
    cape_180 = (float*)malloc(grid_size_x*sizeof(float));
    cape_90 = (float*)malloc(grid_size_x*sizeof(float));
    cape_3000 = (float*)malloc(grid_size_x*sizeof(float));
    cin_surf = (float*)malloc(grid_size_x*sizeof(float));
    cin_255 = (float*)malloc(grid_size_x*sizeof(float));
    cin_180 = (float*)malloc(grid_size_x*sizeof(float));
    cin_90 = (float*)malloc(grid_size_x*sizeof(float));
    li_500 = (float*)malloc(grid_size_x*sizeof(float));
    li_4layer = (float*)malloc(grid_size_x*sizeof(float));
    low_cloud = (float*)malloc(grid_size_x*sizeof(float));
    mid_cloud = (float*)malloc(grid_size_x*sizeof(float));
    high_cloud = (float*)malloc(grid_size_x*sizeof(float));
    total_cloud = (float*)malloc(grid_size_x*sizeof(float));
    cloud_base = (float*)malloc(grid_size_x*sizeof(float));
    cloud_top = (float*)malloc(grid_size_x*sizeof(float));
    terrain = (float*)malloc(grid_size_x*sizeof(float));
    ice = (float*)malloc(grid_size_x*sizeof(float));
    snow = (float*)malloc(grid_size_x*sizeof(float));
    temp_50 = (float*)malloc(grid_size_x*sizeof(float));
    temp_75 = (float*)malloc(grid_size_x*sizeof(float));
    temp_100 = (float*)malloc(grid_size_x*sizeof(float));
    temp_125 = (float*)malloc(grid_size_x*sizeof(float));
    temp_150 = (float*)malloc(grid_size_x*sizeof(float));
    temp_175 = (float*)malloc(grid_size_x*sizeof(float));
    temp_200 = (float*)malloc(grid_size_x*sizeof(float));
    temp_225 = (float*)malloc(grid_size_x*sizeof(float));
    temp_250 = (float*)malloc(grid_size_x*sizeof(float));
    temp_275 = (float*)malloc(grid_size_x*sizeof(float));
    temp_300 = (float*)malloc(grid_size_x*sizeof(float));
    temp_325 = (float*)malloc(grid_size_x*sizeof(float));
    temp_350 = (float*)malloc(grid_size_x*sizeof(float));
    temp_375 = (float*)malloc(grid_size_x*sizeof(float));
    temp_400 = (float*)malloc(grid_size_x*sizeof(float));
    temp_425 = (float*)malloc(grid_size_x*sizeof(float));
    temp_450 = (float*)malloc(grid_size_x*sizeof(float));
    temp_475 = (float*)malloc(grid_size_x*sizeof(float));
    temp_500 = (float*)malloc(grid_size_x*sizeof(float));
    temp_525 = (float*)malloc(grid_size_x*sizeof(float));
    temp_550 = (float*)malloc(grid_size_x*sizeof(float));
    temp_575 = (float*)malloc(grid_size_x*sizeof(float));
    temp_600 = (float*)malloc(grid_size_x*sizeof(float));
    temp_625 = (float*)malloc(grid_size_x*sizeof(float));
    temp_650 = (float*)malloc(grid_size_x*sizeof(float));
    temp_675 = (float*)malloc(grid_size_x*sizeof(float));
    temp_700 = (float*)malloc(grid_size_x*sizeof(float));
    temp_725 = (float*)malloc(grid_size_x*sizeof(float));
    temp_750 = (float*)malloc(grid_size_x*sizeof(float));
    temp_775 = (float*)malloc(grid_size_x*sizeof(float));
    temp_800 = (float*)malloc(grid_size_x*sizeof(float));
    temp_825 = (float*)malloc(grid_size_x*sizeof(float));
    temp_850 = (float*)malloc(grid_size_x*sizeof(float));
    temp_875 = (float*)malloc(grid_size_x*sizeof(float));
    temp_900 = (float*)malloc(grid_size_x*sizeof(float));
    temp_925 = (float*)malloc(grid_size_x*sizeof(float));
    temp_950 = (float*)malloc(grid_size_x*sizeof(float));
    temp_975 = (float*)malloc(grid_size_x*sizeof(float));
    temp_1000 = (float*)malloc(grid_size_x*sizeof(float));
    temp_1013 = (float*)malloc(grid_size_x*sizeof(float));
    rh_50 = (float*)malloc(grid_size_x*sizeof(float));
    rh_75 = (float*)malloc(grid_size_x*sizeof(float));
    rh_100 = (float*)malloc(grid_size_x*sizeof(float));
    rh_125 = (float*)malloc(grid_size_x*sizeof(float));
    rh_150 = (float*)malloc(grid_size_x*sizeof(float));
    rh_175 = (float*)malloc(grid_size_x*sizeof(float));
    rh_200 = (float*)malloc(grid_size_x*sizeof(float));
    rh_225 = (float*)malloc(grid_size_x*sizeof(float));
    rh_250 = (float*)malloc(grid_size_x*sizeof(float));
    rh_275 = (float*)malloc(grid_size_x*sizeof(float));
    rh_300 = (float*)malloc(grid_size_x*sizeof(float));
    rh_325 = (float*)malloc(grid_size_x*sizeof(float));
    rh_350 = (float*)malloc(grid_size_x*sizeof(float));
    rh_375 = (float*)malloc(grid_size_x*sizeof(float));
    rh_400 = (float*)malloc(grid_size_x*sizeof(float));
    rh_425 = (float*)malloc(grid_size_x*sizeof(float));
    rh_450 = (float*)malloc(grid_size_x*sizeof(float));
    rh_475 = (float*)malloc(grid_size_x*sizeof(float));
    rh_500 = (float*)malloc(grid_size_x*sizeof(float));
    rh_525 = (float*)malloc(grid_size_x*sizeof(float));
    rh_550 = (float*)malloc(grid_size_x*sizeof(float));
    rh_575 = (float*)malloc(grid_size_x*sizeof(float));
    rh_600 = (float*)malloc(grid_size_x*sizeof(float));
    rh_625 = (float*)malloc(grid_size_x*sizeof(float));
    rh_650 = (float*)malloc(grid_size_x*sizeof(float));
    rh_675 = (float*)malloc(grid_size_x*sizeof(float));
    rh_700 = (float*)malloc(grid_size_x*sizeof(float));
    rh_725 = (float*)malloc(grid_size_x*sizeof(float));
    rh_750 = (float*)malloc(grid_size_x*sizeof(float));
    rh_775 = (float*)malloc(grid_size_x*sizeof(float));
    rh_800 = (float*)malloc(grid_size_x*sizeof(float));
    rh_825 = (float*)malloc(grid_size_x*sizeof(float));
    rh_850 = (float*)malloc(grid_size_x*sizeof(float));
    rh_875 = (float*)malloc(grid_size_x*sizeof(float));
    rh_900 = (float*)malloc(grid_size_x*sizeof(float));
    rh_925 = (float*)malloc(grid_size_x*sizeof(float));
    rh_950 = (float*)malloc(grid_size_x*sizeof(float));
    rh_975 = (float*)malloc(grid_size_x*sizeof(float));
    rh_1000 = (float*)malloc(grid_size_x*sizeof(float));
    rh_1013 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_50 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_75 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_100 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_125 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_150 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_175 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_200 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_225 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_250 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_275 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_300 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_325 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_350 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_375 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_400 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_425 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_450 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_475 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_500 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_525 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_550 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_575 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_600 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_625 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_650 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_675 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_700 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_725 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_750 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_775 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_800 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_825 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_850 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_875 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_900 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_925 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_950 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_975 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_1000 = (float*)malloc(grid_size_x*sizeof(float));
    dwpt_1013 = (float*)malloc(grid_size_x*sizeof(float));


    // Loop through the bins and create data frames for each bin that has
    // at least one grid cell in it

    bin_number = 0;

    for (bin_lat=80.0; bin_lat>20.0; bin_lat-=BIN_SIZE)
      {
      for (bin_lon=-180.0; bin_lon<180.0; bin_lon+=BIN_SIZE)
        {
	if ((bin_lon < -60.0) || (bin_lon >= 155.0))
          {

          // Count the number of footprints in the bin

          num_in_bin = 0;

          for (y=0; y<grid_size_y; y++)
            {
            num_in_row[y] = 0;

            start_2d[0] = y;
            start_2d[1] = 0;

            count_2d[0] = 1;
            count_2d[1] = grid_size_x;

            start_3d[0] = 0;
            start_3d[1] = y;
            start_3d[2] = 0;  // x

            count_3d[0] = 1;
            count_3d[1] = grid_size_x;
            count_3d[2] = 1;

            status = nc_get_vara_double(ncid, lat_id, start_2d, count_2d, lats);
            status = nc_get_vara_double(ncid, lon_id, start_2d, count_2d, lons);

            for (x=0; x<grid_size_x; x++)
              {
              if (lons[x] > 180.0)
                lons[x] = -180.0 + (lons[x] - 180.0);

              if (pointIsWithinBin(lats[x], lons[x], bin_lat, bin_lon) == TRUE)
                {
                num_in_bin++;
                num_in_row[y]++;
                }
              }  // for (x=0...
            }  // for (y=0...


          // Process this bin if there is at least one point within it

          if (num_in_bin > 0)
            {
            bin_number++;

            printf("BIN:  %d   %f   %f    %d\n", bin_number, bin_lat, bin_lon, num_in_bin);

            // Create a data frame header

            for (n=0; n<HDDF_DATA_VALUES; n++)
              buffer[n] = -32768;

            buffer[0] = 3;
            buffer[1] = bin_number;
            buffer[2] = num_in_bin;

            buffer[4] = (short)((bin_lat - BIN_SIZE) * 128.0);
            buffer[5] = (short)((bin_lat - BIN_SIZE) * 128.0);
            buffer[6] = (short)((bin_lat) * 128.0);
            buffer[7] = (short)((bin_lat) * 128.0);

            buffer[8]  = (short)((bin_lon) * 128.0);
            buffer[9]  = (short)((bin_lon + BIN_SIZE) * 128.0);
            buffer[10] = (short)((bin_lon + BIN_SIZE) * 128.0);
            buffer[11] = (short)((bin_lon) * 128.0);

            buffer[12] = date_to_process / 10000;
            buffer[13] = date_to_process % 10000;
            buffer[14] = time_to_process * 100;

            number_of_output_records++;

            offset = (long)(number_of_output_records - 1) * (long)HDDF_RECORD_LENGTH;
            fseek(out, offset, SEEK_SET);

            fwrite(&buffer, HDDF_RECORD_LENGTH, 1, out);


            // Process the footprints that fall within the bin

            for (y=0; y<grid_size_y; y++)
              {
              if (num_in_row[y] > 0)
                {
                start_2d[0] = y;
                start_2d[1] = 0;

                count_2d[0] = 1;
                count_2d[1] = grid_size_x;

                start_3d[0] = 0;
                start_3d[1] = y;
                start_3d[2] = 0;  // x

                count_3d[0] = 1;
                count_3d[1] = 1;
                count_3d[2] = grid_size_x;

                status = nc_get_vara_double(ncid, lat_id, start_2d, count_2d, lats);
                status = nc_get_vara_double(ncid, lon_id, start_2d, count_2d, lons);
                status = nc_get_vara_float(ncid, terrain_id, start_3d, count_3d, terrain);
                status = nc_get_vara_float(ncid, surf_press_id, start_3d, count_3d, surf_pressure);
                status = nc_get_vara_float(ncid, surf_temp_id, start_3d, count_3d, surf_temperature);
                status = nc_get_vara_float(ncid, prate_id, start_3d, count_3d, precip_rate);
                status = nc_get_vara_float(ncid, total_precip_id, start_3d, count_3d, total_precip);
                status = nc_get_vara_float(ncid, tpw_id, start_3d, count_3d, tpw);
                status = nc_get_vara_float(ncid, cape_surf_id, start_3d, count_3d, cape_surf);
                status = nc_get_vara_float(ncid, cape_255_id, start_3d, count_3d, cape_255);
                status = nc_get_vara_float(ncid, cape_180_id, start_3d, count_3d, cape_180);
                status = nc_get_vara_float(ncid, cape_90_id, start_3d, count_3d, cape_90);
                status = nc_get_vara_float(ncid, cape_3000_id, start_3d, count_3d, cape_3000);
                status = nc_get_vara_float(ncid, cin_surf_id, start_3d, count_3d, cin_surf);
                status = nc_get_vara_float(ncid, cin_255_id, start_3d, count_3d, cin_255);
                status = nc_get_vara_float(ncid, cin_180_id, start_3d, count_3d, cin_180);
                status = nc_get_vara_float(ncid, cin_90_id, start_3d, count_3d, cin_90);
                status = nc_get_vara_float(ncid, li_500_id, start_3d, count_3d, li_500);
                status = nc_get_vara_float(ncid, li_4layer_id, start_3d, count_3d, li_4layer);
                status = nc_get_vara_float(ncid, low_cloud_id, start_3d, count_3d, low_cloud);
                status = nc_get_vara_float(ncid, mid_cloud_id, start_3d, count_3d, mid_cloud);
                status = nc_get_vara_float(ncid, high_cloud_id, start_3d, count_3d, high_cloud);
                status = nc_get_vara_float(ncid, total_cloud_id, start_3d, count_3d, total_cloud);
                status = nc_get_vara_float(ncid, cloud_base_id, start_3d, count_3d, cloud_base);
                status = nc_get_vara_float(ncid, cloud_top_id, start_3d, count_3d, cloud_top);
                status = nc_get_vara_float(ncid, ice_id, start_3d, count_3d, ice);
                status = nc_get_vara_float(ncid, snow_id, start_3d, count_3d, snow);

                status = nc_get_vara_float(ncid, temp_ids[0], start_3d, count_3d, temp_50);
                status = nc_get_vara_float(ncid, temp_ids[1], start_3d, count_3d, temp_75);
                status = nc_get_vara_float(ncid, temp_ids[2], start_3d, count_3d, temp_100);
                status = nc_get_vara_float(ncid, temp_ids[3], start_3d, count_3d, temp_125);
                status = nc_get_vara_float(ncid, temp_ids[4], start_3d, count_3d, temp_150);
                status = nc_get_vara_float(ncid, temp_ids[5], start_3d, count_3d, temp_175);
                status = nc_get_vara_float(ncid, temp_ids[6], start_3d, count_3d, temp_200);
                status = nc_get_vara_float(ncid, temp_ids[7], start_3d, count_3d, temp_225);
                status = nc_get_vara_float(ncid, temp_ids[8], start_3d, count_3d, temp_250);
                status = nc_get_vara_float(ncid, temp_ids[9], start_3d, count_3d, temp_275);
                status = nc_get_vara_float(ncid, temp_ids[10], start_3d, count_3d, temp_300);
                status = nc_get_vara_float(ncid, temp_ids[11], start_3d, count_3d, temp_325);
                status = nc_get_vara_float(ncid, temp_ids[12], start_3d, count_3d, temp_350);
                status = nc_get_vara_float(ncid, temp_ids[13], start_3d, count_3d, temp_375);
                status = nc_get_vara_float(ncid, temp_ids[14], start_3d, count_3d, temp_400);
                status = nc_get_vara_float(ncid, temp_ids[15], start_3d, count_3d, temp_425);
                status = nc_get_vara_float(ncid, temp_ids[16], start_3d, count_3d, temp_450);
                status = nc_get_vara_float(ncid, temp_ids[17], start_3d, count_3d, temp_475);
                status = nc_get_vara_float(ncid, temp_ids[18], start_3d, count_3d, temp_500);
                status = nc_get_vara_float(ncid, temp_ids[19], start_3d, count_3d, temp_525);
                status = nc_get_vara_float(ncid, temp_ids[20], start_3d, count_3d, temp_550);
                status = nc_get_vara_float(ncid, temp_ids[21], start_3d, count_3d, temp_575);
                status = nc_get_vara_float(ncid, temp_ids[22], start_3d, count_3d, temp_600);
                status = nc_get_vara_float(ncid, temp_ids[23], start_3d, count_3d, temp_625);
                status = nc_get_vara_float(ncid, temp_ids[24], start_3d, count_3d, temp_650);
                status = nc_get_vara_float(ncid, temp_ids[25], start_3d, count_3d, temp_675);
                status = nc_get_vara_float(ncid, temp_ids[26], start_3d, count_3d, temp_700);
                status = nc_get_vara_float(ncid, temp_ids[27], start_3d, count_3d, temp_725);
                status = nc_get_vara_float(ncid, temp_ids[28], start_3d, count_3d, temp_750);
                status = nc_get_vara_float(ncid, temp_ids[29], start_3d, count_3d, temp_775);
                status = nc_get_vara_float(ncid, temp_ids[30], start_3d, count_3d, temp_800);
                status = nc_get_vara_float(ncid, temp_ids[31], start_3d, count_3d, temp_825);
                status = nc_get_vara_float(ncid, temp_ids[32], start_3d, count_3d, temp_850);
                status = nc_get_vara_float(ncid, temp_ids[33], start_3d, count_3d, temp_875);
                status = nc_get_vara_float(ncid, temp_ids[34], start_3d, count_3d, temp_900);
                status = nc_get_vara_float(ncid, temp_ids[35], start_3d, count_3d, temp_925);
                status = nc_get_vara_float(ncid, temp_ids[36], start_3d, count_3d, temp_950);
                status = nc_get_vara_float(ncid, temp_ids[37], start_3d, count_3d, temp_975);
                status = nc_get_vara_float(ncid, temp_ids[38], start_3d, count_3d, temp_1000);
                status = nc_get_vara_float(ncid, temp_ids[39], start_3d, count_3d, temp_1013);

                status = nc_get_vara_float(ncid, rh_ids[0], start_3d, count_3d, rh_50);
                status = nc_get_vara_float(ncid, rh_ids[1], start_3d, count_3d, rh_75);
                status = nc_get_vara_float(ncid, rh_ids[2], start_3d, count_3d, rh_100);
                status = nc_get_vara_float(ncid, rh_ids[3], start_3d, count_3d, rh_125);
                status = nc_get_vara_float(ncid, rh_ids[4], start_3d, count_3d, rh_150);
                status = nc_get_vara_float(ncid, rh_ids[5], start_3d, count_3d, rh_175);
                status = nc_get_vara_float(ncid, rh_ids[6], start_3d, count_3d, rh_200);
                status = nc_get_vara_float(ncid, rh_ids[7], start_3d, count_3d, rh_225);
                status = nc_get_vara_float(ncid, rh_ids[8], start_3d, count_3d, rh_250);
                status = nc_get_vara_float(ncid, rh_ids[9], start_3d, count_3d, rh_275);
                status = nc_get_vara_float(ncid, rh_ids[10], start_3d, count_3d, rh_300);
                status = nc_get_vara_float(ncid, rh_ids[11], start_3d, count_3d, rh_325);
                status = nc_get_vara_float(ncid, rh_ids[12], start_3d, count_3d, rh_350);
                status = nc_get_vara_float(ncid, rh_ids[13], start_3d, count_3d, rh_375);
                status = nc_get_vara_float(ncid, rh_ids[14], start_3d, count_3d, rh_400);
                status = nc_get_vara_float(ncid, rh_ids[15], start_3d, count_3d, rh_425);
                status = nc_get_vara_float(ncid, rh_ids[16], start_3d, count_3d, rh_450);
                status = nc_get_vara_float(ncid, rh_ids[17], start_3d, count_3d, rh_475);
                status = nc_get_vara_float(ncid, rh_ids[18], start_3d, count_3d, rh_500);
                status = nc_get_vara_float(ncid, rh_ids[19], start_3d, count_3d, rh_525);
                status = nc_get_vara_float(ncid, rh_ids[20], start_3d, count_3d, rh_550);
                status = nc_get_vara_float(ncid, rh_ids[21], start_3d, count_3d, rh_575);
                status = nc_get_vara_float(ncid, rh_ids[22], start_3d, count_3d, rh_600);
                status = nc_get_vara_float(ncid, rh_ids[23], start_3d, count_3d, rh_625);
                status = nc_get_vara_float(ncid, rh_ids[24], start_3d, count_3d, rh_650);
                status = nc_get_vara_float(ncid, rh_ids[25], start_3d, count_3d, rh_675);
                status = nc_get_vara_float(ncid, rh_ids[26], start_3d, count_3d, rh_700);
                status = nc_get_vara_float(ncid, rh_ids[27], start_3d, count_3d, rh_725);
                status = nc_get_vara_float(ncid, rh_ids[28], start_3d, count_3d, rh_750);
                status = nc_get_vara_float(ncid, rh_ids[29], start_3d, count_3d, rh_775);
                status = nc_get_vara_float(ncid, rh_ids[30], start_3d, count_3d, rh_800);
                status = nc_get_vara_float(ncid, rh_ids[31], start_3d, count_3d, rh_825);
                status = nc_get_vara_float(ncid, rh_ids[32], start_3d, count_3d, rh_850);
                status = nc_get_vara_float(ncid, rh_ids[33], start_3d, count_3d, rh_875);
                status = nc_get_vara_float(ncid, rh_ids[34], start_3d, count_3d, rh_900);
                status = nc_get_vara_float(ncid, rh_ids[35], start_3d, count_3d, rh_925);
                status = nc_get_vara_float(ncid, rh_ids[36], start_3d, count_3d, rh_950);
                status = nc_get_vara_float(ncid, rh_ids[37], start_3d, count_3d, rh_975);
                status = nc_get_vara_float(ncid, rh_ids[38], start_3d, count_3d, rh_1000);
                status = nc_get_vara_float(ncid, rh_ids[39], start_3d, count_3d, rh_1013);

                status = nc_get_vara_float(ncid, dwpt_ids[0], start_3d, count_3d, dwpt_50);
                status = nc_get_vara_float(ncid, dwpt_ids[1], start_3d, count_3d, dwpt_75);
                status = nc_get_vara_float(ncid, dwpt_ids[2], start_3d, count_3d, dwpt_100);
                status = nc_get_vara_float(ncid, dwpt_ids[3], start_3d, count_3d, dwpt_125);
                status = nc_get_vara_float(ncid, dwpt_ids[4], start_3d, count_3d, dwpt_150);
                status = nc_get_vara_float(ncid, dwpt_ids[5], start_3d, count_3d, dwpt_175);
                status = nc_get_vara_float(ncid, dwpt_ids[6], start_3d, count_3d, dwpt_200);
                status = nc_get_vara_float(ncid, dwpt_ids[7], start_3d, count_3d, dwpt_225);
                status = nc_get_vara_float(ncid, dwpt_ids[8], start_3d, count_3d, dwpt_250);
                status = nc_get_vara_float(ncid, dwpt_ids[9], start_3d, count_3d, dwpt_275);
                status = nc_get_vara_float(ncid, dwpt_ids[10], start_3d, count_3d, dwpt_300);
                status = nc_get_vara_float(ncid, dwpt_ids[11], start_3d, count_3d, dwpt_325);
                status = nc_get_vara_float(ncid, dwpt_ids[12], start_3d, count_3d, dwpt_350);
                status = nc_get_vara_float(ncid, dwpt_ids[13], start_3d, count_3d, dwpt_375);
                status = nc_get_vara_float(ncid, dwpt_ids[14], start_3d, count_3d, dwpt_400);
                status = nc_get_vara_float(ncid, dwpt_ids[15], start_3d, count_3d, dwpt_425);
                status = nc_get_vara_float(ncid, dwpt_ids[16], start_3d, count_3d, dwpt_450);
                status = nc_get_vara_float(ncid, dwpt_ids[17], start_3d, count_3d, dwpt_475);
                status = nc_get_vara_float(ncid, dwpt_ids[18], start_3d, count_3d, dwpt_500);
                status = nc_get_vara_float(ncid, dwpt_ids[19], start_3d, count_3d, dwpt_525);
                status = nc_get_vara_float(ncid, dwpt_ids[20], start_3d, count_3d, dwpt_550);
                status = nc_get_vara_float(ncid, dwpt_ids[21], start_3d, count_3d, dwpt_575);
                status = nc_get_vara_float(ncid, dwpt_ids[22], start_3d, count_3d, dwpt_600);
                status = nc_get_vara_float(ncid, dwpt_ids[23], start_3d, count_3d, dwpt_625);
                status = nc_get_vara_float(ncid, dwpt_ids[24], start_3d, count_3d, dwpt_650);
                status = nc_get_vara_float(ncid, dwpt_ids[25], start_3d, count_3d, dwpt_675);
                status = nc_get_vara_float(ncid, dwpt_ids[26], start_3d, count_3d, dwpt_700);
                status = nc_get_vara_float(ncid, dwpt_ids[27], start_3d, count_3d, dwpt_725);
                status = nc_get_vara_float(ncid, dwpt_ids[28], start_3d, count_3d, dwpt_750);
                status = nc_get_vara_float(ncid, dwpt_ids[29], start_3d, count_3d, dwpt_775);
                status = nc_get_vara_float(ncid, dwpt_ids[30], start_3d, count_3d, dwpt_800);
                status = nc_get_vara_float(ncid, dwpt_ids[31], start_3d, count_3d, dwpt_825);
                status = nc_get_vara_float(ncid, dwpt_ids[32], start_3d, count_3d, dwpt_850);
                status = nc_get_vara_float(ncid, dwpt_ids[33], start_3d, count_3d, dwpt_875);
                status = nc_get_vara_float(ncid, dwpt_ids[34], start_3d, count_3d, dwpt_900);
                status = nc_get_vara_float(ncid, dwpt_ids[35], start_3d, count_3d, dwpt_925);
                status = nc_get_vara_float(ncid, dwpt_ids[36], start_3d, count_3d, dwpt_950);
                status = nc_get_vara_float(ncid, dwpt_ids[37], start_3d, count_3d, dwpt_975);
                status = nc_get_vara_float(ncid, dwpt_ids[38], start_3d, count_3d, dwpt_1000);
                status = nc_get_vara_float(ncid, dwpt_ids[39], start_3d, count_3d, dwpt_1013);

                // Loop through the footprints in the row

                for (x=0; x<grid_size_x; x++)
                  {
                  if (lons[x] > 180.0)
                    lons[x] = -180.0 + (lons[x] - 180.0);

                  if (pointIsWithinBin(lats[x], lons[x], bin_lat, bin_lon) == TRUE)
                    {
                    for (n=0; n<HDDF_DATA_VALUES; n++)
                      buffer[n] = -32768;

                    buffer[0] = 0;

                    // Latitude and longitude

                    buffer[1] = (short)(lats[x] / 1);
                    buffer[2] = (short)((lats[x] - (int)lats[x]) * 10000.0);

                    buffer[3] = (short)(lons[x] / 1);
                    buffer[4] = (short)((lons[x] - (int)lons[x]) * 10000.0);

                    // Date and time

                    buffer[5]  = date_to_process / 10000;
                    buffer[6]  = date_to_process % 10000;
                    buffer[7]  = time_to_process;
                    buffer[8] = 0;

                    // Terrain

                    buffer[11] = (short)terrain[x];

                    // Ice cover

                    buffer[12] = (short)ice[x];

                    // Snow cover

                    buffer[13] = (short)(snow[x] * 100.0);

                    // Precipitation rate

                    buffer[14] = (short)(precip_rate[x] * 100.0);

                    // Total precipitation

                    buffer[15] = (short)(total_precip[x] * 100.0);

                    // TPW

                    buffer[16] = (short)(tpw[x] * 100.0);

                    // CAPE: surface

                    buffer[17] = (short)(cape_surf[x] * 1.0);

                    // CAPE: 255mb

                    buffer[18] = (short)(cape_255[x] * 1.0);

                    // CAPE: 180mb

                    buffer[19] = (short)(cape_180[x] * 1.0);

                    // CAPE: 90mb

                    buffer[20] = (short)(cape_90[x] * 1.0);

                    // CAPE: 3000m

                    buffer[21] = (short)(cape_3000[x] * 1.0);

                    // CIN: surface

                    buffer[22] = (short)(cin_surf[x] * 10.0);

                    // CAPE: 255mb

                    buffer[23] = (short)(cin_255[x] * 10.0);

                    // CIN: 180mb

                    buffer[24] = (short)(cin_180[x] * 10.0);

                    // CIN: 90mb

                    buffer[25] = (short)(cin_90[x] * 10.0);

                    // LI: 500mb

                    buffer[26] = (short)(li_500[x] * 10.0);

                    // LI: 4 layer

                    buffer[27] = (short)(li_4layer[x] * 1.0);

                    // Low cloud cover

                    buffer[28] = (short)(low_cloud[x] * 10.0);

                    // Mid cloud cover

                    buffer[29] = (short)(mid_cloud[x] * 10.0);

                    // High cloud cover

                    buffer[30] = (short)(high_cloud[x] * 10.0);

                    // Total cloud cover

                    buffer[31] = (short)(total_cloud[x] * 10.0);

                    // Cloud top pressure

                    buffer[32] = (short)(cloud_top[x] / 100.0 * 10.0);

                    // Cloud base pressure

                    buffer[33] = (short)(cloud_base[x] / 100.0 * 10.0);

                    // Surface pressure

                    buffer[34] = (short)(surf_pressure[x] / 100.0 * 10.0);

                    // Surface temperature

                    buffer[35] = (short)(surf_temperature[x] * 64.0);

                    // Pressures

                    for (level=0; level<40; level++)
                      {
                      buffer[36+level] = (short)(pressures[level] * 10.0);
                      }

                    // Temperatures

                    buffer[76] = (short)(temp_50[x] * 64.0);
                    buffer[77] = (short)(temp_75[x] * 64.0);
                    buffer[78] = (short)(temp_100[x] * 64.0);
                    buffer[79] = (short)(temp_125[x] * 64.0);
                    buffer[80] = (short)(temp_150[x] * 64.0);
                    buffer[81] = (short)(temp_175[x] * 64.0);
                    buffer[82] = (short)(temp_200[x] * 64.0);
                    buffer[83] = (short)(temp_225[x] * 64.0);
                    buffer[84] = (short)(temp_250[x] * 64.0);
                    buffer[85] = (short)(temp_275[x] * 64.0);
                    buffer[86] = (short)(temp_300[x] * 64.0);
                    buffer[87] = (short)(temp_325[x] * 64.0);
                    buffer[88] = (short)(temp_350[x] * 64.0);
                    buffer[89] = (short)(temp_375[x] * 64.0);
                    buffer[90] = (short)(temp_400[x] * 64.0);
                    buffer[91] = (short)(temp_425[x] * 64.0);
                    buffer[92] = (short)(temp_450[x] * 64.0);
                    buffer[93] = (short)(temp_475[x] * 64.0);
                    buffer[94] = (short)(temp_500[x] * 64.0);
                    buffer[95] = (short)(temp_525[x] * 64.0);
                    buffer[96] = (short)(temp_550[x] * 64.0);
                    buffer[97] = (short)(temp_575[x] * 64.0);
                    buffer[98] = (short)(temp_600[x] * 64.0);
                    buffer[99] = (short)(temp_625[x] * 64.0);
                    buffer[100] = (short)(temp_650[x] * 64.0);
                    buffer[101] = (short)(temp_675[x] * 64.0);
                    buffer[102] = (short)(temp_700[x] * 64.0);
                    buffer[103] = (short)(temp_725[x] * 64.0);
                    buffer[104] = (short)(temp_750[x] * 64.0);
                    buffer[105] = (short)(temp_775[x] * 64.0);
                    buffer[106] = (short)(temp_800[x] * 64.0);
                    buffer[107] = (short)(temp_825[x] * 64.0);
                    buffer[108] = (short)(temp_850[x] * 64.0);
                    buffer[109] = (short)(temp_875[x] * 64.0);
                    buffer[110] = (short)(temp_900[x] * 64.0);
                    buffer[111] = (short)(temp_925[x] * 64.0);
                    buffer[112] = (short)(temp_950[x] * 64.0);
                    buffer[113] = (short)(temp_975[x] * 64.0);
                    buffer[114] = (short)(temp_1000[x] * 64.0);
                    buffer[115] = (short)(temp_1013[x] * 64.0);

                    // Water vapors

                    buffer[116] = (short)(log(convertToWVMR(pressures[0],dwpt_50[x])) * 1024.0);
                    buffer[117] = (short)(log(convertToWVMR(pressures[1],dwpt_75[x])) * 1024.0);
                    buffer[118] = (short)(log(convertToWVMR(pressures[2],dwpt_100[x])) * 1024.0);
                    buffer[119] = (short)(log(convertToWVMR(pressures[3],dwpt_125[x])) * 1024.0);
                    buffer[120] = (short)(log(convertToWVMR(pressures[4],dwpt_150[x])) * 1024.0);
                    buffer[121] = (short)(log(convertToWVMR(pressures[5],dwpt_175[x])) * 1024.0);
                    buffer[122] = (short)(log(convertToWVMR(pressures[6],dwpt_200[x])) * 1024.0);
                    buffer[123] = (short)(log(convertToWVMR(pressures[7],dwpt_225[x])) * 1024.0);
                    buffer[124] = (short)(log(convertToWVMR(pressures[8],dwpt_250[x])) * 1024.0);
                    buffer[125] = (short)(log(convertToWVMR(pressures[9],dwpt_275[x])) * 1024.0);
                    buffer[126] = (short)(log(convertToWVMR(pressures[10],dwpt_300[x])) * 1024.0);
                    buffer[127] = (short)(log(convertToWVMR(pressures[11],dwpt_325[x])) * 1024.0);
                    buffer[128] = (short)(log(convertToWVMR(pressures[12],dwpt_350[x])) * 1024.0);
                    buffer[129] = (short)(log(convertToWVMR(pressures[13],dwpt_375[x])) * 1024.0);
                    buffer[130] = (short)(log(convertToWVMR(pressures[14],dwpt_400[x])) * 1024.0);
                    buffer[131] = (short)(log(convertToWVMR(pressures[15],dwpt_425[x])) * 1024.0);
                    buffer[132] = (short)(log(convertToWVMR(pressures[16],dwpt_450[x])) * 1024.0);
                    buffer[133] = (short)(log(convertToWVMR(pressures[17],dwpt_475[x])) * 1024.0);
                    buffer[134] = (short)(log(convertToWVMR(pressures[18],dwpt_500[x])) * 1024.0);
                    buffer[135] = (short)(log(convertToWVMR(pressures[19],dwpt_525[x])) * 1024.0);
                    buffer[136] = (short)(log(convertToWVMR(pressures[20],dwpt_550[x])) * 1024.0);
                    buffer[137] = (short)(log(convertToWVMR(pressures[21],dwpt_575[x])) * 1024.0);
                    buffer[138] = (short)(log(convertToWVMR(pressures[22],dwpt_600[x])) * 1024.0);
                    buffer[139] = (short)(log(convertToWVMR(pressures[23],dwpt_625[x])) * 1024.0);
                    buffer[140] = (short)(log(convertToWVMR(pressures[24],dwpt_650[x])) * 1024.0);
                    buffer[141] = (short)(log(convertToWVMR(pressures[25],dwpt_675[x])) * 1024.0);
                    buffer[142] = (short)(log(convertToWVMR(pressures[26],dwpt_700[x])) * 1024.0);
                    buffer[143] = (short)(log(convertToWVMR(pressures[27],dwpt_725[x])) * 1024.0);
                    buffer[144] = (short)(log(convertToWVMR(pressures[28],dwpt_750[x])) * 1024.0);
                    buffer[145] = (short)(log(convertToWVMR(pressures[29],dwpt_775[x])) * 1024.0);
                    buffer[146] = (short)(log(convertToWVMR(pressures[30],dwpt_800[x])) * 1024.0);
                    buffer[147] = (short)(log(convertToWVMR(pressures[31],dwpt_825[x])) * 1024.0);
                    buffer[148] = (short)(log(convertToWVMR(pressures[32],dwpt_850[x])) * 1024.0);
                    buffer[149] = (short)(log(convertToWVMR(pressures[33],dwpt_875[x])) * 1024.0);
                    buffer[150] = (short)(log(convertToWVMR(pressures[34],dwpt_900[x])) * 1024.0);
                    buffer[151] = (short)(log(convertToWVMR(pressures[35],dwpt_925[x])) * 1024.0);
                    buffer[152] = (short)(log(convertToWVMR(pressures[36],dwpt_950[x])) * 1024.0);
                    buffer[153] = (short)(log(convertToWVMR(pressures[37],dwpt_975[x])) * 1024.0);
                    buffer[154] = (short)(log(convertToWVMR(pressures[38],dwpt_1000[x])) * 1024.0);
                    buffer[155] = (short)(log(convertToWVMR(pressures[39],dwpt_1013[x])) * 1024.0);

                    // Relative humidities

                    buffer[156] = (short)(rh_50[x] * 100.0);
                    buffer[157] = (short)(rh_75[x] * 100.0);
                    buffer[158] = (short)(rh_100[x] * 100.0);
                    buffer[159] = (short)(rh_125[x] * 100.0);
                    buffer[160] = (short)(rh_150[x] * 100.0);
                    buffer[161] = (short)(rh_175[x] * 100.0);
                    buffer[162] = (short)(rh_200[x] * 100.0);
                    buffer[163] = (short)(rh_225[x] * 100.0);
                    buffer[164] = (short)(rh_250[x] * 100.0);
                    buffer[165] = (short)(rh_275[x] * 100.0);
                    buffer[166] = (short)(rh_300[x] * 100.0);
                    buffer[167] = (short)(rh_325[x] * 100.0);
                    buffer[168] = (short)(rh_350[x] * 100.0);
                    buffer[169] = (short)(rh_375[x] * 100.0);
                    buffer[170] = (short)(rh_400[x] * 100.0);
                    buffer[171] = (short)(rh_425[x] * 100.0);
                    buffer[172] = (short)(rh_450[x] * 100.0);
                    buffer[173] = (short)(rh_475[x] * 100.0);
                    buffer[174] = (short)(rh_500[x] * 100.0);
                    buffer[175] = (short)(rh_525[x] * 100.0);
                    buffer[176] = (short)(rh_550[x] * 100.0);
                    buffer[177] = (short)(rh_575[x] * 100.0);
                    buffer[178] = (short)(rh_600[x] * 100.0);
                    buffer[179] = (short)(rh_625[x] * 100.0);
                    buffer[180] = (short)(rh_650[x] * 100.0);
                    buffer[181] = (short)(rh_675[x] * 100.0);
                    buffer[182] = (short)(rh_700[x] * 100.0);
                    buffer[183] = (short)(rh_725[x] * 100.0);
                    buffer[184] = (short)(rh_750[x] * 100.0);
                    buffer[185] = (short)(rh_775[x] * 100.0);
                    buffer[186] = (short)(rh_800[x] * 100.0);
                    buffer[187] = (short)(rh_825[x] * 100.0);
                    buffer[188] = (short)(rh_850[x] * 100.0);
                    buffer[189] = (short)(rh_875[x] * 100.0);
                    buffer[190] = (short)(rh_900[x] * 100.0);
                    buffer[191] = (short)(rh_925[x] * 100.0);
                    buffer[192] = (short)(rh_950[x] * 100.0);
                    buffer[193] = (short)(rh_975[x] * 100.0);
                    buffer[194] = (short)(rh_1000[x] * 100.0);
                    buffer[195] = (short)(rh_1013[x] * 100.0);

                    // Write the record to the output file

                    number_of_output_records++;

                    offset = (long)(number_of_output_records - 1) * (long)HDDF_RECORD_LENGTH;
                    fseek(out, offset, SEEK_SET);
                    fwrite(&buffer, HDDF_RECORD_LENGTH, 1, out);
                    }
                  }  // for (x=0...
                }  // if (num_in_row[y]...
              }  // for (y=0;;;
            }  // if (num_in_bin > 0...
          }  // if (bin_lon < 60...
        }  // for (bin_lon=0...
      }  // for (bin_lat=90...


    // Free the memory that was allocated for the parameters

    free(lats);
    free(lons);
    free(surf_pressure);
    free(surf_temperature);
    free(precip_rate);
    free(total_precip);
    free(tpw);
    free(cape_surf);
    free(cape_255);
    free(cape_180);
    free(cape_90);
    free(cape_3000);
    free(cin_surf);
    free(cin_255);
    free(cin_180);
    free(cin_90);
    free(li_500);
    free(li_4layer);
    free(low_cloud);
    free(mid_cloud);
    free(high_cloud);
    free(total_cloud);
    free(cloud_base);
    free(cloud_top);
    free(terrain);
    free(ice);
    free(snow);
    free(temp_50);
    free(temp_75);
    free(temp_100);
    free(temp_125);
    free(temp_150);
    free(temp_175);
    free(temp_200);
    free(temp_225);
    free(temp_250);
    free(temp_275);
    free(temp_300);
    free(temp_325);
    free(temp_350);
    free(temp_375);
    free(temp_400);
    free(temp_425);
    free(temp_450);
    free(temp_475);
    free(temp_500);
    free(temp_525);
    free(temp_550);
    free(temp_575);
    free(temp_600);
    free(temp_625);
    free(temp_650);
    free(temp_675);
    free(temp_700);
    free(temp_725);
    free(temp_750);
    free(temp_775);
    free(temp_800);
    free(temp_825);
    free(temp_850);
    free(temp_875);
    free(temp_900);
    free(temp_925);
    free(temp_950);
    free(temp_975);
    free(temp_1000);
    free(temp_1013);
    free(rh_50);
    free(rh_75);
    free(rh_100);
    free(rh_125);
    free(rh_150);
    free(rh_175);
    free(rh_200);
    free(rh_225);
    free(rh_250);
    free(rh_275);
    free(rh_300);
    free(rh_325);
    free(rh_350);
    free(rh_375);
    free(rh_400);
    free(rh_425);
    free(rh_450);
    free(rh_475);
    free(rh_500);
    free(rh_525);
    free(rh_550);
    free(rh_575);
    free(rh_600);
    free(rh_625);
    free(rh_650);
    free(rh_675);
    free(rh_700);
    free(rh_725);
    free(rh_750);
    free(rh_775);
    free(rh_800);
    free(rh_825);
    free(rh_850);
    free(rh_875);
    free(rh_900);
    free(rh_925);
    free(rh_950);
    free(rh_975);
    free(rh_1000);
    free(rh_1013);
    free(dwpt_50);
    free(dwpt_75);
    free(dwpt_100);
    free(dwpt_125);
    free(dwpt_150);
    free(dwpt_175);
    free(dwpt_200);
    free(dwpt_225);
    free(dwpt_250);
    free(dwpt_275);
    free(dwpt_300);
    free(dwpt_325);
    free(dwpt_350);
    free(dwpt_375);
    free(dwpt_400);
    free(dwpt_425);
    free(dwpt_450);
    free(dwpt_475);
    free(dwpt_500);
    free(dwpt_525);
    free(dwpt_550);
    free(dwpt_575);
    free(dwpt_600);
    free(dwpt_625);
    free(dwpt_650);
    free(dwpt_675);
    free(dwpt_700);
    free(dwpt_725);
    free(dwpt_750);
    free(dwpt_775);
    free(dwpt_800);
    free(dwpt_825);
    free(dwpt_850);
    free(dwpt_875);
    free(dwpt_900);
    free(dwpt_925);
    free(dwpt_950);
    free(dwpt_975);
    free(dwpt_1000);
    free(dwpt_1013);

    // Free the memory used by the footprint_recs array and close the netCDF file

    free(num_in_row);

    status = nc_close(ncid);
    }  // if (status==NC_NOERR -- nc_open...


//printf("MIN:  %f\n", min);
//printf("MAX:  %f\n", max);
  }



// ===============================================================================
float convertToWVMR(float pressure, float dwpt)
  {
  double saturation_pressure;
  float wvmr = -32768.0;

  if ((pressure != -32768.0) && (dwpt != -32768.0))
    {
    saturation_pressure = 6.1078 * exp((dwpt-273.15) / ((dwpt-273.15)+238.3) * 17.2694);
    wvmr = (float)((622.0 * saturation_pressure) / (pressure - saturation_pressure));
    }

  return wvmr;
  }



// ===============================================================================
void getVariableIDS(int ncid)
  {
  int  var_id, status, n;
  char var_name[100];

  int pressures[] = { 50, 75, 100, 125, 150, 175, 200, 225,
                      250, 275, 300, 325, 350, 375, 400, 425,
                      450, 475, 500, 525, 550, 575, 600, 625,
                      650, 675, 700, 725, 750, 775, 800, 825,
                      850, 875, 900, 925, 950, 975, 1000, 1013 };

  status = nc_inq_varid(ncid, "latitude", &lat_id);
  status = nc_inq_varid(ncid, "longitude", &lon_id);
  status = nc_inq_varid(ncid, "surface_pressure", &surf_press_id);
  status = nc_inq_varid(ncid, "surface_temperature", &surf_temp_id);
  status = nc_inq_varid(ncid, "precipitation_rate", &prate_id);
  status = nc_inq_varid(ncid, "total_precipitation", &total_precip_id);
  status = nc_inq_varid(ncid, "tpw", &tpw_id);
  status = nc_inq_varid(ncid, "cape_surface", &cape_surf_id);
  status = nc_inq_varid(ncid, "cape_255-0mb", &cape_255_id);
  status = nc_inq_varid(ncid, "cape_180-0mb", &cape_180_id);
  status = nc_inq_varid(ncid, "cape_90-0mb", &cape_90_id);
  status = nc_inq_varid(ncid, "cape_0-3000mb", &cape_3000_id);
  status = nc_inq_varid(ncid, "cin_surface", &cin_surf_id);
  status = nc_inq_varid(ncid, "cin_255-0mb", &cin_255_id);
  status = nc_inq_varid(ncid, "cin_180-0mb", &cin_180_id);
  status = nc_inq_varid(ncid, "cin_90-0mb", &cin_90_id);
  status = nc_inq_varid(ncid, "lifted_index_500-1000mb", &li_500_id);
  status = nc_inq_varid(ncid, "lifted_index_4_layer", &li_4layer_id);
  status = nc_inq_varid(ncid, "low_cloud_layer", &low_cloud_id);
  status = nc_inq_varid(ncid, "middle_cloud_layer", &mid_cloud_id);
  status = nc_inq_varid(ncid, "high_cloud_layer", &high_cloud_id);
  status = nc_inq_varid(ncid, "total_cloud_layer", &total_cloud_id);
  status = nc_inq_varid(ncid, "cloud_base_pressure", &cloud_base_id);
  status = nc_inq_varid(ncid, "cloud_top_pressure", &cloud_top_id);
  status = nc_inq_varid(ncid, "terrain", &terrain_id);
  status = nc_inq_varid(ncid, "ice_cover", &ice_id);
  status = nc_inq_varid(ncid, "snow_cover", &snow_id);

  for (n=0; n<40; n++)
    {
    sprintf(var_name, "temperature_%d_mb", pressures[n]);
    status = nc_inq_varid(ncid, var_name, &var_id);
    temp_ids[n] = var_id;

    sprintf(var_name, "rh_%d_mb", pressures[n]);
    status = nc_inq_varid(ncid, var_name, &var_id);
    rh_ids[n] = var_id;

    sprintf(var_name, "dewpoint_temp_%d_mb", pressures[n]);
    status = nc_inq_varid(ncid, var_name, &var_id);
    dwpt_ids[n] = var_id;
    }
  }



// ===============================================================================
int pointIsWithinBin(float lat, float lon, float bin_lat, float bin_lon)
  {
  int is_in_bin = FALSE;

  if ((lat <= bin_lat) && (lat >= (bin_lat-BIN_SIZE)) &&
      (lon >= bin_lon) && (lon <= (bin_lon+BIN_SIZE)))
    {
    is_in_bin = TRUE;
    }

  return is_in_bin;
  }



// ===============================================================================
void convertTime(double fov_time, int *yyyy, int *mmdd, int *hh, int *mmss)
  {
  time_t     epoch_time;
  struct tm *date;

  epoch_time = (time_t)fov_time;
  date = localtime(&epoch_time);

  // Return the values

  *yyyy = 1900 + date->tm_year;;
  *mmdd = ((date->tm_mon + 1) * 100) + date->tm_mday;
  *hh   = date->tm_hour;
  *mmss = (date->tm_min * 100) + date->tm_sec;
  }

// end of file
