PROGRAM ecmwfBinaryToDailyFile
   IMPLICIT NONE

! ***************************************************************
!
!  This program reads in records from ECMWF binary files for four
!  synoptic times and creates an ECMWF Analysis Data File (ECDF).
!  The data are broken down into 2592 5 X 5-degree data frames.
!  Each data frame contains data from all four synoptic times.
!
! ***************************************************************

   INTEGER, PARAMETER :: short = selected_int_kind(3)

   INTEGER(4), PARAMETER :: reclength=764
   INTEGER(4), PARAMETER :: numframes=2592

   INTEGER(4) :: outfile, infile00, infile06, infile12
   INTEGER(4) :: infile18, framenum

   REAL(4), DIMENSION(numframes) :: startlat, endlat, startlon, endlon

   REAL(4) :: lonstart

   INTEGER(4), DIMENSION(reclength / 4) :: fileheader
   INTEGER(4), DIMENSION(numframes) :: recstart
   INTEGER(4) :: ioerr, YYYY, MMDD, word, outrec, startrec, inrec, latline

   INTEGER(KIND=short), DIMENSION(reclength / 2) :: frameheader, datarecord

   LOGICAL :: isinfile, isfile00, isfile06, isfile12, isfile18

   CHARACTER(LEN=100) erchar

   outfile = 11
   infile00 = 20
   infile06 = 21
   infile12 = 22
   infile18 = 23

   isinfile = .FALSE.

!-- Check to see that at least one input file is present.  If no input file is present,
!   exit without creating an output file.

!-- Open the 00Z ECMWF binary file
     
   OPEN(infile00,access="DIRECT",form="UNFORMATTED",recl=reclength, &
        status="OLD",iomsg=erchar,iostat=ioerr)

   IF (ioerr == 0) THEN
      isinfile = .TRUE.
      isfile00 = .TRUE.
   ELSE
      isfile00 = .FALSE.
      PRINT *,'Error ',ioerr,' opening infile00 '
      PRINT *,'Iomsg ',erchar
      CLOSE (infile00)
   END IF

!-- Open the 06Z ECMWF binary file
     
   OPEN(infile06,access="DIRECT",form="UNFORMATTED",recl=reclength, &
        status="OLD",iomsg=erchar,iostat=ioerr)

   IF (ioerr == 0) THEN
      isinfile = .TRUE.
      isfile06 = .TRUE.
   ELSE
      isfile06 = .FALSE.
      PRINT *,'Error ',ioerr,' opening infile06 '
      PRINT *,'Iomsg ',erchar
      CLOSE (infile06)
   END IF

!-- Open the 12Z ECMWF binary file
     
   OPEN(infile12,access="DIRECT",form="UNFORMATTED",recl=reclength, &
        status="OLD",iomsg=erchar,iostat=ioerr)

   IF (ioerr == 0) THEN
      isinfile = .TRUE.
      isfile12 = .TRUE.
   ELSE
      isfile12 = .FALSE.
      PRINT *,'Error ',ioerr,' opening infile12 '
      PRINT *,'Iomsg ',erchar
      CLOSE (infile12)
   END IF

!-- Open the 18Z ECMWF binary file
     
   OPEN(infile18,access="DIRECT",form="UNFORMATTED",recl=reclength, &
        status="OLD",iomsg=erchar,iostat=ioerr)

   IF (ioerr == 0) THEN
      isinfile = .TRUE.
      isfile18 = .TRUE.
   ELSE
      isfile12 = .FALSE.
      PRINT *,'Error ',ioerr,' opening infile18 '
      PRINT *,'Iomsg ',erchar
      CLOSE (infile18)
   END IF

!-- If no synoptic input files are available, DO not create an output file.

   IF (.NOT.(isinfile)) THEN
      PRINT *,'No synoptic input files are available.'
      GO TO 1000
   END IF

!-- Populate the data frame vertices arrays

   lonstart = 0.0

   DO framenum = 1, 72
      startlat(framenum) = 90.0
      endlat(framenum) = 85.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 73, 144
      startlat(framenum) = 84.75
      endlat(framenum) = 80.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 145, 216
      startlat(framenum) = 79.75
      endlat(framenum) = 75.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 217, 288
      startlat(framenum) = 74.75
      endlat(framenum) = 70.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 289, 360
      startlat(framenum) = 69.75
      endlat(framenum) = 65.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 361, 432
      startlat(framenum) = 64.75
      endlat(framenum) = 60.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 433, 504
      startlat(framenum) = 59.75
      endlat(framenum) = 55.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 505, 576
      startlat(framenum) = 54.75
      endlat(framenum) = 50.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 577, 648
      startlat(framenum) = 49.75
      endlat(framenum) = 45.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 649, 720
      startlat(framenum) = 44.75
      endlat(framenum) = 40.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 721, 792
      startlat(framenum) = 39.75
      endlat(framenum) = 35.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 793, 864
      startlat(framenum) = 34.75
      endlat(framenum) = 30.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 865, 936
      startlat(framenum) = 29.75
      endlat(framenum) = 25.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 937, 1008
      startlat(framenum) = 24.75
      endlat(framenum) = 20.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1009, 1080
      startlat(framenum) = 19.75
      endlat(framenum) = 15.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1081, 1152
      startlat(framenum) = 14.75
      endlat(framenum) = 10.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1153, 1224
      startlat(framenum) = 9.75
      endlat(framenum) = 5.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1225, 1296
      startlat(framenum) = 4.75
      endlat(framenum) = 0.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1297, 1368
      startlat(framenum) = -0.25
      endlat(framenum) = -5.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1369, 1440
      startlat(framenum) = -5.25
      endlat(framenum) = -10.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1441, 1512
      startlat(framenum) = -10.25
      endlat(framenum) = -15.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1513, 1584
      startlat(framenum) = -15.25
      endlat(framenum) = -20.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1585, 1656
      startlat(framenum) = -20.25
      endlat(framenum) = -25.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1657, 1728
      startlat(framenum) = -25.25
      endlat(framenum) = -30.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1729, 1800
      startlat(framenum) = -30.25
      endlat(framenum) = -35.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1801, 1872
      startlat(framenum) = -35.25
      endlat(framenum) = -40.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1873, 1944
      startlat(framenum) = -40.25
      endlat(framenum) = -45.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 1945, 2016
      startlat(framenum) = -45.25
      endlat(framenum) = -50.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 2017, 2088
      startlat(framenum) = -50.25
      endlat(framenum) = -55.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 2089, 2160
      startlat(framenum) = -55.25
      endlat(framenum) = -60.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 2161, 2232
      startlat(framenum) = -60.25
      endlat(framenum) = -60.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 2233, 2304
      startlat(framenum) = -65.25
      endlat(framenum) = -70.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 2305, 2376
      startlat(framenum) = -70.25
      endlat(framenum) = -75.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 2377, 2448
      startlat(framenum) = -75.25
      endlat(framenum) = -80.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 2449, 2520
      startlat(framenum) = -80.25
      endlat(framenum) = -85.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

   lonstart = 0.0

   DO framenum = 2521, 2592
      startlat(framenum) = -85.25
      endlat(framenum) = -90.0
      startlon(framenum) = lonstart
      IF (startlon(framenum) > 180.0) startlon(framenum) = startlon(framenum) - 360.0
      endlon(framenum) = lonstart + 4.75
      IF (endlon(framenum) > 180.0) endlon(framenum) = endlon(framenum) - 360.0
      lonstart = lonstart + 5.0
   END DO

!  Calculate the starting record for each data frame

   startrec = 0

   DO framenum = 1, 72
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 21 * 1440

   DO framenum = 73, 144
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 41 * 1440

   DO framenum = 145, 216
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 61 * 1440

   DO framenum = 217, 288
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 81 * 1440

   DO framenum = 289, 360
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 101 * 1440

   DO framenum = 361, 432
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 121 * 1440

   DO framenum = 433, 504
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 141 * 1440

   DO framenum = 505, 576
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 161 * 1440

   DO framenum = 577, 648
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 181 * 1440

   DO framenum = 649, 720
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 201 * 1440

   DO framenum = 721, 792
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 221 * 1440

   DO framenum = 793, 864
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 241 * 1440

   DO framenum = 865, 936
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 261 * 1440

   DO framenum = 937, 1008
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 281 * 1440

   DO framenum = 1009, 1080
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 301 * 1440

   DO framenum = 1081, 1152
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 321 * 1440

   DO framenum = 1153, 1224
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 341 * 1440

   DO framenum = 1225, 1296
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 361 * 1440

   DO framenum = 1297, 1368
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 381 * 1440

   DO framenum = 1369, 1440
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 401 * 1440

   DO framenum = 1441, 1512
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 421 * 1440

   DO framenum = 1513, 1584
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 441 * 1440

   DO framenum = 1585, 1656
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 461 * 1440

   DO framenum = 1657, 1728
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 481 * 1440

   DO framenum = 1729, 1800
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 501 * 1440

   DO framenum = 1801, 1872
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 521 * 1440

   DO framenum = 1873, 1944
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 541 * 1440

   DO framenum = 1945, 2016
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 561 * 1440

   DO framenum = 2017, 2088
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 581 * 1440

   DO framenum = 2089, 2160
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 601 * 1440

   DO framenum = 2161, 2232
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 621 * 1440

   DO framenum = 2233, 2304
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 641 * 1440

   DO framenum = 2305, 2376
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 661 * 1440

   DO framenum = 2377, 2448
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 681 * 1440

   DO framenum = 2449, 2520
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

   startrec = 701 * 1440

   DO framenum = 2521, 2592
      recstart(framenum) = startrec
      startrec = startrec + 20
   END DO

!  Read in the Year (YYYY) and Month/Day (MMDD) from user input.

   READ(5,*) YYYY
   READ(5,*) MMDD 


!  Open the output file for direct access write

   OPEN(outfile,access="DIRECT",form="UNFORMATTED",recl=reclength, &
        iomsg=erchar,iostat=ioerr)

   IF (ioerr /= 0) THEN
      PRINT *,'Error ',ioerr,' opening output file '
      PRINT *,'Iomsg ',erchar
      GO TO 1000
   END IF

!  Initialize the file header

   DO word = 1, (reclength / 4)
      fileheader(word) = 0
   END DO

!-- Fill in header values

!   File type ("ECDF")
   fileheader(1) = 69676870
!   File version
   fileheader(2) = 10
!   Number of records will be updated at the end of processing
!
!   Length of header record (bytes)
   fileheader(4) = reclength
!   Length of data record (bytes)
   fileheader(5) = reclength
!   Date of file update (YYYYMMDD)
   fileheader(6) = (YYYY*10000) + MMDD

!-- Write the header record to the output file

   outrec = 1
   WRITE(outfile,rec=outrec,iostat=ioerr) fileheader

! ----------------------------------------------------------------------

!-- Process data frames 1-72. Each frame contains 21 X 20 points for each of
!   four synoptic times (21 X 20 X 4 = 1680 points)

   DO framenum = 1, 72

!-- Initialize the first data frame header

      DO word = 1, (reclength / 2)
         frameheader(word) = 0
      END DO

!-- Fill in the data frame header values

!   Record type   
      frameheader(1) = 3
!   Data frame number
      frameheader(2) = framenum
!   Number of records in the frame
      frameheader(3) = 1680
!   Latitudes of frame vertices
      frameheader(5) = NINT(startlat(framenum) * 128.0)
      frameheader(6) = NINT(startlat(framenum) * 128.0)
      frameheader(7) = NINT(endlat(framenum) * 128.0)
      frameheader(8) = NINT(endlat(framenum) * 128.0)
!   Longitudes of frame vertices
      frameheader(9) = NINT(startlon(framenum) * 128.0)
      frameheader(10) = NINT(endlon(framenum) * 128.0)
      frameheader(11) = NINT(endlon(framenum) * 128.0)
      frameheader(12) = NINT(startlon(framenum) * 128.0)

!-- Write the data frame header to the output file

      outrec = outrec + 1
      WRITE(outfile,rec=outrec,iostat=ioerr) frameheader

!-- Copy records for the data frame from the 00Z ECMWF binary file and write them to
!   the output file

      IF (isfile00) THEN
         startrec = recstart(framenum)
         DO latline = 1, 21
            DO inrec = startrec+1, startrec+20
               READ(infile00,rec=inrec,iostat=ioerr) datarecord
               outrec = outrec + 1
               WRITE(outfile,rec=outrec,iostat=ioerr) datarecord
            END DO
            startrec = startrec + 1440
         END DO
      END IF

!-- Copy records for the data frame from the 06Z ECMWF binary file and write them to
!   the output file

      IF (isfile06) THEN
         startrec = recstart(framenum)
         DO latline = 1, 21
            DO inrec = startrec+1, startrec+20
               READ(infile06,rec=inrec,iostat=ioerr) datarecord
               outrec = outrec + 1
               WRITE(outfile,rec=outrec,iostat=ioerr) datarecord
            END DO
            startrec = startrec + 1440
         END DO
      END IF

!-- Copy records for the data from the 12Z ECMWF binary file and write them to
!   the output file

      IF (isfile12) THEN
         startrec = recstart(framenum)
         DO latline = 1, 21
            DO inrec = startrec+1, startrec+20
               READ(infile12,rec=inrec,iostat=ioerr) datarecord
               outrec = outrec + 1
               WRITE(outfile,rec=outrec,iostat=ioerr) datarecord
            END DO
            startrec = startrec + 1440
         END DO
      END IF

!-- Copy records for the data frame from the 18Z ECMWF binary file and write them to
!   the output file

      IF (isfile18) THEN
         startrec = recstart(framenum)
         DO latline = 1, 21
            DO inrec = startrec+1, startrec+20
               READ(infile18,rec=inrec,iostat=ioerr) datarecord
               outrec = outrec + 1
               WRITE(outfile,rec=outrec,iostat=ioerr) datarecord
            END DO
            startrec = startrec + 1440
         END DO
      END IF

   END DO

!-- Process data frames 73-2592. Each frame contains 20 X 20 points for each of
!   four synoptic times (20 X 20 X 4 = 16 points)

   DO framenum = 73, 2592

!-- Initialize the first data frame header

      DO word = 1, (reclength / 2)
         frameheader(word) = 0
      END DO

!-- Fill in the data frame header values

!   Record type   
      frameheader(1) = 3
!   Data frame number
      frameheader(2) = framenum
!   Number of records in the frame
      frameheader(3) = 1600
!   Latitudes of frame vertices
      frameheader(5) = NINT(startlat(framenum) * 128.0)
      frameheader(6) = NINT(startlat(framenum) * 128.0)
      frameheader(7) = NINT(endlat(framenum) * 128.0)
      frameheader(8) = NINT(endlat(framenum) * 128.0)
!   Longitudes of frame vertices
      frameheader(9) = NINT(startlon(framenum) * 128.0)
      frameheader(10) = NINT(endlon(framenum) * 128.0)
      frameheader(11) = NINT(endlon(framenum) * 128.0)
      frameheader(12) = NINT(startlon(framenum) * 128.0)


!-- Write the data frame header to the output file

      outrec = outrec + 1
      WRITE(outfile,rec=outrec,iostat=ioerr) frameheader

!-- Copy records for the data frame from the 00Z ECMWF binary file and write them to
!   the output file

      IF (isfile00) THEN
         startrec = recstart(framenum)
         DO latline = 1, 20
            DO inrec = startrec+1, startrec+20
               READ(infile00,rec=inrec,iostat=ioerr) datarecord
               outrec = outrec + 1
               WRITE(outfile,rec=outrec,iostat=ioerr) datarecord
            END DO
            startrec = startrec + 1440
         END DO
      END IF

!-- Copy records for the data frame from the 06Z ECMWF binary file and write them to
!   the output file

      IF (isfile06) THEN
         startrec = recstart(framenum)
         DO latline = 1, 20
            DO inrec = startrec+1, startrec+20
               READ(infile06,rec=inrec,iostat=ioerr) datarecord
               outrec = outrec + 1
               WRITE(outfile,rec=outrec,iostat=ioerr) datarecord
            END DO
            startrec = startrec + 1440
         END DO
      END IF

!-- Copy records for the data from the 12Z ECMWF binary file and write them to
!   the output file

      IF (isfile12) THEN
         startrec = recstart(framenum)
         DO latline = 1, 20
            DO inrec = startrec+1, startrec+20
               READ(infile12,rec=inrec,iostat=ioerr) datarecord
               outrec = outrec + 1
               WRITE(outfile,rec=outrec,iostat=ioerr) datarecord
            END DO
            startrec = startrec + 1440
         END DO
      END IF

!-- Copy records for the data frame from the 18Z ECMWF binary file and write them to
!   the output file

      IF (isfile18) THEN
         startrec = recstart(framenum)
         DO latline = 1, 20
            DO inrec = startrec+1, startrec+20
               READ(infile18,rec=inrec,iostat=ioerr) datarecord
               outrec = outrec + 1
               WRITE(outfile,rec=outrec,iostat=ioerr) datarecord
            END DO
            startrec = startrec + 1440
         END DO
      END IF

   END DO

   fileheader(3) = outrec
   WRITE(outfile,rec=1,iostat=ioerr) fileheader

   PRINT *,'Last record = ',outrec

!-- Close the files

   CLOSE(outfile)
   IF (isfile00) CLOSE(infile00)
   IF (isfile06) CLOSE(infile06)
   IF (isfile12) CLOSE(infile12)
   IF (isfile18) CLOSE(infile18)

1000 continue

   CALL EXIT(0)

END PROGRAM ecmwfBinaryToDailyFile
