/*
   Name- processMirs

   Language- C     Type- MAIN

   Version- 1.0    Date-  1/16/2007   Programmer- Mike Pettey (IMSG)
   Version- 2.0    Date- 11/01/2013   Programmer- Mike Pettey (IMSG)
   Version- 3.0    Date-  2/04/2015   Programmer- Mike Pettey (IMSG)

   Function- This program extracts selected data from a IDDF
             and writes the data to an ODS file.  The selected
             data is defined in the run script.  Also in the run
             script are the starting date (YYYYMMDD) and hour and
             the ending date and hour.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "ods7_capture.h"


#define  RECORD_LENGTH_MIRS       4800
#define  NUM_HEADER_VALUES_MIRS   RECORD_LENGTH_MIRS/4
#define  NUM_DATA_VALUES_MIRS     RECORD_LENGTH_MIRS/2

#define  SWAP_IN   FALSE
#define  SWAP_OUT  TRUE


void getMirsLayerData(struct parameter_info *parm_list, short *buffer, 
		      float *retrieved_temp_layers, float *retrieved_wvap_layers);

double getSpecialMirsValue(long parm_number, short buffer[NUM_DATA_VALUES_MIRS],
			   float *retrieved_layer_temps, float *retrieved_layer_wvaps);

void copyParameterInformation(struct parameter_info *parm_list, FILE *ods);

void createProfileInfoFile();

long secondsSinceYear(int year, long retrieval_date, long retrieval_time);



void processMirs(int date_to_process, float footprint_size, struct parameter_info *parameter_list)
  {
  int    mirs_header[NUM_HEADER_VALUES_MIRS];
  short  mirs_buffer[NUM_DATA_VALUES_MIRS];
  char   out_buffer_i1[BUFFER_SIZE];
  short  out_buffer_i2[BUFFER_SIZE];
  int    out_buffer_i4[BUFFER_SIZE];
  long   out_buffer_i8[BUFFER_SIZE];
  int    number_of_mirs_records, last_mirs_rec_written;
  int    first_record, last_record;
  int    mirs_record;
  double value;

  long   seconds, earliest_seconds, latest_seconds;
  int    number_of_retrievals;
  int    earliest_date, earliest_time;
  int    latest_date, latest_time;
  int    retrieval_year, retrieval_mmdd;
  int    retrieval_hour, retrieval_mmss;
  long   retrieval_date, retrieval_time;

  char   i1temp;
  long   offset, i8temp;
  int    i4temp, i, j, n, count, status, ival;
  short  i2temp;
  float  lat, lon, pct;

  char   ods_file_name[1000], set_name[100], parmdefs_file_name[1000];
  char   filename_string[1000];
  char   c, char12[12], char36[36];
  char   month_string[10];
  int    buffer_index, block_index, output_index, block_number;
  int    num_values_written;
  int    use_fov, pass_both;

  float  retrieved_layer_temps[30], retrieved_layer_wvaps[20];

  struct parameter_info *ptr;

  int    month, day, year, hour, minute, second, today;

  time_t  date_time;
  struct  tm  *ts;   

  FILE   *mirs, *ods, *in;


  // Create the parameter_info.xml file with the data from each parameter.
  // While doing this, pull out the file name used for each parameter.

  sprintf(ods_file_name, "out.dir/parameter_info.xml");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS parameter_info.xml file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fprintf(ods, "<?xml version=\"1.0\" standalone=\"yes\" ?>\n\n");

  fprintf(ods, "<parameter_info>\n");

  ptr = parameter_list;

  while (ptr != NULL)
    {
    copyParameterInformation(ptr, ods);
    ptr = ptr->next;
    }

  fprintf(ods, "</parameter_info>\n");
  fclose(ods);


  // Create sub-directories for each parameter

  //sprintf(set_name, "%d", date_to_process);
  sprintf(set_name, "set01");

  sprintf(ods_file_name, "out.dir/%s", set_name);
  mkdir(ods_file_name, 0777);

  ptr = parameter_list;

  while (ptr != NULL)
    {
    sprintf(ods_file_name, "out.dir/%s/%s", set_name, ptr->file_name);
    mkdir(ods_file_name, 0777);
    ptr = ptr->next;
    }


  // Open the MIRS

  if ((mirs=fopen("in.file","r")) == NULL)
    {
    //createEmptyFile(capture_info, parameter_list);

    printf("\n\nThe MIRS could not be opened for input.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  // Position the read pointer to the start of the first record

  offset = 0;
  fseek(mirs, offset, SEEK_SET);

  // Read the header record

  fread(&mirs_header, RECORD_LENGTH_MIRS, 1, mirs);

  if (SWAP_IN == TRUE)
    {
    for (n=0; n<NUM_HEADER_VALUES_MIRS; n++)
      mirs_header[n] = htonl(mirs_header[n]);
    }


  number_of_mirs_records = mirs_header[2];

  printf("%s file:\n", system_description);
  printf("   Number of records in the file:  %d\n",number_of_mirs_records);

  first_record = 2;
  last_record  = number_of_mirs_records;


  block_number = 0;

  num_values_written = 0;

  // Initialize the earliest and latest seconds and the lat/lon range

  latest_seconds   = 0;
  earliest_seconds = secondsSinceYear(2000,20291231,235959);

  // Loop through the data records

  buffer_index = 0;
  output_index = 0;

  for (mirs_record=first_record; mirs_record<=last_record; mirs_record++)
    {
    pct = (float)mirs_record / (float)last_record * 100.0;

    if ((mirs_record%1000)==0)
      printf("   %d    %d    %.1f%\n", mirs_record, last_record, pct);

    // Position the read pointer to the start of the next record

    //offset = (long)(mirs_record - 1) * (long)RECORD_LENGTH_MIRS;
    //fseek(mirs, offset, SEEK_SET);

    // Read the data record

    fread(&mirs_buffer, RECORD_LENGTH_MIRS, 1, mirs);

    if (SWAP_IN == TRUE)
      {
      for (n=0; n<NUM_DATA_VALUES_MIRS; n++)
        mirs_buffer[n] = htons(mirs_buffer[n]);
      }


    if (mirs_buffer[0] == 0)
      {

      // Unpack the date, latitude and longitude

      retrieval_date = (mirs_buffer[5] * 10000) + mirs_buffer[6];

      lat = mirs_buffer[3] / 128.0;
      lon = mirs_buffer[4] / 128.0;

      // If the retrieval is within both windows, count it

      use_fov = FALSE;

      if ((date_to_process == -32768) || (retrieval_date == date_to_process))
	{
	if ((lat >= bottom_latitude) && (lat <= top_latitude))
  	  {
	  if ((left_longitude <= right_longitude))
	    {
	    if ((lon >= left_longitude) && (lon <= right_longitude))
	      {
	      use_fov = TRUE;
              }
	    }
	  else
	    { 
	    if ((lon <= right_longitude) || (lon >= left_longitude))
	      {
	      use_fov = TRUE;
	      }
	    }
	  }  // if (lats[fov]...
	}  // if (date_to_process...


      // Some MIRS footprints have an hour value of 24. These will be filtered out.

      if (mirs_buffer[7] >= 24)
	use_fov = FALSE;


      if (use_fov == TRUE)
	{

        // Keep track of the earliest and latest time

        retrieval_date = (mirs_buffer[5] * 10000) + mirs_buffer[6];
        retrieval_time = (mirs_buffer[7] * 10000) + mirs_buffer[8];

        seconds = secondsSinceYear(2000,retrieval_date,retrieval_time);

        if (seconds < earliest_seconds)
          {
          earliest_seconds = seconds;
          earliest_date    = retrieval_date;
          earliest_time    = retrieval_time;
          }

        if (seconds > latest_seconds)
          {
          latest_seconds = seconds;
          latest_date    = retrieval_date;
          latest_time    = retrieval_time;
          }

	// Call the subroutine getMirsLayerData which will convert some or all
	// of the temperature and moisture data from levels to layers

	getMirsLayerData(parameter_list, mirs_buffer,
			 retrieved_layer_temps, retrieved_layer_wvaps);

        // Fill the buffers

        ptr = parameter_list;

        while (ptr != NULL)
	  {
          if (ptr->word_number > 0)
            {
            value = mirs_buffer[ptr->word_number - 1];
            }  // if (word_number > 0...
          else
            {
	    value = getSpecialMirsValue(ptr->word_number, mirs_buffer,
					retrieved_layer_temps, retrieved_layer_wvaps);
            }  // else word_number < 0...

	  if (ptr->data_type == BYTE)
	    ptr->byte_buffer[buffer_index] = (char)value;
	  else if (ptr->data_type == SHORT)
	    ptr->short_buffer[buffer_index] = (short)value;
	  else if (ptr->data_type == INT)
	    ptr->int_buffer[buffer_index] = (int)value;
	  else if (ptr->data_type == FLOAT)
	    ptr->short_buffer[buffer_index] = (short)value;
	  else if (ptr->data_type == DOUBLE)
	    ptr->short_buffer[buffer_index] = (short)value;

          ptr = ptr->next;
  	  }

        buffer_index++;


        // If the buffers are full, write them to the file

        if (buffer_index >= BUFFER_SIZE)
          {
          buffer_index = 0;

          block_index = 0;
          ptr = parameter_list;

          while (ptr != NULL)
	    {
            sprintf(ods_file_name, "out.dir/%s/%s/block_%d.dat", set_name, ptr->file_name, block_number);

	    if ((ods=fopen(ods_file_name,"w")) == NULL)
	      {
	      printf("\n\nThe ODS block data file file %s could not be opened for output.\n",
		     ods_file_name);
	      printf("Execution ending.\n\n");
	      exit(1);
	      }


	    if (ptr->data_type == BYTE)
	      {
	      for (i=0; i<BUFFER_SIZE; i++)
		{
		out_buffer_i1[i] = ptr->byte_buffer[i];
		}

	      fwrite(out_buffer_i1, 1, BUFFER_SIZE, ods);
	      }

	    else if ((ptr->data_type == SHORT) || (ptr->data_type == FLOAT))
	      {
	      for (i=0; i<BUFFER_SIZE; i++)
		{
                if (SWAP_OUT == FALSE)
		  out_buffer_i2[i] = ptr->short_buffer[i];
		else
		  out_buffer_i2[i] = htons(ptr->short_buffer[i]);
		}

	      fwrite(out_buffer_i2, 2, BUFFER_SIZE, ods);
	      }

	    else if ((ptr->data_type == INT) || (ptr->data_type == DOUBLE))
	      {
	      for (i=0; i<BUFFER_SIZE; i++)
		{
                if (SWAP_OUT == FALSE)
		  out_buffer_i4[i] = ptr->int_buffer[i];
		else
		  out_buffer_i4[i] = htonl(ptr->int_buffer[i]);
		}

	      fwrite(out_buffer_i4, 4, BUFFER_SIZE, ods);
	      }

	    fclose(ods);

	    block_index++;
	    ptr = ptr->next;
	    }  // while (ptr != NULL...

	  num_values_written += BUFFER_SIZE;
          output_index = output_index + BUFFER_SIZE;

	  block_number++;
          }  // if (buffer_index...
	}  // if (use_fov...
      }  // if (buffer[0]...
    }  // for (mirs_record=first_record...


  // Write the remainder of the data to the file

  if (buffer_index > 0)
    {
    block_index = 0;
    ptr = parameter_list;

    while (ptr != NULL)
      {
      sprintf(ods_file_name, "out.dir/%s/%s/block_%d.dat", set_name, ptr->file_name, block_number);

      if ((ods=fopen(ods_file_name,"w")) == NULL)
	{
        printf("\n\nThe ODS block data file file %s could not be opened for output.\n",
	       ods_file_name);
	printf("Execution ending.\n\n");
	exit(1);
	}

      if (ptr->data_type == BYTE)
	{
	for (i=0; i<BUFFER_SIZE; i++)
	  {
	  out_buffer_i1[i] = ptr->byte_buffer[i];
	  }

	fwrite(out_buffer_i1, 1, BUFFER_SIZE, ods);
	}

      else if ((ptr->data_type == SHORT) || (ptr->data_type == FLOAT))
	{
	for (i=0; i<BUFFER_SIZE; i++)
	  {
	  if (SWAP_OUT == FALSE)
	    out_buffer_i2[i] = ptr->short_buffer[i];
	  else
	    out_buffer_i2[i] = htons(ptr->short_buffer[i]);
	  }

	fwrite(out_buffer_i2, 2, BUFFER_SIZE, ods);
	}

      else if ((ptr->data_type == INT) || (ptr->data_type == DOUBLE))
	{
	for (i=0; i<BUFFER_SIZE; i++)
	  {
	  if (SWAP_OUT == FALSE)
	    out_buffer_i4[i] = ptr->int_buffer[i];
	  else
	    out_buffer_i4[i] = htonl(ptr->int_buffer[i]);
	  }

	fwrite(out_buffer_i4, 4, BUFFER_SIZE, ods);
	}

      fclose(ods);

      block_index++;
      ptr = ptr->next;
      }

    num_values_written += block_index;
    block_number++;
    }


  printf("\nNumber of values written to the output file:  %d\n", num_values_written);

  // Print the earliest and latest times found

  year   = earliest_date / 10000;
  month  = (earliest_date % 10000) / 100;
  day    = earliest_date % 100;
  hour   = earliest_time / 10000;
  minute = (earliest_time % 10000) / 100;
  second = earliest_time % 100;

  printf("\nEarliest retrieval found:  %d/%2d/%d %2d:%2d:%2d\n",
          month,day,year,hour,minute,second);

  year   = latest_date / 10000;

  month  = (latest_date % 10000) / 100;
  day    = latest_date % 100;
  hour   = latest_time / 10000;
  minute = (latest_time % 10000) / 100;
  second = latest_time % 100;

  printf("Latest retrieval found:    %d/%2d/%d %2d:%2d:%2d\n\n",
	  month,day,year,hour,minute,second);


  // Get the current date

  time (&date_time);         
  ts = localtime(&date_time);

  month  = ts->tm_mon + 1;    
  day    = ts->tm_mday;       
  year   = ts->tm_year;       

  if (year > 56)
    year = 1900 + year;
  else
    year = 2000 + year;

  today = year*10000 + month*100 + day;


  // Create the ODS header record

  sprintf(ods_file_name, "out.dir/header.xml");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS header.xml file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fprintf(ods, "<?xml version=\"1.0\" standalone=\"yes\" ?>\n\n");

  fprintf(ods, "<header>\n");
  fprintf(ods, "  <file_type>ODS7</file_type>\n\n");

  fprintf(ods, "  <file_creation_date>%d</file_creation_date>\n", today);
  fprintf(ods, "  <file_modification_date>%d</file_modification_date>\n\n", today);

  fprintf(ods, "  <oldest_data>\n");
  fprintf(ods, "    <date>%d</date>\n", earliest_date);
  fprintf(ods, "    <time>%06d</time>\n", earliest_time);
  fprintf(ods, "  </oldest_data>\n\n");

  fprintf(ods, "  <newest_data>\n");
  fprintf(ods, "    <date>%d</date>\n", latest_date);
  fprintf(ods, "    <time>%06d</time>\n", latest_time);
  fprintf(ods, "  </newest_data>\n\n");

  fprintf(ods, "  <data_format>binary</data_format>\n\n");

  fprintf(ods, "  <data_type>sequential</data_type>\n");
  fprintf(ods, "  <footprint_size>%.2f</footprint_size>\n\n", footprint_size);

  fprintf(ods, "  <earth_coverage>\n");
  fprintf(ods, "    <left_longitude>%.2f</left_longitude>\n", left_longitude);
  fprintf(ods, "    <right_longitude>%.2f</right_longitude>\n", right_longitude);
  fprintf(ods, "    <top_latitude>%.2f</top_latitude>\n", top_latitude);
  fprintf(ods, "    <bottom_latitude>%.2f</bottom_latitude>\n", bottom_latitude);
  fprintf(ods, "  </earth_coverage>\n\n");

  fprintf(ods, "  <system_name>%s</system_name>\n", system_description);
  fprintf(ods, "</header>\n");

  fclose(ods);


  // Create the set_info.xml file

  sprintf(ods_file_name, "out.dir/set_info.xml");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS set_info.xml file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fprintf(ods, "<?xml version=\"1.0\" standalone=\"yes\" ?>\n\n");

  fprintf(ods, "<set_info>\n");
  fprintf(ods, "  <set>\n");

  fprintf(ods, "    <set_name>%s</set_name>\n", set_name);

  year  = date_to_process / 10000;
  month = (date_to_process % 10000) / 100;
  day   = date_to_process % 100;

  if (month == 1)
    sprintf(month_string, "January");
  else if (month == 2)
    sprintf(month_string, "February");
  else if (month == 3)
    sprintf(month_string, "March");
  else if (month == 4)
    sprintf(month_string, "April");
  else if (month == 5)
    sprintf(month_string, "May");
  else if (month == 6)
    sprintf(month_string, "June");
  else if (month == 7)
    sprintf(month_string, "July");
  else if (month == 8)
    sprintf(month_string, "August");
  else if (month == 9)
    sprintf(month_string, "September");
  else if (month == 10)
    sprintf(month_string, "October");
  else if (month == 11)
    sprintf(month_string, "November");
  else if (month == 12)
    sprintf(month_string, "December");

  fprintf(ods, "    <description>%d/%02d/%4d</description>\n", month, day, year);
  fprintf(ods, "    <date_string>%s %d, %d</date_string>\n", month_string, day, year);

  fprintf(ods, "    <oldest_data>\n");
  fprintf(ods, "      <date>%d</date>\n", earliest_date);
  fprintf(ods, "      <time>%06d</time>\n", earliest_time);
  fprintf(ods, "    </oldest_data>\n\n");

  fprintf(ods, "    <newest_data>\n");
  fprintf(ods, "      <date>%d</date>\n", latest_date);
  fprintf(ods, "      <time>%06d</time>\n", latest_time);
  fprintf(ods, "    </newest_data>\n\n");

  fprintf(ods, "    <system_name>%s</system_name>\n", system_description);
  fprintf(ods, "  </set>\n");
  fprintf(ods, "</set_info>\n");
  fclose(ods);


  // Create the profile_info.xml file
  
  createProfileInfoFile();


  // Create the byte_check.dat file

  sprintf(ods_file_name, "out.dir/byte_check.dat");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS byte_check.dat file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  n = 79688355;

  if (SWAP_OUT == TRUE)
    n = htonl(n);

  fwrite(&n, 4, 1, ods);

  fclose(ods);
  }

// end of file
