/*
   Name- EcmwfOzonesondeCollocator.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  7/04/2023   Programmer- Mike Pettey (IMSG)

   Function- This program extracts data from GOES ABI netCDF files
             and copies the data to a GOES ABI Daily Data File (GDDF).
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>

#include <dirent.h>
//#include <errno.h>

#define  MAX_TIME_DIFF       6.0
#define  MAX_DISTANCE_DIFF 200.0

#define  TRUE      0
#define  FALSE     1


//void processCollocations(char *dir_name, int start_date, int end_date);
//void processFile(char *file_name, int start_date, int end_date);

void processEcmwfFile(int collocation_date, int collocation_time,
                      float collocation_lat, float collocation_lon,
                      int nprovs_platform_id, char *nprovs_platform_name);

int findClosestFootprint(int collocation_date, int collocation_time,
                         float collocation_lat, float collocation_lon);

float calculateDistanceDifference(float lat1, float lon1, float lat2, float lon2);

float calculateTimeDifference(int date1, int time1, int date2, int time2);

//void convertEpochTimeToDateAndTime(double epoch_time, int *yr, int *md, int *hr, int *ms);
//
//int fileEndsWith(char *file_name, char *token);

int defineVariable(int nc_group_id, char *var_name, nc_type var_type, int ndims, int *dimids,
                   char *attr_string, char *attr);

void writeArrayFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float *value);
void writeAttributeText(int grp, char *attrVariable, char *attrValue);
//void writeVariableByte(int group_id, int var_id, size_t *index, size_t *num_vals, char value);
void writeVariableFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float value);
void writeVariableInteger(int group_id, int var_id, size_t *index, size_t *num_vals, int value);
void writeVariableShort(int group_id, int var_id, size_t *index, size_t *num_vals, short value);


//int filter(const struct dirent *name)
//  {
//  return 1;
//  }



int main(int argc, char *argv[])
  {
  int     retval, nc_id, var_id, nprovs_platform_id;
  int     collocation_date, collocation_time;
  float   collocation_lat, collocation_lon;
  char    nprovs_platform_name[100];
  size_t  index[1];

  index[0] = 0;

  // Print an initial message

  printf("\nCollocating ECMWF data to Ozonesondes\n\n");

  // Unpack the platform ID and platform name

  nprovs_platform_id = atoi(argv[1]);
  sprintf(nprovs_platform_name, "%s", argv[2]);

  // Unpack the date, time, latitude and longitude from the ozonesonde file

  retval = nc_open("ozonesonde.in", NC_NOWRITE, &nc_id);

  if (retval == NC_NOERR)
    {

    // Unpack the date

    retval = nc_inq_varid(nc_id, "date", &var_id);
    retval = nc_get_var1_int(nc_id, var_id, index, &collocation_date);

    // Unpack the time

    retval = nc_inq_varid(nc_id, "time", &var_id);
    retval = nc_get_var1_int(nc_id, var_id, index, &collocation_time);

    // Unpack the latitude

    retval = nc_inq_varid(nc_id, "latitude", &var_id);
    retval = nc_get_var1_float(nc_id, var_id, index, &collocation_lat);

    // Unpack the longitude

    retval = nc_inq_varid(nc_id, "longitude", &var_id);
    retval = nc_get_var1_float(nc_id, var_id, index, &collocation_lon);

    // Close the input netCDF file and free the allocated memory

    nc_close(nc_id);

    printf("Collocation Date:      %d\n", collocation_date);
    printf("Collocation Time:      %d\n", collocation_time);
    printf("Collocation Latitude:  %0.1f\n", collocation_lat);
    printf("Collocation Longitude: %0.1f\n", collocation_lon);


    // Process the input ECMWF file

    processEcmwfFile(collocation_date, collocation_time, collocation_lat, collocation_lon,
                     nprovs_platform_id, nprovs_platform_name);

    }  //  if (retval == NC_NOERR)...
  else
    {
    printf("\n\n*** ERROR ***\n\n");
    printf("The ozonesonde netCDF file could not be read.\n");
    printf("Execution is stopping.\n\n");
    }

  printf("\nProcessing completed.\n\n");
  }



// ===============================================================================
void processEcmwfFile(int collocation_date, int collocation_time,
                      float collocation_lat, float collocation_lon,
                      int nprovs_platform_id, char *nprovs_platform_name)
  {
  int     n, retval, out_id;
  int     closest_footprint_index;
  int     header_length, record_length;
  int     yyyymmdd, hhmmss;
  int     vid_lat, vid_lon, vid_date, vid_time;
  int     vid_press, vid_temp, vid_wvmr, vid_ozone;
  int     vid_tpw, vid_skin_temp;
  int     vid_qc_temp, vid_qc_wvap, vid_qc_ozone;
  int     level_dim, dimid_scalar[1], dimid_levels[2];

  long    offset;
  short   buffer[382];
  float   latitude, longitude, time_diff, distance_diff;
  float   level_values[91];
  size_t  index_scalar[1], num_vals_scalar[1];
  size_t  index_1D[1], num_vals_1D[1];

  char    platform_id[5];
  FILE    *in;


  // Search all of the NUCAPS granules to find a footprint that most closely matches
  // the time and location of the ozonesonde

  closest_footprint_index = findClosestFootprint(collocation_date, collocation_time,
                                                 collocation_lat, collocation_lon);

  // If a closest footprint was found, then begin copying the data to a
  // new netCDF file

  if (closest_footprint_index != -32768)
    {

    // Read the record of the closest footprint

    if ((in=fopen("ecmwf.in","r")) == NULL)
      {
      printf("\n\nThe input file could not be opened.\n");
      printf("Execution ending.\n\n");
      exit(1);
      }

    offset = 12;
    fseek(in, offset, SEEK_SET);
    fread(&header_length, 4, 1, in);
    fread(&record_length, 4, 1, in);

    offset = (long)header_length + ((long)(closest_footprint_index-1) * (long)record_length);
    fseek(in, offset, SEEK_SET);
    fread(&buffer, 382, 2, in);

    fclose(in);

    // Display information about the collocation closeness

    yyyymmdd = (buffer[5] * 10000) + buffer[6];
    hhmmss   = (buffer[7] * 10000) + buffer[8];

    time_diff = fabs(calculateTimeDifference(yyyymmdd, hhmmss, collocation_date, collocation_time));

    latitude  = buffer[3] / 128.0;
    longitude = buffer[4] / 128.0;

    distance_diff = calculateDistanceDifference(latitude, longitude,
                                                collocation_lat, collocation_lon);

    printf("\nCollocation Made:\n");
    printf("   Collocation Time:   %d  %d\n", collocation_date, collocation_time);
    printf("   Footprint Time:     %d  %d\n", yyyymmdd, hhmmss);
    printf("   Time Difference:    %f hours\n\n", time_diff);
    printf("   Collocation Loc:    %.2f   %.2f\n", collocation_lat, collocation_lon);
    printf("   Footprint Loc:      %.2f   %.2f\n", latitude, longitude);
    printf("   Dist Difference:    %f km\n\n", distance_diff);


    // Create a new netCDF file

    retval = nc_create("ecmwf.out", NC_NETCDF4, &out_id);

    if (retval == NC_NOERR)
      {
      sprintf(platform_id, "%d", nprovs_platform_id);
      writeAttributeText(out_id, "NPROVS_Platform_ID", platform_id);
      writeAttributeText(out_id, "NPROVS_Platform_Type", "800");
      writeAttributeText(out_id, "NPROVS_Platform_Name", nprovs_platform_name);

      // Define the dimensions

      retval = nc_def_dim(out_id, "Levels", 91, &level_dim);
      dimid_levels[0] = level_dim;

      // Define the variables

      vid_lat = defineVariable(out_id, "latitude", NC_FLOAT, 0, 0,
                               "units", "degrees_north");

      vid_lon = defineVariable(out_id, "longitude", NC_FLOAT, 0, 0,
                               "units", "degrees_east");

      vid_date = defineVariable(out_id, "date", NC_INT, 0, 0, "format", "yyyymmdd");

      vid_time = defineVariable(out_id, "time", NC_INT, 0, 0, "format", "hhmmss");

      vid_press = defineVariable(out_id, "pressure", NC_FLOAT, 1, dimid_levels,
                                 "units", "hPa");

      vid_temp = defineVariable(out_id, "temperature", NC_FLOAT, 1, dimid_levels,
                                "units", "K");

      vid_wvmr = defineVariable(out_id, "water_vapor_mixing_ratio", NC_FLOAT, 1, dimid_levels,
                                  "units", "g/kg");

      vid_ozone = defineVariable(out_id, "ozone_mixing_ratio", NC_FLOAT, 1, dimid_levels,
                                 "units", "kg/kg");

      vid_tpw = defineVariable(out_id, "total_precipitable_water", NC_FLOAT, 0, 0,
                               NULL, NULL);

      vid_skin_temp = defineVariable(out_id, "skin_temperature", NC_FLOAT, 0, 0,
                                     "units", "K");

      vid_qc_wvap = defineVariable(out_id, "nprovs_moisture_qc", NC_SHORT, 0, 0,
                                   "values", "0=pass, 1=fail");


      // Index and num_vals are used to save a single value within the
      // variable array

      index_scalar[0]    = 0;
      num_vals_scalar[0] = 1;

      index_1D[0]    = 0;
      num_vals_1D[0] = 91;

      // Latitude and longitude

      if (buffer[3] != -32768)
        latitude  = buffer[3] / 128.0;
      else
        latitude = -32768.0;

      if (buffer[4] != -32768)
        longitude = buffer[4] / 128.0;
      else
        longitude = -32768.0;

      writeVariableFloat(out_id, vid_lat, index_scalar, num_vals_scalar, latitude);
      writeVariableFloat(out_id, vid_lon, index_scalar, num_vals_scalar, longitude);

      // Date and time

      if ((buffer[5] != -32768) && (buffer[6] != -32768))
        yyyymmdd = (buffer[5] * 10000) + buffer[6];
      else
        yyyymmdd = -32768;

      if ((buffer[7] != -32768) && (buffer[8] != -32768))
        hhmmss = (buffer[7] * 10000) + buffer[8];
      else
        hhmmss = -32768;

      writeVariableInteger(out_id, vid_date, index_scalar, num_vals_scalar, yyyymmdd);
      writeVariableInteger(out_id, vid_time, index_scalar, num_vals_scalar, hhmmss);

      // Save the tpw and skin temperature

      if (buffer[282] > 0)
        writeVariableFloat(out_id, vid_tpw, index_scalar, num_vals_scalar, (buffer[282]/100.0));
      else
        writeVariableFloat(out_id, vid_tpw, index_scalar, num_vals_scalar, -32768.0);

      if (buffer[283] > 0)
        writeVariableFloat(out_id, vid_skin_temp, index_scalar, num_vals_scalar, (buffer[283]/64.0));
      else
        writeVariableFloat(out_id, vid_skin_temp, index_scalar, num_vals_scalar, -32768.0);

      // Pressures

      for (n=0; n<91; n++)
        {
        if (buffer[9+n] == -32768)
          level_values[n] = -32768.0;
        else if (n < 36)
          level_values[n] = buffer[9+n] / 100.0;
        else
          level_values[n] = buffer[9+n] / 10.0;

        //printf("ECMWF PRESS:  %d  %d   %f\n", recnum, n, level_values[n]);
        }

      writeArrayFloat(out_id, vid_press, index_1D, num_vals_1D, level_values);

      // Temperatures

      for (n=0; n<91; n++)
        {
        if (buffer[100+n] == -32768)
          level_values[n] = -32768.0;
        else
          level_values[n] = buffer[100+n] / 64.0;
        }

      writeArrayFloat(out_id, vid_temp, index_1D, num_vals_1D, level_values);

      // Water vapor mixing ratio

      for (n=0; n<91; n++)
        {
        if (buffer[191+n] == -32768)
          level_values[n] = -32768.0;
        else
          //level_values[n] = (float)(exp(buffer[191+n]/1024.0));
          level_values[n] = buffer[191+n] / 1000.0;
          //level_values[n] = buffer[191+n];

        //printf("ECMWF WVMR:  %d   %d   %f\n", n, buffer[191+n], level_values[n]);
        }

      writeArrayFloat(out_id, vid_wvmr, index_1D, num_vals_1D, level_values);

      // Ozone mixing ratio

      for (n=0; n<91; n++)
        {
        if (buffer[284+n] == -32768)
          {
          level_values[n] = -32768.0;
          }
        else
          {
          //level_values[n] = (float)(exp(buffer[284+n]/1024.0));
          //level_values[n] = (float)(exp(buffer[284+n]/64.0)) * 1000.0;
          //level_values[n] = (float)(exp(buffer[284+n]/64.0));

          //level_values[n] = (28.9644 / 47.9982) * 1000000000F * exp(buffer[284+n]/64.0);
          level_values[n] = (28.9644 / 47.9982) * exp(buffer[284+n]/64.0);
          }

        //printf("ECMWF OZONE:  %d   %d   %f\n", n, buffer[284+n], level_values[n]);
        }

      writeArrayFloat(out_id, vid_ozone, index_1D, num_vals_1D, level_values);

      // QC flags

      writeVariableShort(out_id, vid_qc_wvap, index_scalar, num_vals_scalar, buffer[371]);

      nc_close(out_id);
      }  // if (retval == NC_NOERR (output_open)...
    }  // if (closest_footprint_index...
  else
    {
    printf("\n\nNo footprints were found within %0.1f km and %0.1f hours.\n", MAX_DISTANCE_DIFF,
                                                                              MAX_TIME_DIFF);
    printf("No collocation has been made for this ozonesonde.\n\n");
    }
  }



// ===============================================================================
int findClosestFootprint(int collocation_date, int collocation_time,
                         float collocation_lat, float collocation_lon)
  {
  int    closest_index;
  int    num_records_in_file, header_length, record_length;
  int    recnum, yyyymmdd, hhmmss;
  short  buffer[9];
  long   offset;
  float  time_diff, distance_diff;
  float  latitude, longitude;
  float  time_factor, closeness, closest_value;
  FILE   *in;

  closest_index = -32768;

  closest_value = 99999999999.9;

  // Open the ECMWF file

  if ((in=fopen("ecmwf.in","r")) == NULL)
    {
    printf("\n\nThe input file could not be opened.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  // Unpack the number of records and record lengths from the header

  offset = 8;
  fseek(in, offset, SEEK_SET);
  fread(&num_records_in_file, 4, 1, in);
  fread(&header_length, 4, 1, in);
  fread(&record_length, 4, 1, in);

  // Loop through all of the records in the file to find one that is closest
  // to the ozonesonde

  for (recnum=1; recnum<num_records_in_file; recnum++)
    {
    offset = (long)header_length + ((long)(recnum-1) * (long)record_length);
    fseek(in, offset, SEEK_SET);

    fread(&buffer, 9, 2, in);

    // Only process the record if it is a data record

    if (buffer[0] == 0)
      {

      // Extract the date and time and calculate the time difference between this
      // footprint and the collocation

      yyyymmdd = (buffer[5] * 10000) + buffer[6];
      hhmmss   = (buffer[7] * 10000) + buffer[8];

      time_diff = fabs(calculateTimeDifference(yyyymmdd, hhmmss, collocation_date, collocation_time));

      // Continue processing the footprint if the time difference is less
      // than the maximum allowed time difference

      if (time_diff <= MAX_TIME_DIFF)
        {

        // Unpack the latitude and longitude and calculate the distance
        // difference between this footprint and the collocation

        latitude  = buffer[3] / 128.0;
        longitude = buffer[4] / 128.0;

        distance_diff = calculateDistanceDifference(latitude, longitude,
                                                    collocation_lat, collocation_lon);

        // Continue processing the footprint if the distance difference is less than MAX_DISTANCE_DIFF

        if (distance_diff <= MAX_DISTANCE_DIFF)
          {

          // Calculate the closeness value

          time_factor = 30.0;
          closeness = distance_diff + (time_diff * time_factor);

          if (closeness < closest_value)
            {
            closest_value = closeness;
            closest_index = recnum;
            }  // if (closeness < closest_value...
          }  // if (dist_diff <= MAX_DISTANCE_DIFF...
        }  // if (time_diff <= MAX_TIME_DIFF...
      }  // if (record_type == 0...
    }  // for (recnum=1...

  // Close the ECMWF file

  fclose(in);

  return closest_index;
  }


// ===============================================================================


//---------------------------------------------------------------------
//int fileEndsWith(char *file_name, char *token)
//  {
//  int i, file_length, token_length;
//  int ends_with = TRUE;
//
//  file_length  = strlen(file_name);
//  token_length = strlen(token);
//
//  for (i=0; i<token_length; i++)
//    {
//    if (file_name[file_length-i] != token[token_length-i])
//      ends_with = FALSE;
//    }
//
//  return ends_with;
//  }

// end of file
