/*
   Name- processECMWF02.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  1/30/2017   Programmer- Mike Pettey (IMSG)

   Function- This program extracts data from P72 EDR files in HDF5 format
             and writes the data to an EDGE orbital file.  The selected
             data is defined in the run script.  Also in the run
             script are the starting date (YYYYMMDD) and hour and
             the ending date and hour.
*/


#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <math.h>
#include <netcdf.h>

#include "nprovs_to_netcdf.h"


//#define ERR(e)
//  {
//  printf("Error: %s, error num = %d\n", nc_strerror(e),e); 
//  return 2;
//  }

#define ERR(e) {printf("Error: %s, error num = %d\n", nc_strerror(e),e); return 2;}


//#define PRESSURE_NAME PRESSURE
#define NDIMS 1
//#define MISSING_VALUE -32768


int defineVariable(int nc_group_id, char *var_name, nc_type var_type, int ndims, int *dimids, char *attr_string, char *attr);

void writeVariableFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float value);

void writeVariableShort(int group_id, int var_id, size_t *index, size_t *num_vals, short value);

void writeVariableInteger(int group_id, int var_id, size_t *index, size_t *num_vals, long value);

void writeArrayFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float *value);



processECMWF02(FILE *in, int header_length, int record_length, int number_of_collocations,
	       int group, char *nprovs_name, char *description, int group_id, char *dir_name, 
	       int group_type, int group_length, long group_offset, int nc_date_group)
  {
  int     n, i, recnum, retval;
  int     nc_group_id, collocation_group_id;
  size_t  num_colls, index[1], num_vals[1], index_2D[2], num_vals_2D[2];

  float   missing_float, latitude, longitude, level_values[91];
  int     year, month, day, hour, minute, second;
  long    offset, yyyymmdd, hhmmss;
  char    col_dir_name[50];

  char    *string[1];

  short   *buffer;

  int     col_scalar_dim;
  int     scalar_dim, dim_levels;

  int     dimid_scalar[1], col_dimid_scalar[1];
  int     dimid_levels[2];

  int     col_vid_lat, col_vid_lon, col_vid_date, col_vid_time;
  int     vid_lat, vid_lon, vid_date, vid_time;

  int     vid_press, vid_temp, vid_wvmr, vid_ozone;
  int     vid_tpw, vid_skin_temp;
  int     vid_qc_temp, vid_qc_wvap, vid_qc_ozone;


  num_colls = (size_t)number_of_collocations;

  missing_float = -32768.0;

  // If this is the baseline platform, then create another subgroup
  // for the collocation information

  if (group == 0)
    {
    sprintf(col_dir_name, "Collocation_Info\0");

    if ((retval = nc_def_grp(nc_date_group, col_dir_name, &collocation_group_id)))
      {
      if (retval != -42) 
        ERR(retval);
      }

    writeAttributeShort(collocation_group_id, "Baseline_Data_Type", group_type);
    writeAttributeText(collocation_group_id, "Baseline_Platform", description);
    writeAttributeShort(collocation_group_id, "Baseline_Platform_ID", group_id);
    }


  // Create a subgroup for this system platform

  if ((retval = nc_def_grp(nc_date_group, dir_name, &nc_group_id)))
    {
    //if (retval != -42) 
      ERR(retval);
    }

  // Set the platform type attribute to the data group type

  writeAttributeText(nc_group_id, "Platform_Name", description);
  writeAttributeShort(nc_group_id, "Platform_ID", group_id);
  writeAttributeShort(nc_group_id, "Platform_Type", group_type);
  writeAttributeText(nc_group_id, "Platform_Data_Source", "ECMWF Version 2");
  writeAttributeText(nc_group_id, "Platform_NPROVS_Source_Name", nprovs_name);


  // Define the dimensions

  if ((retval = nc_def_dim(nc_group_id, "Num_Collocations", num_colls, &scalar_dim)))
    ERR(retval);

  dimid_scalar[0] = scalar_dim;

  if ((retval = nc_def_dim(nc_group_id, "Levels", 91, &dim_levels)))
    ERR(retval);

  dimid_levels[0] = scalar_dim;
  dimid_levels[1] = dim_levels;

  if (group == 0)
    {
    if ((retval = nc_def_dim(collocation_group_id, "Num_Collocations", num_colls, &col_scalar_dim)))
      ERR(retval);

    col_dimid_scalar[0] = col_scalar_dim;
    }


  // Define the variables

  vid_lat = defineVariable(nc_group_id, "latitude", NC_FLOAT, 1, dimid_scalar, "units", "degrees_north");

  vid_lon = defineVariable(nc_group_id, "longitude", NC_FLOAT, 1, dimid_scalar, "units", "degrees_east");

  vid_date = defineVariable(nc_group_id, "date", NC_INT, 1, dimid_scalar, "format", "yyyymmdd");

  vid_time = defineVariable(nc_group_id, "time", NC_INT, 1, dimid_scalar, "format", "hhmmss");

  vid_press = defineVariable(nc_group_id, "pressure_level", NC_FLOAT, 2, dimid_levels, "units", "hPa");

  vid_temp = defineVariable(nc_group_id, "temperature", NC_FLOAT, 2, dimid_levels, "units", "K");

  vid_wvmr = defineVariable(nc_group_id, "water_vapor_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
                              "units", "g/kg");

  vid_ozone = defineVariable(nc_group_id, "ozone_mixing_ratio", NC_FLOAT, 2, dimid_levels,
			       "units", "kg/kg");

  vid_tpw = defineVariable(nc_group_id, "total_precipitable_water", NC_FLOAT, 1, dimid_scalar, NULL, NULL);

  vid_skin_temp = defineVariable(nc_group_id, "skin_temperature", NC_FLOAT, 1, dimid_scalar,
				   "units", "K");

  vid_qc_temp = defineVariable(nc_group_id, "nprovs_temperature_qc", NC_SHORT, 1, dimid_scalar, 
                               "values", "0=pass, 1=fail");

  vid_qc_wvap = defineVariable(nc_group_id, "nprovs_moisture_qc", NC_SHORT, 1, dimid_scalar, 
                               "values", "0=pass, 1=fail");

  vid_qc_ozone = defineVariable(nc_group_id, "nprovs_ozone_qc", NC_SHORT, 1, dimid_scalar, 
                                "values", "0=pass, 1=fail");


  if (group == 0)
    {
    col_vid_lat = defineVariable(collocation_group_id, "latitude", NC_FLOAT, 1, col_dimid_scalar, 
				   "units", "degrees_north");
    col_vid_lon = defineVariable(collocation_group_id, "longitude", NC_FLOAT, 1, col_dimid_scalar, 
				   "units", "degrees_east");
    col_vid_date = defineVariable(collocation_group_id, "date", NC_INT, 1, col_dimid_scalar, "format", "yyyymmdd");
    col_vid_time = defineVariable(collocation_group_id, "time", NC_INT, 1, col_dimid_scalar, "format", "hhmmss");
    }



  // Allocate the memory used by the data buffer

  buffer = (short *)malloc(group_length);


  // Loop through each record in the file and read the data for 
  // this data group into the buffer

  for (recnum=0; recnum<number_of_collocations; recnum++)
    {
    offset = (long)header_length + ((long)recnum * (long)record_length) + group_offset;
    fseek(in, offset, SEEK_SET);
    fread(buffer, group_length, 1, in);

    //for (i=0; i<group_length/2; i++)
    //  printf("processECMWF02record: buffer[%d] = %d\n",i,buffer[i]);

    // Index and num_vals are used to save a single value within the
    // variable array

    index[0]    = recnum;
    num_vals[0] = 1;

    index_2D[0] = recnum;
    index_2D[1] = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 91;

    // Latitude and longitude

    if (buffer[3] != -32768)
      latitude  = buffer[3] / 128.0;
    else
      latitude = -32768.0;

    if (buffer[4] != -32768)
      longitude = buffer[4] / 128.0;
    else
      longitude = -32768.0;

    writeVariableFloat(nc_group_id, vid_lat, index, num_vals, latitude);
    writeVariableFloat(nc_group_id, vid_lon, index, num_vals, longitude);

    // Date and time

    if ((buffer[5] != -32768) && (buffer[6] != -32768) && (buffer[7] != -32768) && (buffer[8] != -32768))
      {
      year      = buffer[5];
      month     = buffer[6] / 100;
      day       = buffer[6] % 100;
      hour      = buffer[7];
      minute    = buffer[8] / 100;
      second    = buffer[8] % 100;

      yyyymmdd = (year * 10000) + (month * 100) + day;
      hhmmss   = (hour * 10000) + (minute * 100) + second;
      }
    else
      {
      yyyymmdd = -32768;
      hhmmss   = -32768;
      }

    writeVariableInteger(nc_group_id, vid_date, index, num_vals, yyyymmdd);
    writeVariableInteger(nc_group_id, vid_time, index, num_vals, hhmmss);

    // If this is the baseline system, save the collocation information

    if (group == 0)
      {
      writeVariableFloat(collocation_group_id, col_vid_lat, index, num_vals, latitude);
      writeVariableFloat(collocation_group_id, col_vid_lon, index, num_vals, longitude);
      writeVariableInteger(collocation_group_id, col_vid_date, index, num_vals, yyyymmdd);
      writeVariableInteger(collocation_group_id, col_vid_time, index, num_vals, hhmmss);
      }

    // Save the tpw and skin temperature

    if (buffer[282] > 0)
      writeVariableFloat(nc_group_id, vid_tpw, index, num_vals, (buffer[282]/100.0));
    else
      writeVariableFloat(nc_group_id, vid_tpw, index, num_vals, -32768.0);

    if (buffer[283] > 0)
      writeVariableFloat(nc_group_id, vid_skin_temp, index, num_vals, (buffer[283]/64.0));
    else
      writeVariableFloat(nc_group_id, vid_skin_temp, index, num_vals, -32768.0);

    // Pressures

    for (n=0; n<91; n++)
      {
      if (buffer[9+n] == -32768)
	level_values[n] = -32768.0;
      else if (n < 36)
	level_values[n] = buffer[9+n] / 100.0;
      else
	level_values[n] = buffer[9+n] / 10.0;

      //printf("ECMWF PRESS:  %d  %d   %f\n", recnum, n, level_values[n]);
      }

    writeArrayFloat(nc_group_id, vid_press, index_2D, num_vals_2D, level_values);

    // Temperatures

    for (n=0; n<91; n++)
      {
      if (buffer[100+n] == -32768)
	level_values[n] = -32768.0;
      else
	level_values[n] = buffer[100+n] / 64.0;
      }

    writeArrayFloat(nc_group_id, vid_temp, index_2D, num_vals_2D, level_values);

    // Water vapor mixing ratio

    for (n=0; n<91; n++)
      {
      if (buffer[191+n] == -32768)
	level_values[n] = -32768.0;
      else
	//level_values[n] = (float)(exp(buffer[191+n]/1024.0));
	level_values[n] = buffer[191+n] / 1000.0;

      //printf("ECMWF WVMR:  %d  %d   %d   %f\n", recnum, n, buffer[191+n], level_values[n]);
      }

    writeArrayFloat(nc_group_id, vid_wvmr, index_2D, num_vals_2D, level_values);

    // Ozone mixing ratio

    for (n=0; n<91; n++)
      {
      if (buffer[284+n] == -32768)
	level_values[n] = -32768.0;
      else
	level_values[n] = (float)(exp(buffer[284+n]/1024.0));

      //printf("ECMWF OZONE:  %d  %d   %d   %f\n", recnum, n, buffer[284+n], level_values[n]);
      }

    writeArrayFloat(nc_group_id, vid_ozone, index_2D, num_vals_2D, level_values);

    // QC flags

    writeVariableShort(nc_group_id, vid_qc_temp, index, num_vals, buffer[370]);
    writeVariableShort(nc_group_id, vid_qc_wvap, index, num_vals, buffer[371]);
    writeVariableShort(nc_group_id, vid_qc_ozone, index, num_vals, buffer[372]);
    }  // for (recnum=0...


  // Free the memory used by the buffer

  free(buffer);
  }

// end of file
