/*
   Name- NucapsCollocator.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  3/03/2025   Programmer- Mike Pettey (STC)

   Function- This program extracts data from NUCAPS granules and collocates
             the data to selected collocations times and locations.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>

#include <dirent.h>
//#include <errno.h>

#define  MAX_TIME_DIFF       6.0
#define  MAX_DISTANCE_DIFF 200.0

#define  TRUE      0
#define  FALSE     1


struct collocation_data
  {
  int   date;
  int   time;
  float latitude;
  float longitude;

  float closest_value;
  char  closest_granule[500];
  int   closest_footprint_index;
  };

struct collocation_data *collocations;
int num_collocations;


void createCollocationArray(int date_to_process);
void doCollocation();
void buildOutputFile(int date_to_process, int nprovs_platform_id, char *nprovs_platform_name);

//void processNucapsFile(int collocation_date, int collocation_time,
//                       float collocation_lat, float collocation_lon,
//                       int nprovs_platform_id, char *nprovs_platform_name);

//void findClosestFootprint(int collocation_date, int collocation_time,
//                          float collocation_lat, float collocation_lon,
//                          char *closest_file_name, int *closest_index);

float calculateDistanceDifference(float lat1, float lon1, float lat2, float lon2);

float calculateTimeDifference(int date1, int time1, int date2, int time2);

void convertEpochTimeToDateAndTime(double epoch_time, int *yr, int *md, int *hr, int *ms);

int fileEndsWith(char *file_name, char *token);

int defineVariable(int nc_group_id, char *var_name, nc_type var_type, int ndims, int *dimids,
                   char *attr_string, char *attr);

void writeAttributeText(int grp, char *attrVariable, char *attrValue);
//void writeVariableByte(int group_id, int var_id, size_t *index, size_t *num_vals, char value);
//void writeVariableFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float value);
//void writeVariableInteger(int group_id, int var_id, size_t *index, size_t *num_vals, int value);
//void writeVariableShort(int group_id, int var_id, size_t *index, size_t *num_vals, short value);


int filter(const struct dirent *name)
  {
  return 1;
  }



int main(int argc, char *argv[])
  {
  int  date_to_process, nprovs_platform_id;
  char nprovs_platform_name[100];

  
  // Print an initial message

  printf("\nCollocating NUCAPS data to Ozonesondes\n\n");

  // Unpack the date to process

  date_to_process = atoi(argv[1]);
  
  // Unpack the platform ID and platform name

  nprovs_platform_id = atoi(argv[2]);
  sprintf(nprovs_platform_name, "%s", argv[3]);

  printf("Date to process:  %d\n\n", date_to_process);
  printf("NPROVS Platform ID:    %d\n", nprovs_platform_id);
  printf("NPROVS Platform Name:  %s\n\n", nprovs_platform_name);


  // Create the collocation array that will contain information about each
  // collocation

  createCollocationArray(date_to_process);


  // Loop through all of the NUCAPS granules to collocate the closest ones
  // to each collocation

  doCollocation();


  // Create the output file and copy the data to it

  buildOutputFile(date_to_process, nprovs_platform_id, nprovs_platform_name);
  



  
    // Process the input NUCAPS file

//    processNucapsFile(collocation_date, collocation_time, collocation_lat, collocation_lon,
//                      nprovs_platform_id, nprovs_platform_name);


  // Free the memory used for the collocation array

  free(collocations);
  
  printf("\nProcessing completed.\n\n");
  }



//-------------------------------------------------------------------------------
void createCollocationArray(int date_to_process)
  {
  int    coll, retval, nc_id, var_id, dim_id;
  int    *collocation_dates, *collocation_times;
  float  *collocation_lats, *collocation_lons;
  char   file_name[500];
  size_t dimlen;
  size_t index[1], num_vals[1];

  // Open the Collocation_Info file

  sprintf(file_name, "out.dir/%d/Collocation_Info.nc", date_to_process);

  retval = nc_open(file_name, NC_NOWRITE, &nc_id);

  if (retval != NC_NOERR)
    {
    printf("\n\n*** ERROR ***\n\n");
    printf("The Collocaton_Info netCDF file could not be read.\n");
    printf("File name:  %s\n", file_name);
    printf("Execution is stopping.\n\n");
    exit(1);
    }

  // Unpack the number of collocations

  retval = nc_inq_dimid(nc_id, "Num_Collocations", &dim_id);
  retval = nc_inq_dimlen(nc_id, dim_id, &dimlen);

  num_collocations = (int)dimlen;

  printf("Number of collocations:  %d\n", num_collocations);

  index[0]    = 0;
  num_vals[0] = (size_t)num_collocations;
      
  // Unpack the collocation dates

  collocation_dates = (int*)malloc(num_collocations*(sizeof(int)));

  retval = nc_inq_varid(nc_id, "date", &var_id);
  retval = nc_get_vara_int(nc_id, var_id, index, num_vals, collocation_dates);

  // Unpack the collocation times

  collocation_times = (int*)malloc(num_collocations*(sizeof(int)));

  retval = nc_inq_varid(nc_id, "time", &var_id);
  retval = nc_get_vara_int(nc_id, var_id, index, num_vals, collocation_times);

  // Unpack the collocation latitudes

  collocation_lats = (float*)malloc(num_collocations*(sizeof(float)));

  retval = nc_inq_varid(nc_id, "latitude", &var_id);
  retval = nc_get_vara_float(nc_id, var_id, index, num_vals, collocation_lats);

  // Unpack the collocation longitudes

  collocation_lons = (float*)malloc(num_collocations*(sizeof(float)));

  retval = nc_inq_varid(nc_id, "longitude", &var_id);
  retval = nc_get_vara_float(nc_id, var_id, index, num_vals, collocation_lons);


  // Create the collocation array and copy the data into it

  collocations = (struct collocation_data*)malloc(num_collocations*sizeof(struct collocation_data));

  for (coll=0; coll<num_collocations; coll++)
    {
    collocations[coll].date      = collocation_dates[coll];
    collocations[coll].time      = collocation_times[coll];
    collocations[coll].latitude  = collocation_lats[coll];
    collocations[coll].longitude = collocation_lons[coll];

    collocations[coll].closest_value           = 99999999.9;
    collocations[coll].closest_footprint_index = -32768;
    }  // for (coll=0...

  
  // Close the Collocation_Info file

  nc_close(nc_id);

  // Free the collocation data memory

  free(collocation_lons);
  free(collocation_lats);
  free(collocation_times);
  free(collocation_dates);
  }



//-------------------------------------------------------------------------------
void doCollocation()
  {
  int     retval, nc_id, var_id, dim_id;
  int     num_footprints, fp, coll;
  int     dir_index;
  int     yyyy, mmdd, hh, mmss;
  int     *footprint_dates, *footprint_times;
  float   time_diff, dist_diff, time_factor, closeness;
  float   *footprint_lats, *footprint_lons;
  double  *footprint_epoch_times;
  char    file_name[1000];
  size_t  dimlen;
  size_t  index[1], num_vals[1];
  struct dirent **namelist;


  // Get the list of granule files in the input directory

  dir_index = scandir("in.dir", &namelist, filter, alphasort);

  if (dir_index == -1)
    {
    printf("Error with scandir:  in.dir\n");
    exit(1);
    }

  while (dir_index--)
    {
    sprintf(file_name, "in.dir/%s", namelist[dir_index]->d_name);

    // Process all netCDF granule files

    if (fileEndsWith(file_name, ".nc") == TRUE)
      {
      printf("Process File:  %s\n", file_name);

      // Open the granule

      retval = nc_open(file_name, NC_NOWRITE, &nc_id);

      if (retval == NC_NOERR)
	{

	// Unpack the number of footprints

	retval = nc_inq_dimid(nc_id, "Number_of_CrIS_FORs", &dim_id);
	retval = nc_inq_dimlen(nc_id, dim_id, &dimlen);

	num_footprints = (int)dimlen;

	index[0]    = 0;
	num_vals[0] = (size_t)num_footprints;

	// Unpack the footprint latitudes

	footprint_lats = (float*)malloc(num_footprints*(sizeof(float)));

	retval = nc_inq_varid(nc_id, "Latitude", &var_id);
	retval = nc_get_vara_float(nc_id, var_id, index, num_vals, footprint_lats);

	// Unpack the footprint longitudes

	footprint_lons = (float*)malloc(num_footprints*(sizeof(float)));

	retval = nc_inq_varid(nc_id, "Longitude", &var_id);
	retval = nc_get_vara_float(nc_id, var_id, index, num_vals, footprint_lons);

	// Unpack the footprint epoch times

	footprint_epoch_times = (double*)malloc(num_footprints*(sizeof(double)));

	retval = nc_inq_varid(nc_id, "Time", &var_id);
	retval = nc_get_vara_double(nc_id, var_id, index, num_vals, footprint_epoch_times);

	// Convert the epoch times to date and time

	footprint_dates = (int*)malloc(num_footprints*sizeof(int));
	footprint_times = (int*)malloc(num_footprints*sizeof(int));

	for (fp=0; fp<num_footprints; fp++)
	  {
	  convertEpochTimeToDateAndTime(footprint_epoch_times[fp], &yyyy, &mmdd, &hh, &mmss);
	  footprint_dates[fp] = (yyyy * 10000) + mmdd;
	  footprint_times[fp] = (hh * 10000) + mmss;
	  }
	
	// Close the granule file

	nc_close(nc_id);

	// For each collocation, calculate the time and location distances to
	// each footprint and keep track of the closest ones

	for (coll=0; coll<num_collocations; coll++)
	  {
	  for (fp=0; fp<num_footprints; fp++)
	    {
	    time_diff = fabs(calculateTimeDifference(footprint_dates[fp], footprint_times[fp],
						     collocations[coll].date, collocations[coll].time));

            // Continue processing the footprint if the time difference is less than MAX_TIME_DIFF

            if (time_diff <= MAX_TIME_DIFF)
              {
              dist_diff = calculateDistanceDifference(footprint_lats[fp], footprint_lons[fp],
                                                      collocations[coll].latitude, collocations[coll].longitude);

	      // Continue processing the footprint if the distance difference is less than MAX_DISTANCE_DIFF

	      if (dist_diff <= MAX_DISTANCE_DIFF)
		{
                time_factor = 30.0;
		closeness = dist_diff + (time_diff * time_factor);

		if (closeness < collocations[coll].closest_value)
		  {
                  collocations[coll].closest_value = closeness;
		  collocations[coll].closest_footprint_index = fp;
		  sprintf(collocations[coll].closest_granule, "%s", file_name);
		  }  // if (closeness...
		}  // if (dist_diff < MAX...
	      }  // if (time_diff < MAX...
	    }  // for (fp=0...
	  }  // for (coll=0...

	// Free the memory used to store the footprint data

	free(footprint_times);
	free(footprint_dates);
	free(footprint_epoch_times);
	free(footprint_lons);
	free(footprint_lats);
	}  // if (retval==NC_NOERR (nc_open...
      }  // if (fileEndsWith...
    }  // while (dir_index---...
  }



//-------------------------------------------------------------------------------
void buildOutputFile(int date_to_process, int nprovs_platform_id, char *nprovs_platform_name)
  {
  int     n, i, coll, index;
  int     retval, in_id, out_id, dim_id;
  int     in_var_id, out_var_id;
  int     in_var_ndims, in_var_natts, in_var_dimids[NC_MAX_VAR_DIMS];
  int     out_var_ndims, out_var_dimids[NC_MAX_VAR_DIMS];
  int     date_id, time_id;
  int     varnum, num_dims, num_variables, num_attributes, recdim;
  int     num_dice_dim_id, num_values;
  int     yyyymmdd, hhmmss, yyyy, mmdd, hh, mmss;
  double  epoch_time;
  char    output_file_name[1000];
  char    platform_id[10];
  char    var_name[NC_MAX_NAME], attr_name[NC_MAX_NAME], dim_name[NC_MAX_NAME];
  size_t  dim_length;
  size_t  *in_index, *in_count, *out_index, *out_count;
  nc_type var_type;

  char           *byte_array;
  unsigned char  *u_byte_array;
//  const signed char *character_array;
  signed char character;
  short          *i2val_array;
  unsigned short *u_i2val_array;
  int            *i4val_array;
  unsigned int   *u_i4val_array;
  long           *i8val_array;
  long           *u_i8val_array;
  float          *fval_array;
  double         *dval_array;
  char           *string;

  /*
  int     n, index, varnum, retval, in_id, out_id;
  int     closest_footprint_index;
  int     num_dice_dim_id;
  int     num_attributes, num_dims, num_variables;
  int     dim_id, recdim;
  char    closest_file_name[1000];
  char    var_name[NC_MAX_NAME], attr_name[NC_MAX_NAME], dim_name[NC_MAX_NAME];
  size_t  dim_length;

  char  platform_id[5];
  */

  

  // Create the output file name

  sprintf(output_file_name, "out.dir/%d/%s_%d.nc", date_to_process, nprovs_platform_name, nprovs_platform_id);

  for (i=0; i<strlen(output_file_name); i++)
    {
    if (output_file_name[i] == ' ')
      output_file_name[i] = '_';
    }
  
  //printf("Output Name:  %s\n", output_file_name);

  // Create the output netCDF file

  retval = nc_create(output_file_name, NC_NETCDF4, &out_id);

  if (retval == NC_NOERR)
    {

    // Add global attributes

    sprintf(platform_id, "%d", nprovs_platform_id);
    writeAttributeText(out_id, "NPROVS_Platform_ID", platform_id);
    writeAttributeText(out_id, "NPROVS_Platform_Type", "600");
    writeAttributeText(out_id, "NPROVS_Platform_Name", nprovs_platform_name);

    writeAttributeText(out_id, "Conventions", "CF-1.8");
    writeAttributeText(out_id, "Metadata_Conventions", "CF-1.8, Unidata Dataset Discovery v1.0");
    writeAttributeText(out_id, "standard_name_vocabulary", "CF Standard Name Table (version 17, 24 March 2011)");
    writeAttributeText(out_id, "project", "NESDIS Common Cloud Framework");
    writeAttributeText(out_id, "title", "NUCAPS_EDR");
    writeAttributeText(out_id, "summary", "NUCAPS retrieved profiles of temperature, moisture, and trace gases derived from CrIS and ATMS");
    writeAttributeText(out_id, "data_source_institution", "DOC/NOAA/NESDIS/OSPO > Office of Satellite and Product Operations, NESDIS, NOAA, U.S. Department of Commerce.");
    writeAttributeText(out_id, "institution", "DOC/NOAA/NESDIS/STAR > Center for Satellite Applications and Research, NESDIS, NOAA, U.S. Department of Commerce.");
    writeAttributeText(out_id, "naming_authority", "gov.noaa.nesdis.ncei.");
    writeAttributeText(out_id, "instrument_name", "CrIS, ATMS");
    writeAttributeText(out_id, "instrument", "CrIS, ATMS");
    writeAttributeText(out_id, "data_source_creator_name", "DOC/NOAA/NESDIS/STAR > NUCAPS Team, Center for Satellite Applications and Research, NESDIS, NOAA, U.S. Department of Commerce");
    writeAttributeText(out_id, "creator_name", "DOC/NOAA/NESDIS/STAR > NPROVS Team, Center for Satellite Applications and Research, NESDIS, NOAA, U.S. Department of Commerce");
    writeAttributeText(out_id, "keywords", "soundings, temperature, water vapor, ozone, trace gases");
    writeAttributeText(out_id, "history", "Created by NUCAPS version 3.0, Release 2.0");
    writeAttributeText(out_id, "processing_level", "NOAA Level 2 data");
    writeAttributeText(out_id, "geospatial_lat_units", "degrees_north");
    writeAttributeText(out_id, "geospatial_lon_units", "degrees_east");
    writeAttributeText(out_id, "data_source_date_created", "2025-03-01T13:12:33Z");
    writeAttributeText(out_id, "instrument_resolution", "CrIS 2211 channels");

    
    // Find the first collocation that was matched to a footprint and
    // extract netCDF file information from the granule that was matched
    // to the collocation.

    coll = 0;

    while ((coll < num_collocations) && (collocations[coll].closest_footprint_index == -32768))
      {
      coll++;
      }

    if (coll < num_collocations)
      {

      // Open the first granule

      retval = nc_open(collocations[coll].closest_granule, NC_NOWRITE, &in_id);

      if (retval == NC_NOERR)
	{

        // Obtain the number of dimensions, attributes and variables

        ncinquire(in_id, &num_dims, &num_variables, &num_attributes, &recdim);

        // Copy the attributes

        //for (n=0; n<num_attributes; n++)
        //  {
        //  nc_inq_attname(in_id, NC_GLOBAL, n, attr_name);
        //  nc_copy_att(in_id, NC_GLOBAL, attr_name, out_id, NC_GLOBAL);
        //  }

        // Copy the dimensions (except for the number of dice)

        for (n=0; n<num_dims; n++)
          {
          nc_inq_dim(in_id, n, dim_name, &dim_length);

          //if (strcmp(dim_name, "Number_of_Dice") != 0)
          if (strcmp(dim_name, "Number_of_CrIS_FORs") != 0)
            nc_def_dim(out_id, dim_name, (int)dim_length, &dim_id);
          else
            num_dice_dim_id = n;
          }

        // Copy the variables

        for (varnum=0; varnum<num_variables; varnum++)
          {
          retval = nc_inq_varname(in_id, varnum, var_name);
          retval = nc_inq_var(in_id, varnum, 0, &var_type, &in_var_ndims, in_var_dimids, &in_var_natts);
          retval = nc_inq_varid(in_id, var_name, &in_var_id);

          //printf("VAR %d:  %s   %d   %d   %d\n", varnum, var_name, (int)var_type, in_var_ndims, in_var_natts);

          // Update the dimensions with one fewer dimension

          out_var_ndims = in_var_ndims - 1;

          index = -1;

          for (n=0; n<in_var_ndims; n++)
            {
            if (in_var_dimids[n] != num_dice_dim_id)
              {
              index++;

              nc_inq_dim(in_id, in_var_dimids[n], dim_name, &dim_length);
              nc_inq_dimid(out_id, dim_name, &dim_id);
              out_var_dimids[index] = dim_id;
              }
            }

          // Define the variable

          if (strcmp(var_name, "Time") == 0)
            {
            date_id = defineVariable(out_id, "Date", NC_INT, 0, 0, "format", "YYYYMMDD");
            time_id = defineVariable(out_id, "Time", NC_INT, 0, 0, "format", "HHMMSS");
            }
          else
            {
            if (out_var_ndims != 0)
              retval = nc_def_var(out_id, var_name, var_type, out_var_ndims, out_var_dimids, &out_var_id);
            else
              retval = nc_def_var(out_id, var_name, var_type, 0, 0, &out_var_id);

            // Copy the variable attributes

            retval = nc_inq_varnatts(in_id, in_var_id, &num_attributes);

            for (n=0; n<num_attributes; n++)
              {
              nc_inq_attname(in_id, in_var_id, n, attr_name);
              nc_copy_att(in_id, in_var_id, attr_name, out_id, out_var_id);
              }
            }

          // Set up the index and count arrays

          in_index = (size_t*)malloc(in_var_ndims*sizeof(size_t));
          in_count = (size_t*)malloc(in_var_ndims*sizeof(size_t));

          in_index[0] = collocations[coll].closest_footprint_index;
          in_count[0] = 1;

          for (n=1; n<in_var_ndims; n++)
            {
            in_index[n] = 0;

            nc_inq_dim(in_id, in_var_dimids[n], dim_name, &dim_length);
            in_count[n] = dim_length;
            }

          out_index = (size_t*)malloc((out_var_ndims+1)*sizeof(size_t));
          out_count = (size_t*)malloc((out_var_ndims+1)*sizeof(size_t));

          out_index[0] = 0;
          out_count[0] = 1;

          for (n=0; n<out_var_ndims; n++)
            {
            out_index[n] = 0;

            nc_inq_dim(out_id, out_var_dimids[n], dim_name, &dim_length);
            out_count[n] = dim_length;
            }

          // Calculate the number of data values to read for this variable

          num_values = 1;

          if (out_var_ndims != 0)
            {
            for (n=0; n<out_var_ndims; n++)
              {
              nc_inq_dim(out_id, out_var_dimids[n], dim_name, &dim_length);
              num_values = num_values * (int)dim_length;
              }
            }

          // If the variable is "Time" then intercept the copying to convert
          // it from a single variable for Epoch time to two variables
          // for date and time

          if (strcmp(var_name, "Time") == 0)
            {
            retval = nc_get_vara_double(in_id, in_var_id, in_index, in_count, &epoch_time);

            convertEpochTimeToDateAndTime(epoch_time, &yyyy, &mmdd, &hh, &mmss);

            yyyymmdd = (yyyy * 10000) + mmdd;
            hhmmss   = (hh * 10000) + mmss;

            retval = nc_put_var1_int(out_id, date_id, out_index, &yyyymmdd);
            retval = nc_put_var1_int(out_id, time_id, out_index, &hhmmss);
            }
          else
            {

            // Copy the data based on the data type

            if (var_type == NC_FLOAT)
              {
              fval_array = (float*)malloc(num_values*sizeof(float));

              retval = nc_get_vara_float(in_id, in_var_id, in_index, in_count, fval_array);
              retval = nc_put_vara_float(out_id, out_var_id, out_index, out_count, fval_array);

              free (fval_array);
              }  // if (var_type == NC_FLOAT...
            else if (var_type == NC_DOUBLE)
              {
              dval_array = (double*)malloc(num_values*sizeof(double));

              retval = nc_get_vara_double(in_id, in_var_id, in_index, in_count, dval_array);
              retval = nc_put_vara_double(out_id, out_var_id, out_index, out_count, dval_array);

              free (dval_array);
              }  // if (var_type == NC_DOUBLE...
            else if ((var_type == NC_INT) || (var_type == NC_LONG))
              {
              i4val_array = (int*)malloc(num_values*sizeof(int));

              retval = nc_get_vara_int(in_id, in_var_id, in_index, in_count, i4val_array);
              retval = nc_put_vara_int(out_id, out_var_id, out_index, out_count, i4val_array);

              free (i4val_array);
              }  // if (var_type == NC_INT...
            else if (var_type == NC_SHORT)
              {
              i2val_array = (short*)malloc(num_values*sizeof(short));

              retval = nc_get_vara_short(in_id, in_var_id, in_index, in_count, i2val_array);
              retval = nc_put_vara_short(out_id, out_var_id, out_index, out_count, i2val_array);

              free (i2val_array);
              }  // if (var_type == NC_SHORT...
            else if (var_type == NC_INT64)
              {
              i8val_array = (long*)malloc(num_values*sizeof(long));

              retval = nc_get_vara_long(in_id, in_var_id, in_index, in_count, i8val_array);
              retval = nc_put_vara_long(out_id, out_var_id, out_index, out_count, i8val_array);

              free (i8val_array);
              }  // if (var_type == NC_INT64...
            else if (var_type == NC_CHAR)
              {
              //character_array = (constant signed char*)malloc(num_values*sizeof(constant signed char));

              //retval = nc_get_vara_schar(in_id, in_var_id, in_index, in_count, &character);
              //retval = nc_put_vara_schar(out_id, out_var_id, out_index, out_count, character);
              //printf("A char variable was found:  <%c>\n", character);

              //free (character_array);
              }  // if (var_type == NC_CHAR...
            }  // else (variable name is not Time...

//  char           *byte_array;
//  unsigned char  *u_byte_array;
//  char           *character_array;
//  unsigned short *u_i2val_array;
//  unsigned int   *u_i4val_array;
//  long           *u_i8val_array;
//  char           *string;

//NC_BYTE 8-bit signed integer
//NC_UBYTE 8-bit unsigned integer *
//NC_CHAR 8-bit character
//NC_USHORT 16-bit unsigned integer *
//NC_UINT 32-bit unsigned integer *
//NC_UINT64 64-bit unsigned integer *
//NC_STRING variable length character string +


          free(out_count);
          free(out_index);
          free(in_count);
          free(in_index);

          }  // for (varnum=0...




	
	  
	// Unpack the number of footprints

	//retval = nc_inq_dimid(nc_id, "Number_of_CrIS_FORs", &dim_id);
	//retval = nc_inq_dimlen(nc_id, dim_id, &dimlen);

	nc_close(in_id);
	}  // if (retval = no_err on nc_open...
      }  // if (coll < num_collocations...





//:metadata_link = "NUCAPS-EDR_v3r2_j01_s202503011200029_e202503011200327_c202503011312330.nc" ;
//:day_night_data_flag = "night" ;
//:ascend_descend_data_flag = 1 ;

    
//  for (int coll=0; coll<num_collocations; coll++)
//    if (collocations[coll].closest_footprint_index != -32768)
//      printf("Coll:  %d    %d   %s\n", coll, collocations[coll].closest_footprint_index, collocations[coll].closest_granule);



    // Close the output file

    nc_close(out_id);

    }  // if (retval = no error on nc_create...
  }



//-------------------------------------------------------------------------------
//-------------------------------------------------------------------------------

/*

// ===============================================================================
void processNucapsFile(int collocation_date, int collocation_time,
                       float collocation_lat, float collocation_lon,
                       int nprovs_platform_id, char *nprovs_platform_name)
  {
  int     n, index, varnum, retval, in_id, out_id, in_var_id, out_var_id;
  int     date_id, time_id;
  int     num_values;
  int     closest_footprint_index;
  int     num_dice_dim_id;
  int     num_attributes, num_dims, num_variables;
  int     in_var_ndims, in_var_natts, in_var_dimids[NC_MAX_VAR_DIMS];
  int     dim_id, recdim;
  int     out_var_ndims, out_var_dimids[NC_MAX_VAR_DIMS];
  int     yyyymmdd, hhmmss, yyyy, mmdd, hh, mmss;
  double  epoch_time;
  char    closest_file_name[1000];
  char    var_name[NC_MAX_NAME], attr_name[NC_MAX_NAME], dim_name[NC_MAX_NAME];
  size_t  dim_length;
  size_t  *in_index, *in_count, *out_index, *out_count;
  nc_type var_type;

  char           *byte_array;
  unsigned char  *u_byte_array;
//  const signed char *character_array;
  signed char character;
  short          *i2val_array;
  unsigned short *u_i2val_array;
  int            *i4val_array;
  unsigned int   *u_i4val_array;
  long           *i8val_array;
  long           *u_i8val_array;
  float          *fval_array;
  double         *dval_array;
  char           *string;

  char  platform_id[5];


  // Search all of the NUCAPS granules to find a footprint that most closely matches
  // the time and location of the ozonesonde

  findClosestFootprint(collocation_date, collocation_time, collocation_lat, collocation_lon,
                       closest_file_name, &closest_footprint_index);


  //printf("Closest Footprint:   %d    %d  %d    %f  %f\n", closest_footprint_index, collocation_date, collocation_time, collocation_lat, collocation_lon);


  // If a closest footprint was found, then begin copying the data to a
  // new netCDF file

  if (closest_footprint_index != -32768)
    {

    // Open the granule that contains the closest footprint

    retval = nc_open(closest_file_name, NC_NOWRITE, &in_id);

    if (retval == NC_NOERR)
      {

      // Create a new netCDF file

      retval = nc_create("nucaps.out", NC_NETCDF4, &out_id);

      if (retval == NC_NOERR)
        {
        sprintf(platform_id, "%d", nprovs_platform_id);
        writeAttributeText(out_id, "NPROVS_Platform_ID", platform_id);
        writeAttributeText(out_id, "NPROVS_Platform_Type", "600");
        writeAttributeText(out_id, "NPROVS_Platform_Name", nprovs_platform_name);

        // Obtain the number of dimensions, attributes and variables

        ncinquire(in_id, &num_dims, &num_variables, &num_attributes, &recdim);

        // Copy the attributes

        for (n=0; n<num_attributes; n++)
          {
          nc_inq_attname(in_id, NC_GLOBAL, n, attr_name);
          nc_copy_att(in_id, NC_GLOBAL, attr_name, out_id, NC_GLOBAL);
          }

//******** Change the time_coverage_start and time_coverage_end later

        // Copy the dimensions (except for the number of dice)

        for (n=0; n<num_dims; n++)
          {
          nc_inq_dim(in_id, n, dim_name, &dim_length);

          if (strcmp(dim_name, "Number_of_Dice") != 0)
            nc_def_dim(out_id, dim_name, (int)dim_length, &dim_id);
          else
            num_dice_dim_id = n;
          }

        // Copy the variables

        for (varnum=0; varnum<num_variables; varnum++)
          {
          retval = nc_inq_varname(in_id, varnum, var_name);
          retval = nc_inq_var(in_id, varnum, 0, &var_type, &in_var_ndims, in_var_dimids, &in_var_natts);
          retval = nc_inq_varid(in_id, var_name, &in_var_id);

          //printf("VAR %d:  %s   %d   %d   %d\n", varnum, var_name, (int)var_type, in_var_ndims, in_var_natts);

          // Update the dimensions with one fewer dimension

          out_var_ndims = in_var_ndims - 1;

          index = -1;

          for (n=0; n<in_var_ndims; n++)
            {
            if (in_var_dimids[n] != num_dice_dim_id)
              {
              index++;

              nc_inq_dim(in_id, in_var_dimids[n], dim_name, &dim_length);
              nc_inq_dimid(out_id, dim_name, &dim_id);
              out_var_dimids[index] = dim_id;
              }
            }

          // Define the variable

          if (strcmp(var_name, "Time") == 0)
            {
            date_id = defineVariable(out_id, "Date", NC_INT, 0, 0, "format", "YYYYMMDD");
            time_id = defineVariable(out_id, "Time", NC_INT, 0, 0, "format", "HHMMSS");
            }
          else
            {
            if (out_var_ndims != 0)
              retval = nc_def_var(out_id, var_name, var_type, out_var_ndims, out_var_dimids, &out_var_id);
            else
              retval = nc_def_var(out_id, var_name, var_type, 0, 0, &out_var_id);

            // Copy the variable attributes

            retval = nc_inq_varnatts(in_id, in_var_id, &num_attributes);

            for (n=0; n<num_attributes; n++)
              {
              nc_inq_attname(in_id, in_var_id, n, attr_name);
              nc_copy_att(in_id, in_var_id, attr_name, out_id, out_var_id);
              }
            }

          // Set up the index and count arrays

          in_index = (size_t*)malloc(in_var_ndims*sizeof(size_t));
          in_count = (size_t*)malloc(in_var_ndims*sizeof(size_t));

          in_index[0] = closest_footprint_index;
          in_count[0] = 1;

          for (n=1; n<in_var_ndims; n++)
            {
            in_index[n] = 0;

            nc_inq_dim(in_id, in_var_dimids[n], dim_name, &dim_length);
            in_count[n] = dim_length;
            }

          out_index = (size_t*)malloc((out_var_ndims+1)*sizeof(size_t));
          out_count = (size_t*)malloc((out_var_ndims+1)*sizeof(size_t));

          out_index[0] = 0;
          out_count[0] = 1;

          for (n=0; n<out_var_ndims; n++)
            {
            out_index[n] = 0;

            nc_inq_dim(out_id, out_var_dimids[n], dim_name, &dim_length);
            out_count[n] = dim_length;
            }

          // Calculate the number of data values to read for this variable

          num_values = 1;

          if (out_var_ndims != 0)
            {
            for (n=0; n<out_var_ndims; n++)
              {
              nc_inq_dim(out_id, out_var_dimids[n], dim_name, &dim_length);
              num_values = num_values * (int)dim_length;
              }
            }

          // If the variable is "Time" then intercept the copying to convert
          // it from a single variable for Epoch time to two variables
          // for date and time

          if (strcmp(var_name, "Time") == 0)
            {
            retval = nc_get_vara_double(in_id, in_var_id, in_index, in_count, &epoch_time);

            convertEpochTimeToDateAndTime(epoch_time, &yyyy, &mmdd, &hh, &mmss);

            yyyymmdd = (yyyy * 10000) + mmdd;
            hhmmss   = (hh * 10000) + mmss;

            retval = nc_put_var1_int(out_id, date_id, out_index, &yyyymmdd);
            retval = nc_put_var1_int(out_id, time_id, out_index, &hhmmss);
            }
          else
            {

            // Copy the data based on the data type

            if (var_type == NC_FLOAT)
              {
              fval_array = (float*)malloc(num_values*sizeof(float));

              retval = nc_get_vara_float(in_id, in_var_id, in_index, in_count, fval_array);
              retval = nc_put_vara_float(out_id, out_var_id, out_index, out_count, fval_array);

              free (fval_array);
              }  // if (var_type == NC_FLOAT...
            else if (var_type == NC_DOUBLE)
              {
              dval_array = (double*)malloc(num_values*sizeof(double));

              retval = nc_get_vara_double(in_id, in_var_id, in_index, in_count, dval_array);
              retval = nc_put_vara_double(out_id, out_var_id, out_index, out_count, dval_array);

              free (dval_array);
              }  // if (var_type == NC_DOUBLE...
            else if ((var_type == NC_INT) || (var_type == NC_LONG))
              {
              i4val_array = (int*)malloc(num_values*sizeof(int));

              retval = nc_get_vara_int(in_id, in_var_id, in_index, in_count, i4val_array);
              retval = nc_put_vara_int(out_id, out_var_id, out_index, out_count, i4val_array);

              free (i4val_array);
              }  // if (var_type == NC_INT...
            else if (var_type == NC_SHORT)
              {
              i2val_array = (short*)malloc(num_values*sizeof(short));

              retval = nc_get_vara_short(in_id, in_var_id, in_index, in_count, i2val_array);
              retval = nc_put_vara_short(out_id, out_var_id, out_index, out_count, i2val_array);

              free (i2val_array);
              }  // if (var_type == NC_SHORT...
            else if (var_type == NC_INT64)
              {
              i8val_array = (long*)malloc(num_values*sizeof(long));

              retval = nc_get_vara_long(in_id, in_var_id, in_index, in_count, i8val_array);
              retval = nc_put_vara_long(out_id, out_var_id, out_index, out_count, i8val_array);

              free (i8val_array);
              }  // if (var_type == NC_INT64...
            else if (var_type == NC_CHAR)
              {
              //character_array = (constant signed char*)malloc(num_values*sizeof(constant signed char));

              //retval = nc_get_vara_schar(in_id, in_var_id, in_index, in_count, &character);
              //retval = nc_put_vara_schar(out_id, out_var_id, out_index, out_count, character);
              //printf("A char variable was found:  <%c>\n", character);

              //free (character_array);
              }  // if (var_type == NC_CHAR...
            }  // else (variable name is not Time...

//  char           *byte_array;
//  unsigned char  *u_byte_array;
//  char           *character_array;
//  unsigned short *u_i2val_array;
//  unsigned int   *u_i4val_array;
//  long           *u_i8val_array;
//  char           *string;

//NC_BYTE 8-bit signed integer
//NC_UBYTE 8-bit unsigned integer *
//NC_CHAR 8-bit character
//NC_USHORT 16-bit unsigned integer *
//NC_UINT 32-bit unsigned integer *
//NC_UINT64 64-bit unsigned integer *
//NC_STRING variable length character string +


          free(out_count);
          free(out_index);
          free(in_count);
          free(in_index);

          }  // for (varnum=0...


////    //    // Granule file name
////    //
////    //    for (i=0; i<100; i++)
////    //      granule_name[i] = (char)buffer[6314+i];
////    //
////    //    granule_name[100] = '\0';
////    //
////    ////    if (strlen(granule_name) == 0)
////    ////      {
////    ////      for (i=0; i<=99; i++)
////    ////        granule_name[i] = ' ';
////    ////      }
////    //
////    //    string[0] = granule_name;
////    //
////    //    if (retval = nc_put_vara_string(nc_id, vid_granulename, index, num_vals, (const char **)string))
////    //      ERR(retval);
////    //
////    //
////    //    // Program version and release
////    //
////    //    if (buffer[6414] != -32768)
////    //      writeVariableFloat(nc_id, vid_version, index, num_vals, (buffer[6414]/100.0));
////    //    else
////    //      writeVariableFloat(nc_id, vid_version, index, num_vals, -32768.0);
////    //
////    //    if (buffer[6415] != -32768)
////    //      writeVariableFloat(nc_id, vid_release, index, num_vals, (buffer[6415]/100.0));
////    //    else
////    //      writeVariableFloat(nc_id, vid_release, index, num_vals, -32768.0);
////    //    }  // for (recnum=0...



        nc_close(out_id);
        }  // if (retval == NC_NOERR (output_open)...

      nc_close(in_id);
      }  // if (retval == NC_NOERR (input open)...
    }  // if (closest_footprint_index...
  else
    {
    printf("\n\nNo footprints were found within %0.1f km and %0.1f hours.\n", MAX_DISTANCE_DIFF,
                                                                              MAX_TIME_DIFF);
    printf("No collocation has been made for this ozonesonde.\n\n");
    }
  }



// ===============================================================================
//void processCollocations(char *dir_name, int start_date, int end_date)
//  {
//  struct dirent **namelist;
//  int           n;
//  char          full_name[1000];
//
//  //printf("\n\nDIR NAME: %s\n", dir_name);
//
//  n = scandir(dir_name, &namelist, filter, alphasort);
//
//  if (n == -1)
//    {
//    printf("Error with scandir:  %s\n", dir_name);
//    exit(1);
//    }
//
//  while (n--)
//    {
//    //printf("\n\nDIR NAME: %s\n", namelist[n]->d_name);
//
//    if (namelist[n]->d_type == DT_DIR)
//      {
//      if ((strcmp(namelist[n]->d_name, ".") != 0) && (strcmp(namelist[n]->d_name, "..") != 0))
//        {
//        sprintf(full_name, "%s/%s", dir_name, namelist[n]->d_name);
//        processCollocations(full_name, start_date, end_date);
//        }
//      }
//    else if (namelist[n]->d_type == DT_REG)
//      {
//      sprintf(full_name, "%s/%s", dir_name, namelist[n]->d_name);
//      processFile(full_name, start_date, end_date);
//      }
//
//    free(namelist[n]);
//    }
//
//  free(namelist);
//  }



//void processFile(char *file_name, int start_date, int end_date)
//  {
//  int     n, retval, nc_id;
//  int     launch_date;
//  char    *date_string, full_file_name[1000], matching_ending[25];
//  size_t  attr_length;
//  struct dirent **namelist;
//
//
//  if (fileEndsWith(file_name, ".nc") == TRUE)
//    {
//
//    // Open netCDF file and pull out date, time, lat and lon
//
//    retval = nc_open(file_name, NC_NOWRITE, &nc_id);
//
//    if (retval == NC_NOERR)
//      {
//
//      // Unpack the date
//
//      retval = nc_inq_attlen(nc_id, NC_GLOBAL, "Date_(yyyymmdd)", &attr_length);
//
//      date_string = (char *)malloc(attr_length + 1);
//
//      retval = nc_get_att_text(nc_id, NC_GLOBAL, "Date_(yyyymmdd)", date_string);
//
//      date_string[attr_length] = '\0';
//
//      launch_date = atoi(date_string);
//
//      free(date_string);
//
//      // Close the input netCDF file
//
//      nc_close(nc_id);
//
//
//      // Process the file if the launch date is within the date range
//
//      if ((launch_date >= start_date) && (launch_date <= end_date))
//        {
//          n = scandir("nucaps.dir", &namelist, filter, alphasort);
//
//          if (n == -1)
//            {
//            printf("Error with scandir:  nucaps.dir\n");
//            exit(1);
//            }
//
//          while (n--)
//            {
//            if (namelist[n]->d_type == DT_REG)
//              {
//              sprintf(full_file_name, "nucaps.dir/%s", namelist[n]->d_name);
//              sprintf(matching_ending, "%d.iddf", launch_date);
//
//              if (fileEndsWith(full_file_name, matching_ending) == TRUE)
//                {
//                processNucapsFile(full_file_name);
//                }
//              }
//
//            free(namelist[n]);
//            }
//
//          free(namelist);
//        }  // if (launch_date...
//      }  // if (retval (nc_open...
//    }  // if (fileEndsWith...
//  }


*/

//---------------------------------------------------------------------
int fileEndsWith(char *file_name, char *token)
  {
  int i, file_length, token_length;
  int ends_with = TRUE;

  file_length  = strlen(file_name);
  token_length = strlen(token);

  for (i=0; i<token_length; i++)
    {
    if (file_name[file_length-i] != token[token_length-i])
      ends_with = FALSE;
    }

  return ends_with;
  }

// end of file
