/*
   Name- processNUCAPS02.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  1/30/2017   Programmer- Mike Pettey (IMSG)

   Function- This program extracts data from P72 EDR files in HDF5 format
             and writes the data to an EDGE orbital file.  The selected
             data is defined in the run script.  Also in the run
             script are the starting date (YYYYMMDD) and hour and
             the ending date and hour.
*/


#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <math.h>
#include <netcdf.h>

#include "nprovs_to_netcdf.h"



#define ERR(e) {printf("Error: %s, error num = %d\n", nc_strerror(e),e); return 2;}

#define NDIMS 1


int defineVariable(int nc_group_id, char *var_name, nc_type var_type, int ndims, int *dimids, char *attr_string, char *attr);

void writeVariableFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float value);

void writeVariableShort(int group_id, int var_id, size_t *index, size_t *num_vals, short value);

void writeVariableInteger(int group_id, int var_id, size_t *index, size_t *num_vals, long value);

void writeArrayFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float *value);

float* calculateRelativeHumidities(float *temp_pressures, float *temps, int num_temp_levels, 
				   float *wvap_pressures, float *wvaps, int num_wvap_levels,
                                   float surf_pressure);

float calculateTPW(float *wvap_pressures, float *wvaps, int num_wvap_levels,
		   float surf_pressure, float surf_wvap);



processNUCAPS02(FILE *in, int header_length, int record_length, int number_of_collocations,
		int group, char *nprovs_name, char *description, int group_id, char *dir_name, 
		int group_type, int group_length, long group_offset, int nc_date_group)
  {
  int     n, i, recnum, retval;
  int     nc_group_id, collocation_group_id;
  size_t  num_colls, index[1], num_vals[1], index_2D[2], num_vals_2D[2];

  float   missing_float, latitude, longitude, level_values[100];
  float   temp_pressures[100], temps[100], wvap_pressures[100], wvaps[100], surf_pressure;
  float   *relative_humidities, tpw;
  float   rspares[262], clouds[8], spectral_values[16];
  short   ispares[129];
  int     year, month, day, hour, minute, second;
  long    offset, yyyymmdd, hhmmss;
  char    col_dir_name[50], byte_values[100], granule_name[101];

  char    *string[1];

  short   *buffer;

  int     col_scalar_dim;
  int     scalar_dim, dim_levels;
  int     dim_spectralpts, dim_hingepts, dim_cldlayers;
  int     dim_stabparms, dim_ispares, dim_rspares;

  int     dimid_scalar[1], col_dimid_scalar[1];
  int     dimid_levels[2], dimid_spectralpts[2], dimid_hingepts[2], dimid_cldlayers[2];
  int     dimid_stabparms[2], dimid_ispares[2], dimid_rspares[2];

  int     col_vid_lat, col_vid_lon, col_vid_date, col_vid_time;
  int     vid_lat, vid_lon, vid_date, vid_time;
  int     vid_node, vid_granule, vid_scanline, vid_fov, vid_dice, vid_qc;
  int     vid_solzen, vid_viewangle, vid_satheight, vid_avgco2;
  int     vid_surfheight, vid_landfrac, vid_surfclass, vid_surfemmis;
  int     vid_surfpress, vid_skintemp, vid_mit_skintemp, vid_fg_skintemp;
  int     vid_ispares, vid_rspares, vid_numspectral, vid_numhinge, vid_cldhinge;
  int     vid_numcld, vid_ctp, vid_ctf;
  int     vid_pressure, vid_effpress, vid_temp, vid_mittemp, vid_fgtemp;
  int     vid_wvmr, vid_mitwvmr, vid_fgwvmr, vid_wvcd, vid_mitwvcd, vid_fgwvcd;
  int     vid_rh, vid_mitrh, vid_fgrh, vid_tpw, vid_mittpw, vid_fgtpw;
  int     vid_o3lcd, vid_fgo3lcd, vid_o3mr, vid_fgo3mr;
  int     vid_h2olcd, vid_h2omr, vid_iceflag, vid_colcd, vid_comr;
  int     vid_ch4lcd, vid_ch4mr, vid_co2mr, vid_hno3lcd, vid_hno3mr;
  int     vid_n2olcd, vid_n2omr, vid_so2lcd, vid_so2mr;
  int     vid_mwfreq, vid_mwemiss, vid_mitemiss;
  int     vid_irhpfreq, vid_fgirhpfreq, vid_irsurfemiss, vid_fgirsurfemiss;
  int     vid_surfreflect, vid_stability, vid_cldfreq, vid_cldemiss, vid_cldreflect;
  int     vid_granulename, vid_version, vid_release;


  num_colls = (size_t)number_of_collocations;

  missing_float = -32768.0;

  // If this is the baseline platform, then create another subgroup
  // for the collocation information

  if (group == 0)
    {
    sprintf(col_dir_name, "Collocation_Info\0");

    if ((retval = nc_def_grp(nc_date_group, col_dir_name, &collocation_group_id)))
      {
      if (retval != -42) 
        ERR(retval);
      }

    writeAttributeShort(collocation_group_id, "Baseline_Data_Type", group_type);
    writeAttributeText(collocation_group_id, "Baseline_Platform", description);
    writeAttributeShort(collocation_group_id, "Baseline_Platform_ID", group_id);
    }


  // Create a subgroup for this system platform

  if ((retval = nc_def_grp(nc_date_group, dir_name, &nc_group_id)))
    {
    //if (retval != -42) 
      ERR(retval);
    }

  // Set the platform type attribute to the data group type

  writeAttributeText(nc_group_id, "Platform_Name", description);
  writeAttributeShort(nc_group_id, "Platform_ID", group_id);
  writeAttributeShort(nc_group_id, "Platform_Type", group_type);
  writeAttributeText(nc_group_id, "Platform_Data_Source", "NUCAPS Version 2");
  writeAttributeText(nc_group_id, "Platform_NPROVS_Source_Name", nprovs_name);


  // Define the dimensions

  if ((retval = nc_def_dim(nc_group_id, "Num_Collocations", num_colls, &scalar_dim)))
    ERR(retval);

  dimid_scalar[0] = scalar_dim;

  // levels

  if ((retval = nc_def_dim(nc_group_id, "Levels", 100, &dim_levels)))
    ERR(retval);

  dimid_levels[0] = scalar_dim;
  dimid_levels[1] = dim_levels;

  // spectral points

  if ((retval = nc_def_dim(nc_group_id, "Spectral_Pts", 16, &dim_spectralpts)))
    ERR(retval);

  dimid_spectralpts[0] = scalar_dim;
  dimid_spectralpts[1] = dim_spectralpts;

  // hinge points

  if ((retval = nc_def_dim(nc_group_id, "Hinge_Pts", 100, &dim_hingepts)))
    ERR(retval);

  dimid_hingepts[0] = scalar_dim;
  dimid_hingepts[1] = dim_hingepts;

  // cloud layers

  if ((retval = nc_def_dim(nc_group_id, "Cloud_Layers", 8, &dim_cldlayers)))
    ERR(retval);

  dimid_cldlayers[0] = scalar_dim;
  dimid_cldlayers[1] = dim_cldlayers;

  // stability parameters

  if ((retval = nc_def_dim(nc_group_id, "Stability_Parameters", 16, &dim_stabparms)))
    ERR(retval);

  dimid_stabparms[0] = scalar_dim;
  dimid_stabparms[1] = dim_stabparms;

  // i spares

  if ((retval = nc_def_dim(nc_group_id, "Ispares", 129, &dim_ispares)))
    ERR(retval);

  dimid_ispares[0] = scalar_dim;
  dimid_ispares[1] = dim_ispares;

  // r spares

  if ((retval = nc_def_dim(nc_group_id, "Rspares", 262, &dim_rspares)))
    ERR(retval);

  dimid_rspares[0] = scalar_dim;
  dimid_rspares[1] = dim_rspares;

  if (group == 0)
    {
    if ((retval = nc_def_dim(collocation_group_id, "Num_Collocations", num_colls, &col_scalar_dim)))
      ERR(retval);

    col_dimid_scalar[0] = col_scalar_dim;
    }


  // Define the variables

  if (group == 0)
    {
    col_vid_lat = defineVariable(collocation_group_id, "latitude", NC_FLOAT, 1, col_dimid_scalar, 
				   "units", "degrees_north");
    col_vid_lon = defineVariable(collocation_group_id, "longitude", NC_FLOAT, 1, col_dimid_scalar, 
				   "units", "degrees_east");
    col_vid_date = defineVariable(collocation_group_id, "date", NC_INT, 1, col_dimid_scalar, "format", "yyyymmdd");
    col_vid_time = defineVariable(collocation_group_id, "time", NC_INT, 1, col_dimid_scalar, "format", "hhmmss");
    }

  vid_lat = defineVariable(nc_group_id, "latitude", NC_FLOAT, 1, dimid_scalar, "units", "degrees_north");

  vid_lon = defineVariable(nc_group_id, "longitude", NC_FLOAT, 1, dimid_scalar, "units", "degrees_east");

  vid_date = defineVariable(nc_group_id, "date", NC_INT, 1, dimid_scalar, "format", "yyyymmdd");

  vid_time = defineVariable(nc_group_id, "time", NC_INT, 1, dimid_scalar, "format", "hhmmss");

  vid_node = defineVariable(nc_group_id, "orbital_node", NC_BYTE, 1, dimid_scalar,
			    "values", "0=ascending, 1=descending");

  vid_granule = defineVariable(nc_group_id, "granule_number", NC_SHORT, 1, dimid_scalar,
			       NULL, NULL);

  vid_scanline = defineVariable(nc_group_id, "scan_line", NC_SHORT, 1, dimid_scalar,
				NULL, NULL);

  vid_fov = defineVariable(nc_group_id, "field_of_view", NC_SHORT, 1, dimid_scalar,
			   NULL, NULL);

  vid_dice = defineVariable(nc_group_id, "index_within_granule", NC_SHORT, 1, dimid_scalar,
			    NULL, NULL);

  vid_qc = defineVariable(nc_group_id, "retrieval_quality_flag", NC_BYTE, 1, dimid_scalar,
			  "values", "0=accepted, 1=reject_physical, 2=reject_MIT, 4=reject_NOAA_reg, 8=reject_iMIT, 9=rej_phy_and_iMIT, 16=reject_iNOAA, 17=reject_phy_and_iNOAA, 24=reject_iMIT_and_iNOAA, 25=reject_phy_and_iMIT_and_iNOAA");

  vid_solzen = defineVariable(nc_group_id, "solar_zenith_angle", NC_FLOAT, 1, dimid_scalar,
			      "units", "deg");

  vid_viewangle = defineVariable(nc_group_id, "view_angle", NC_FLOAT, 1, dimid_scalar,
				 "units", "deg");

  vid_satheight = defineVariable(nc_group_id, "satellite_height", NC_FLOAT, 1, dimid_scalar,
				 "units", "km");

  vid_avgco2 = defineVariable(nc_group_id, "column_averaged_CO2", NC_FLOAT, 1, dimid_scalar,
			      "units", "ppm");

  vid_surfheight = defineVariable(nc_group_id, "surface_height", NC_SHORT, 1, dimid_scalar,
				  "units", "m");

  vid_landfrac = defineVariable(nc_group_id, "land_fraction", NC_FLOAT, 1, dimid_scalar, NULL, NULL);

  vid_surfclass = defineVariable(nc_group_id, "microwave_surface_class", NC_BYTE, 1, dimid_scalar, 
				 "values", "0=coast, 1=land, 2=sea, 3=ice, 4=ice, 5=snow, 6=glacier, 7=snow");

  vid_surfemmis = defineVariable(nc_group_id, "surface_emmisivity", NC_FLOAT, 1, dimid_scalar,
				 NULL, NULL);

  vid_surfpress = defineVariable(nc_group_id, "surface_pressure", NC_FLOAT, 1, dimid_scalar,
				 "units", "hPa");

  vid_skintemp = defineVariable(nc_group_id, "skin_temperature", NC_FLOAT, 1, dimid_scalar,
				"units", "K");

  vid_mit_skintemp = defineVariable(nc_group_id, "microwave_skin_temperature", NC_FLOAT, 1, dimid_scalar,
				    "units", "K");

  vid_fg_skintemp = defineVariable(nc_group_id, "first_guess_skin_temperature", NC_FLOAT, 1, dimid_scalar,
				   "units", "K");

  vid_ispares = defineVariable(nc_group_id, "i_spare", NC_SHORT, 2, dimid_ispares, NULL, NULL);

  vid_rspares = defineVariable(nc_group_id, "r_spare", NC_FLOAT, 2, dimid_rspares, NULL, NULL);

  vid_numspectral = defineVariable(nc_group_id, "number_of_microwave_spectral_points_per_fov", NC_SHORT, 1, dimid_scalar,
				   NULL, NULL);

  vid_numhinge = defineVariable(nc_group_id, "number_of_surf_emiss_hinge_points_per_fov", NC_SHORT, 1, dimid_scalar,
				NULL, NULL);

  vid_cldhinge = defineVariable(nc_group_id, "number_of_cloud_emiss_hinge_points_per_fov", NC_SHORT, 1, dimid_scalar,
				NULL, NULL);

  vid_numcld = defineVariable(nc_group_id, "number_of_cloud_layers_per_fov", NC_SHORT, 1, dimid_scalar,
			      NULL, NULL);

  vid_ctp = defineVariable(nc_group_id, "cloud_top_pressure", NC_FLOAT, 2, dimid_cldlayers, "unit", "hPa");

  vid_ctf = defineVariable(nc_group_id, "cloud_top_fraction", NC_FLOAT, 2, dimid_cldlayers, NULL, NULL);

  vid_pressure = defineVariable(nc_group_id, "pressure", NC_FLOAT, 2, dimid_levels, "units", "hPa");

  vid_effpress = defineVariable(nc_group_id, "effective_pressure", NC_FLOAT, 2, dimid_levels, "units", "hPa");

  vid_temp = defineVariable(nc_group_id, "temperature", NC_FLOAT, 2, dimid_levels, "units", "K");

  vid_mittemp = defineVariable(nc_group_id, "microwave_temperature", NC_FLOAT, 2, dimid_levels, "units", "K");

  vid_fgtemp = defineVariable(nc_group_id, "first_guess_temperature", NC_FLOAT, 2, dimid_levels, "units", "K");

  vid_wvmr = defineVariable(nc_group_id, "water_vapor_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			    "units", "g/g");

  vid_mitwvmr = defineVariable(nc_group_id, "microwave_water_vapor_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			       "units", "g/g");

  vid_fgwvmr = defineVariable(nc_group_id, "first_guess_water_vapor_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			      "units", "g/g");

  //vid_wvcd = defineVariable(nc_group_id, "water_vapor_column_density", NC_FLOAT, 2, dimid_levels, 
  //			    "units", "mol/cm^2");
  vid_wvcd = defineVariable(nc_group_id, "water_vapor_column_density", NC_FLOAT, 2, dimid_levels, 
			    "units", "dobson");

  //vid_mitwvcd = defineVariable(nc_group_id, "microwave_water_vapor_column_density", NC_FLOAT, 2, dimid_levels, 
  //			       "units", "mol/cm^2");
  vid_mitwvcd = defineVariable(nc_group_id, "microwave_water_vapor_column_density", NC_FLOAT, 2, dimid_levels, 
			       "units", "dobson");

  //vid_fgwvcd = defineVariable(nc_group_id, "first_guess_water_vapor_column_density", NC_FLOAT, 2, dimid_levels, 
  //			      "units", "mol/cm^2");
  vid_fgwvcd = defineVariable(nc_group_id, "first_guess_water_vapor_column_density", NC_FLOAT, 2, dimid_levels, 
			      "units", "dobson");

  //vid_o3lcd = defineVariable(nc_group_id, "ozone_layer_column_density", NC_FLOAT, 2, dimid_levels, 
  //			     "units", "mol/cm^2");
  vid_o3lcd = defineVariable(nc_group_id, "ozone_layer_column_density", NC_FLOAT, 2, dimid_levels, 
			     "units", "dobson");

  //vid_fgo3lcd = defineVariable(nc_group_id, "first_guess_ozone_layer_column_density", NC_FLOAT, 2, dimid_levels, 
  //			       "units", "mol/cm^2");
  vid_fgo3lcd = defineVariable(nc_group_id, "first_guess_ozone_layer_column_density", NC_FLOAT, 2, dimid_levels, 
			       "units", "dobson");

  vid_o3mr = defineVariable(nc_group_id, "ozone_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			    "units", "ppb");

  vid_fgo3mr = defineVariable(nc_group_id, "first_guess_ozone_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			      "units", "ppb");

  //vid_h2olcd = defineVariable(nc_group_id, "liquid_water_layer_column_density", NC_FLOAT, 2, dimid_levels, 
  //			      "units", "mol/cm^2");
  vid_h2olcd = defineVariable(nc_group_id, "liquid_water_layer_column_density", NC_FLOAT, 2, dimid_levels, 
			      "units", "dobson");

  vid_h2omr = defineVariable(nc_group_id, "liquid_water_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			     "units", "g/g");

  vid_iceflag = defineVariable(nc_group_id, "ice_liquid_flag", NC_BYTE, 2, dimid_levels, 
			       "values", "0=water, 1=ice");

  //vid_colcd = defineVariable(nc_group_id, "carbon_monoxide_layer_column_density", NC_FLOAT, 2, dimid_levels, 
  //			     "units", "mol/cm^2");
  vid_colcd = defineVariable(nc_group_id, "carbon_monoxide_layer_column_density", NC_FLOAT, 2, dimid_levels, 
			     "units", "dobson");

  vid_comr = defineVariable(nc_group_id, "carbon_monoxide_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			     "units", "ppb");

  //vid_ch4lcd = defineVariable(nc_group_id, "methane_layer_column_density", NC_FLOAT, 2, dimid_levels, 
  //			      "units", "mol/cm^2");
  vid_ch4lcd = defineVariable(nc_group_id, "methane_layer_column_density", NC_FLOAT, 2, dimid_levels, 
			      "units", "dobson");

  vid_ch4mr = defineVariable(nc_group_id, "methane_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			      "units", "ppb");

  vid_co2mr = defineVariable(nc_group_id, "carbon_dioxide_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			     "units", "ppb");

  //vid_hno3lcd = defineVariable(nc_group_id, "nitric_acid_layer_column_density", NC_FLOAT, 2, dimid_levels, 
  //			       "units", "mol/cm^2");
  vid_hno3lcd = defineVariable(nc_group_id, "nitric_acid_layer_column_density", NC_FLOAT, 2, dimid_levels, 
			       "units", "dobson");

  vid_hno3mr = defineVariable(nc_group_id, "nitric_acid_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			      "units", "ppb");

  //vid_n2olcd = defineVariable(nc_group_id, "nitrous_oxide_layer_column_density", NC_FLOAT, 2, dimid_levels, 
  //			       "units", "mol/cm^2");
  vid_n2olcd = defineVariable(nc_group_id, "nitrous_oxide_layer_column_density", NC_FLOAT, 2, dimid_levels, 
			       "units", "dobson");

  vid_n2omr = defineVariable(nc_group_id, "nitrous_oxide_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			     "units", "ppb");

  //vid_so2lcd = defineVariable(nc_group_id, "sulfur_dioxide_layer_column_density", NC_FLOAT, 2, dimid_levels, 
  //			      "units", "mol/cm^2");
  vid_so2lcd = defineVariable(nc_group_id, "sulfur_dioxide_layer_column_density", NC_FLOAT, 2, dimid_levels, 
			      "units", "dobson");

  vid_so2mr = defineVariable(nc_group_id, "sulfur_dioxide_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
			     "units", "ppb");

  vid_mwfreq = defineVariable(nc_group_id, "microwave_frequency", NC_FLOAT, 2, dimid_spectralpts, 
			      "units", "cm^-1");

  vid_mwemiss = defineVariable(nc_group_id, "microwave_emissivity", NC_FLOAT, 2, dimid_spectralpts, 
			       "units", "cm^-1");

  vid_mitemiss = defineVariable(nc_group_id, "mit_microwave_emissivity", NC_FLOAT, 2, dimid_spectralpts, 
				"units", "cm^-1");

  vid_irhpfreq = defineVariable(nc_group_id, "ir_emiss_hinge_pt_frequency", NC_FLOAT, 2, dimid_levels, 
				"units", "cm^-1");

  vid_fgirhpfreq = defineVariable(nc_group_id, "fg_ir_emiss_hinge_pt_frequency", NC_FLOAT, 2, dimid_levels, 
				  "units", "cm^-1");

  vid_irsurfemiss = defineVariable(nc_group_id, "ir_surface_emissivity", NC_FLOAT, 2, dimid_levels, 
				   NULL, NULL);

  vid_fgirsurfemiss = defineVariable(nc_group_id, "fg_ir_surface_emissivity", NC_FLOAT, 2, dimid_levels, 
				     NULL, NULL);

  vid_surfreflect = defineVariable(nc_group_id, "ir_surface_reflectance", NC_FLOAT, 2, dimid_levels, 
				   "units", "percent");

  vid_stability = defineVariable(nc_group_id, "stability_parameters", NC_FLOAT, 2, dimid_stabparms, 
				 NULL, NULL);

//  vid_cldfreq = defineVariable(nc_group_id, "cloud_ir_frequency", NC_FLOAT, 2, dimid_stabparms, 
//			       "units", "cm^-1");

//  vid_cldemiss = defineVariable(nc_group_id, "cloud_ir_emissivity", NC_FLOAT, 2, dimid_stabparms, 
//				NULL, NULL);

//  vid_cldreflect = defineVariable(nc_group_id, "cloud_ir_reflectivity", NC_FLOAT, 2, dimid_stabparms, 
//				  "units", "percent");


  vid_rh = defineVariable(nc_group_id, "relative_humidity", NC_FLOAT, 2, dimid_levels, 
			  "units", "percent");

  vid_mitrh = defineVariable(nc_group_id, "microwave_relative_humidity", NC_FLOAT, 2, dimid_levels, 
			     "units", "percent");

  vid_fgrh = defineVariable(nc_group_id, "first_guess_relative_humidity", NC_FLOAT, 2, dimid_levels, 
			    "units", "percent");

  vid_tpw = defineVariable(nc_group_id, "total_precipitable_water", NC_FLOAT, 1, dimid_scalar,
			   "units", "mm");

  vid_mittpw = defineVariable(nc_group_id, "microwave_total_precipitable_water", NC_FLOAT, 1, dimid_scalar,
			      "units", "mm");

  vid_fgtpw = defineVariable(nc_group_id, "first_guess_total_precipitable_water", NC_FLOAT, 1, dimid_scalar,
			     "units", "mm");

  vid_granulename = defineVariable(nc_group_id, "granule_file_name", NC_STRING, 1, dimid_scalar, 
				   "string_length", "101");

  vid_version = defineVariable(nc_group_id, "program_version", NC_FLOAT, 1, dimid_scalar, NULL, NULL);

  vid_release = defineVariable(nc_group_id, "program_release", NC_FLOAT, 1, dimid_scalar, NULL, NULL);


  // Allocate the memory used by the data buffer

  buffer = (short *)malloc(group_length);


  // Loop through each record in the file and read the data for 
  // this data group into the buffer

  for (recnum=0; recnum<number_of_collocations; recnum++)
    {
    offset = (long)header_length + ((long)recnum * (long)record_length) + group_offset;
    fseek(in, offset, SEEK_SET);
    fread(buffer, group_length, 1, in);

    //for (i=0; i<group_length/2; i++)
    //  printf("processNUCAPS02record: buffer[%d] = %d\n",i,buffer[i]);

    // Index and num_vals are used to save a single value within the
    // variable array

    index[0]    = recnum;
    num_vals[0] = 1;

    index_2D[0] = recnum;
    index_2D[1] = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 91;

    // Latitude and longitude

    if (buffer[3] != -32768)
      latitude  = buffer[3] / 128.0;
    else
      latitude = -32768.0;

    if (buffer[4] != -32768)
      longitude = buffer[4] / 128.0;
    else
      longitude = -32768.0;

    writeVariableFloat(nc_group_id, vid_lat, index, num_vals, latitude);
    writeVariableFloat(nc_group_id, vid_lon, index, num_vals, longitude);

    // Date and time

    if ((buffer[5] != -32768) && (buffer[6] != -32768) && (buffer[7] != -32768) && (buffer[8] != -32768))
      {
      year      = buffer[5];
      month     = buffer[6] / 100;
      day       = buffer[6] % 100;
      hour      = buffer[7];
      minute    = buffer[8] / 100;
      second    = buffer[8] % 100;

      yyyymmdd = (year * 10000) + (month * 100) + day;
      hhmmss   = (hour * 10000) + (minute * 100) + second;
      }
    else
      {
      yyyymmdd = -32768;
      hhmmss   = -32768;
      }

    writeVariableInteger(nc_group_id, vid_date, index, num_vals, yyyymmdd);
    writeVariableInteger(nc_group_id, vid_time, index, num_vals, hhmmss);

    // If this is the baseline system, save the collocation information

    if (group == 0)
      {
      writeVariableFloat(collocation_group_id, col_vid_lat, index, num_vals, latitude);
      writeVariableFloat(collocation_group_id, col_vid_lon, index, num_vals, longitude);
      writeVariableInteger(collocation_group_id, col_vid_date, index, num_vals, yyyymmdd);
      writeVariableInteger(collocation_group_id, col_vid_time, index, num_vals, hhmmss);
      }

    // Save the orbital node

    if (buffer[11] != -32768)
      writeVariableByte(nc_group_id, vid_node, index, num_vals, (char)buffer[11]);
    else
      writeVariableByte(nc_group_id, vid_node, index, num_vals, (char)-128);

    writeVariableShort(nc_group_id, vid_granule, index, num_vals, buffer[12]);
    writeVariableShort(nc_group_id, vid_scanline, index, num_vals, buffer[13]);
    writeVariableShort(nc_group_id, vid_fov, index, num_vals, buffer[14]);
    writeVariableShort(nc_group_id, vid_dice, index, num_vals, buffer[15]);

    // Retrieval quality flag

    if (buffer[16] != -32768)
      writeVariableByte(nc_group_id, vid_qc, index, num_vals, (char)buffer[16]);
    else
      writeVariableByte(nc_group_id, vid_qc, index, num_vals, (char)-128);

    if (buffer[19] != -32768)
      writeVariableFloat(nc_group_id, vid_solzen, index, num_vals, (buffer[19]/128.0));
    else
      writeVariableFloat(nc_group_id, vid_solzen, index, num_vals, -32768.0);

    if (buffer[20] != -32768)
      writeVariableFloat(nc_group_id, vid_viewangle, index, num_vals, (buffer[20]/64.0));
    else
      writeVariableFloat(nc_group_id, vid_viewangle, index, num_vals, -32768.0);

    if (buffer[21] != -32768)
      writeVariableFloat(nc_group_id, vid_satheight, index, num_vals, (buffer[21]/10.0));
    else
      writeVariableFloat(nc_group_id, vid_satheight, index, num_vals, -32768.0);

    if (buffer[22] != -32768)
      writeVariableFloat(nc_group_id, vid_avgco2, index, num_vals, (buffer[22]/10.0));
    else
      writeVariableFloat(nc_group_id, vid_avgco2, index, num_vals, -32768.0);

    writeVariableShort(nc_group_id, vid_surfheight, index, num_vals, buffer[23]);

    if (buffer[24] != -32768)
      writeVariableFloat(nc_group_id, vid_landfrac, index, num_vals, (buffer[24]/100.0));
    else
      writeVariableFloat(nc_group_id, vid_landfrac, index, num_vals, -32768.0);

    if (buffer[25] != -32768)
      writeVariableByte(nc_group_id, vid_surfclass, index, num_vals, (char)buffer[25]);
    else
      writeVariableByte(nc_group_id, vid_surfclass, index, num_vals, (char)-128);

    if (buffer[26] != -32768)
      writeVariableFloat(nc_group_id, vid_surfemmis, index, num_vals, (buffer[26]/100.0));
    else
      writeVariableFloat(nc_group_id, vid_surfemmis, index, num_vals, -32768.0);

    if (buffer[27] != -32768)
      writeVariableFloat(nc_group_id, vid_surfpress, index, num_vals, (buffer[27]/10.0));
    else
      writeVariableFloat(nc_group_id, vid_surfpress, index, num_vals, -32768.0);

    if ((buffer[28] != -32768) && (buffer[28] != 6400))
      writeVariableFloat(nc_group_id, vid_skintemp, index, num_vals, (buffer[28]/64.0));
    else
      writeVariableFloat(nc_group_id, vid_skintemp, index, num_vals, -32768.0);

    if (buffer[29] != -32768)
      writeVariableFloat(nc_group_id, vid_mit_skintemp, index, num_vals, (buffer[29]/64.0));
    else
      writeVariableFloat(nc_group_id, vid_mit_skintemp, index, num_vals, -32768.0);

    if (buffer[30] != -32768)
      writeVariableFloat(nc_group_id, vid_fg_skintemp, index, num_vals, (buffer[30]/64.0));
    else
      writeVariableFloat(nc_group_id, vid_fg_skintemp, index, num_vals, -32768.0);


    // I Spares

    index_2D[0] = recnum;
    index_2D[1] = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 129;

    for (n=0; n<129; n++)
      ispares[n] = buffer[31+n];

    writeArrayShort(nc_group_id, vid_ispares, index_2D, num_vals_2D, ispares);


    // R Spares

    index_2D[0] = recnum;
    index_2D[1] = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 262;

    for (n=0; n<262; n++)
      {
      if ((buffer[160+n] == -32768) || (buffer[160+n] == -9999))
    	rspares[n] = -32768.0;
      else
    	rspares[n] = buffer[160+n] / 1.0;
      }

    writeArrayFloat(nc_group_id, vid_rspares, index_2D, num_vals_2D, rspares);

    // Number of spectral points and cloud parameters

    writeVariableShort(nc_group_id, vid_numspectral, index, num_vals, buffer[422]);

    writeVariableShort(nc_group_id, vid_numhinge, index, num_vals, buffer[423]);

    writeVariableShort(nc_group_id, vid_cldhinge, index, num_vals, buffer[424]);

    writeVariableShort(nc_group_id, vid_numcld, index, num_vals, buffer[425]);

    index_2D[0]    = recnum;
    index_2D[1]    = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 8;

    for (n=0; n<8; n++)
      {
      if (buffer[426+n] == -32768)
    	clouds[n] = -32768.0;
      else
    	clouds[n] = buffer[426+n] / 10.0;
      }

    writeArrayFloat(nc_group_id, vid_ctp, index_2D, num_vals_2D, clouds);

    for (n=0; n<8; n++)
      {
      if (buffer[434+n] == -32768)
    	clouds[n] = -32768.0;
      else
    	clouds[n] = buffer[434+n] / 100.0;
      }

    writeArrayFloat(nc_group_id, vid_ctf, index_2D, num_vals_2D, clouds);


    // Pressures

    index_2D[0]    = recnum;
    index_2D[1]    = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 100;

    for (n=0; n<100; n++)
      {
      if (buffer[450+n] == -32768)
    	level_values[n] = -32768.0;
      else if (n < 20)
    	level_values[n] = buffer[450+n] / 1000.0;
      else
    	level_values[n] = buffer[450+n] / 10.0;

      //if (level_values[n] != -32768.0)
      //printf("NUCAPS PRESS:  %d  %d   %f\n", recnum, n, level_values[n]);
      }

    writeArrayFloat(nc_group_id, vid_pressure, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[550+n] == -32768)
    	level_values[n] = -32768.0;
      else if (n < 20)
    	level_values[n] = buffer[550+n] / 1000.0;
      else
    	level_values[n] = buffer[550+n] / 10.0;
      }

    writeArrayFloat(nc_group_id, vid_effpress, index_2D, num_vals_2D, level_values);

    // Temperatures

    for (n=0; n<100; n++)
      {
      if (buffer[650+n] != -32768)
	level_values[n] = buffer[650+n] / 64.0;
      else
	level_values[n] = -32768.0;
      }

    writeArrayFloat(nc_group_id, vid_temp, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[750+n] != -32768)
	level_values[n] = buffer[750+n] / 64.0;
      else
	level_values[n] = -32768.0;
      }

    writeArrayFloat(nc_group_id, vid_mittemp, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[850+n] != -32768)
	level_values[n] = buffer[850+n] / 64.0;
      else
	level_values[n] = -32768.0;
      }

    writeArrayFloat(nc_group_id, vid_fgtemp, index_2D, num_vals_2D, level_values);


    // Water vapor mixing ratios

    for (n=0; n<100; n++)
      {
      if (buffer[950+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[950+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_wvmr, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1050+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1050+n]/1024.0));

      //if (level_values[n] != -32768.0)
      //printf("NUCAPS WVMR:  %d  %d   %d   %f\n", recnum, n, buffer[1050+n], level_values[n]);
      }

    writeArrayFloat(nc_group_id, vid_mitwvmr, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1150+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1150+n]/1024.0));

      //if (level_values[n] != -32768.0)
      //printf("NUCAPS WVMR:  %d  %d   %d   %f\n", recnum, n, buffer[1150+n], level_values[n]);
      }

    writeArrayFloat(nc_group_id, vid_fgwvmr, index_2D, num_vals_2D, level_values);

    // Water vapor column densities

    for (n=0; n<100; n++)
      {
      if (buffer[1250+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1250+n]/1024.0));

      //if (level_values[n] != -32768.0)
      //printf("NUCAPS WVCD:  %d  %d   %d   %f\n", recnum, n, buffer[1250+n], level_values[n]);
      }

    writeArrayFloat(nc_group_id, vid_wvcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1350+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1350+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_mitwvcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1450+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1450+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_fgwvcd, index_2D, num_vals_2D, level_values);

    // Ozone

    for (n=0; n<100; n++)
      {
      if (buffer[1550+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1550+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_o3lcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1650+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1650+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_fgo3lcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1750+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1750+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_o3mr, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1850+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1850+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_fgo3mr, index_2D, num_vals_2D, level_values);

    // Liquid water

    for (n=0; n<100; n++)
      {
      if (buffer[1950+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1950+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_h2olcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1950+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[1950+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_h2omr, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2150+n] != -32768)
	byte_values[n] = (char)buffer[2150+n];
      else
	byte_values[n] = (char)-128;
      }

    writeArrayByte(nc_group_id, vid_iceflag, index_2D, num_vals_2D, byte_values);

    // Carbon monoxide

    for (n=0; n<100; n++)
      {
      if (buffer[2250+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[2250+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_colcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2350+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[2350+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_comr, index_2D, num_vals_2D, level_values);

    // Methane and carbon dioxide

    for (n=0; n<100; n++)
      {
      if (buffer[2450+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[2450+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_ch4lcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2550+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[2550+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_ch4mr, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2650+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[2650+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_co2mr, index_2D, num_vals_2D, level_values);

    // Nitric acid

    for (n=0; n<100; n++)
      {
      if (buffer[2750+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[2750+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_hno3lcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2850+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[2850+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_hno3mr, index_2D, num_vals_2D, level_values);

    // Nitrous oxide

    for (n=0; n<100; n++)
      {
      if (buffer[2950+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[2950+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_n2olcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3050+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[3050+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_n2omr, index_2D, num_vals_2D, level_values);

    // Sulfur dioxide

    for (n=0; n<100; n++)
      {
      if (buffer[3150+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[3150+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_so2lcd, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3250+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = (float)(exp(buffer[3250+n]/1024.0));
      }

    writeArrayFloat(nc_group_id, vid_so2mr, index_2D, num_vals_2D, level_values);


    // Spectral points

    index_2D[0]    = recnum;
    index_2D[1]    = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 16;

    for (n=0; n<16; n++)
      {
      if (buffer[3350+n] == -32768)
    	spectral_values[n] = -32768.0;
      else
    	spectral_values[n] = buffer[3350+n] / 1.0;
      }

    writeArrayFloat(nc_group_id, vid_mwfreq, index_2D, num_vals_2D, spectral_values);

    for (n=0; n<16; n++)
      {
      if (buffer[3366+n] == -32768)
    	spectral_values[n] = -32768.0;
      else
    	spectral_values[n] = buffer[3366+n] / 100.0;
      }

    writeArrayFloat(nc_group_id, vid_mwemiss, index_2D, num_vals_2D, spectral_values);

    for (n=0; n<16; n++)
      {
      if (buffer[3382+n] == -32768)
    	spectral_values[n] = -32768.0;
      else
    	spectral_values[n] = buffer[3382+n] / 100.0;
      }

    writeArrayFloat(nc_group_id, vid_mitemiss, index_2D, num_vals_2D, spectral_values);


    // Hinge point frequencies

    index_2D[0]    = recnum;
    index_2D[1]    = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 100;

    for (n=0; n<100; n++)
      {
      if (buffer[3398+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = buffer[3398+n] / 1.0;
      }

    writeArrayFloat(nc_group_id, vid_irhpfreq, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3498+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = buffer[3498+n] / 1.0;
      }

    writeArrayFloat(nc_group_id, vid_fgirhpfreq, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3598+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = buffer[3598+n] / 100.0;
      }

    writeArrayFloat(nc_group_id, vid_irsurfemiss, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3698+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = buffer[3698+n] / 100.0;
      }

    writeArrayFloat(nc_group_id, vid_fgirsurfemiss, index_2D, num_vals_2D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3798+n] == -32768)
    	level_values[n] = -32768.0;
      else
    	level_values[n] = buffer[3798+n] / 100.0;
      }

    writeArrayFloat(nc_group_id, vid_surfreflect, index_2D, num_vals_2D, level_values);

    // Stability parameters

    index_2D[0]    = recnum;
    index_2D[1]    = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 16;

    for (n=0; n<16; n++)
      {
      if ((buffer[3898+n] == -32768) || (buffer[3898+n] == -999))
    	spectral_values[n] = -32768.0;
      else
    	spectral_values[n] = buffer[3898+n] / 1.0;
      }

    writeArrayFloat(nc_group_id, vid_stability, index_2D, num_vals_2D, spectral_values);



//***********
//***********
// cloud frequency values
//***********
//***********


    // NPROVS calculated values beginning with the relative humidity

    index_2D[0]    = recnum;
    index_2D[1]    = 0;
    num_vals_2D[0] = 1;
    num_vals_2D[1] = 100;

    // Unpack the pressures

    for (n=0; n<100; n++)
      {
      if (buffer[450+n] == -32768)
    	temp_pressures[n] = -32768.0;
      else if (n < 20)
    	temp_pressures[n] = buffer[450+n] / 1000.0;
      else
    	temp_pressures[n] = buffer[450+n] / 10.0;

      if (buffer[550+n] == -32768)
    	wvap_pressures[n] = -32768.0;
      else if (n < 20)
    	wvap_pressures[n] = buffer[550+n] / 1000.0;
      else
    	wvap_pressures[n] = buffer[550+n] / 10.0;
      }  // for (n=0;

    // Unpack the surface pressure

    if (buffer[27] != -32768)
      surf_pressure = buffer[27] / 10.0;
    else
      surf_pressure = -32768.0;


    // Calculate the observed relative humidities and tpw

    for (n=0; n<100; n++)
      {
      if (buffer[650+n] != -32768)
	temps[n] = buffer[650+n] / 64.0;
      else
	temps[n] = -32768.0;

      if (buffer[950+n] == -32768)
    	wvaps[n] = -32768.0;
      else
    	wvaps[n] = (float)(exp(buffer[950+n]/1024.0));
      }  // for (n=0...

    relative_humidities = calculateRelativeHumidities(temp_pressures, temps, 100, 
						      wvap_pressures, wvaps, 100, surf_pressure);

    writeArrayFloat(nc_group_id, vid_rh, index_2D, num_vals_2D, relative_humidities);

    free(relative_humidities);
    
    tpw = calculateTPW(wvap_pressures, wvaps, 100, surf_pressure, -32768.0);
    writeVariableFloat(nc_group_id, vid_tpw, index, num_vals, tpw);

    // Calculate the microwave relative humidities and tpw

    for (n=0; n<100; n++)
      {
      if (buffer[750+n] != -32768)
	temps[n] = buffer[750+n] / 64.0;
      else
	temps[n] = -32768.0;

      if (buffer[1050+n] == -32768)
    	wvaps[n] = -32768.0;
      else
    	wvaps[n] = (float)(exp(buffer[1050+n]/1024.0));
      }  // for (n=0...

    relative_humidities = calculateRelativeHumidities(temp_pressures, temps, 100, 
						      wvap_pressures, wvaps, 100, surf_pressure);

    writeArrayFloat(nc_group_id, vid_mitrh, index_2D, num_vals_2D, relative_humidities);

    free(relative_humidities);
    
    tpw = calculateTPW(wvap_pressures, wvaps, 100, surf_pressure, -32768.0);
    writeVariableFloat(nc_group_id, vid_mittpw, index, num_vals, tpw);


    // Calculate the first guess relative humidities and tpw

    for (n=0; n<100; n++)
      {
      if (buffer[850+n] != -32768)
	temps[n] = buffer[850+n] / 64.0;
      else
	temps[n] = -32768.0;

      if (buffer[1150+n] == -32768)
    	wvaps[n] = -32768.0;
      else
    	wvaps[n] = (float)(exp(buffer[1150+n]/1024.0));
      }  // for (n=0...

    relative_humidities = calculateRelativeHumidities(temp_pressures, temps, 100, 
						      wvap_pressures, wvaps, 100, surf_pressure);

    writeArrayFloat(nc_group_id, vid_fgrh, index_2D, num_vals_2D, relative_humidities);

    free(relative_humidities);
    
    tpw = calculateTPW(wvap_pressures, wvaps, 100, surf_pressure, -32768.0);
    writeVariableFloat(nc_group_id, vid_fgtpw, index, num_vals, tpw);


    // Granule file name

    for (i=0; i<100; i++)
      granule_name[i] = (char)buffer[6314+i];

    granule_name[100] = '\0';

//    if (strlen(granule_name) == 0)
//      {
//      for (i=0; i<=99; i++)
//        granule_name[i] = ' ';
//      }

    string[0] = granule_name;

    if (retval = nc_put_vara_string(nc_group_id, vid_granulename, index, num_vals, (const char **)string))
      ERR(retval);


    // Program version and release

    if (buffer[6414] != -32768)
      writeVariableFloat(nc_group_id, vid_version, index, num_vals, (buffer[6414]/100.0));
    else
      writeVariableFloat(nc_group_id, vid_version, index, num_vals, -32768.0);

    if (buffer[6415] != -32768)
      writeVariableFloat(nc_group_id, vid_release, index, num_vals, (buffer[6415]/100.0));
    else
      writeVariableFloat(nc_group_id, vid_release, index, num_vals, -32768.0);
    }  // for (recnum=0...


  // Free the memory used by the buffer

  free(buffer);
  }

// end of file
