/*
   Name- NucapsBinaryOzonesondeCollocator.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  5/15/2023   Programmer- Mike Pettey (IMSG)

   Function- This program extracts data from GOES ABI netCDF files
             and copies the data to a GOES ABI Daily Data File (GDDF).
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>

#include <dirent.h>
#include <errno.h>

#define  MAX_TIME_DIFF  6.0
#define  MAX_DIST_DIFF  200.0

#define  TRUE      0
#define  FALSE     1


void processNucapsFile(int collocation_date, int collocation_time,
                       float collocation_lat, float collocation_lon,
                       int nprovs_platform_id, char *nprovs_platform_name);

float calculateDistanceDifference(float lat1, float lon1, float lat2, float lon2);
float calculateTimeDifference(int date1, int time1, int date2, int time2);

float* calculateRelativeHumidities(float *pressures, float *temperatures, int num_temp_levels,
                                   float *eff_pressures, float *water_vapors, int num_wvap_levels,
                                   float surface_pressure);

float calculateTPW(float *pressures, float *water_vapors, int num_wvap_levels,
       float surface_pressure, float surface_wvmr);

int defineVariable(int nc_id, char *var_name, nc_type var_type, int ndims,
                   int *dimids, char *attr_string, char *attr);

void writeArrayByte(int group_id, int var_id, size_t *index, size_t *num_vals, char *value);
void writeArrayFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float *value);
void writeArrayShort(int group_id, int var_id, size_t *index, size_t *num_vals, short *value);
int writeAttributeText(int grp, char *attrVariable, char *attrValue);
void writeVariableByte(int group_id, int var_id, size_t *index, size_t *num_vals, char value);
void writeVariableFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float value);
void writeVariableInteger(int group_id, int var_id, size_t *index, size_t *num_vals, int value);
void writeVariableShort(int group_id, int var_id, size_t *index, size_t *num_vals, short value);



int main(int argc, char *argv[])
  {
  int     retval, nc_id, var_id, nprovs_platform_id;
  int     collocation_date, collocation_time;
  float   collocation_lat, collocation_lon;
  char    nprovs_platform_name[100];
  size_t  index[1];

  index[0] = 0;

  // Print an initial message

  printf("\nCollocating NUCAPS data to Ozonesondes\n\n");

  // Unpack the platform ID and platform name

  nprovs_platform_id = atoi(argv[1]);
  sprintf(nprovs_platform_name, "%s", argv[2]);


  // Unpack the date, time, latitude and longitude from the ozonesonde file

  retval = nc_open("ozonesonde.in", NC_NOWRITE, &nc_id);

  if (retval == NC_NOERR)
    {

    // Unpack the date

    retval = nc_inq_varid(nc_id, "date", &var_id);
    retval = nc_get_var1_int(nc_id, var_id, index, &collocation_date);

    // Unpack the time

    retval = nc_inq_varid(nc_id, "time", &var_id);
    retval = nc_get_var1_int(nc_id, var_id, index, &collocation_time);

    // Unpack the latitude

    retval = nc_inq_varid(nc_id, "latitude", &var_id);
    retval = nc_get_var1_float(nc_id, var_id, index, &collocation_lat);

    // Unpack the longitude

    retval = nc_inq_varid(nc_id, "longitude", &var_id);
    retval = nc_get_var1_float(nc_id, var_id, index, &collocation_lon);

    // Close the input netCDF file and free the allocated memory

    nc_close(nc_id);

    printf("Collocation Date:      %d\n", collocation_date);
    printf("Collocation Time:      %d\n", collocation_time);
    printf("Collocation Latitude:  %0.1f\n", collocation_lat);
    printf("Collocation Longitude: %0.1f\n", collocation_lon);


    // Process the input NUCAPS file

    processNucapsFile(collocation_date, collocation_time, collocation_lat, collocation_lon,
                      nprovs_platform_id, nprovs_platform_name);

    }  //  if (retval == NC_NOERR)...
  else
    {
    printf("\n\n*** ERROR ***\n\n");
    printf("The ozonesonde netCDF file could not be read.\n");
    printf("Execution is stopping.\n\n");
    }

  printf("\nProcessing completed.\n\n");
  }



// ===============================================================================
void processNucapsFile(int collocation_date, int collocation_time,
                       float collocation_lat, float collocation_lon,
                       int nprovs_platform_id, char *nprovs_platform_name)
  {
  int  retval;
  int  nc_id;
  int  dim_levels, dim_spectralpts, dim_hingepts, dim_cldlayers;
  int  dim_stabparms, dim_ispares, dim_rspares;
  int  dimid_levels[1], dimid_spectralpts[1], dimid_hingepts[1], dimid_cldlayers[1];;
  int  dimid_stabparms[1], dimid_ispares[1], dimid_rspares[1];

  size_t  index[1], num_vals[1], index_1D[1], num_vals_1D[1];
//
//  float   missing_float;
//  char    granule_name[101];
//  char    *string[1];

  int     vid_lat, vid_lon, vid_date, vid_time;
  int     vid_node, vid_granule, vid_scanline, vid_fov, vid_dice, vid_qc;
  int     vid_solzen, vid_viewangle, vid_satheight, vid_avgco2;
  int     vid_surfheight, vid_landfrac, vid_surfclass, vid_surfemmis;
  int     vid_surfpress, vid_skintemp, vid_mit_skintemp, vid_fg_skintemp;
  int     vid_ispares, vid_rspares, vid_numspectral, vid_numhinge, vid_cldhinge;
  int     vid_numcld, vid_ctp, vid_ctf;
  int     vid_pressure, vid_effpress, vid_temp, vid_mittemp, vid_fgtemp;
  int     vid_wvmr, vid_mitwvmr, vid_fgwvmr, vid_wvcd, vid_mitwvcd, vid_fgwvcd;
  int     vid_rh, vid_mitrh, vid_fgrh, vid_tpw, vid_mittpw, vid_fgtpw;
  int     vid_o3lcd, vid_fgo3lcd, vid_o3mr, vid_fgo3mr;
  int     vid_h2olcd, vid_h2omr, vid_iceflag, vid_colcd, vid_comr;
  int     vid_ch4lcd, vid_ch4mr, vid_co2mr, vid_hno3lcd, vid_hno3mr;
  int     vid_n2olcd, vid_n2omr, vid_so2lcd, vid_so2mr;
  int     vid_mwfreq, vid_mwemiss, vid_mitemiss;
  int     vid_irhpfreq, vid_fgirhpfreq, vid_irsurfemiss, vid_fgirsurfemiss;
  int     vid_surfreflect, vid_stability, vid_cldfreq, vid_cldemiss, vid_cldreflect;
//  int     vid_granulename, vid_version, vid_release;

  int     n, recnum, num_nucaps_records;
  int     year, month, day, hour, minute, second, yyyymmdd, hhmmss;
  short   small_buffer[9], buffer[6500];
  short   ispares[129];
  long    offset, closest_footprint_offset;
  float   latitude, longitude, time_factor;
  float   distance_diff, time_diff, closeness, closest_value;
  float   *relative_humidities, tpw;
  float   temp_pressures[100], temps[100], wvap_pressures[100], wvaps[100], surf_pressure;
  float   rspares[262], clouds[8], spectral_values[16];
  float   level_values[100];
  char    byte_values[100], platform_id[5];
  FILE    *in;


  // Search the input NUCAPS file for a footprint that most closely matches
  // the time and location of the ozonesonde

  closest_value = 9999999.9;
  closest_footprint_offset = -1;

  if ((in=fopen("nucaps.in","r")) == NULL)
    {
    printf("\n\nThe input NUCAPS file could not be opened.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fseek(in, 8, SEEK_SET);
  fread(&num_nucaps_records, 4, 1, in);

  for (recnum=1; recnum<num_nucaps_records; recnum++)
    {
    offset = (long)recnum * (long)13000;
    fseek(in, offset, SEEK_SET);
    fread(small_buffer, 9, 2, in);

    if ((small_buffer[0] == 0) && (small_buffer[3] != -32768) && (small_buffer[4] != -32768) &&
        (small_buffer[5] != -32768) && (small_buffer[6] != -32768) &&
        (small_buffer[7] != -32768) && (small_buffer[8] != -32768))
      {
      latitude  = small_buffer[3] / 128.0;
      longitude = small_buffer[4] / 128.0;
      yyyymmdd  = (small_buffer[5] * 10000) + small_buffer[6];
      hhmmss    = (small_buffer[7] * 10000) + small_buffer[8];

      time_diff = calculateTimeDifference(yyyymmdd, hhmmss, collocation_date, collocation_time);
      time_diff = fabs(time_diff);

      if (time_diff < MAX_TIME_DIFF)
        {
        distance_diff = calculateDistanceDifference(latitude, longitude,
                                                    collocation_lat, collocation_lon);

        if (distance_diff < MAX_DIST_DIFF)
          {
          time_factor = 30.0;
          closeness = distance_diff + (time_diff * time_factor);

          if (closeness < closest_value)
            {
            closest_value = closeness;
            closest_footprint_offset = offset;
            }

//printf("DIFFS:  %d    %f    %f    %d %06d    %d %06d\n", recnum, time_diff, distance_diff, yyyymmdd,
//                                                         hhmmss, collocation_date, collocation_time);

          }  // if (distance_diff < MAX_DIST_DIFF...
        }  // if (time_diff < MAX_TIME_DIFF...
      }  // if (small_buffer[0]...
    }  // for (recnum=1...


  // If a closest footprint was found, then begin copying the data to a
  // new netCDF file

  if (closest_footprint_offset != -1)
    {

    // Create a new netCDF file

    retval = nc_create("nucaps.out", NC_NETCDF4, &nc_id);

    if (retval == NC_NOERR)
      {
      sprintf(platform_id, "%d", nprovs_platform_id);
      writeAttributeText(nc_id, "NPROVS_Platform_ID", platform_id);
      writeAttributeText(nc_id, "NPROVS_Platform_Type", "600");
      writeAttributeText(nc_id, "NPROVS_Platform_Name", nprovs_platform_name);
      writeAttributeText(nc_id, "NPROVS_Program", "NucapsNOAABinaryOzonesondeCollocator");
  //    writeAttributeText(nc_id, "Platform_Name", description);
  //    writeAttributeShort(nc_id, "Platform_ID", group_id);
  //    writeAttributeShort(nc_id, "Platform_Type", group_type);
  //    writeAttributeText(nc_id, "Platform_Data_Source", "NUCAPS Version 2");
  //    writeAttributeText(nc_id, "Platform_NPROVS_Source_Name", nprovs_name);


    // Define the dimensions
    // levels

    retval = nc_def_dim(nc_id, "Levels", 100, &dim_levels);
    dimid_levels[0] = dim_levels;

    // spectral points

    retval = nc_def_dim(nc_id, "Spectral_Pts", 16, &dim_spectralpts);
    dimid_spectralpts[0] = dim_spectralpts;

    // hinge points

    retval = nc_def_dim(nc_id, "Hinge_Pts", 100, &dim_hingepts);
    dimid_hingepts[0] = dim_hingepts;

    // cloud layers

    retval = nc_def_dim(nc_id, "Cloud_Layers", 8, &dim_cldlayers);
    dimid_cldlayers[0] = dim_cldlayers;

    // stability parameters

    retval = nc_def_dim(nc_id, "Stability_Parameters", 16, &dim_stabparms);
    dimid_stabparms[0] = dim_stabparms;

    // i spares

    retval = nc_def_dim(nc_id, "Ispares", 129, &dim_ispares);
    dimid_ispares[0] = dim_ispares;

    // r spares

    retval = nc_def_dim(nc_id, "Rspares", 262, &dim_rspares);
    dimid_rspares[0] = dim_rspares;


    // Define the variables

    vid_lat = defineVariable(nc_id, "Latitude", NC_FLOAT, 0, 0, "units", "degrees_north");

    vid_lon = defineVariable(nc_id, "Longitude", NC_FLOAT, 0, 0, "units", "degrees_east");

    vid_date = defineVariable(nc_id, "Date", NC_INT, 0, 0, "format", "yyyymmdd");

    vid_time = defineVariable(nc_id, "Time", NC_INT, 0, 0, "format", "hhmmss");

    vid_node = defineVariable(nc_id, "Ascending_Descending", NC_BYTE, 0, 0,
                              "values", "0=ascending, 1=descending");

    vid_granule = defineVariable(nc_id, "granule_number", NC_SHORT, 0, 0, NULL, NULL);

    vid_scanline = defineVariable(nc_id, "scan_line", NC_SHORT, 0, 0, NULL, NULL);

    vid_fov = defineVariable(nc_id, "CrIS_FORs", NC_SHORT, 0, 0, NULL, NULL);

    vid_dice = defineVariable(nc_id, "index_within_granule", NC_SHORT, 0, 0, NULL, NULL);

    vid_qc = defineVariable(nc_id, "retrieval_quality_flag", NC_BYTE, 0, 0,
          "values", "0=accepted, 1=reject_physical, 2=reject_MIT, 4=reject_NOAA_reg, 8=reject_iMIT, 9=rej_phy_and_iMIT, 16=reject_iNOAA, 17=reject_phy_and_iNOAA, 24=reject_iMIT_and_iNOAA, 25=reject_phy_and_iMIT_and_iNOAA");

    vid_solzen = defineVariable(nc_id, "Solar_Zenith", NC_FLOAT, 0, 0, "units", "deg");

    vid_viewangle = defineVariable(nc_id, "View_Angle", NC_FLOAT, 0, 0, "units", "deg");

    vid_satheight = defineVariable(nc_id, "Satellite_Height", NC_FLOAT, 0, 0, "units", "km");

    vid_avgco2 = defineVariable(nc_id, "Mean_CO2", NC_FLOAT, 0, 0, "units", "ppm");

    vid_surfheight = defineVariable(nc_id, "Topography", NC_SHORT, 0, 0, "units", "m");

    vid_landfrac = defineVariable(nc_id, "Land_Fraction", NC_FLOAT, 0, 0, NULL, NULL);

    vid_surfclass = defineVariable(nc_id, "MW_Surface_Class", NC_BYTE, 0, 0,
           "values", "0=coast, 1=land, 2=sea, 3=ice, 4=ice, 5=snow, 6=glacier, 7=snow");

    vid_surfemmis = defineVariable(nc_id, "MW_Surface_Emis", NC_FLOAT, 0, 0, NULL, NULL);

    vid_surfpress = defineVariable(nc_id, "Surface_Pressure", NC_FLOAT, 0, 0, "units", "hPa");

    vid_skintemp = defineVariable(nc_id, "Skin_Temperature", NC_FLOAT, 0, 0, "units", "K");

    vid_mit_skintemp = defineVariable(nc_id, "MIT_Skin_Temperature", NC_FLOAT, 0, 0,
              "units", "K");

    vid_fg_skintemp = defineVariable(nc_id, "FG_Skin_Temperature", NC_FLOAT, 0, 0,
             "units", "K");

    vid_ispares = defineVariable(nc_id, "Ispare_Field", NC_SHORT, 1, dimid_ispares, NULL, NULL);

    vid_rspares = defineVariable(nc_id, "Rspre_Field", NC_FLOAT, 1, dimid_rspares, NULL, NULL);

    vid_numspectral = defineVariable(nc_id, "N_Smw_Per_FOV",
                                     NC_SHORT, 0, 0, NULL, NULL);

    vid_numhinge = defineVariable(nc_id, "nemis_Per_FOV",
                                  NC_SHORT, 0, 0, NULL, NULL);

    vid_cldhinge = defineVariable(nc_id, "ncemis_Per_FOV",
                                  NC_SHORT, 0, 0, NULL, NULL);

    vid_numcld = defineVariable(nc_id, "ncld_Per_FOV", NC_SHORT, 0, 0,
                                NULL, NULL);

    vid_ctp = defineVariable(nc_id, "Cloud_Top_Pressure", NC_FLOAT, 1, dimid_cldlayers,
                             "unit", "hPa");

    vid_ctf = defineVariable(nc_id, "Cloud_Top_Fraction", NC_FLOAT, 1, dimid_cldlayers, NULL, NULL);

    vid_pressure = defineVariable(nc_id, "Pressure", NC_FLOAT, 1, dimid_levels, "units", "hPa");

    vid_effpress = defineVariable(nc_id, "Effective_Pressure", NC_FLOAT, 1, dimid_levels,
                                  "units", "hPa");

    vid_temp = defineVariable(nc_id, "Temperature", NC_FLOAT, 1, dimid_levels, "units", "K");

    vid_mittemp = defineVariable(nc_id, "MIT_Temperature", NC_FLOAT, 1, dimid_levels,
                                 "units", "K");

    vid_fgtemp = defineVariable(nc_id, "FG_Temperature", NC_FLOAT, 1, dimid_levels,
                                "units", "K");

    vid_wvmr = defineVariable(nc_id, "H2O_MR", NC_FLOAT, 1, dimid_levels,
                              "units", "g/g");

    vid_mitwvmr = defineVariable(nc_id, "MIT_H2O_MR", NC_FLOAT, 1,
                                 dimid_levels, "units", "g/g");

    vid_fgwvmr = defineVariable(nc_id, "FG_H2O_MR", NC_FLOAT, 1,
                                dimid_levels, "units", "g/g");

    vid_wvcd = defineVariable(nc_id, "H2O", NC_FLOAT, 1, dimid_levels,
                              "units", "kg/kg");

    vid_mitwvcd = defineVariable(nc_id, "MIT_H2O", NC_FLOAT, 1,
                                 dimid_levels, "units", "kg/kg");

    vid_fgwvcd = defineVariable(nc_id, "FG_H2O", NC_FLOAT, 1,
                                dimid_levels, "units", "kg/kg");

    vid_o3lcd = defineVariable(nc_id, "O3", NC_FLOAT, 1, dimid_levels,
                               "units", "mol/cm^2");

    vid_fgo3lcd = defineVariable(nc_id, "FG_O3", NC_FLOAT, 1,
                                 dimid_levels, "units", "mol/cm^2");

    vid_o3mr = defineVariable(nc_id, "O3_MR", NC_FLOAT, 1, dimid_levels,
                              "units", "ppb");

    vid_fgo3mr = defineVariable(nc_id, "FG_O3_MR", NC_FLOAT, 1,
                                dimid_levels, "units", "ppb");

    vid_h2olcd = defineVariable(nc_id, "Liquid_H2O", NC_FLOAT, 1,
                                dimid_levels, "units", "mol/cm^2");

    vid_h2omr = defineVariable(nc_id, "Liquid_H2O_MR", NC_FLOAT, 1,
                               dimid_levels, "units", "g/g");

    vid_iceflag = defineVariable(nc_id, "Ice_Liquid_Flag", NC_BYTE, 1, dimid_levels,
                                 "values", "0=water, 1=ice");

    vid_colcd = defineVariable(nc_id, "CO", NC_FLOAT, 1,
                               dimid_levels, "units", "mol/cm^2");

    vid_comr = defineVariable(nc_id, "CO_MR", NC_FLOAT, 1,
                              dimid_levels, "units", "ppb");

    vid_ch4lcd = defineVariable(nc_id, "CH4", NC_FLOAT, 1,
                                dimid_levels, "units", "mol/cm^2");

    vid_ch4mr = defineVariable(nc_id, "CH4_MR", NC_FLOAT, 1, dimid_levels,
                               "units", "ppb");

    vid_co2mr = defineVariable(nc_id, "CO2_MR", NC_FLOAT, 1,
                               dimid_levels, "units", "ppb");

    vid_hno3lcd = defineVariable(nc_id, "HNO3", NC_FLOAT, 1,
                                 dimid_levels, "units", "mol/cm^2");

    vid_hno3mr = defineVariable(nc_id, "HNO3_MR", NC_FLOAT, 1, dimid_levels,
                                "units", "ppb");

    vid_n2olcd = defineVariable(nc_id, "N2O", NC_FLOAT, 1,
                                dimid_levels, "units", "mol/cm^2");

    vid_n2omr = defineVariable(nc_id, "N2O_MR", NC_FLOAT, 1,
                               dimid_levels, "units", "ppb");

    vid_so2lcd = defineVariable(nc_id, "SO2", NC_FLOAT, 1,
                                dimid_levels, "units", "mol/cm^2");

    vid_so2mr = defineVariable(nc_id, "SO2_MR", NC_FLOAT, 1,
                               dimid_levels, "units", "ppb");

    vid_mwfreq = defineVariable(nc_id, "MW_Frequency", NC_FLOAT, 1, dimid_spectralpts,
                                "units", "cm^-1");

    vid_mwemiss = defineVariable(nc_id, "MW_Emis", NC_FLOAT, 1, dimid_spectralpts,
                                 "units", "cm^-1");

    vid_mitemiss = defineVariable(nc_id, "MIT_MW_Emis", NC_FLOAT, 1, dimid_spectralpts,
                                  "units", "cm^-1");

    vid_irhpfreq = defineVariable(nc_id, "IR_Emis_Freq", NC_FLOAT, 1, dimid_levels,
                                  "units", "cm^-1");

    vid_fgirhpfreq = defineVariable(nc_id, "FG_IR_Emis_Freq", NC_FLOAT, 1,
                                    dimid_levels, "units", "cm^-1");

    vid_irsurfemiss = defineVariable(nc_id, "IR_Surface_Emis", NC_FLOAT, 1, dimid_levels,
                                     NULL, NULL);

    vid_fgirsurfemiss = defineVariable(nc_id, "FG_IR_Surface_Emis", NC_FLOAT, 1, dimid_levels,
                                       NULL, NULL);

    vid_surfreflect = defineVariable(nc_id, "IR_Surface_Refl", NC_FLOAT, 1, dimid_levels,
                                     "units", "percent");

    vid_stability = defineVariable(nc_id, "Stability", NC_FLOAT, 1, dimid_stabparms,
                                   NULL, NULL);

  //  vid_cldfreq = defineVariable(nc_id, "cloud_ir_frequency", NC_FLOAT, 1, dimid_stabparms,
  //             "units", "cm^-1");

  //  vid_cldemiss = defineVariable(nc_id, "cloud_ir_emissivity", NC_FLOAT, 1, dimid_stabparms,
  //        NULL, NULL);

  //  vid_cldreflect = defineVariable(nc_id, "cloud_ir_reflectivity", NC_FLOAT, 1, dimid_stabparms,
  //          "units", "percent");

    vid_rh = defineVariable(nc_id, "relative_humidity", NC_FLOAT, 1, dimid_levels,
                            "units", "percent");

    vid_mitrh = defineVariable(nc_id, "microwave_relative_humidity", NC_FLOAT, 1,
                               dimid_levels, "units", "percent");

    vid_fgrh = defineVariable(nc_id, "first_guess_relative_humidity", NC_FLOAT, 1,
                              dimid_levels, "units", "percent");

    vid_tpw = defineVariable(nc_id, "total_precipitable_water", NC_FLOAT, 0, 0,
                             "units", "mm");

    vid_mittpw = defineVariable(nc_id, "microwave_total_precipitable_water", NC_FLOAT, 0, 0,
                                "units", "mm");

    vid_fgtpw = defineVariable(nc_id, "first_guess_total_precipitable_water", NC_FLOAT, 0, 0,
                               "units", "mm");

  //  vid_granulename = defineVariable(nc_id, "granule_file_name", NC_STRING, 0, 0,
  //           "string_length", "101");

  //  vid_version = defineVariable(nc_id, "program_version", NC_FLOAT, 0, 0, NULL, NULL);

  //  vid_release = defineVariable(nc_id, "program_release", NC_FLOAT, 0, 0, NULL, NULL);


    // Read the NUCAPS data record for the closest footprint

    fseek(in, closest_footprint_offset, SEEK_SET);
    fread(buffer, 6500, 2, in);

    // Index and num_vals are used to save a single value within the
    // variable array

    index[0]    = 0;
    num_vals[0] = 1;

    // Latitude and longitude

    if (buffer[3] != -32768)
      latitude  = buffer[3] / 128.0;
    else
      latitude = -32768.0;

    if (buffer[4] != -32768)
      longitude = buffer[4] / 128.0;
    else
      longitude = -32768.0;

    writeVariableFloat(nc_id, vid_lat, index, num_vals, latitude);
    writeVariableFloat(nc_id, vid_lon, index, num_vals, longitude);

    // Date and time

    if ((buffer[5] != -32768) && (buffer[6] != -32768) && (buffer[7] != -32768) && (buffer[8] != -32768))
      {
      year      = buffer[5];
      month     = buffer[6] / 100;
      day       = buffer[6] % 100;
      hour      = buffer[7];
      minute    = buffer[8] / 100;
      second    = buffer[8] % 100;

      yyyymmdd = (year * 10000) + (month * 100) + day;
      hhmmss   = (hour * 10000) + (minute * 100) + second;
      }
    else
      {
      yyyymmdd = -32768;
      hhmmss   = -32768;
      }

    writeVariableInteger(nc_id, vid_date, index, num_vals, yyyymmdd);
    writeVariableInteger(nc_id, vid_time, index, num_vals, hhmmss);

    // Print a message about the collocation

    time_diff = calculateTimeDifference(yyyymmdd, hhmmss, collocation_date, collocation_time);
    time_diff = fabs(time_diff);

    distance_diff = calculateDistanceDifference(latitude, longitude,
                                                collocation_lat, collocation_lon);

    printf("\n\nA collocation has been made.\n");
    printf("Time Difference:     %0.1f hours\n", time_diff);
    printf("Distance Difference: %0.1f km\n\n", distance_diff);


    // Save the orbital node

    if (buffer[11] != -32768)
      writeVariableByte(nc_id, vid_node, index, num_vals, (char)buffer[11]);
    else
      writeVariableByte(nc_id, vid_node, index, num_vals, (char)-128);

    writeVariableShort(nc_id, vid_granule, index, num_vals, buffer[12]);
    writeVariableShort(nc_id, vid_scanline, index, num_vals, buffer[13]);
    writeVariableShort(nc_id, vid_fov, index, num_vals, buffer[14]);
    writeVariableShort(nc_id, vid_dice, index, num_vals, buffer[15]);

    // Retrieval quality flag

    if (buffer[16] != -32768)
      writeVariableByte(nc_id, vid_qc, index, num_vals, (char)buffer[16]);
    else
      writeVariableByte(nc_id, vid_qc, index, num_vals, (char)-128);

    if (buffer[19] != -32768)
      writeVariableFloat(nc_id, vid_solzen, index, num_vals, (buffer[19]/128.0));
    else
      writeVariableFloat(nc_id, vid_solzen, index, num_vals, -32768.0);

    if (buffer[20] != -32768)
      writeVariableFloat(nc_id, vid_viewangle, index, num_vals, (buffer[20]/64.0));
    else
      writeVariableFloat(nc_id, vid_viewangle, index, num_vals, -32768.0);

    if (buffer[21] != -32768)
      writeVariableFloat(nc_id, vid_satheight, index, num_vals, (buffer[21]/10.0));
    else
      writeVariableFloat(nc_id, vid_satheight, index, num_vals, -32768.0);

    if (buffer[22] != -32768)
      writeVariableFloat(nc_id, vid_avgco2, index, num_vals, (buffer[22]/10.0));
    else
      writeVariableFloat(nc_id, vid_avgco2, index, num_vals, -32768.0);

    writeVariableShort(nc_id, vid_surfheight, index, num_vals, buffer[23]);

    if (buffer[24] != -32768)
      writeVariableFloat(nc_id, vid_landfrac, index, num_vals, (buffer[24]/100.0));
    else
      writeVariableFloat(nc_id, vid_landfrac, index, num_vals, -32768.0);

    if (buffer[25] != -32768)
      writeVariableByte(nc_id, vid_surfclass, index, num_vals, (char)buffer[25]);
    else
      writeVariableByte(nc_id, vid_surfclass, index, num_vals, (char)-128);

    if (buffer[26] != -32768)
      writeVariableFloat(nc_id, vid_surfemmis, index, num_vals, (buffer[26]/100.0));
    else
      writeVariableFloat(nc_id, vid_surfemmis, index, num_vals, -32768.0);

    if (buffer[27] != -32768)
      writeVariableFloat(nc_id, vid_surfpress, index, num_vals, (buffer[27]/10.0));
    else
      writeVariableFloat(nc_id, vid_surfpress, index, num_vals, -32768.0);

    if ((buffer[28] != -32768) && (buffer[28] != 6400))
      writeVariableFloat(nc_id, vid_skintemp, index, num_vals, (buffer[28]/64.0));
    else
      writeVariableFloat(nc_id, vid_skintemp, index, num_vals, -32768.0);

    if (buffer[29] != -32768)
      writeVariableFloat(nc_id, vid_mit_skintemp, index, num_vals, (buffer[29]/64.0));
    else
      writeVariableFloat(nc_id, vid_mit_skintemp, index, num_vals, -32768.0);

    if (buffer[30] != -32768)
      writeVariableFloat(nc_id, vid_fg_skintemp, index, num_vals, (buffer[30]/64.0));
    else
      writeVariableFloat(nc_id, vid_fg_skintemp, index, num_vals, -32768.0);


    // I Spares

    index_1D[0] = 0;
    num_vals_1D[0] = 129;

    for (n=0; n<129; n++)
      ispares[n] = buffer[31+n];

    writeArrayShort(nc_id, vid_ispares, index_1D, num_vals_1D, ispares);


    // R Spares

    num_vals_1D[0] = 262;

    for (n=0; n<262; n++)
      {
      if ((buffer[160+n] == -32768) || (buffer[160+n] == -9999))
      rspares[n] = -32768.0;
      else
      rspares[n] = buffer[160+n] / 1.0;
      }

    writeArrayFloat(nc_id, vid_rspares, index_1D, num_vals_1D, rspares);

    // Number of spectral points and cloud parameters

    writeVariableShort(nc_id, vid_numspectral, index, num_vals, buffer[422]);
    writeVariableShort(nc_id, vid_numhinge, index, num_vals, buffer[423]);
    writeVariableShort(nc_id, vid_cldhinge, index, num_vals, buffer[424]);
    writeVariableShort(nc_id, vid_numcld, index, num_vals, buffer[425]);

    num_vals_1D[0] = 8;

    for (n=0; n<8; n++)
      {
      if (buffer[426+n] == -32768)
      clouds[n] = -32768.0;
      else
      clouds[n] = buffer[426+n] / 10.0;
      }

    writeArrayFloat(nc_id, vid_ctp, index_1D, num_vals_1D, clouds);

    for (n=0; n<8; n++)
      {
      if (buffer[434+n] == -32768)
      clouds[n] = -32768.0;
      else
      clouds[n] = buffer[434+n] / 100.0;
      }

    writeArrayFloat(nc_id, vid_ctf, index_1D, num_vals_1D, clouds);


    // Pressures

    num_vals_1D[0] = 100;

    for (n=0; n<100; n++)
      {
      if (buffer[450+n] == -32768)
        level_values[n] = -32768.0;
      else if (n < 20)
        level_values[n] = buffer[450+n] / 1000.0;
      else
        level_values[n] = buffer[450+n] / 10.0;

      //if (level_values[n] != -32768.0)
      //printf("NUCAPS PRESS:  %d  %d   %f\n", recnum, n, level_values[n]);
      }

    writeArrayFloat(nc_id, vid_pressure, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[550+n] == -32768)
        level_values[n] = -32768.0;
      else if (n < 20)
        level_values[n] = buffer[550+n] / 1000.0;
      else
        level_values[n] = buffer[550+n] / 10.0;
      }

    writeArrayFloat(nc_id, vid_effpress, index_1D, num_vals_1D, level_values);

    // Temperatures

    for (n=0; n<100; n++)
      {
      if (buffer[650+n] != -32768)
        level_values[n] = buffer[650+n] / 64.0;
      else
        level_values[n] = -32768.0;
      }

    writeArrayFloat(nc_id, vid_temp, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[750+n] != -32768)
        level_values[n] = buffer[750+n] / 64.0;
      else
        level_values[n] = -32768.0;
      }

    writeArrayFloat(nc_id, vid_mittemp, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[850+n] != -32768)
        level_values[n] = buffer[850+n] / 64.0;
      else
        level_values[n] = -32768.0;
      }

    writeArrayFloat(nc_id, vid_fgtemp, index_1D, num_vals_1D, level_values);


    // Water vapor mixing ratios

    for (n=0; n<100; n++)
      {
      if (buffer[950+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[950+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_wvmr, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1050+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[1050+n]/1024.0));

      //if (level_values[n] != -32768.0)
      //printf("NUCAPS WVMR:  %d  %d   %d   %f\n", recnum, n, buffer[1050+n], level_values[n]);
      }

    writeArrayFloat(nc_id, vid_mitwvmr, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1150+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[1150+n]/1024.0));

      //if (level_values[n] != -32768.0)
      //printf("NUCAPS WVMR:  %d  %d   %d   %f\n", recnum, n, buffer[1150+n], level_values[n]);
      }

    writeArrayFloat(nc_id, vid_fgwvmr, index_1D, num_vals_1D, level_values);

    // Water vapor column densities

    for (n=0; n<100; n++)
      {
      if (buffer[1250+n] == -32768)
      level_values[n] = -32768.0;
      else
      level_values[n] = (float)(exp(buffer[1250+n]/1024.0));

      //if (level_values[n] != -32768.0)
      //printf("NUCAPS WVCD:  %d  %d   %d   %f\n", recnum, n, buffer[1250+n], level_values[n]);
      }

    writeArrayFloat(nc_id, vid_wvcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1350+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[1350+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_mitwvcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1450+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[1450+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_fgwvcd, index_1D, num_vals_1D, level_values);

    // Ozone

    for (n=0; n<100; n++)
      {
      if (buffer[1550+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[1550+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_o3lcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1650+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[1650+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_fgo3lcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1750+n] == -32768)
        level_values[n] = -32768.0;
      else
        //level_values[n] = (float)(exp(buffer[1750+n]/1024.0));
        level_values[n] = (float)(pow(10.0, (buffer[1750+n]/1024.0)));
      }

    writeArrayFloat(nc_id, vid_o3mr, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1850+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[1850+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_fgo3mr, index_1D, num_vals_1D, level_values);

    // Liquid water

    for (n=0; n<100; n++)
      {
      if (buffer[1950+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[1950+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_h2olcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[1950+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[1950+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_h2omr, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2150+n] != -32768)
        byte_values[n] = (char)buffer[2150+n];
      else
        byte_values[n] = (char)-128;
      }

    writeArrayByte(nc_id, vid_iceflag, index_1D, num_vals_1D, byte_values);

    // Carbon monoxide

    for (n=0; n<100; n++)
      {
      if (buffer[2250+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[2250+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_colcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2350+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[2350+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_comr, index_1D, num_vals_1D, level_values);

    // Methane and carbon dioxide

    for (n=0; n<100; n++)
      {
      if (buffer[2450+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[2450+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_ch4lcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2550+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[2550+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_ch4mr, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2650+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[2650+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_co2mr, index_1D, num_vals_1D, level_values);

    // Nitric acid

    for (n=0; n<100; n++)
      {
      if (buffer[2750+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[2750+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_hno3lcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[2850+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[2850+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_hno3mr, index_1D, num_vals_1D, level_values);

    // Nitrous oxide

    for (n=0; n<100; n++)
      {
      if (buffer[2950+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[2950+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_n2olcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3050+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[3050+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_n2omr, index_1D, num_vals_1D, level_values);

    // Sulfur dioxide

    for (n=0; n<100; n++)
      {
      if (buffer[3150+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[3150+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_so2lcd, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3250+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = (float)(exp(buffer[3250+n]/1024.0));
      }

    writeArrayFloat(nc_id, vid_so2mr, index_1D, num_vals_1D, level_values);


    // Spectral points

    num_vals_1D[0] = 16;

    for (n=0; n<16; n++)
      {
      if (buffer[3350+n] == -32768)
        spectral_values[n] = -32768.0;
      else
        spectral_values[n] = buffer[3350+n] / 1.0;
      }

    writeArrayFloat(nc_id, vid_mwfreq, index_1D, num_vals_1D, spectral_values);

    for (n=0; n<16; n++)
      {
      if (buffer[3366+n] == -32768)
        spectral_values[n] = -32768.0;
      else
        spectral_values[n] = buffer[3366+n] / 100.0;
      }

    writeArrayFloat(nc_id, vid_mwemiss, index_1D, num_vals_1D, spectral_values);

    for (n=0; n<16; n++)
      {
      if (buffer[3382+n] == -32768)
        spectral_values[n] = -32768.0;
      else
        spectral_values[n] = buffer[3382+n] / 100.0;
      }

    writeArrayFloat(nc_id, vid_mitemiss, index_1D, num_vals_1D, spectral_values);


    // Hinge point frequencies

    num_vals_1D[0] = 100;

    for (n=0; n<100; n++)
      {
      if (buffer[3398+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = buffer[3398+n] / 1.0;
      }

    writeArrayFloat(nc_id, vid_irhpfreq, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3498+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = buffer[3498+n] / 1.0;
      }

    writeArrayFloat(nc_id, vid_fgirhpfreq, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3598+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = buffer[3598+n] / 100.0;
      }

    writeArrayFloat(nc_id, vid_irsurfemiss, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3698+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = buffer[3698+n] / 100.0;
      }

    writeArrayFloat(nc_id, vid_fgirsurfemiss, index_1D, num_vals_1D, level_values);

    for (n=0; n<100; n++)
      {
      if (buffer[3798+n] == -32768)
        level_values[n] = -32768.0;
      else
        level_values[n] = buffer[3798+n] / 100.0;
      }

    writeArrayFloat(nc_id, vid_surfreflect, index_1D, num_vals_1D, level_values);

    // Stability parameters

    num_vals_1D[0] = 16;

    for (n=0; n<16; n++)
      {
      if ((buffer[3898+n] == -32768) || (buffer[3898+n] == -999))
        spectral_values[n] = -32768.0;
      else
        spectral_values[n] = buffer[3898+n] / 1.0;
      }

    writeArrayFloat(nc_id, vid_stability, index_1D, num_vals_1D, spectral_values);



//***********
//***********
// cloud frequency values
//***********
//***********


    // NPROVS calculated values beginning with the relative humidity

    num_vals_1D[0] = 100;

    // Unpack the pressures

    for (n=0; n<100; n++)
      {
      if (buffer[450+n] == -32768)
        temp_pressures[n] = -32768.0;
      else if (n < 20)
        temp_pressures[n] = buffer[450+n] / 1000.0;
      else
        temp_pressures[n] = buffer[450+n] / 10.0;

      if (buffer[550+n] == -32768)
        wvap_pressures[n] = -32768.0;
      else if (n < 20)
        wvap_pressures[n] = buffer[550+n] / 1000.0;
      else
        wvap_pressures[n] = buffer[550+n] / 10.0;
      }  // for (n=0;

    // Unpack the surface pressure

    if (buffer[27] != -32768)
      surf_pressure = buffer[27] / 10.0;
    else
      surf_pressure = -32768.0;


    // Calculate the observed relative humidities and tpw

    for (n=0; n<100; n++)
      {
      if (buffer[650+n] != -32768)
        temps[n] = buffer[650+n] / 64.0;
      else
        temps[n] = -32768.0;

      if (buffer[950+n] == -32768)
        wvaps[n] = -32768.0;
      else
        wvaps[n] = (float)(exp(buffer[950+n]/1024.0));
      }  // for (n=0...

    relative_humidities = calculateRelativeHumidities(temp_pressures, temps, 100,
                                                      wvap_pressures, wvaps, 100, surf_pressure);

    writeArrayFloat(nc_id, vid_rh, index_1D, num_vals_1D, relative_humidities);

    free(relative_humidities);

    tpw = calculateTPW(wvap_pressures, wvaps, 100, surf_pressure, -32768.0);
    writeVariableFloat(nc_id, vid_tpw, index, num_vals, tpw);

    // Calculate the microwave relative humidities and tpw

    for (n=0; n<100; n++)
      {
      if (buffer[750+n] != -32768)
        temps[n] = buffer[750+n] / 64.0;
      else
        temps[n] = -32768.0;

      if (buffer[1050+n] == -32768)
         wvaps[n] = -32768.0;
      else
         wvaps[n] = (float)(exp(buffer[1050+n]/1024.0));
      }  // for (n=0...

    relative_humidities = calculateRelativeHumidities(temp_pressures, temps, 100,
                                                      wvap_pressures, wvaps, 100, surf_pressure);

    writeArrayFloat(nc_id, vid_mitrh, index_1D, num_vals_1D, relative_humidities);

    free(relative_humidities);

    tpw = calculateTPW(wvap_pressures, wvaps, 100, surf_pressure, -32768.0);
    writeVariableFloat(nc_id, vid_mittpw, index, num_vals, tpw);


    // Calculate the first guess relative humidities and tpw

    for (n=0; n<100; n++)
      {
      if (buffer[850+n] != -32768)
        temps[n] = buffer[850+n] / 64.0;
      else
        temps[n] = -32768.0;

      if (buffer[1150+n] == -32768)
        wvaps[n] = -32768.0;
      else
        wvaps[n] = (float)(exp(buffer[1150+n]/1024.0));
      }  // for (n=0...

    relative_humidities = calculateRelativeHumidities(temp_pressures, temps, 100,
                                                      wvap_pressures, wvaps, 100, surf_pressure);

    writeArrayFloat(nc_id, vid_fgrh, index_1D, num_vals_1D, relative_humidities);

    free(relative_humidities);

    tpw = calculateTPW(wvap_pressures, wvaps, 100, surf_pressure, -32768.0);
    writeVariableFloat(nc_id, vid_fgtpw, index, num_vals, tpw);


//    // Granule file name
//
//    for (i=0; i<100; i++)
//      granule_name[i] = (char)buffer[6314+i];
//
//    granule_name[100] = '\0';
//
////    if (strlen(granule_name) == 0)
////      {
////      for (i=0; i<=99; i++)
////        granule_name[i] = ' ';
////      }
//
//    string[0] = granule_name;
//
//    if (retval = nc_put_vara_string(nc_id, vid_granulename, index, num_vals, (const char **)string))
//      ERR(retval);
//
//
//    // Program version and release
//
//    if (buffer[6414] != -32768)
//      writeVariableFloat(nc_id, vid_version, index, num_vals, (buffer[6414]/100.0));
//    else
//      writeVariableFloat(nc_id, vid_version, index, num_vals, -32768.0);
//
//    if (buffer[6415] != -32768)
//      writeVariableFloat(nc_id, vid_release, index, num_vals, (buffer[6415]/100.0));
//    else
//      writeVariableFloat(nc_id, vid_release, index, num_vals, -32768.0);
//    }  // for (recnum=0...



      nc_close(nc_id);
      }  // if (retval == NC_NOERR...

    }  // if (closest_footprint != -1)
  else
    {
    printf("\n\nNo footprints were found within %0.1f km and %0.1f hours.\n", MAX_DIST_DIFF,
                                                                              MAX_TIME_DIFF);
    printf("No collocation has been made for this ozonesonde.\n\n");
    }

  fclose(in);
  }

// end of file
