/*
   Name- OzoneCollocationFileBuilder.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  5/19/2023   Programmer- Mike Pettey (IMSG)

   Function- This program extracts data from GOES ABI netCDF files
             and copies the data to a GOES ABI Daily Data File (GDDF).
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>

#include <dirent.h>
#include <errno.h>

#define  TRUE      0
#define  FALSE     1

#define  OZONESONDE  0
#define  NUCAPS      600
#define  ECMWF       800


struct alpha_sort
  {
  char value[500];
  struct alpha_sort *next;
  };


int createOutputFile();
struct alpha_sort *readAndSortDateDirectories(int start_date, int end_date);
struct alpha_sort* readAndSortPlatforms(char *collocation_dir_name);
void processCollocationsInDateDir(int collocation_group_id, char *date);
void processCollocation(int collocation_group_id, char *collocation_dir_name, char *site_name);
void processPlatform(int collocation_group_id, char *platform_dir_name, char *platform_file_name);
int processOzonesonde(int collocation_group_id, char *file_name, int group_id);
void processECMWF(char *file_name, int group_id);
void processNUCAPS(char *file_name, int group_id);
void addAvailablePlatforms(int nc_id);

int defineVariable(int nc_group_id, char *var_name, nc_type var_type, int ndims, int *dimids,
                   char *attr_string, char *attr);

void writeAttributeInteger(int grp, char *attrVariable, int attrValue);
void writeAttributeLong(int grp, char *attrVariable, long attrValue);
void writeAttributeText(int grp, char *attrVariable, char *attrValue);
void writeVariableInteger(int group_id, int var_id, size_t *index, size_t *num_vals, int value);


int  available_platform[10000], available_platform_type[10000];
int  number_of_collocations;
int  earliest_date, latest_date;
char available_platform_name[10000][1000];



int main(int argc, char *argv[])
  {
  int  n, start_date, end_date;
  int  retval, nc_id, date_group_id;
  struct alpha_sort *sorted_dates;
  struct alpha_sort *ptr;

  // Print an initial message

  printf("\nBuilding an NPROVS Ozone Collocation File\n\n");

  start_date = atoi(argv[1]);
  end_date   = atoi(argv[2]);

  printf("   Start Date:  %d\n", start_date);
  printf("   End Date:    %d\n", end_date);

  // Initialize the available platform array

  for (n=0; n<10000; n++)
    {
    available_platform[n] = FALSE;
    available_platform_type[n] = -32768;
    sprintf(available_platform_name[n], "%c", '\0');
    }

  number_of_collocations = 0;

  earliest_date =  99999999;
  latest_date   = -99999999;

  // Create a new output netCDF file

  nc_id = createOutputFile();

  if (nc_id != -32768)
    {

    // Create a group for the collocations

    retval = nc_def_grp(nc_id, "collocations", &date_group_id);

    if (retval != NC_NOERR)
      {
      printf("%d\n", retval);
      printf("\n***A group for the collocations could not be created in the netCDF file.\n");
      printf("Execution cannot continue.\n\n");
      exit(1);
      }


    // Read and sort the date directories

    sorted_dates = readAndSortDateDirectories(start_date, end_date);

    // Loop through the dates and process every collocation

    ptr = sorted_dates;

    while (ptr != NULL)
      {
      processCollocationsInDateDir(date_group_id, ptr->value);
      ptr = ptr->next;
      }


    // Add a group that will contain information about the
    // available platforms

    addAvailablePlatforms(nc_id);


    // Free the date structures

    ptr = sorted_dates;

    while (ptr != NULL)
      {
      ptr = ptr->next;
      free(sorted_dates);
      sorted_dates = ptr;
      }

    // Save the number of collocations and date range as global attributes

    writeAttributeInteger(nc_id, "number_of_collocations", number_of_collocations);
    writeAttributeInteger(nc_id, "data_start_date_(yyyymmdd)", earliest_date);
    writeAttributeInteger(nc_id, "data_end_date_(yyyymmdd)", latest_date);

    // Save the baseline platform id and name

    writeAttributeInteger(nc_id, "baseline_platform_id", 0);
    writeAttributeText(nc_id, "baseline_platform_name", "Ozonesonde");


    // Close the netCDF file

    nc_close(nc_id);
    }  // if (nc_id != -32768)...

  printf("\nProcessing completed.\n\n");
  }



// ===============================================================================
int createOutputFile()
  {
  int     retval, nc_id;
  int     month, day, year;
  char    date_string[50];
  time_t  date_time;
  struct  tm  *ts;

  // Open the output file

  retval = nc_create("out.file", NC_NETCDF4, &nc_id);

  if (retval == NC_NOERR)
    {

    // Add some of the global attributes

    writeAttributeText(nc_id, "project", "NOAA Products Validation System (NPROVS)");
    writeAttributeText(nc_id, "title", "NPROVS Ozone Collocation File");
    writeAttributeLong(nc_id, "file_version_number", (long)1);

    writeAttributeText(nc_id, "summary", "Multiple data platforms (satellites, sondes, forecasts, etc.) collocated for cross-comparison and validation purposes");

    writeAttributeText(nc_id, "creator_name", "DOC/NOAA/NESDIS/STAR > NPROVS Team, Center for Satellite Applications and Research, NESDIS, NOAA, U.S. Department of Commerce");
    writeAttributeText(nc_id, "publisher_name", "DOC/NOAA/NESDIS/STAR > NPROVS Team, Center for Satellite Applications and Research, NESDIS, NOAA, U.S. Department of Commerce");
    writeAttributeText(nc_id, "institution", "DOC/NOAA/NESDIS/STAR");
    writeAttributeText(nc_id, "naming_authority", "gov.noaa.nesdis.star");

//  writeAttributeText(nc_id, "missing_integer_value", "-32768");
//  writeAttributeText(nc_id, "missing_float_value", "-32768.0");
//  writeAttributeText(nc_id, "missing_byte_value", "-128");
//
//  //writeAttributeLong(nc_id, "missing_value_(integer)", (long)-32768);
//  //writeAttributeFloat(nc_id, "missing_value_(float)", (float)-32768.0);
//  //writeAttributeShort(nc_id, "missing_value_(byte)", (short)-128);
//
//  writeAttributeLong(nc_id, "File_Version_Number", NETCDF4_VERSION_NUMBER);
//
//  int num_collocation_attributes = collocations.getNumberOfAttributes();
//
//  for (n=0; n<num_collocation_attributes; n++)
//    {
//    str1 = collocations.getAttributeName(n);
//    attr_name = &str1[0];
//
//    str2 = collocations.getAttributeValue(n);
//    attr_value = &str2[0];
//
//    writeAttributeText(nc_id, attr_name, attr_value);
//    }

    // Add the file creation date as an attribute

    time (&date_time);
    ts = localtime(&date_time);

    month  = ts->tm_mon + 1;
    day    = ts->tm_mday;
    year   = ts->tm_year;

    if (year > 56)
      year = 1900 + year;
    else
      year = 2000 + year;

    if (month == 1)
      sprintf(date_string, "%d January %d", day, year);
    else if (month == 2)
      sprintf(date_string, "%d February %d", day, year);
    else if (month == 3)
      sprintf(date_string, "%d March %d", day, year);
    else if (month == 4)
      sprintf(date_string, "%d April %d", day, year);
    else if (month == 5)
      sprintf(date_string, "%d May %d", day, year);
    else if (month == 6)
      sprintf(date_string, "%d June %d", day, year);
    else if (month == 7)
      sprintf(date_string, "%d July %d", day, year);
    else if (month == 8)
      sprintf(date_string, "%d August %d", day, year);
    else if (month == 9)
      sprintf(date_string, "%d September %d", day, year);
    else if (month == 10)
      sprintf(date_string, "%d October %d", day, year);
    else if (month == 11)
      sprintf(date_string, "%d November %d", day, year);
    else if (month == 12)
      sprintf(date_string, "%d December %d", day, year);

    writeAttributeText(nc_id, "file_creation_date", date_string);
    writeAttributeText(nc_id, "file_modification_date", date_string);
    }
  else
    {
    nc_id = -32768;
    }

  return nc_id;
  }



// ===============================================================================
void addAvailablePlatforms(int nc_id)
  {
  int     n, platform_index, num_available_platforms;
  int     vid_platform_id, vid_platform_type, vid_platform_name;
  int     platform_dim, dimid_platform[1];
  char    platform_name[100];
  char    *string[1];
  size_t  index[1], num_vals[1];


  // Count the number of available platforms

  num_available_platforms = 0;

  for (n=0; n<10000; n++)
    {
    if (available_platform[n] == TRUE)
      num_available_platforms++;
    }


  // Write the platform variables

  nc_def_dim(nc_id, "number_of_platforms", num_available_platforms, &platform_dim);
  dimid_platform[0] = platform_dim;

  vid_platform_id = defineVariable(nc_id, "platform_id", NC_INT, 1, dimid_platform, NULL, NULL);
  vid_platform_type = defineVariable(nc_id, "platform_type", NC_INT, 1, dimid_platform, NULL, NULL);
  vid_platform_name = defineVariable(nc_id, "platform_name", NC_STRING, 1, dimid_platform, NULL, NULL);

  platform_index = -1;

  for (n=0; n<10000; n++)
    {
    if (available_platform[n] == TRUE)
      {
      platform_index++;

      index[0]    = platform_index;
      num_vals[0] = 1;

      // Platform ID

      writeVariableInteger(nc_id, vid_platform_id, index, num_vals, n);

      // Platform type

      writeVariableInteger(nc_id, vid_platform_type, index, num_vals, available_platform_type[n]);

      // Platform name

      sprintf(platform_name, "%s", available_platform_name[n]);
      string[0] = platform_name;

      nc_put_vara_string(nc_id, vid_platform_name, index, num_vals, (const char **)string);
      }  // if (available_platform[n]...
    }  // for (n=0...

  }



// ===============================================================================
struct alpha_sort* readAndSortDateDirectories(int start_date, int end_date)
  {
  struct alpha_sort *sorted_dates;
  int    directory_date;
  char   date_name[100];
  struct dirent *date_entry;
  struct alpha_sort *new_date;
  struct alpha_sort *ptr;
  DIR    *collocations_dir;

  sorted_dates = NULL;


  collocations_dir = opendir("in.dir");

  if (collocations_dir == NULL)  // opendir returns NULL if couldn't open directory
    {
    printf("\n\nERROR:  Could not open the collocation directory\n");
    printf("Execution ending\n\n");

    exit(1);
    }

  while ((date_entry = readdir(collocations_dir)) != NULL)
    {
    sprintf(date_name, "%s", date_entry->d_name);

    if (date_name[0] != '.')
      {
      directory_date = atoi(date_name);

      // If the directory date is within the date range, then open it
      // and insert the date string into the sorted array

      if ((directory_date >= start_date) && (directory_date <= end_date))
        {
        new_date = (struct alpha_sort*)malloc(sizeof(struct alpha_sort));
        sprintf(new_date->value, "%s", date_name);
        new_date->next = NULL;

        if (sorted_dates == NULL)
          {
          sorted_dates = new_date;
          }
        else if (strcmp(date_name, sorted_dates->value) < 0)
          {
          new_date->next = sorted_dates;
          sorted_dates = new_date;
          }
        else
          {
          ptr = sorted_dates;

          while (ptr != NULL)
            {
            if (ptr->next == NULL)
              {
              ptr->next = new_date;
              ptr = NULL;
              }
            else
              {
              if (strcmp(date_name, ptr->next->value) < 0)
                {
                new_date->next = ptr->next;
                ptr->next = new_date;
                ptr = NULL;
                }
              else
                {
                ptr = ptr->next;
                }
              }
            }
          }
        }
      }
    }

  closedir(collocations_dir);

  return sorted_dates;
  }



void processCollocationsInDateDir(int collocation_group_id, char *date)
  {
  int    retval, date_group_id;
  char   site_name[100];
  char   date_directory_name[500];
  char   site_directory_name[500];
  struct dirent *entry;
  DIR    *date_dir;

  // Create a subgroup for the date

  retval = nc_def_grp(collocation_group_id, date, &date_group_id);

  if (retval != NC_NOERR)
    {
    printf("Error: %d\n", retval);
    printf("\n***A group for the date could not be created in the netCDF file.\n");
    printf("Platform name:  %s\n", date);
    printf("Execution cannot continue.\n\n");
    exit(1);
    }

  // Search the input date directory for collocations within it

  sprintf(date_directory_name, "in.dir/%s", date);
  date_dir = opendir(date_directory_name);
printf("Processing Date:  %s\n", date);
  if (date_dir == NULL)  // opendir returns NULL if couldn't open directory
    {
    printf("\n\nERROR:  Could not open the collocation date directory:  %s\n", date);
    printf("Execution ending\n\n");

    exit(1);
    }

  while ((entry = readdir(date_dir)) != NULL)
    {
    sprintf(site_name, "%s", entry->d_name);

    if (site_name[0] != '.')
      {
      sprintf(site_directory_name, "%s/%s", date_directory_name, entry->d_name);
      processCollocation(date_group_id, site_directory_name, entry->d_name);
      }
    }

  closedir(date_dir);
  }



// ===============================================================================
void processCollocation(int date_group_id, char *collocation_dir_name, char *site_name)
  {
  int    n, index, retval, collocation_group_id;
  char   file_name[100], clean_site_name[1000];
  struct alpha_sort *sorted_platforms;
  struct alpha_sort *ptr;
  struct dirent *entry;
  DIR    *site_dir;

  printf("\nProcessing Collocation:  %s\n", site_name);

  // Copy the site name into a new string and remove any invalid characters.
  // This is necessary become some sonde locations used single and double quotes.

  for (n=0; n<1000; n++)
    clean_site_name[n] = '\0';

  index = -1;

  for (n=0; n<strlen(site_name); n++)
    {
    if ((site_name[n] != '\'') && (site_name[n] != '\"'))
      {
      index++;
      clean_site_name[index] = site_name[n];
      }
    }


  // Create a subgroup for this collocation

  retval = nc_def_grp(date_group_id, clean_site_name, &collocation_group_id);

  if (retval != NC_NOERR)
      {
      printf("Error: %d\n", retval);
      printf("\n***A group for the collocation could not be created in the netCDF file.\n");
      printf("Platform name:  %s\n", clean_site_name);
      printf("Execution cannot continue.\n\n");
      exit(1);
      }


  // Sort the platforms into alphabetical order

  sorted_platforms = readAndSortPlatforms(collocation_dir_name);

  // Loop through the platforms and process each one

  ptr = sorted_platforms;

  while (ptr != NULL)
    {
    processPlatform(collocation_group_id, collocation_dir_name, ptr->value);
    ptr = ptr->next;
    }

  number_of_collocations++;

  // Free the memory used to sort the platforms

  ptr = sorted_platforms;

  while (ptr != NULL)
    {
    ptr = ptr->next;
    free(sorted_platforms);
    sorted_platforms = ptr;
    }
  }



struct alpha_sort* readAndSortPlatforms(char *collocation_dir_name)
  {
  char   file_name[100];
  struct dirent *entry;
  struct alpha_sort *sorted_platforms;
  struct alpha_sort *new_platform;
  struct alpha_sort *ptr;
  DIR    *collocations_dir;

  sorted_platforms = NULL;


  collocations_dir = opendir(collocation_dir_name);

  if (collocations_dir == NULL)  // opendir returns NULL if couldn't open directory
    {
    printf("\n\nERROR:  Could not open the collocation directory\n");
    printf("Execution ending\n\n");

    exit(1);
    }

  while ((entry = readdir(collocations_dir)) != NULL)
    {
    sprintf(file_name, "%s", entry->d_name);

    if (file_name[0] != '.')
      {
      new_platform = (struct alpha_sort*)malloc(sizeof(struct alpha_sort));
      sprintf(new_platform->value, "%s", file_name);
      new_platform->next = NULL;

      if (sorted_platforms == NULL)
        {
        sorted_platforms = new_platform;
        }
      else if (strcmp(file_name, sorted_platforms->value) < 0)
        {
        new_platform->next = sorted_platforms;
        sorted_platforms = new_platform;
        }
      else
        {
        ptr = sorted_platforms;

        while (ptr != NULL)
          {
          if (ptr->next == NULL)
            {
            ptr->next = new_platform;
            ptr = NULL;
            }
          else
            {
            if (strcmp(file_name, ptr->next->value) < 0)
              {
              new_platform->next = ptr->next;
              ptr->next = new_platform;
              ptr = NULL;
              }
            else
              {
              ptr = ptr->next;
              }
            }
          }
        }
      }
    }

  closedir(collocations_dir);

  return sorted_platforms;
  }



// ===============================================================================
void processPlatform(int collocation_group_id, char *platform_dir_name, char *platform_file_name)
  {
  int     i, n, index, plat_id, in_id, collocation_date;
  int     retval, platform_id, platform_type;
  char    platform_name[1000], clean_platform_name[1000], file_name[1000];
  char    *attr_value;
  size_t  attr_length;

  // Create a subgroup for the platform

  for (i=0; i<100; i++)
    platform_name[i] = '\0';

  strncpy(platform_name, platform_file_name, (strlen(platform_file_name)-3));

  // Copy the platform name into a new string and remove any invalid characters.
  // This is necessary become some sonde locations used single and double quotes.

  for (n=0; n<1000; n++)
    clean_platform_name[n] = '\0';

  index = -1;

  for (n=0; n<strlen(platform_name); n++)
    {
    if ((platform_name[n] != '\'') && (platform_name[n] != '\"'))
      {
      index++;
      clean_platform_name[index] = platform_name[n];
      }
    }

  printf("   Platform:  <%s>\n", clean_platform_name);

  retval = nc_def_grp(collocation_group_id, clean_platform_name, &plat_id);

  if (retval != NC_NOERR)
    {
    printf("Error: %d\n", retval);
    printf("\n***A group for the platform could not be created in the netCDF file.\n");
    printf("Platform name:  %s\n", clean_platform_name);
    printf("Execution cannot continue.\n\n");
    exit(1);
    }


  // Open the platform netCDF and read the data type

  sprintf(file_name, "%s/%s", platform_dir_name, platform_file_name);

  retval = nc_open(file_name, NC_NOWRITE, &in_id);

  if (retval == NC_NOERR)
    {

    // Platform ID

    retval = nc_inq_attlen(in_id, NC_GLOBAL, "NPROVS_Platform_ID", &attr_length);

    attr_value = (char *)malloc(attr_length + 1);

    retval = nc_get_att_text(in_id, NC_GLOBAL, "NPROVS_Platform_ID", attr_value);

    attr_value[attr_length] = '\0';

    platform_id = atoi(attr_value);

    free(attr_value);

    // Add the platform id to the list of available platforms

    if ((platform_id >= 0) && (platform_id < 10000))
      available_platform[platform_id] = TRUE;

    // Platform Type

    retval = nc_inq_attlen(in_id, NC_GLOBAL, "NPROVS_Platform_Type", &attr_length);

    attr_value = (char *)malloc(attr_length + 1);

    retval = nc_get_att_text(in_id, NC_GLOBAL, "NPROVS_Platform_Type", attr_value);

    attr_value[attr_length] = '\0';

    platform_type = atoi(attr_value);

    free(attr_value);


    // Platform Name

    retval = nc_inq_attlen(in_id, NC_GLOBAL, "NPROVS_Platform_Name", &attr_length);

    attr_value = (char *)malloc(attr_length + 1);

    retval = nc_get_att_text(in_id, NC_GLOBAL, "NPROVS_Platform_Name", attr_value);

    attr_value[attr_length] = '\0';

    sprintf(clean_platform_name, "%s", attr_value);

    free(attr_value);

    nc_close(in_id);


    // Based on the platform type, process the appropriate file

    available_platform_type[platform_id] = platform_type;
    sprintf(available_platform_name[platform_id], "%s", clean_platform_name);

    switch (platform_type)
      {
      case OZONESONDE:
	{
	collocation_date = processOzonesonde(collocation_group_id, file_name, plat_id);

	if (collocation_date < earliest_date)
	  earliest_date = collocation_date;

	if (collocation_date > latest_date)
	  latest_date = collocation_date;

	break;
	}

      case NUCAPS:
	{
	processNUCAPS(file_name, plat_id);
	break;
	}

      case ECMWF:
	{
	processECMWF(file_name, plat_id);
	break;
	}

      default:
	printf("*** UNKNOWN platform data type:  %d\n", platform_type);
      }
    }

  }

// end of file
