/*
   Name- SondeCollocator.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  3/03/2025   Programmer- Mike Pettey (STC)

   Function- This program extracts data from unified radiosonde files and creates
             a netCDF file.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>

#include <dirent.h>
#include <errno.h>

#define  TRUE      0
#define  FALSE     1

#define SIGNIFICANT 0
#define STANDARD    1


struct merged_data
  {
  int   std_or_sig;
  float pressure;
  float temp;
  float radcor_temp;
  float cfsr_temp;
  float cfsr_reanal_temp;
  float wvmr;
  float radcor_wvmr;
  float cfsr_wvmr;
  float cfsr_reanal_wvmr;
  float drift_time;
  float drift_latitude;
  float drift_longitude;
  float geo_height;
  float wind_dir;
  float wind_speed;
  int   pressure_qc;
  int   temp_qc;
  int   humidity_qc;
  int   height_qc;
  int   wind_qc;
  int   missing_flag;
  float background_temp;
  float background_wvmr;

  struct merged_data *next;
  };


int   *collocation_dates, *collocation_times;
float *collocation_lats, *collocation_lons;

int  scalar_dim, dim_levels, dim_lpw, dim_tropo, dim_fcst;
int  dimid_scalar[1];
int  dimid_levels[2], dimid_lpw[2], dimid_tropo[2], dimid_fcst[2];

int  vid_sonde_id, vid_lat, vid_lon, vid_date, vid_time, vid_synoptic_hour;
int  vid_elevation, vid_report_type, vid_instrument_type, vid_terrain;
int  vid_radcor, vid_lo_interp_press, vid_hi_interp_press;
int  vid_qc_flag, vid_nprovs_qc, vid_vert_ext, vid_prof_cap, vid_superad, vid_inversion;
int  vid_tpw_cat, vid_moisture_cap, vid_surf_data_flag, vid_max_inv_depth;
int  vid_calc_tropo_pres, vid_sbl_flag, vid_hi_interp_moisture;
int  vid_wvmr_anomaly, vid_wvmr_event_1, vid_wvmr_extreme_1,  vid_wvmr_event_2;
int  vid_wvmr_neg_event, vid_wvmr_extreme_neg, vid_utls_anomaly, vid_inversion_type;
int  vid_dwpt_depress, vid_duplicate, vid_tier_2, vid_screen_adjust;
int  vid_sbl_top, vid_tropo_press, vid_surf_press, vid_surf_temp;
int  vid_surf_wvmr, vid_surf_dwpt, vid_low_wvmr, vid_high_wvmr;
int  vid_suan, vid_gcos, vid_gruan, vid_tpw, vid_lpw;
int  vid_utls_pressure, vid_temp_extrap, vid_climate_limit, vid_dwpt_flag;
int  vid_vert_extent, vid_temp_gap, vid_moisture_gap, vid_ncep_screen;
int  vid_low_cloud_amount, vid_low_cloud_type, vid_low_cloud_fraction;
int  vid_radcor_code, vid_low_cloud_base, vid_low_cloud_base_lut;
int  vid_mid_cloud_type, vid_high_cloud_type;
int  vid_vtemp_flag, vid_text_flag, vid_solzen, vid_daynight;
int  vid_gfs_lat, vid_gfs_lon, vid_gfs_date, vid_gfs_time, vid_gfs_distance;
int  vid_gfs_press, vid_gfs_temps, vid_gfs_surf_temp, vid_gfs_tropo_temp, vid_gfs_relhum;
int  vid_gfs_u_wind, vid_gfs_v_wind, vid_gfs_tropo_u_wind, vid_gfs_tropo_v_wind;
int  vid_gfs_height, vid_gfs_surf_height, vid_gfs_tropo_height, vid_gfs_surf_press;
int  vid_gfs_tropo_press, vid_gfs_sea_press;
int  vid_temp_cap_press, vid_moisture_cap_press, vid_gfs_temp_cap_flag, vid_gfs_temp_cap_press;
int  vid_gfs_moisture_cap_flag, vid_gfs_moisture_cap_press, vid_cfsr_temp_cap_flag;
int  vid_cfsr_temp_cap_press, vid_cfsr_moisture_cap_flag, vid_cfsr_moisture_cap_press;
int  vid_reanal_temp_cap_flag, vid_reanal_temp_cap_press;
int  vid_reanal_moisture_cap_flag, vid_reanal_moisture_cap_press;

int  vid_press, vid_temps, vid_radcor_temps, vid_cfsr_temps, vid_cfsr_reanal_temps;
int  vid_wvmrs, vid_radcor_wvmrs, vid_cfsr_wvmrs, vid_cfsr_reanal_wvmrs;
int  vid_drift_times, vid_drift_lats, vid_drift_lons, vid_heights;
int  vid_wind_dirs, vid_wind_speeds, vid_press_qcs, vid_temp_qcs;
int  vid_humid_qcs, vid_height_qcs, vid_wind_qcs, vid_missing_flags, vid_std_sig_flags;
int  vid_back_temps, vid_back_wvmrs;
int  vid_tropo_rep_press, vid_tropo_calc_press, vid_tropo_deflect, vid_tropo_qc;
int  vid_surf_radcor_temp, vid_surf_press_qc;
int  vid_surf_temp_qc, vid_surf_humid_qc, vid_surf_wind_dir, vid_surf_wind_speed;



void mkdir(char *dir, int permissions);

int countCollocations(int day_to_process, char *file_name, int *sonde_file_type);
int createOutputFile(char *output_dir_name, int day_to_process);
void defineVariables(int nc_id, int num_collocations);
int processSonde03(char *file_name, int day_to_process, int nc_id, int num_written);
void createCollocationInfo(int day_to_process, int num_collocations);

float dwptToWvmr(float pressure, float dwpt);

int defineVariable(int nc_id, char *var_name, nc_type var_type, int ndims, int *dimids,
                   char *attr_string, char *attr);

void writeArrayByte(int group_id, int var_id, size_t *index, size_t *num_vals, char *value);
void writeArrayFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float *value);
void writeArrayInteger(int group_id, int var_id, size_t *index, size_t *num_vals, int *value);
void writeAttributeShort(int nc_id, char *name, short value);
void writeAttributeText(int nc_id, char *name, char *value);
void writeVariableByte(int group_id, int var_id, size_t *index, size_t *num_vals, char value);
void writeVariableFloat(int group_id, int var_id, size_t *index, size_t *num_vals, float value);
void writeVariableInteger(int group_id, int var_id, size_t *index, size_t *num_vals, int value);
void writeVariableShort(int group_id, int var_id, size_t *index, size_t *num_vals, short value);



int main(int argc, char *argv[])
  {
  int   day_before, day_to_process, day_after;
  int   num_collocations, num_before, num_baseday, num_after;
  int   nc_id, sonde_before_file_type, sonde_baseday_file_type, sonde_after_file_type;
  int   num_written;
  char  output_dir_name[1000];
  DIR   *dir;

  // Print an initial message

  printf("\nReading unified radiosonde data and creating a netCDF granule\n\n");

  // Unpack the dates from the command line

  day_before     = atoi(argv[1]);
  day_to_process = atoi(argv[2]);
  day_after      = atoi(argv[3]);

  printf("Day before:        %d\n", day_before);
  printf("Date to process:   %d\n", day_to_process);
  printf("Day after:         %d\n", day_after);


  // Create a directory for the date if it does not exist

  sprintf(output_dir_name, "out.dir/%d", day_to_process);

  dir = opendir(output_dir_name);

  if (dir)
    {
    closedir(dir);
    }
  else if (ENOENT == errno)
    {
    mkdir(output_dir_name, 0700);
    }
  else
    {
    printf("The directory check could not be done for an unknown reason.\n");
    printf("Execution will stop.\n\n");
    exit(1);
    }


  // Read the radiosonde data to count the number of observations that
  // occurred on the day to process

  num_before  = countCollocations(day_to_process, "raob_before.file", &sonde_before_file_type);
  num_baseday = countCollocations(day_to_process, "raob_baseday.file", &sonde_baseday_file_type);
  num_after   = countCollocations(day_to_process, "raob_after.file", &sonde_after_file_type);

  num_collocations = num_before + num_baseday + num_after;

  printf("\nTotal collocations:  %d\n", num_collocations);
  printf("   Day before:       %d\n", num_before);
  printf("   Baseday:          %d\n", num_baseday);
  printf("   Day after:        %d\n", num_after);


  // Allocate the arrays into which the collocation data will be put

  collocation_dates = (int*)malloc(num_collocations*sizeof(int));
  collocation_times = (int*)malloc(num_collocations*sizeof(int));
  collocation_lats  = (float*)malloc(num_collocations*sizeof(float));
  collocation_lons  = (float*)malloc(num_collocations*sizeof(float));

  // Create the output netCDF file

  nc_id = createOutputFile(output_dir_name, day_to_process);


  // Define the variables

  defineVariables(nc_id, num_collocations);


  // Process the appropriate type of sonde format

  num_written = 0;

  if (sonde_before_file_type == 3)
    num_written = processSonde03("raob_before.file", day_to_process, nc_id, num_written);

  if (sonde_baseday_file_type == 3)
    num_written = processSonde03("raob_baseday.file", day_to_process, nc_id, num_written);

  if (sonde_before_file_type == 3)
    num_written = processSonde03("raob_after.file", day_to_process, nc_id, num_written);

  // Close the output file

  nc_close(nc_id);


  // Create the Collocation_info file

  createCollocationInfo(day_to_process, num_collocations);

  // Free the collocation data memory

  free(collocation_lons);
  free(collocation_lats);
  free(collocation_times);
  free(collocation_dates);
  
  printf("\nProcessing completed.\n\n");
  }



//----------------------------------------------------------------
int countCollocations(int day_to_process, char *file_name, int *file_type)
  {
  int   num_collocations;
  int   header_length, record_length, num_records;
  int   rec, date;
  short buffer[100];
  long  offset;
  FILE  *in;

  num_collocations = 0;

  // Open the input file

  if ((in=fopen(file_name,"r")) == NULL)
    {
    printf("\n\nThe input file could not be opened.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  // Extract information from the header

  fseek(in, 8, SEEK_SET);
  fread(&header_length, 4, 1, in);

  fseek(in, 16, SEEK_SET);
  fread(&record_length, 4, 1, in);

  fseek(in, 24, SEEK_SET);
  fread(&num_records, 4, 1, in);

  if (record_length == 4000)
    *file_type = 1;
  else if (record_length == 6000)
    *file_type = 2;
  else if (record_length == 6500)
    *file_type = 3;
  else if (record_length == 8500)
    *file_type = 5;

  // Loop through the data records in the file

  for (rec=0; rec<num_records; rec++)
    {
    offset = (long)header_length + ((long)rec * (long)record_length);
    fseek(in, offset, SEEK_SET);
    fread(buffer, 2, 100, in);

    date = (buffer[35]*10000) + (buffer[36]*100) + buffer[37];

    if (date == day_to_process)
      num_collocations++;
    }

  fclose(in);

  return num_collocations;
  }



//----------------------------------------------------------------
int createOutputFile(char *output_dir_name, int day_to_process)
  {
  int     nc_id, retval;
  int     month, day, year;
  char    output_file_name[1000], month_string[25], date_string[100];
  time_t  date_time;
  struct  tm  *ts;   

  //sprintf(output_file_name, "%s/sonde_%d.nc", output_dir_name, day_to_process);
  sprintf(output_file_name, "%s/sonde.nc", output_dir_name);

  retval = nc_create(output_file_name, NC_NETCDF4, &nc_id);

  if (retval == NC_NOERR)
    {
    writeAttributeText(nc_id, "project", "NOAA Products Validation System (NPROVS)");
    writeAttributeText(nc_id, "title", "Sonde data");
    writeAttributeText(nc_id, "file_version_number", "1");

    writeAttributeText(nc_id, "summary", "Multiple data platforms (satellites, sondes, forecasts, etc.) collocated for cross-comparison and validation purposes");

    writeAttributeText(nc_id, "creator_name", "DOC/NOAA/NESDIS/STAR > NPROVS Team, Center for Satellite Applications and Research, NESDIS, NOAA, U.S. Department of Commerce");
    writeAttributeText(nc_id, "publisher_name", "DOC/NOAA/NESDIS/STAR > NPROVS Team, Center for Satellite Applications and Research, NESDIS, NOAA, U.S. Department of Commerce");
    writeAttributeText(nc_id, "institution", "DOC/NOAA/NESDIS/STAR");
    writeAttributeText(nc_id, "naming_authority", "gov.noaa.nesdis.star");

    writeAttributeText(nc_id, "missing_integer_value", "-32768");
    writeAttributeText(nc_id, "missing_float_value", "-32768.0");
    writeAttributeText(nc_id, "missing_byte_value", "-128");

    writeAttributeText(nc_id, "NPROVS_Platform_ID", "0");
    writeAttributeText(nc_id, "NPROVS_Platform_Type", "0");
    writeAttributeText(nc_id, "NPROVS_Platform_Name", "Sonde");

    sprintf(date_string, "%d", day_to_process);
    writeAttributeText(nc_id, "Date_of_data", date_string);

    // Add the file creation date as an attribute

    time (&date_time);         
    ts = localtime(&date_time);

    month  = ts->tm_mon + 1;    
    day    = ts->tm_mday;       
    year   = ts->tm_year;       

    if (year > 56)
      year = 1900 + year;
    else
      year = 2000 + year;

    if (month == 1)
      sprintf(month_string, "January");
    else if (month == 2)
      sprintf(month_string, "February");
    else if (month == 3)
      sprintf(month_string, "March");
    else if (month == 4)
      sprintf(month_string, "April");
    else if (month == 5)
      sprintf(month_string, "May");
    else if (month == 6)
      sprintf(month_string, "June");
    else if (month == 7)
      sprintf(month_string, "July");
    else if (month == 8)
      sprintf(month_string, "August");
    else if (month == 9)
      sprintf(month_string, "September");
    else if (month == 10)
      sprintf(month_string, "October");
    else if (month == 11)
      sprintf(month_string, "November");
    else if (month == 12)
      sprintf(month_string, "December");

    sprintf(date_string, "%d %s %d", day, month_string, year);

    writeAttributeText(nc_id, "file_creation_date", date_string);
    }  // if (retval...

  return nc_id;
  }



//----------------------------------------------------------------
void defineVariables(int nc_id, int num_collocations)
  {
  int  retval;

  // Define the dimensions

  retval = nc_def_dim(nc_id, "Num_Collocations", num_collocations, &scalar_dim);

  dimid_scalar[0] = scalar_dim;

  retval = nc_def_dim(nc_id, "Levels", 118, &dim_levels);
  retval = nc_def_dim(nc_id, "LPW_Layers", 18, &dim_lpw);
  retval = nc_def_dim(nc_id, "Tropopause_Levels", 5, &dim_tropo);
  retval = nc_def_dim(nc_id, "GFS_Levels", 26, &dim_fcst);

  dimid_levels[0] = scalar_dim;
  dimid_levels[1] = dim_levels;

  dimid_lpw[0] = scalar_dim;
  dimid_lpw[1] = dim_lpw;

  dimid_tropo[0] = scalar_dim;
  dimid_tropo[1] = dim_tropo;

  dimid_fcst[0] = scalar_dim;
  dimid_fcst[1] = dim_fcst;


  // Define the variables

  vid_lat = defineVariable(nc_id, "latitude", NC_FLOAT, 1, dimid_scalar, "units", "degrees_north");

  vid_lon = defineVariable(nc_id, "longitude", NC_FLOAT, 1, dimid_scalar, "units", "degrees_east");

  vid_date = defineVariable(nc_id, "date", NC_INT, 1, dimid_scalar, "format", "yyyymmdd");

  vid_time = defineVariable(nc_id, "time", NC_INT, 1, dimid_scalar, "format", "hhmmss");

  vid_sonde_id = defineVariable(nc_id, "sonde_station_id", NC_STRING, 1, dimid_scalar, 
                                "string_length", "9");

  vid_synoptic_hour = defineVariable(nc_id, "synoptic_hour", NC_BYTE, 1, dimid_scalar,
                                     "values", "1 = 00Z, 2 = 06Z, 3 = 12Z, 4 = 18Z");

  vid_elevation = defineVariable(nc_id, "elevation", NC_INT, 1, dimid_scalar, NULL, NULL);

  vid_report_type = defineVariable(nc_id, "report_type", NC_SHORT, 1, dimid_scalar, NULL, NULL);

  vid_instrument_type = defineVariable(nc_id, "instrument_type", NC_SHORT, 1, dimid_scalar, 
					 "source", "NCEP Common Code Table S-2: Radiosonde/Sounding System Used");

  vid_terrain = defineVariable(nc_id, "terrain", NC_BYTE, 1, dimid_scalar, "values", 
				 "0=Island Coast, 1=Mainland Inland, 2=Mainland Coast, 3=Island Inland, 4=Ship");

  vid_lo_interp_press = defineVariable(nc_id, "lowest_interpolated_pressure", NC_FLOAT, 1, dimid_scalar, 
					 "units", "hPa");

  vid_hi_interp_press = defineVariable(nc_id, "highest_interpolated_pressure", NC_FLOAT, 1, dimid_scalar, 
					 "units", "hPa");

  vid_qc_flag = defineVariable(nc_id, "sonde_qc", NC_BYTE, 1, dimid_scalar, "values", "0=pass, 1=fail");

  vid_nprovs_qc = defineVariable(nc_id, "nprovs_qc", NC_BYTE, 1, dimid_scalar, "values", "0=pass, 1=fail");

  vid_radcor = defineVariable(nc_id, "ncep_radiation_corrected_flag", NC_BYTE, 1, dimid_scalar,
                              "values", "0=not corrected, 1=corrected");

  vid_vert_ext = defineVariable(nc_id, "vertical_extent_flag", NC_BYTE, 1, dimid_scalar, 
				  "values", "0=greater than 5 km, 1=less than 5 km");

  vid_prof_cap = defineVariable(nc_id, "temperature_profile_capped", NC_BYTE, 1, dimid_scalar, 
				  "values", "0=not capped, 1=capped");

  vid_superad = defineVariable(nc_id, "profile_superadiabatic", NC_BYTE, 1, dimid_scalar, 
				 "values", "0=not superadiabatic, 1=marginal, 2=moderate, 3=severe");

  vid_inversion = defineVariable(nc_id, "profile_inversion", NC_BYTE, 1, dimid_scalar, 
				   "values", "0=no inversion, 1=depth<2.5km, 2=depth>2.5km");

  vid_tpw_cat = defineVariable(nc_id, "total_precipitable_water_category", NC_BYTE, 1, dimid_scalar, 
				 "values", "0=tpw < 0 mm, 1=between 0 and 15mm, 2=between 15 and 30mm, 3=between 30 and 45 mm, 4=between 45 and 60mm, 5=greater than 60mm");

  vid_moisture_cap = defineVariable(nc_id, "moisture_profile_cap_flag", NC_BYTE, 1, dimid_scalar, 
				      "values", "0=not capped, 1=capped");

  vid_surf_data_flag = defineVariable(nc_id, "surface_data_flag", NC_BYTE, 1, dimid_scalar, 
					"values", "0=not missing, 1=missing");

  vid_max_inv_depth = defineVariable(nc_id, "maximum_inversion_depth", NC_FLOAT, 1, dimid_scalar, 
				       "units", "km");

  vid_calc_tropo_pres = defineVariable(nc_id, "tropopause_pressure_(calculated)", NC_FLOAT, 1, dimid_scalar, 
				    "units", "hPa");

  vid_sbl_flag = defineVariable(nc_id, "moisture_surface_boundary_layer", NC_BYTE, 1, dimid_scalar, 
				  "values", "0=no anomalies, 1=anomaly detected");

  vid_hi_interp_moisture = defineVariable(nc_id, "highest_interpolated_moisture_pressure", NC_FLOAT, 1, 
					    dimid_scalar, "units", "hPa");

  vid_wvmr_anomaly = defineVariable(nc_id, "wvmr_anomaly", NC_BYTE, 1, dimid_scalar, 
				    "values", "0=no anomalies, 1=anomaly detected");

  vid_wvmr_event_1 = defineVariable(nc_id, "wvmr_event_1_detected", NC_BYTE, 1, dimid_scalar, 
				    "values", "0=no, 1=yes");

  vid_wvmr_extreme_1 = defineVariable(nc_id, "wvmr_extreme_event_1_detected", NC_BYTE, 1, dimid_scalar, 
				      "values", "0=no, 1=yes");

  vid_wvmr_event_2 = defineVariable(nc_id, "wvmr_event_2_detected", NC_BYTE, 1, dimid_scalar, 
				    "values", "0=no, 1=yes");

  vid_wvmr_neg_event = defineVariable(nc_id, "wvmr_negative_event_detected", NC_BYTE, 1, dimid_scalar, 
				    "values", "0=no, 1=yes");

  vid_wvmr_extreme_neg = defineVariable(nc_id, "wvmr_extreme_negative_event_detected", NC_BYTE, 1, 
                                        dimid_scalar, "values", "0=no, 1=yes");

  vid_utls_anomaly = defineVariable(nc_id, "utls_anomaly_detected", NC_BYTE, 1, dimid_scalar, 
				    "values", "0=no, 1=yes");

  vid_inversion_type = defineVariable(nc_id, "inversion_type", NC_BYTE, 1, dimid_scalar, 
				      "values", "0=no inversion, 1=surface inversion");

  vid_dwpt_depress = defineVariable(nc_id, "constant_dewpoint_depression", NC_BYTE, 1, dimid_scalar, 
				    "values", "0=none, 1=all within 5 deg, 2=all within 2.5 deg, 3=all within 1 deg");

  vid_duplicate = defineVariable(nc_id, "duplicate_sonde_flag", NC_BYTE, 1, dimid_scalar, 
				    "values", "0=no duplicate, 1=accepted duplicate, 2=rejected duplicate");

  vid_tier_2 = defineVariable(nc_id, "moisture_screening_tier_2_score", NC_BYTE, 1, dimid_scalar, 
			      NULL, NULL);

  vid_screen_adjust = defineVariable(nc_id, "moisture_screening_adjustment", NC_BYTE, 1, dimid_scalar, 
				     "values", "0=flag not set, 1=additional anomalies found");

  vid_sbl_top = defineVariable(nc_id, "top_of_surface_boundary_layer", NC_FLOAT, 1, dimid_scalar, 
			       "units", "hPa");

  vid_tropo_press = defineVariable(nc_id, "tropopause_pressure", NC_FLOAT, 1, dimid_scalar, 
				   "units", "hPa");

  vid_surf_press = defineVariable(nc_id, "surface_pressure", NC_FLOAT, 1, dimid_scalar, 
				  "units", "hPa");

  vid_surf_temp = defineVariable(nc_id, "surface_temperature", NC_FLOAT, 1, dimid_scalar, 
				 "units", "K");

  vid_surf_wvmr = defineVariable(nc_id, "surface_water_vapor_mixing_ratio", NC_FLOAT, 1, dimid_scalar, 
				 "units", "g/kg");

  vid_surf_dwpt = defineVariable(nc_id, "surface_dewpoint_temperature", NC_FLOAT, 1, dimid_scalar, 
				 "units", "K");

  vid_low_wvmr = defineVariable(nc_id, "lowest_level_water_vapor_mixing_ratio", NC_FLOAT, 1, dimid_scalar, 
				"units", "g/kg");

  vid_high_wvmr = defineVariable(nc_id, "highest_level_water_vapor_mixing_ratio", NC_FLOAT, 1, dimid_scalar, 
				 "units", "g/kg");

  vid_suan = defineVariable(nc_id, "suan_network_flag", NC_BYTE, 1, dimid_scalar, 
			    "values", "0=not in network, 1=in network");

  vid_gcos = defineVariable(nc_id, "gcos_network_flag", NC_BYTE, 1, dimid_scalar, 
			    "values", "0=not in network, 1=in network");

  vid_gruan = defineVariable(nc_id, "gruan_network_flag", NC_BYTE, 1, dimid_scalar, 
		 	     "values", "0=not in network, 1=in network");

  vid_tpw = defineVariable(nc_id, "total_precipitable_water", NC_FLOAT, 1, dimid_scalar, 
			   "units", "mm");

  vid_lpw = defineVariable(nc_id, "layer_precipitable_water", NC_FLOAT, 2, dimid_lpw, 
			   "units", "mm");

  vid_utls_pressure = defineVariable(nc_id, "pressure_above_utls_anomaly", NC_FLOAT, 1, dimid_scalar, 
			             "units", "hPa");

  vid_temp_extrap = defineVariable(nc_id, "temperature_extrapolation_pressure", NC_FLOAT, 1, dimid_scalar, 
			           "units", "hPa");

  vid_climate_limit = defineVariable(nc_id, "climatological_limits_flag", NC_BYTE, 1, dimid_scalar, 
			             "values", "0=passed, 1=failed");

  vid_dwpt_flag = defineVariable(nc_id, "dewpoint_temperature_flag", NC_BYTE, 1, dimid_scalar, 
			         "values", "0=passed, 1=failed");

  vid_vert_extent = defineVariable(nc_id, "uncapped_vertical_extent_flag", NC_BYTE, 1, dimid_scalar, 
			           "values", "0=passed, 1=failed");

  vid_temp_gap = defineVariable(nc_id, "temperature_data_gap_flag", NC_BYTE, 1, dimid_scalar, 
			        "values", "0=no gap, 1=gap exists");

  vid_moisture_gap = defineVariable(nc_id, "moisture_data_gap_flag", NC_BYTE, 1, dimid_scalar, 
			            "values", "0=no gap, 1=gap exists");

  vid_ncep_screen = defineVariable(nc_id, "ncep_screening_flag", NC_BYTE, 1, dimid_scalar, 
			           "values", "0=no screen, 1=at least one level screened");

  vid_tropo_deflect = defineVariable(nc_id, "tropopause_deflection_point_pressure", NC_FLOAT, 2, dimid_tropo, 
			             "units", "hPa");

  vid_tropo_rep_press = defineVariable(nc_id, "reported_tropopause_pressure", NC_FLOAT, 2, dimid_tropo, 
			               "units", "hPa");

  vid_tropo_calc_press = defineVariable(nc_id, "calculated_tropopause_pressure", NC_FLOAT, 2, dimid_tropo, 
			                "units", "hPa");

  vid_tropo_qc = defineVariable(nc_id, "tropopause_qc_flag", NC_BYTE, 1, dimid_scalar, 
			        "values", "0=count<=2, 1=count>2, 2=pressure>=500hPa");

  vid_surf_radcor_temp = defineVariable(nc_id, "surface_radcor_temperature", NC_FLOAT, 1, dimid_scalar, 
			        "units", "K");

  vid_surf_press_qc = defineVariable(nc_id, "surface_pressure_qc", NC_BYTE, 1, dimid_scalar, 
			             NULL, NULL);

  vid_surf_temp_qc = defineVariable(nc_id, "surface_temperature_qc", NC_BYTE, 1, dimid_scalar, 
			            NULL, NULL);

  vid_surf_humid_qc = defineVariable(nc_id, "surface_relative_humidity_qc", NC_BYTE, 1, dimid_scalar, 
			             NULL, NULL);

  vid_surf_wind_dir = defineVariable(nc_id, "surface_wind_direction", NC_FLOAT, 1, dimid_scalar, 
			             "units", "degrees");

  vid_surf_wind_speed = defineVariable(nc_id, "surface_wind_speed", NC_FLOAT, 1, dimid_scalar, 
				       "units", "knots");

  vid_low_cloud_amount = defineVariable(nc_id, "low_cloud_amount", NC_SHORT, 1, dimid_scalar, 
				        NULL, NULL);

  vid_low_cloud_type = defineVariable(nc_id, "low_cloud_type", NC_SHORT, 1, dimid_scalar, 
				      NULL, NULL);

  vid_low_cloud_fraction = defineVariable(nc_id, "low_cloud_fraction", NC_FLOAT, 1, dimid_scalar, 
				          "units", "pct");

  vid_radcor_code = defineVariable(nc_id, "radiation_correction_code", NC_SHORT, 1, dimid_scalar, 
				   NULL, NULL);

  vid_low_cloud_base = defineVariable(nc_id, "base_of_low_cloud_height", NC_SHORT, 1, dimid_scalar, 
				      "units", "m");

  vid_low_cloud_base_lut = defineVariable(nc_id, "base_of_low_cloud_height_(lut)", NC_SHORT, 1, dimid_scalar, 
				          NULL, NULL);

  vid_mid_cloud_type = defineVariable(nc_id, "mid_cloud_type", NC_SHORT, 1, dimid_scalar, 
				      NULL, NULL);

  vid_high_cloud_type = defineVariable(nc_id, "high_cloud_type", NC_SHORT, 1, dimid_scalar, 
				       NULL, NULL);

  vid_vtemp_flag = defineVariable(nc_id, "virtual_temperature_conversion_flag", NC_BYTE, 1, dimid_scalar, 
				  "values", "1=conversion performed");

  vid_text_flag = defineVariable(nc_id, "text_file_report_flag", NC_BYTE, 1, dimid_scalar, 
				 "values", "0=not special, 1=special");

  vid_solzen = defineVariable(nc_id, "launch_point_solar_zenith_angle", NC_FLOAT, 1, dimid_scalar, 
			      "units", "degrees");

  vid_daynight = defineVariable(nc_id, "day_night_flag", NC_BYTE, 1, dimid_scalar, 
				"values", "0=day, 1=dusk, 2=night");

  vid_gfs_lat = defineVariable(nc_id, "gfs_analysis_latitude", NC_FLOAT, 1, dimid_scalar, 
			       "units", "degrees_north");

  vid_gfs_lon = defineVariable(nc_id, "gfs_analysis_longitude", NC_FLOAT, 1, dimid_scalar, 
			       "units", "degrees_east");

  vid_gfs_date = defineVariable(nc_id, "gfs_analysis_date", NC_INT, 1, dimid_scalar, 
				"formats", "YYYYMMDD");

  vid_gfs_time = defineVariable(nc_id, "gfs_analysis_time", NC_BYTE, 1, dimid_scalar, 
				"formats", "HH");

  vid_gfs_distance = defineVariable(nc_id, "gfs_distance_from_sonde", NC_FLOAT, 1, dimid_scalar, 
				    "units", "km");

  vid_gfs_press = defineVariable(nc_id, "gfs_analysis_pressure", NC_FLOAT, 2, dimid_fcst, 
				 "units", "hPa");

  vid_gfs_temps = defineVariable(nc_id, "gfs_analysis_temperature", NC_FLOAT, 2, dimid_fcst, 
				 "units", "K");

  vid_gfs_surf_temp = defineVariable(nc_id, "gfs_analysis_surface_temperature", NC_FLOAT, 1, dimid_scalar, 
				     "units", "K");

  vid_gfs_tropo_temp = defineVariable(nc_id, "gfs_analysis_tropopause_temperature", NC_FLOAT, 1, dimid_scalar, 
				      "units", "K");

  vid_gfs_relhum = defineVariable(nc_id, "gfs_analysis_relative_humidity", NC_FLOAT, 2, dimid_fcst, 
				  "units", "pct");

  vid_gfs_u_wind = defineVariable(nc_id, "gfs_analysis_u_component_wind", NC_FLOAT, 2, dimid_fcst, 
				  "units", "m/s");

  vid_gfs_v_wind = defineVariable(nc_id, "gfs_analysis_v_component_wind", NC_FLOAT, 2, dimid_fcst, 
				  "units", "m/s");

  vid_gfs_tropo_u_wind = defineVariable(nc_id, "gfs_analysis_tropopause_u_component_wind", NC_FLOAT, 1, dimid_scalar, 
					"units", "m/s");

  vid_gfs_tropo_v_wind = defineVariable(nc_id, "gfs_analysis_tropopause_v_component_wind", NC_FLOAT, 1, dimid_scalar, 
					"units", "m/s");

  vid_gfs_height = defineVariable(nc_id, "gfs_analysis_geopotential_height", NC_FLOAT, 2, dimid_fcst, 
				  "units", "m");

  vid_gfs_surf_height = defineVariable(nc_id, "gfs_analysis_surface_geopotential_height", NC_FLOAT, 1, dimid_scalar, 
				       "units", "m");

  vid_gfs_tropo_height = defineVariable(nc_id, "gfs_analysis_tropopause_geopotential_height", NC_FLOAT, 1, 
                                        dimid_scalar, "units", "m");

  vid_gfs_surf_press = defineVariable(nc_id, "gfs_analysis_surface_pressure", NC_FLOAT, 1, dimid_scalar, 
				      "units", "hPa");

  vid_gfs_tropo_press = defineVariable(nc_id, "gfs_analysis_tropopause_pressure", NC_FLOAT, 1, dimid_scalar, 
				       "units", "hPa");

  vid_gfs_sea_press = defineVariable(nc_id, "gfs_analysis_sea_level_pressure", NC_FLOAT, 1, dimid_scalar, 
				     "units", "hPa");


  vid_temp_cap_press = defineVariable(nc_id, "temperature_cap_pressure", NC_FLOAT, 1, dimid_scalar, 
				      "units", "hPa");

  vid_moisture_cap_press = defineVariable(nc_id, "moisture_cap_pressure", NC_FLOAT, 1, dimid_scalar, 
					  "units", "hPa");

  vid_gfs_temp_cap_flag = defineVariable(nc_id, "gfs_temperature_cap_flag", NC_BYTE, 1, dimid_scalar, 
					 "values", "0=not capped, 1=capped");

  vid_gfs_temp_cap_press = defineVariable(nc_id, "gfs_temperature_cap_pressure", NC_FLOAT, 1, dimid_scalar, 
					  "units", "hPa");

  vid_gfs_moisture_cap_flag = defineVariable(nc_id, "gfs_moisture_cap_flag", NC_BYTE, 1, dimid_scalar, 
					     "values", "0=not capped, 1=capped");

  vid_gfs_moisture_cap_press = defineVariable(nc_id, "gfs_moisture_cap_pressure", NC_FLOAT, 1, dimid_scalar, 
					      "units", "hPa");

  vid_cfsr_temp_cap_flag = defineVariable(nc_id, "cfsr_temperature_cap_flag", NC_BYTE, 1, dimid_scalar, 
					  "values", "0=not capped, 1=capped");

  vid_cfsr_temp_cap_press = defineVariable(nc_id, "cfsr_temperature_cap_pressure", NC_FLOAT, 1, dimid_scalar, 
					   "units", "hPa");

  vid_cfsr_moisture_cap_flag = defineVariable(nc_id, "cfsr_moisture_cap_flag", NC_BYTE, 1, dimid_scalar, 
					      "values", "0=not capped, 1=capped");

  vid_cfsr_moisture_cap_press = defineVariable(nc_id, "cfsr_moisture_cap_pressure", NC_FLOAT, 1, dimid_scalar, 
					       "units", "hPa");

  vid_reanal_temp_cap_flag = defineVariable(nc_id, "cfsr_reanalysis_temperature_cap_flag", NC_BYTE, 1, dimid_scalar, 
					    "values", "0=not capped, 1=capped");

  vid_reanal_temp_cap_press = defineVariable(nc_id, "cfsr_reanalysis_temperature_cap_pressure", NC_FLOAT, 1, 
					     dimid_scalar, "units", "hPa");

  vid_reanal_moisture_cap_flag = defineVariable(nc_id, "cfsr_reanalysis_moisture_cap_flag", NC_BYTE, 1, dimid_scalar, 
						"values", "0=not capped, 1=capped");

  vid_reanal_moisture_cap_press = defineVariable(nc_id, "cfsr_reanalysis_moisture_cap_pressure", NC_FLOAT, 1, 
						 dimid_scalar, "units", "hPa");

  // Merged standard and significant level variables

  vid_std_sig_flags = defineVariable(nc_id, "standard_or_significant_level_flag", NC_BYTE, 2, dimid_levels, 
                                     "values", "0=Significant Level, 1=Standard Level");

  vid_press = defineVariable(nc_id, "pressure", NC_FLOAT, 2, dimid_levels, "units", "hPa");

  vid_temps = defineVariable(nc_id, "temperature", NC_FLOAT, 2, dimid_levels, "units", "K");

  vid_radcor_temps = defineVariable(nc_id, "radiation_corrected_temperature", NC_FLOAT, 2, dimid_levels, 
				    "units", "K");

  vid_cfsr_temps = defineVariable(nc_id, "cfsr_forecast_temperature", NC_FLOAT, 2, dimid_levels, 
				  "units", "K");

  vid_cfsr_reanal_temps = defineVariable(nc_id, "cfsr_reanalysis_temperature", NC_FLOAT, 2, dimid_levels, 
					 "units", "K");

  vid_wvmrs = defineVariable(nc_id, "water_vapor_mixing_ratio", NC_FLOAT, 2, dimid_levels, "units", "g/kg");

  vid_radcor_wvmrs = defineVariable(nc_id, "radiation_corrected_water_vapor_mixing_ratio", NC_FLOAT, 2, 
				    dimid_levels, "units", "g/kg");

  vid_cfsr_wvmrs = defineVariable(nc_id, "cfsr_forecast_water_vapor_mixing_ratio", NC_FLOAT, 2, dimid_levels, 
				  "units", "g/kg");

  vid_cfsr_reanal_wvmrs = defineVariable(nc_id, "cfsr_reanalysis_water_vapor_mixing_ratio", NC_FLOAT, 2, 
					 dimid_levels, "units", "g/kg");

  vid_back_temps = defineVariable(nc_id, "gfs_forecast_background_temperature", NC_FLOAT, 2, 
				  dimid_levels, "units", "K");

  vid_back_wvmrs = defineVariable(nc_id, "gfs_forecast_background_water_vapor_mixing_ratio", NC_FLOAT, 2, 
				  dimid_levels, "units", "g/kg");

  vid_drift_times = defineVariable(nc_id, "balloon_drift_time_delta", NC_FLOAT, 2, dimid_levels, 
				   "units", "hours");

  vid_drift_lats = defineVariable(nc_id, "balloon_drift_latitude", NC_FLOAT, 2, dimid_levels, 
				  "units", "degrees_north");

  vid_drift_lons = defineVariable(nc_id, "balloon_drift_longitude", NC_FLOAT, 2, dimid_levels, 
				  "units", "degrees_east");

  vid_heights = defineVariable(nc_id, "geopotential_height", NC_FLOAT, 2, dimid_levels, 
			       "units", "meters");

  vid_wind_dirs = defineVariable(nc_id, "wind_direction", NC_FLOAT, 2, dimid_levels, NULL, NULL);

  vid_wind_speeds = defineVariable(nc_id, "wind_speed", NC_FLOAT, 2, dimid_levels, "units", "knots");

  vid_press_qcs = defineVariable(nc_id, "pressure_qc_flag", NC_BYTE, 2, dimid_levels, NULL, NULL);

  vid_temp_qcs = defineVariable(nc_id, "temperature_qc_flag", NC_BYTE, 2, dimid_levels, NULL, NULL);

  vid_humid_qcs = defineVariable(nc_id, "humidity_qc_flag", NC_BYTE, 2, dimid_levels, NULL, NULL);

  vid_height_qcs = defineVariable(nc_id, "geopotential_height_qc_flag", NC_BYTE, 2, dimid_levels, 
				  NULL, NULL);

  vid_wind_qcs = defineVariable(nc_id, "wind_qc_flag", NC_BYTE, 2, dimid_levels, NULL, NULL);

  vid_missing_flags = defineVariable(nc_id, "level_missing_flag", NC_BYTE, 2, dimid_levels, NULL, NULL);


  }






//-------------------------------------------------------------------
int processSonde03(char *file_name, int day_to_process, int nc_id, int prev_written)
  {
  int     date, num_written;
  int     n, i, recnum, retval, level, level_inserted;
  int     num_records, header_length, record_length;
  size_t  index[1], num_vals[1], index_2D[2], num_vals_2D[2];

  float   missing_float, latitude, longitude, std_level_pressures[18], gfs_pressures[26], pressure;
  float   lpw[18], tropo[5], gfs[26];
  int     year, month, day, hour, minute, second;
  long    offset, yyyymmdd, hhmmss;

  char    sonde_id[9];
  char    *string[1];

  short   buffer[6500];

  char    merged_std_sig[118], merged_press_qcs[118], merged_temp_qcs[118];
  char    merged_humid_qcs[118], merged_height_qcs[118], merged_wind_qcs[118];
  char    merged_missing_flags[118];
  float   merged_pressures[118], merged_temps[118], merged_radcor_temps[118];
  float   merged_cfsr_temps[118], merged_reanal_temps[118], merged_wvmrs[118];
  float   merged_radcor_wvmrs[118], merged_cfsr_wvmrs[118], merged_reanal_wvmrs[118];
  float   merged_drift_times[118], merged_drift_lats[118], merged_drift_lons[118];
  float   merged_heights[118], merged_wind_dirs[118], merged_wind_speeds[118];
  float   merged_back_temps[118], merged_back_wvmrs[118];

  FILE    *in;

  struct merged_data *merged_levels;
  struct merged_data *new_level;
  struct merged_data *ptr;
  struct merged_data *prev_ptr;

  num_written = prev_written;

  std_level_pressures[0]  = 1000.0;
  std_level_pressures[1]  = 925.0;
  std_level_pressures[2]  = 850.0;
  std_level_pressures[3]  = 700.0;
  std_level_pressures[4]  = 500.0;
  std_level_pressures[5]  = 400.0;
  std_level_pressures[6]  = 300.0;
  std_level_pressures[7]  = 250.0;
  std_level_pressures[8]  = 200.0;
  std_level_pressures[9]  = 150.0;
  std_level_pressures[10] = 100.0;
  std_level_pressures[11] = 70.0;
  std_level_pressures[12] = 50.0;
  std_level_pressures[13] = 30.0;
  std_level_pressures[14] = 20.0;
  std_level_pressures[15] = 10.0;
  std_level_pressures[16] = 7.0;
  std_level_pressures[17] = 5.0;

  gfs_pressures[0]  = 10.0;
  gfs_pressures[1]  = 20.0;
  gfs_pressures[2]  = 30.0;
  gfs_pressures[3]  = 50.0;
  gfs_pressures[4]  = 70.0;
  gfs_pressures[5]  = 100.0;
  gfs_pressures[6]  = 150.0;
  gfs_pressures[7]  = 200.0;
  gfs_pressures[8]  = 250.0;
  gfs_pressures[9]  = 300.0;
  gfs_pressures[10] = 350.0;
  gfs_pressures[11] = 400.0;
  gfs_pressures[12] = 450.0;
  gfs_pressures[13] = 500.0;
  gfs_pressures[14] = 550.0;
  gfs_pressures[15] = 600.0;
  gfs_pressures[16] = 650.0;
  gfs_pressures[17] = 700.0;
  gfs_pressures[18] = 750.0;
  gfs_pressures[19] = 800.0;
  gfs_pressures[20] = 850.0;
  gfs_pressures[21] = 900.0;
  gfs_pressures[22] = 925.0;
  gfs_pressures[23] = 950.0;
  gfs_pressures[24] = 975.0;
  gfs_pressures[25] = 1000.0;

  missing_float = -32768.0;


  // Open the input file

  if ((in=fopen(file_name,"r")) == NULL)
    {
    printf("\n\nThe input file could not be opened.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  // Extract information from the header

  fseek(in, 8, SEEK_SET);
  fread(&header_length, 4, 1, in);

  fseek(in, 16, SEEK_SET);
  fread(&record_length, 4, 1, in);

  fseek(in, 24, SEEK_SET);
  fread(&num_records, 4, 1, in);


  // Loop through the data records in the file

  offset = (long)header_length;
  fseek(in, offset, SEEK_SET);

  for (recnum=0; recnum<num_records; recnum++)
    {
    fread(buffer, record_length, 1, in);

    date = (buffer[35]*10000) + (buffer[36]*100) + buffer[37];

    if (date == day_to_process)
      {
      num_written++;

      // Index and num_vals are used to save a single value within the
      // variable array

      index[0]    = num_written - 1;
      num_vals[0] = 1;

      index_2D[0] = num_written - 1;
      index_2D[1] = 0;
      num_vals_2D[0] = 1;
      num_vals_2D[1] = 118;

      // Unpack and save the Sonde ID

      sonde_id[0] = buffer[29] / 100;
      sonde_id[1] = buffer[29] % 100;
      sonde_id[2] = buffer[30] / 100;
      sonde_id[3] = buffer[30] % 100;
      sonde_id[4] = buffer[31] / 100;
      sonde_id[5] = buffer[31] % 100;
      sonde_id[6] = ' ';
      sonde_id[7] = ' ';
      sonde_id[8] = '\0';

      string[0] = sonde_id;

      retval = nc_put_vara_string(nc_id, vid_sonde_id, index, num_vals, (const char **)string);

      // Latitude and longitude

      latitude  = buffer[39] / 128.0;
      longitude = buffer[40] / 128.0;

      writeVariableFloat(nc_id, vid_lat, index, num_vals, latitude);
      writeVariableFloat(nc_id, vid_lon, index, num_vals, longitude);

      collocation_lats[num_written-1] = latitude;
      collocation_lons[num_written-1] = longitude;
      
      // Date and time

      year      = buffer[35];
      month     = buffer[36];
      day       = buffer[37];
      hour      = buffer[38] / 100;
      minute    = buffer[38] % 100;
      second    = 0;

      yyyymmdd = (year * 10000) + (month * 100) + day;
      hhmmss   = (hour * 10000) + (minute * 100) + second;

      writeVariableInteger(nc_id, vid_date, index, num_vals, yyyymmdd);
      writeVariableInteger(nc_id, vid_time, index, num_vals, hhmmss);

      collocation_dates[num_written-1] = yyyymmdd;
      collocation_times[num_written-1] = hhmmss;

      // Save the station elevation, report type, instrument type
      // and terrain

      if (buffer[2704] != -32768)
	writeVariableByte(nc_id, vid_synoptic_hour, index, num_vals, (char)buffer[2704]);
      else
	writeVariableByte(nc_id, vid_synoptic_hour, index, num_vals, (char)-128);

      writeVariableShort(nc_id, vid_elevation, index, num_vals, buffer[41]);
      writeVariableShort(nc_id, vid_report_type, index, num_vals, buffer[42]);
      writeVariableShort(nc_id, vid_instrument_type, index, num_vals, buffer[43]);

      if (buffer[44] != -32768)
	writeVariableByte(nc_id, vid_terrain, index, num_vals, (char)buffer[44]);
      else
	writeVariableByte(nc_id, vid_terrain, index, num_vals, (char)-128);

      // Lowest and highest pressures

      if (buffer[45] != -32768)
	writeVariableFloat(nc_id, vid_lo_interp_press, index, num_vals, (buffer[45]/10.0));
      else
	writeVariableFloat(nc_id, vid_lo_interp_press, index, num_vals, missing_float);

      if (buffer[46] != -32768)
	writeVariableFloat(nc_id, vid_hi_interp_press, index, num_vals, (buffer[46]/10.0));
      else
	writeVariableFloat(nc_id, vid_hi_interp_press, index, num_vals, missing_float);

      if (buffer[11] != -32768)
	writeVariableByte(nc_id, vid_radcor, index, num_vals, (char)buffer[11]);
      else
	writeVariableByte(nc_id, vid_radcor, index, num_vals, (char)-128);

      if (buffer[47] != -32768)
	writeVariableByte(nc_id, vid_qc_flag, index, num_vals, (char)buffer[47]);
      else
	writeVariableByte(nc_id, vid_qc_flag, index, num_vals, (char)-128);

      writeVariableByte(nc_id, vid_nprovs_qc, index, num_vals, (char)0);

      if (buffer[48] != -32768)
	writeVariableByte(nc_id, vid_vert_ext, index, num_vals, (char)buffer[48]);
      else
	writeVariableByte(nc_id, vid_vert_ext, index, num_vals, (char)-128);

      if (buffer[49] != -32768)
	writeVariableByte(nc_id, vid_prof_cap, index, num_vals, (char)buffer[49]);
      else
	writeVariableByte(nc_id, vid_prof_cap, index, num_vals, (char)-128);

      if ((buffer[50] >= 0) && (buffer[50] <= 3))
	writeVariableByte(nc_id, vid_superad, index, num_vals, (char)buffer[50]);
      else
	writeVariableByte(nc_id, vid_superad, index, num_vals, (char)-128);

      if (buffer[51] != -32768)
	writeVariableByte(nc_id, vid_inversion, index, num_vals, (char)buffer[51]);
      else
	writeVariableByte(nc_id, vid_inversion, index, num_vals, (char)-128);

      if (buffer[52] != -32768)
	writeVariableByte(nc_id, vid_tpw_cat, index, num_vals, (char)buffer[52]);
      else
	writeVariableByte(nc_id, vid_tpw_cat, index, num_vals, (char)-128);

      if (buffer[53] != -32768)
	writeVariableByte(nc_id, vid_moisture_cap, index, num_vals, (char)buffer[53]);
      else
	writeVariableByte(nc_id, vid_moisture_cap, index, num_vals, (char)-128);

      if (buffer[54] != -32768)
	writeVariableByte(nc_id, vid_surf_data_flag, index, num_vals, (char)buffer[54]);
      else
	writeVariableByte(nc_id, vid_surf_data_flag, index, num_vals, (char)-128);

      if (buffer[55] != -32768)
	writeVariableFloat(nc_id, vid_max_inv_depth, index, num_vals, (buffer[55]/10.0));
      else
	writeVariableFloat(nc_id, vid_max_inv_depth, index, num_vals, missing_float);

      writeVariableFloat(nc_id, vid_calc_tropo_pres, index, num_vals, (buffer[56]/1.0));

      if (buffer[57] != -32768)
	writeVariableByte(nc_id, vid_sbl_flag, index, num_vals, (char)buffer[57]);
      else
	writeVariableByte(nc_id, vid_sbl_flag, index, num_vals, (char)-128);

      if (buffer[58] != -32768)
	writeVariableFloat(nc_id, vid_hi_interp_moisture, index, num_vals, (buffer[58]/10.0));
      else
	writeVariableFloat(nc_id, vid_hi_interp_moisture, index, num_vals, missing_float);

      if (buffer[59] != -32768)
	writeVariableByte(nc_id, vid_wvmr_anomaly, index, num_vals, (char)buffer[59]);
      else
	writeVariableByte(nc_id, vid_wvmr_anomaly, index, num_vals, (char)-128);

      if (buffer[60] != -32768)
	writeVariableByte(nc_id, vid_wvmr_event_1, index, num_vals, (char)buffer[60]);
      else
	writeVariableByte(nc_id, vid_wvmr_event_1, index, num_vals, (char)-128);

      if (buffer[61] != -32768)
	writeVariableByte(nc_id, vid_wvmr_extreme_1, index, num_vals, (char)buffer[61]);
      else
	writeVariableByte(nc_id, vid_wvmr_extreme_1, index, num_vals, (char)-128);

      if (buffer[62] != -32768)
	writeVariableByte(nc_id, vid_wvmr_event_2, index, num_vals, (char)buffer[62]);
      else
	writeVariableByte(nc_id, vid_wvmr_event_2, index, num_vals, (char)-128);

      if (buffer[63] != -32768)
	writeVariableByte(nc_id, vid_wvmr_neg_event, index, num_vals, (char)buffer[63]);
      else
	writeVariableByte(nc_id, vid_wvmr_neg_event, index, num_vals, (char)-128);

      if (buffer[64] != -32768)
	writeVariableByte(nc_id, vid_wvmr_extreme_neg, index, num_vals, (char)buffer[64]);
      else
	writeVariableByte(nc_id, vid_wvmr_extreme_neg, index, num_vals, (char)-128);

      if (buffer[65] != -32768)
	writeVariableByte(nc_id, vid_utls_anomaly, index, num_vals, (char)buffer[65]);
      else
	writeVariableByte(nc_id, vid_utls_anomaly, index, num_vals, (char)-128);

      if (buffer[66] != -32768)
	writeVariableByte(nc_id, vid_inversion_type, index, num_vals, (char)buffer[66]);
      else
	writeVariableByte(nc_id, vid_inversion_type, index, num_vals, (char)-128);

      if (buffer[67] != -32768)
	writeVariableByte(nc_id, vid_dwpt_depress, index, num_vals, (char)buffer[67]);
      else
	writeVariableByte(nc_id, vid_dwpt_depress, index, num_vals, (char)-128);

      if (buffer[68] != -32768)
	writeVariableByte(nc_id, vid_duplicate, index, num_vals, (char)buffer[68]);
      else
	writeVariableByte(nc_id, vid_duplicate, index, num_vals, (char)-128);

      if (buffer[69] != -32768)
	writeVariableByte(nc_id, vid_tier_2, index, num_vals, (char)buffer[69]);
      else
	writeVariableByte(nc_id, vid_tier_2, index, num_vals, (char)-128);

      if (buffer[70] != -32768)
	writeVariableByte(nc_id, vid_screen_adjust, index, num_vals, (char)buffer[70]);
      else
	writeVariableByte(nc_id, vid_screen_adjust, index, num_vals, (char)-128);

      writeVariableFloat(nc_id, vid_sbl_top, index, num_vals, (buffer[151]/1.0));
      writeVariableFloat(nc_id, vid_tropo_press, index, num_vals, (buffer[152]/1.0));
      writeVariableFloat(nc_id, vid_surf_press, index, num_vals, (buffer[153]/1.0));

      if (buffer[154] != -32768)
	writeVariableFloat(nc_id, vid_surf_temp, index, num_vals, (buffer[154]/64.0));
      else
	writeVariableFloat(nc_id, vid_surf_temp, index, num_vals, missing_float);

      if (buffer[155] != -32768)
	writeVariableFloat(nc_id, vid_surf_wvmr, index, num_vals, (float)(exp(buffer[155]/1024.0)));
      else
	writeVariableFloat(nc_id, vid_surf_wvmr, index, num_vals, missing_float);

      if (buffer[156] != -32768)
	writeVariableFloat(nc_id, vid_surf_dwpt, index, num_vals, (buffer[156]/64.0));
      else
	writeVariableFloat(nc_id, vid_surf_dwpt, index, num_vals, missing_float);

      if (buffer[157] != -32768)
	writeVariableFloat(nc_id, vid_low_wvmr, index, num_vals, (float)(exp(buffer[157]/1024.0)));
      else
	writeVariableFloat(nc_id, vid_low_wvmr, index, num_vals, missing_float);

      if (buffer[158] != -32768)
	writeVariableFloat(nc_id, vid_high_wvmr, index, num_vals, (float)(exp(buffer[158]/1024.0)));
      else
	writeVariableFloat(nc_id, vid_high_wvmr, index, num_vals, missing_float);

      if (buffer[159] != -32768)
	writeVariableByte(nc_id, vid_suan, index, num_vals, (char)buffer[159]);
      else
	writeVariableByte(nc_id, vid_suan, index, num_vals, (char)-128);

      if (buffer[160] != -32768)
	writeVariableByte(nc_id, vid_gcos, index, num_vals, (char)buffer[160]);
      else
	writeVariableByte(nc_id, vid_gcos, index, num_vals, (char)-128);

      if (buffer[161] != -32768)
	writeVariableByte(nc_id, vid_gruan, index, num_vals, (char)buffer[161]);
      else
	writeVariableByte(nc_id, vid_gruan, index, num_vals, (char)-128);

      if (buffer[162] != -32768)
	writeVariableFloat(nc_id, vid_tpw, index, num_vals, (buffer[162]/100.0));
      else
	writeVariableFloat(nc_id, vid_tpw, index, num_vals, missing_float);

      // Layer precipitable water

      for (n=0; n<18; n++)
	{
        if (buffer[163+n] != -32768)
	  lpw[n] = buffer[163+n] / 100.0;
	else
	  lpw[n] = -32768;
	}

      index_2D[0] = num_written - 1;
      index_2D[1] = 0;
      num_vals_2D[0] = 1;
      num_vals_2D[1] = 18;

      writeArrayFloat(nc_id, vid_lpw, index_2D, num_vals_2D, lpw);

      // More values

      if (buffer[181] != -32768)
	writeVariableFloat(nc_id, vid_utls_pressure, index, num_vals, (buffer[181]/10.0));
      else
	writeVariableFloat(nc_id, vid_utls_pressure, index, num_vals, missing_float);

      if (buffer[182] != -32768)
	writeVariableFloat(nc_id, vid_temp_extrap, index, num_vals, (buffer[182]/10.0));
      else
	writeVariableFloat(nc_id, vid_temp_extrap, index, num_vals, missing_float);

      if (buffer[183] != -32768)
	writeVariableByte(nc_id, vid_climate_limit, index, num_vals, (char)buffer[183]);
      else
	writeVariableByte(nc_id, vid_climate_limit, index, num_vals, (char)-128);

      if (buffer[184] != -32768)
	writeVariableByte(nc_id, vid_dwpt_flag, index, num_vals, (char)buffer[184]);
      else
	writeVariableByte(nc_id, vid_dwpt_flag, index, num_vals, (char)-128);

      if (buffer[185] != -32768)
	writeVariableByte(nc_id, vid_vert_extent, index, num_vals, (char)buffer[185]);
      else
	writeVariableByte(nc_id, vid_vert_extent, index, num_vals, (char)-128);

      if (buffer[186] != -32768)
	writeVariableByte(nc_id, vid_temp_gap, index, num_vals, (char)buffer[186]);
      else
	writeVariableByte(nc_id, vid_temp_gap, index, num_vals, (char)-128);

      if (buffer[187] != -32768)
	writeVariableByte(nc_id, vid_moisture_gap, index, num_vals, (char)buffer[187]);
      else
	writeVariableByte(nc_id, vid_moisture_gap, index, num_vals, (char)-128);

      if (buffer[188] != -32768)
	writeVariableByte(nc_id, vid_ncep_screen, index, num_vals, (char)buffer[188]);
      else
	writeVariableByte(nc_id, vid_ncep_screen, index, num_vals, (char)-128);

      // Tropopause data

      index_2D[0] = num_written - 1;
      index_2D[1] = 0;
      num_vals_2D[0] = 1;
      num_vals_2D[1] = 5;
      
      for (n=0; n<5; n++)
	{
        if (buffer[2699+n] != -32768)
	  tropo[n] = buffer[2699+n] / 10.0;
	else
	  tropo[n] = -32768;
	}

      writeArrayFloat(nc_id, vid_tropo_deflect, index_2D, num_vals_2D, tropo);

      for (n=0; n<5; n++)
	{
        if (buffer[2705+n] != -32768)
	  tropo[n] = buffer[2705+n] / 10.0;
	else
	  tropo[n] = -32768;
	}

      writeArrayFloat(nc_id, vid_tropo_rep_press, index_2D, num_vals_2D, tropo);

      for (n=0; n<5; n++)
	{
	if (buffer[2729+n] != -32768)
	  tropo[n] = buffer[2729+n] / 10.0;
	else
	  tropo[n] = -32768;
	}

      writeArrayFloat(nc_id, vid_tropo_calc_press, index_2D, num_vals_2D, tropo);

      if (buffer[2734] != -32768)
	writeVariableByte(nc_id, vid_tropo_qc, index, num_vals, (char)buffer[2734]);
      else
	writeVariableByte(nc_id, vid_tropo_qc, index, num_vals, (char)-128);

      // Surface values

      if (buffer[2737] != -32768)
	writeVariableFloat(nc_id, vid_surf_radcor_temp, index, num_vals, ((buffer[2737]/10.0)+273.13));
      else
	writeVariableFloat(nc_id, vid_surf_radcor_temp, index, num_vals, missing_float);

      if (buffer[2739] != -32768)
	writeVariableByte(nc_id, vid_surf_press_qc, index, num_vals, (char)buffer[2739]);
      else
	writeVariableByte(nc_id, vid_surf_press_qc, index, num_vals, (char)-128);

      if (buffer[2740] != -32768)
	writeVariableByte(nc_id, vid_surf_temp_qc, index, num_vals, (char)buffer[2740]);
      else
	writeVariableByte(nc_id, vid_surf_temp_qc, index, num_vals, (char)-128);

      if (buffer[2741] != -32768)
	writeVariableByte(nc_id, vid_surf_humid_qc, index, num_vals, (char)buffer[2741]);
      else
	writeVariableByte(nc_id, vid_surf_humid_qc, index, num_vals, (char)-128);

      writeVariableFloat(nc_id, vid_surf_wind_dir, index, num_vals, (buffer[2742]/1.0));
      writeVariableFloat(nc_id, vid_surf_wind_speed, index, num_vals, (buffer[2743]/1.0));

      // Cloud data

      writeVariableShort(nc_id, vid_low_cloud_amount, index, num_vals, (short)buffer[2744]);
      writeVariableShort(nc_id, vid_low_cloud_type, index, num_vals, (short)buffer[2745]);

      if (buffer[2746] != -32768)
	writeVariableFloat(nc_id, vid_low_cloud_fraction, index, num_vals, (buffer[2746]/1000.0));
      else
	writeVariableFloat(nc_id, vid_low_cloud_fraction, index, num_vals, missing_float);

      writeVariableShort(nc_id, vid_radcor_code, index, num_vals, buffer[2747]);
      writeVariableShort(nc_id, vid_low_cloud_base, index, num_vals, buffer[2748]);
      writeVariableShort(nc_id, vid_low_cloud_base_lut, index, num_vals, buffer[2749]);
      writeVariableShort(nc_id, vid_mid_cloud_type, index, num_vals, buffer[2750]);
      writeVariableShort(nc_id, vid_high_cloud_type, index, num_vals, buffer[2751]);

      if (buffer[2752] != -32768)
	writeVariableByte(nc_id, vid_vtemp_flag, index, num_vals, (char)buffer[2752]);
      else
	writeVariableByte(nc_id, vid_vtemp_flag, index, num_vals, (char)-128);

      if (buffer[2753] != -32768)
	writeVariableByte(nc_id, vid_text_flag, index, num_vals, (char)buffer[2753]);
      else
	writeVariableByte(nc_id, vid_text_flag, index, num_vals, (char)-128);

      if (buffer[2754] != -32768)
	writeVariableFloat(nc_id, vid_solzen, index, num_vals, (buffer[2754]/128.0));
      else
	writeVariableFloat(nc_id, vid_solzen, index, num_vals, missing_float);

      if (buffer[2755] != -32768)
	writeVariableByte(nc_id, vid_daynight, index, num_vals, (char)buffer[2755]);
      else
	writeVariableByte(nc_id, vid_daynight, index, num_vals, (char)-128);

      // GFS data

      if (buffer[2757] != -32768)
	writeVariableFloat(nc_id, vid_gfs_lat, index, num_vals, (buffer[2757]/128.0));
      else
	writeVariableFloat(nc_id, vid_gfs_lat, index, num_vals, missing_float);

      if (buffer[2758] != -32768)
	writeVariableFloat(nc_id, vid_gfs_lon, index, num_vals, (buffer[2758]/128.0));
      else
	writeVariableFloat(nc_id, vid_gfs_lon, index, num_vals, missing_float);

      if ((buffer[2759] != -32768) && (buffer[2760] != -32768))
	writeVariableInteger(nc_id, vid_gfs_date, index, num_vals, ((buffer[2759]*10000)+buffer[2760]));
      else
	writeVariableInteger(nc_id, vid_gfs_date, index, num_vals, -32768);

      if (buffer[2761] != -32768)
	writeVariableByte(nc_id, vid_gfs_time, index, num_vals, (char)buffer[2761]);
      else
	writeVariableByte(nc_id, vid_gfs_time, index, num_vals, (char)-128);

      if (buffer[2762] != -32768)
	writeVariableFloat(nc_id, vid_gfs_distance, index, num_vals, (buffer[2762]/10.0));
      else
	writeVariableFloat(nc_id, vid_gfs_distance, index, num_vals, missing_float);

      if (buffer[2790] != -32768)
	writeVariableFloat(nc_id, vid_gfs_surf_temp, index, num_vals, ((buffer[2790]/10.0)+273.13));
      else
	writeVariableFloat(nc_id, vid_gfs_surf_temp, index, num_vals, missing_float);

      if (buffer[2791] != -32768)
	writeVariableFloat(nc_id, vid_gfs_tropo_temp, index, num_vals, ((buffer[2791]/10.0)+273.13));
      else
	writeVariableFloat(nc_id, vid_gfs_tropo_temp, index, num_vals, missing_float);

      if (buffer[2840] != -32768)
	writeVariableFloat(nc_id, vid_gfs_tropo_u_wind, index, num_vals, (buffer[2840]/100.0));
      else
	writeVariableFloat(nc_id, vid_gfs_tropo_u_wind, index, num_vals, missing_float);

      if (buffer[2867] != -32768)
	writeVariableFloat(nc_id, vid_gfs_tropo_v_wind, index, num_vals, (buffer[2867]/100.0));
      else
	writeVariableFloat(nc_id, vid_gfs_tropo_v_wind, index, num_vals, missing_float);

      writeVariableFloat(nc_id, vid_gfs_surf_height, index, num_vals, (buffer[2894]/1.0));
      writeVariableFloat(nc_id, vid_gfs_tropo_height, index, num_vals, (buffer[2895]/1.0));

      if (buffer[2896] != -32768)
	writeVariableFloat(nc_id, vid_gfs_surf_press, index, num_vals, (buffer[2896]/10.0));
      else
	writeVariableFloat(nc_id, vid_gfs_surf_press, index, num_vals, missing_float);

      if (buffer[2897] != -32768)
	writeVariableFloat(nc_id, vid_gfs_tropo_press, index, num_vals, (buffer[2897]/10.0));
      else
	writeVariableFloat(nc_id, vid_gfs_tropo_press, index, num_vals, missing_float);

      if (buffer[2898] != -32768)
	writeVariableFloat(nc_id, vid_gfs_sea_press, index, num_vals, (buffer[2898]/10.0));
      else
	writeVariableFloat(nc_id, vid_gfs_sea_press, index, num_vals, missing_float);

      index_2D[0] = num_written - 1;
      index_2D[1] = 0;
      num_vals_2D[0] = 1;
      num_vals_2D[1] = 26;

      writeArrayFloat(nc_id, vid_gfs_press, index_2D, num_vals_2D, gfs_pressures);

      for (n=0; n<26; n++)
	{
        if (buffer[2763+n] != -32768)
	  gfs[25-n] = (buffer[2763+n] / 10.0) + 273.13;
	else
	  gfs[25-n] = -32768.0;
	}

      writeArrayFloat(nc_id, vid_gfs_temps, index_2D, num_vals_2D, gfs);

      for (n=0; n<26; n++)
	{
        if (buffer[2792+n] != -32768)
	  gfs[25-n] = buffer[2792+n] / 100.0;
	else
	  gfs[25-n] = -32768.0;
	}

      writeArrayFloat(nc_id, vid_gfs_relhum, index_2D, num_vals_2D, gfs);

      for (n=0; n<26; n++)
	{
        if (buffer[2815+n] != -32768)
	  gfs[25-n] = buffer[2815+n] / 100.0;
	else
	  gfs[25-n] = -32768;
	}

      if (buffer[2839] != -32768)
	gfs[0] = buffer[2839] / 100.0;
      else
	gfs[0] = -32768.0;

      writeArrayFloat(nc_id, vid_gfs_u_wind, index_2D, num_vals_2D, gfs);

      for (n=0; n<26; n++)
	{
        if (buffer[2841+n] != -32768)
	  gfs[25-n] = buffer[2841+n] / 100.0;
	else
	  gfs[25-n] = -32768.0;
	}

      writeArrayFloat(nc_id, vid_gfs_v_wind, index_2D, num_vals_2D, gfs);

      for (n=0; n<26; n++)
	{
        if (buffer[2868+n] != -32768)
	  gfs[25-n] = buffer[2868+n] / 1.0;
	else
	  gfs[25-n] = -32768.0;
	}

      writeArrayFloat(nc_id, vid_gfs_height, index_2D, num_vals_2D, gfs);

      // Profile cap values

      if (buffer[2899] != -32768)
	writeVariableFloat(nc_id, vid_temp_cap_press, index, num_vals, (buffer[2899]/10.0));
      else
	writeVariableFloat(nc_id, vid_temp_cap_press, index, num_vals, missing_float);

      if (buffer[2900] != -32768)
	writeVariableFloat(nc_id, vid_moisture_cap_press, index, num_vals, (buffer[2900]/10.0));
      else
	writeVariableFloat(nc_id, vid_moisture_cap_press, index, num_vals, missing_float);

      if (buffer[2901] != -32768)
	writeVariableByte(nc_id, vid_gfs_temp_cap_flag, index, num_vals, (char)buffer[2901]);
      else
	writeVariableByte(nc_id, vid_gfs_temp_cap_flag, index, num_vals, (char)-128);

      if (buffer[2902] != -32768)
	writeVariableFloat(nc_id, vid_gfs_temp_cap_press, index, num_vals, (buffer[2902]/10.0));
      else
	writeVariableFloat(nc_id, vid_gfs_temp_cap_press, index, num_vals, missing_float);

      if (buffer[2903] != -32768)
	writeVariableByte(nc_id, vid_gfs_moisture_cap_flag, index, num_vals, (char)buffer[2903]);
      else
	writeVariableByte(nc_id, vid_gfs_moisture_cap_flag, index, num_vals, (char)-128);

      if (buffer[2904] != -32768)
	writeVariableFloat(nc_id, vid_gfs_moisture_cap_press, index, num_vals, (buffer[2904]/10.0));
      else
	writeVariableFloat(nc_id, vid_gfs_moisture_cap_press, index, num_vals, missing_float);

      if (buffer[2905] != -32768)
	writeVariableByte(nc_id, vid_cfsr_temp_cap_flag, index, num_vals, (char)buffer[2905]);
      else
	writeVariableByte(nc_id, vid_cfsr_temp_cap_flag, index, num_vals, (char)-128);

      if (buffer[2906] != -32768)
	writeVariableFloat(nc_id, vid_cfsr_temp_cap_press, index, num_vals, (buffer[2906]/10.0));
      else
	writeVariableFloat(nc_id, vid_cfsr_temp_cap_press, index, num_vals, missing_float);

      if (buffer[2907] != -32768)
	writeVariableByte(nc_id, vid_cfsr_moisture_cap_flag, index, num_vals, (char)buffer[2907]);
      else
	writeVariableByte(nc_id, vid_cfsr_moisture_cap_flag, index, num_vals, (char)-128);

      if (buffer[2908] != -32768)
	writeVariableFloat(nc_id, vid_cfsr_moisture_cap_press, index, num_vals, (buffer[2908]/10.0));
      else
	writeVariableFloat(nc_id, vid_cfsr_moisture_cap_press, index, num_vals, missing_float);

      if (buffer[2909] != -32768)
	writeVariableByte(nc_id, vid_reanal_temp_cap_flag, index, num_vals, (char)buffer[2909]);
      else
	writeVariableByte(nc_id, vid_reanal_temp_cap_flag, index, num_vals, (char)-128);

      if (buffer[2910] != -32768)
	writeVariableFloat(nc_id, vid_reanal_temp_cap_press, index, num_vals, (buffer[2910]/10.0));
      else
	writeVariableFloat(nc_id, vid_reanal_temp_cap_press, index, num_vals, missing_float);

      if (buffer[2911] != -32768)
	writeVariableByte(nc_id, vid_reanal_moisture_cap_flag, index, num_vals, (char)buffer[2911]);
      else
	writeVariableByte(nc_id, vid_reanal_moisture_cap_flag, index, num_vals, (char)-128);

      if (buffer[2912] != -32768)
	writeVariableFloat(nc_id, vid_reanal_moisture_cap_press, index, num_vals, (buffer[2912]/10.0));
      else
	writeVariableFloat(nc_id, vid_reanal_moisture_cap_press, index, num_vals, missing_float);


      // Merge the standard and significant level data

      merged_levels = NULL;

      for (n=0; n<18; n++)
	{

        // Process this level unless all of the data are missing

	if ((buffer[194+n] != -32768) || (buffer[212+n] != -32768) || (buffer[230+n] != -32768) ||
	    (buffer[248+n] != -32768) || (buffer[266+n] != -32768) || (buffer[284+n] != -32768) ||
	    (buffer[302+n] != -32768) || (buffer[320+n] != -32768) || (buffer[338+n] != -32768) ||
	    (buffer[356+n] != -32768) || (buffer[374+n] != -32768) || (buffer[392+n] != -32768))
	  {
	  new_level = (struct merged_data*)malloc(sizeof (struct merged_data));

	  new_level->std_or_sig = STANDARD;
	  new_level->pressure = std_level_pressures[n];

	  if (buffer[194+n] == -32768)
	    new_level->temp = -32768.0;
	  else
	    new_level->temp = (buffer[194+n] / 10.0) + 273.13;

	  if (buffer[212+n] == -32768)
	    new_level->radcor_temp = -32768.0;
	  else
	    new_level->radcor_temp = (buffer[212+n] / 10.0) + 273.13;

	  if (buffer[230+n] == -32768)
	    new_level->cfsr_temp = -32768.0;
	  else
	    new_level->cfsr_temp = (buffer[230+n] / 10.0) + 273.13;

	  if (buffer[500+n] == -32768)
	    new_level->cfsr_reanal_temp = -32768.0;
	  else
	    new_level->cfsr_reanal_temp = (buffer[500+n] / 10.0) + 273.13;

	  if (buffer[248+n] == -32768)
	    new_level->wvmr = -32768.0;
	  else
	    new_level->wvmr = dwptToWvmr(new_level->pressure, ((buffer[248+n]/10.0)+273.15));

	  if (buffer[266+n] == -32768)
	    new_level->radcor_wvmr = -32768.0;
	  else
	    new_level->radcor_wvmr = dwptToWvmr(new_level->pressure, ((buffer[266+n]/10.0)+273.15));

	  if (buffer[284+n] == -32768)
	    new_level->cfsr_wvmr = -32768.0;
	  else
	    new_level->cfsr_wvmr = buffer[284+n] / 1000.0;

	  if (buffer[518+n] == -32768)
	    new_level->cfsr_reanal_wvmr = -32768.0;
	  else
	    new_level->cfsr_reanal_wvmr = buffer[518+n] / 1000.0;

	  if (buffer[302+n] == -32768)
	    new_level->drift_time = -32768.0;
	  else
	    new_level->drift_time = buffer[302+n] / 100.0;

	  if (buffer[320+n] == -32768)
	    new_level->drift_latitude = -32768.0;
	  else
	    new_level->drift_latitude = buffer[320+n] / 128.0;

	  if (buffer[338+n] == -32768)
	    new_level->drift_longitude = -32768.0;
	  else
	    new_level->drift_longitude = buffer[338+n] / 128.0;
	  
	  if (buffer[356+n] == -32768)
	    new_level->geo_height = -32768.0;
	  else if (n < 11)
	    new_level->geo_height = buffer[356+n] / 1.0;
	  else
	    new_level->geo_height = buffer[356+n] / 10.0;

	  if (buffer[374+n] == -32768)
	    new_level->wind_dir = -32768.0;
	  else
	    new_level->wind_dir = buffer[374+n] / 1.0;

	  if (buffer[392+n] == -32768)
	    new_level->wind_speed = -32768.0;
	  else
	    new_level->wind_speed = buffer[392+n] / 1.0;

	  new_level->pressure_qc  = -32768;
	  new_level->temp_qc      = buffer[410+n];
	  new_level->humidity_qc  = buffer[428+n];
	  new_level->height_qc    = buffer[446+n];
	  new_level->wind_qc      = buffer[464+n];
	  new_level->missing_flag = buffer[482+n];
	  
	  if (buffer[2917+n] == -32768)
	    new_level->background_temp = -32768.0;
	  else
	    new_level->background_temp = (buffer[2917+n] / 10.0) + 273.12;

	  if (buffer[2935+n] == -32768)
	    new_level->background_wvmr = -32768.0;
	  else
	    new_level->background_wvmr = buffer[2935+n] / 1000.0;

	  new_level->next = NULL;


	  // Insert the new level into the sorted merged array

	  if (merged_levels == NULL)
	    {
	    merged_levels = new_level;
	    }
	  else
	    {
	    ptr = merged_levels;
	    prev_ptr = NULL;

	    level_inserted = FALSE;

	    while ((ptr != NULL) && (level_inserted == FALSE))
	      {
	      if (new_level->pressure < ptr->pressure)
		{
	        if (prev_ptr == NULL)
		  {
		  new_level->next = merged_levels;
		  merged_levels = new_level;
		  }
		else
		  {
		    new_level = prev_ptr->next;
		    prev_ptr->next = new_level;
		  }

		level_inserted = TRUE;
		}
	      else if (ptr->next == NULL)
		{
		  ptr->next = new_level;
		  level_inserted = TRUE;
		}
	      else
		{
		  prev_ptr = ptr;
		  ptr = ptr->next;
		}
	      }
	    }  // else (merged_levels != NULL...
	  }  // if (buffer[356+n...
	}  // for (n=0...


      // Significant level data

      for (n=0; n<buffer[536]; n++)
	{

	// Process this level unless all of the data are missing

        if ((buffer[537+n] != -32768) ||
	    (buffer[637+n] != -32768) || (buffer[737+n] != -32768) || (buffer[837+n] != -32768) ||
	    (buffer[937+n] != -32768) || (buffer[1037+n] != -32768) || (buffer[1137+n] != -32768) ||
	    (buffer[1237+n] != -32768) || (buffer[1337+n] != -32768) || (buffer[1437+n] != -32768) ||
	    (buffer[1537+n] != -32768) || (buffer[1637+n] != -32768) || (buffer[1737+n] != -32768))
	  {
	  new_level = (struct merged_data*)malloc(sizeof (struct merged_data));

	  new_level->std_or_sig = SIGNIFICANT;

	  if (buffer[537+n] != -32768)
	    pressure = buffer[537+n] / 10.0;
	  else
	    pressure = -32768.0;

	  new_level->pressure = pressure;

	  if (buffer[637+n] == -32768)
	    new_level->temp = -32768.0;
	  else
	    new_level->temp = (buffer[637+n] / 10.0) + 273.13;

	  if (buffer[737+n] == -32768)
	    new_level->radcor_temp = -32768.0;
	  else
	    new_level->radcor_temp = (buffer[737+n] / 10.0) + 273.13;

	  if (buffer[837+n] == -32768)
	    new_level->cfsr_temp = -32768.0;
	  else
	    new_level->cfsr_temp = (buffer[837+n] / 10.0) + 273.13;

	  if (buffer[2437+n] == -32768)
	    new_level->cfsr_reanal_temp = -32768.0;
	  else
	    new_level->cfsr_reanal_temp = (buffer[2437+n] / 10.0) + 273.13;

	  if (buffer[937+n] == -32768)
	    new_level->wvmr = -32768.0;
	  else
	    new_level->wvmr = dwptToWvmr(new_level->pressure, ((buffer[937+n]/10.0)+273.15));

	  if (buffer[1037+n] == -32768)
	    new_level->radcor_wvmr = -32768.0;
	  else
	    new_level->radcor_wvmr = dwptToWvmr(new_level->pressure, ((buffer[1037+n]/10.0)+273.15));

	  if (buffer[1137+n] == -32768)
	    new_level->cfsr_wvmr = -32768.0;
	  else
	    new_level->cfsr_wvmr = buffer[1137+n] / 1000.0;

	  if (buffer[2537+n] == -32768)
	    new_level->cfsr_reanal_wvmr = -32768.0;
	  else
	    new_level->cfsr_reanal_wvmr = buffer[2537+n] / 1000.0;

	  if (buffer[1237+n] == -32768)
	    new_level->drift_time = -32768.0;
	  else
	    new_level->drift_time = buffer[1237+n] / 100.0;

	  if (buffer[1337+n] == -32768)
	    new_level->drift_latitude = -32768.0;
	  else
	    new_level->drift_latitude = buffer[1337+n] / 128.0;

	  if (buffer[1437+n] == -32768)
	    new_level->drift_longitude = -32768.0;
	  else
	    new_level->drift_longitude = buffer[1437+n] / 128.0;

	  if (buffer[1537+n] == -32768)
	    new_level->geo_height = -32768.0;
	  else if (pressure <= 80.0)
	    new_level->geo_height = buffer[1537+n] / 1.0;
	  else
	    new_level->geo_height = buffer[1537+n] / 10.0;

	  if (buffer[1637+n] == -32768)
	    new_level->wind_dir = -32768.0;
	  else
	    new_level->wind_dir = buffer[1637+n] / 1.0;

	  if (buffer[1737+n] == -32768)
	    new_level->wind_speed = -32768.0;
	  else
	    new_level->wind_speed = buffer[1737+n] / 1.0;

	  new_level->pressure_qc  = buffer[1837+n];
	  new_level->temp_qc      = buffer[1937+n];
	  new_level->humidity_qc  = buffer[2037+n];
	  new_level->height_qc    = buffer[2137+n];
	  new_level->wind_qc      = buffer[2237+n];
	  new_level->missing_flag = buffer[2337+n];

	  if (buffer[2953+n] == -32768)
	    new_level->background_temp = -32768.0;
	  else
	    new_level->background_temp = (buffer[2953+n] / 10.0) + 273.12;

	  if (buffer[3053+n] == -32768)
	    new_level->background_wvmr = -32768.0;
	  else
	    new_level->background_wvmr = buffer[3053+n] / 1000.0;

	  new_level->next = NULL;


	  // Insert the new level into the sorted merged array
	  
	  if (merged_levels == NULL)
	    {
	    merged_levels = new_level;
	    }
	  else
	    {
	    ptr = merged_levels;
	    prev_ptr = NULL;

	    level_inserted = FALSE;

	    while ((ptr != NULL) && (level_inserted == FALSE))
	      {
	      if (new_level->pressure < ptr->pressure)
	        {
		if (prev_ptr == NULL)
		  {
		  new_level->next = merged_levels;
		  merged_levels = new_level;
		  }
		else
		  {
		  new_level->next = prev_ptr->next;
		  prev_ptr->next = new_level;
		  }

		level_inserted = TRUE;
		}
	      else if (ptr->next == NULL)
		{
	        ptr->next = new_level;
		level_inserted = TRUE;
		}
	      else
		{
	        prev_ptr = ptr;
		ptr = ptr->next;
		}
	      }
	    }  // else (merged_levels != NULL...
	  }  // if (buffer[1537+n...
	}  // for (n=0...


      // Put the merged data into arrays in preparation of writing the data
      // to the netCDF file

      for (n=0; n<118; n++)
	{
        merged_std_sig[n]       = (char)-128;
	merged_press_qcs[n]     = (char)-128;
	merged_temp_qcs[n]      = (char)-128;
	merged_humid_qcs[n]     = (char)-128;
	merged_height_qcs[n]    = (char)-128;
	merged_wind_qcs[n]      = (char)-128;
	merged_missing_flags[n] = (char)-128;
	merged_pressures[n]     = -32768.0;
	merged_temps[n]         = -32768.0;
	merged_radcor_temps[n]  = -32768.0;
	merged_cfsr_temps[n]    = -32768.0;
	merged_reanal_temps[n]  = -32768.0;
	merged_wvmrs[n]         = -32768.0;
	merged_radcor_wvmrs[n]  = -32768.0;
	merged_cfsr_wvmrs[n]    = -32768.0;
	merged_reanal_wvmrs[n]  = -32768.0;
	merged_drift_times[n]   = -32768.0;
	merged_drift_lats[n]    = -32768.0;
	merged_drift_lons[n]    = -32768.0;
	merged_heights[n]       = -32768.0;
	merged_wind_dirs[n]     = -32768.0;
	merged_wind_speeds[n]   = -32768.0;
        }
      
      level = -1;

      ptr = merged_levels;

      while (ptr != NULL)
	{
        level++;

	if (ptr->std_or_sig != -32768)
	  merged_std_sig[level] = (char)ptr->std_or_sig;

	if (ptr->pressure_qc != -32768)
	  merged_press_qcs[level] = (char)ptr->pressure_qc;

	if (ptr->temp_qc != -32768)
	  merged_temp_qcs[level] = (char)ptr->temp_qc;

	if (ptr->humidity_qc != -32768)
	  merged_humid_qcs[level] = (char)ptr->humidity_qc;

	if (ptr->height_qc != -32768)
	  merged_height_qcs[level] = (char)ptr->height_qc;

	if (ptr->wind_qc != -32768)
	  merged_wind_qcs[level] = (char)ptr->wind_qc;

	if (ptr->missing_flag != -32768)
	  merged_missing_flags[level] = (char)ptr->missing_flag;

	merged_pressures[level]     = ptr->pressure;
	merged_temps[level]         = ptr->temp;
	merged_radcor_temps[level]  = ptr->radcor_temp;
	merged_cfsr_temps[level]    = ptr->cfsr_temp;
	merged_reanal_temps[level]  = ptr->cfsr_reanal_temp;
	merged_wvmrs[level]         = ptr->wvmr;
	merged_radcor_wvmrs[level]  = ptr->radcor_wvmr;
	merged_cfsr_wvmrs[level]    = ptr->cfsr_wvmr;
	merged_reanal_wvmrs[level]  = ptr->cfsr_reanal_wvmr;
	merged_drift_times[level]   = ptr->drift_time;
	merged_drift_lats[level]    = ptr->drift_latitude;
	merged_drift_lons[level]    = ptr->drift_longitude;
	merged_heights[level]       = ptr->geo_height;
	merged_wind_dirs[level]     = ptr->wind_dir;
	merged_wind_speeds[level]   = ptr->wind_speed;
	merged_back_temps[level]    = ptr->background_temp;
	merged_back_wvmrs[level]    = ptr->background_wvmr;

	prev_ptr = ptr;

	ptr = ptr->next;

	free(prev_ptr);
	}

      // Write the arrays to the netCDF file

      index_2D[0] = num_written - 1;
      index_2D[1] = 0;
      num_vals_2D[0] = 1;
      num_vals_2D[1] = 118;

      writeArrayByte(nc_id, vid_std_sig_flags, index_2D, num_vals_2D, merged_std_sig);
      writeArrayByte(nc_id, vid_press_qcs, index_2D, num_vals_2D, merged_press_qcs);
      writeArrayByte(nc_id, vid_temp_qcs, index_2D, num_vals_2D, merged_temp_qcs);
      writeArrayByte(nc_id, vid_humid_qcs, index_2D, num_vals_2D, merged_humid_qcs);
      writeArrayByte(nc_id, vid_height_qcs, index_2D, num_vals_2D, merged_height_qcs);
      writeArrayByte(nc_id, vid_wind_qcs, index_2D, num_vals_2D, merged_wind_qcs);
      writeArrayByte(nc_id, vid_missing_flags, index_2D, num_vals_2D, merged_missing_flags);
      writeArrayFloat(nc_id, vid_press, index_2D, num_vals_2D, merged_pressures);
      writeArrayFloat(nc_id, vid_temps, index_2D, num_vals_2D, merged_temps);
      writeArrayFloat(nc_id, vid_radcor_temps, index_2D, num_vals_2D, merged_radcor_temps);
      writeArrayFloat(nc_id, vid_cfsr_temps, index_2D, num_vals_2D, merged_cfsr_temps);
      writeArrayFloat(nc_id, vid_cfsr_reanal_temps, index_2D, num_vals_2D, merged_reanal_temps);
      writeArrayFloat(nc_id, vid_back_temps, index_2D, num_vals_2D, merged_back_temps);
      writeArrayFloat(nc_id, vid_back_wvmrs, index_2D, num_vals_2D, merged_back_wvmrs);
      writeArrayFloat(nc_id, vid_wvmrs, index_2D, num_vals_2D, merged_wvmrs);
      writeArrayFloat(nc_id, vid_radcor_wvmrs, index_2D, num_vals_2D, merged_radcor_wvmrs);
      writeArrayFloat(nc_id, vid_cfsr_wvmrs, index_2D, num_vals_2D, merged_cfsr_wvmrs);
      writeArrayFloat(nc_id, vid_cfsr_reanal_wvmrs, index_2D, num_vals_2D, merged_reanal_wvmrs);
      writeArrayFloat(nc_id, vid_drift_times, index_2D, num_vals_2D, merged_drift_times);
      writeArrayFloat(nc_id, vid_drift_lats, index_2D, num_vals_2D, merged_drift_lats);
      writeArrayFloat(nc_id, vid_drift_lons, index_2D, num_vals_2D, merged_drift_lons);
      writeArrayFloat(nc_id, vid_heights, index_2D, num_vals_2D, merged_heights);
      writeArrayFloat(nc_id, vid_wind_dirs, index_2D, num_vals_2D, merged_wind_dirs);
      writeArrayFloat(nc_id, vid_wind_speeds, index_2D, num_vals_2D, merged_wind_speeds);


      }  // if (date == day_to_process...
    }  // for (recnum=0...

  fclose(in);


  printf("Num Written:  %d\n", num_written);
  
  return num_written;
  }



//--------------------------------------------------------------------------------
void createCollocationInfo(int day_to_process, int num_collocations)
  {
  int     retval, ci_id, scalar_dim;
  int     vid_coll_lat, vid_coll_lon, vid_coll_date, vid_coll_time;
  char    info_file_name[25];
  size_t  index_1D[1], num_vals_1D[1];


  sprintf(info_file_name, "out.dir/%d/Collocation_Info.nc", day_to_process);

  // Create the output netCDF file

  retval = nc_create(info_file_name, NC_NETCDF4, &ci_id);

  if (retval == NC_NOERR)
    {

    // Add the attributes

    writeAttributeShort(ci_id, "Baseline_Data_Type", 0);
    writeAttributeText(ci_id, "Baseline_Platform", "Radiosonde");
    writeAttributeShort(ci_id, "Baseline_Platform_ID", 0);

    // Define the dimension

    retval = nc_def_dim(ci_id, "Num_Collocations", num_collocations, &scalar_dim);
    dimid_scalar[0] = scalar_dim;

    // Define the variables

    vid_coll_lat = defineVariable(ci_id, "latitude", NC_FLOAT, 1, dimid_scalar, "units", "degrees_north");
    vid_coll_lon = defineVariable(ci_id, "longitude", NC_FLOAT, 1, dimid_scalar, "units", "degrees_east");
    vid_coll_date = defineVariable(ci_id, "date", NC_INT, 1, dimid_scalar, "format", "yyyymmdd");
    vid_coll_time = defineVariable(ci_id, "time", NC_INT, 1, dimid_scalar, "format", "hhmmss");

    // Write the variable data

    index_1D[0] = 0;
    num_vals_1D[0] = num_collocations;

    writeArrayFloat(ci_id, vid_coll_lat, index_1D, num_vals_1D, collocation_lats);
    writeArrayFloat(ci_id, vid_coll_lon, index_1D, num_vals_1D, collocation_lons);
    writeArrayInteger(ci_id, vid_coll_date, index_1D, num_vals_1D, collocation_dates);
    writeArrayInteger(ci_id, vid_coll_time, index_1D, num_vals_1D, collocation_times);
    
    // Close the output file

    nc_close(ci_id);
    }  // if (retval=nc_create...
  }

// end of file
