#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/*
   Name- calculateFineLayerMeans.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  9/09/2014   Programmer- Mike Pettey (IMSG)

   Function- This subroutine converts a profile from defined at
             pressure levels to one defined at pressure layers.
*/


float* calculateRelativeHumidities(float *pressures, float *temperatures, int num_temp_levels, 
                                   float *eff_pressures, float *water_vapors, int num_wvap_levels,
                                   float surface_pressure)
  {
  int    n, level, index_of_eff_level_above_surface;
  float  press_above, press_below, temp_above, temp_below;
  float  eff_temp, vp1, vp2;
  float *relhums, *eff_temperatures;

  // Initialize the array that will contain the calculated relative humidities

  relhums = (float*)malloc(sizeof(float) * num_wvap_levels);

  for (level=0; level<num_wvap_levels; level++)
    relhums[level] = -32768.0;


  // Find the index of the effective pressure above the surface

  index_of_eff_level_above_surface = -1;

  for (level=0; level<num_wvap_levels; level++)
    {
    if ((eff_pressures[level] != -32768.0) && (eff_pressures[level] < surface_pressure))
      index_of_eff_level_above_surface = level;
    }


  // If a level above the surface was not found, then skip the rest of the processing
  // and return an array of missing values

  if (index_of_eff_level_above_surface != -1)
    {

    // Compute the relative humidities at the effective pressures

    for (level=0; level<num_wvap_levels; level++)
      {

      // Find the temperature and pressure above the effective pressure

      press_above = -32768.0;
      temp_above  = -32768.0;

      for (n=0; n<num_temp_levels; n++)
	{
	if ((pressures[n] != -32768.0) && (temperatures[n] != -32768.0))
	  {
	  if (pressures[n] < eff_pressures[level])
	    {
	    press_above = pressures[n];
	    temp_above  = temperatures[n];
	    }
	  }
	}

      // Make sure that the upper level is above the surface

      if (press_above > surface_pressure)
	{
	press_above = -32768.0;
	temp_above  = -32768.0;
	}


      // Find the temperature and pressure below the effective pressure

      press_below = -32768.0;
      temp_below  = -32768.0;

      for (n=(num_temp_levels-1); n>=0; n--)
	{
	if ((pressures[n] != -32768.0) && (temperatures[n] != -32768.0))
	  {
	  if (pressures[n] > eff_pressures[level])
	    {
	    press_below = pressures[n];
	    temp_below  = temperatures[n];
	    }
	  }
	}

      // Make sure that the lower level is above the surface

      if (press_below > surface_pressure)
	{
	press_below = -32768.0;
	temp_below  = -32768.0;
	}


      // Compute the temperature at the effective level

      if ((press_above == -32768.0) || (temp_above == -32768.0) || 
	  (press_below == -32768.0) || (temp_below == -32768.0))
	{
	eff_temp = -32768.0;
	}
      else
	{
	eff_temp = temp_above + (temp_below-temp_above) / 
	                        log(press_below/press_above) * 
	                        log(eff_pressures[level] / press_above);
	}

      // Compute the relative humidity

      if ((eff_temp != -32768.0) && (water_vapors[level] != -32768.0))
	{
	eff_temp = eff_temp - 273.15;

	vp1 = eff_pressures[level] * water_vapors[level] / (622.0 + water_vapors[level]);

	vp2 = 6.1078 * exp(eff_temp / (eff_temp + 238.3) * 17.2694);

	relhums[level] = vp1 / vp2 * 100.0;
	}  // if (eff_temp != -32768.0...
      }  // for (level=0...
    }  // if (index_of_eff_level_above_surface...


  // Remove relative humidity values that are below the surface

  for (level=0; level<num_wvap_levels; level++)
    {
    if (eff_pressures[level] > surface_pressure)
      relhums[level] = -32768.0;
    }

  return relhums;
  }

// end of file
