/*
   Name- getNUCAPSLayerData.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  9/09/2014   Programmer- Mike Pettey (IMSG)

   Function- This program reads values from an AIRS file, calculates
             a value and returns it.
*/


#include <stdio.h>
#include <math.h>

#include "ods7_capture.h"



void convertFromLevelsToLayers(int num_layers, int num_levels, float *level_pressures,
                               float *level_data, float *layer_data);



void getNUCAPSLayerData(struct parameter_info *parm_list, short *buffer, 
		      float *retrieved_temp_layers, float *guess_temp_layers, float *mw_temp_layers,
		      float *retrieved_wvap_layers, float *guess_wvap_layers, float *mw_wvap_layers)
  {
  int   i, n, level;
  float temp_pressures[100], wvap_pressures[100], data[100];
  int   do_retrieved_temp_layers, do_guess_temp_layers, do_mw_temp_layers;
  int   do_retrieved_wvap_layers, do_guess_wvap_layers, do_mw_wvap_layers;

  struct parameter_info *ptr;


  // Determine if the layer data will be processed at some point.

  do_retrieved_temp_layers = FALSE;
  do_guess_temp_layers     = FALSE;
  do_mw_temp_layers        = FALSE;
  do_retrieved_wvap_layers = FALSE;
  do_guess_wvap_layers     = FALSE;
  do_mw_wvap_layers        = FALSE;

  ptr = parm_list;

  while (ptr != NULL)
    {
      {
      if (ptr->parameter_id == -1500)
        do_retrieved_temp_layers = TRUE;
      else if (ptr->parameter_id == -1501)
	      do_guess_temp_layers = TRUE;
      else if (ptr->parameter_id == -1502)
	      do_mw_temp_layers = TRUE;
      else if (ptr->parameter_id == -3500)
        do_retrieved_wvap_layers = TRUE;
      else if (ptr->parameter_id == -3501)
	      do_guess_wvap_layers = TRUE;
      else if (ptr->parameter_id == -3502)
	      do_mw_wvap_layers = TRUE;
      }

    ptr = ptr->next;
    }

  // Unpack the level pressures

  for (n=0; n<100; n++)
    {
    temp_pressures[n] = -32768.0;
    wvap_pressures[n] = -32768.0;

    if (buffer[n+450] != -32768)
      {
      if (n <= 19)
	temp_pressures[n] = buffer[n+450] / 1000.0;
      else
	temp_pressures[n] = buffer[n+450] / 10.0;
      }

    if (buffer[n+550] != -32768)
      {
      if (n <= 19)
	wvap_pressures[n] = (float)buffer[n+550] / 1000.0;
      else
	wvap_pressures[n] = (float)buffer[n+550] / 10.0;
      }
    }


  // If there is at least one retrieved temp layer, create a layer profile

  if (do_retrieved_temp_layers == TRUE)
    {

    // Unpack the temperatures

    for (n=0; n<100; n++)
      {
      if (buffer[n+650] != -32768)
	      data[n] = buffer[n+650] / 64.0;
      else
	      data[n] = -32768.0;
      }

    // Do the conversion

    convertFromLevelsToLayers(30, 100, temp_pressures, data, retrieved_temp_layers);
    }


  // If there is at least one guess temp layer, create a layer profile

  if (do_guess_temp_layers == TRUE)
    {

    // Unpack the temperatures

    for (n=0; n<100; n++)
      {
      if (buffer[n+850] != -32768)
	data[n] = buffer[n+850] / 64.0;
      else
	data[n] = -32768.0;
      }

    // Do the conversion

    convertFromLevelsToLayers(30, 100, temp_pressures, data, guess_temp_layers);
    }


  // If there is at least one microwave temp layer, create a layer profile

  if (do_mw_temp_layers == TRUE)
    {

    // Unpack the temperatures

    for (n=0; n<100; n++)
      {
      if (buffer[n+1515] != -32768)
	data[n] = buffer[n+750] / 64.0;
      else
	data[n] = -32768.0;
      }

    // Do the conversion

    convertFromLevelsToLayers(30, 100, temp_pressures, data, mw_temp_layers);
    }


  // If there is at least one retrieved wvap layer, create a layer profile

  if (do_retrieved_wvap_layers == TRUE)
    {

    // Unpack the water vapors

    for (i=0; i<100; i++)
      {
      if (buffer[i+950] != -32768)
	data[i] = (float)(exp(buffer[i+950]/1024.0));
      else
	data[i] = -32768.0;

      if (data[i] < 0.001)
	data[i] = -32768.0;
      }

    // Do the conversion

    convertFromLevelsToLayers(20, 100, wvap_pressures, data, retrieved_wvap_layers);
    }


  // If there is at least one guess wvap layer, create a layer profile

  if (do_guess_wvap_layers == TRUE)
    {

    // Unpack the water vapors

    for (i=0; i<100; i++)
      {
      if (buffer[i+1150] != -32768)
	data[i] = (float)(exp(buffer[i+1150]/1024.0));
      else
	data[i] = -32768.0;

      if (data[i] < 0.001)
	data[i] = -32768.0;
      }

    // Do the conversion

    convertFromLevelsToLayers(20, 100, wvap_pressures, data, guess_wvap_layers);
    }


  // If there is at least one mw wvap layer, create a layer profile

  if (do_mw_wvap_layers == TRUE)
    {

    // Unpack the water vapors

    for (i=0; i<100; i++)
      {
      if (buffer[i+1050] != -32768)
	data[i] = (float)(exp(buffer[i+1050]/1024.0));
      else
	data[i] = -32768.0;

      if (data[i] < 0.001)
	data[i] = -32768.0;
      }

    // Do the conversion

    convertFromLevelsToLayers(20, 100, wvap_pressures, data, mw_wvap_layers);
    }
  }

// end of file
