/*
   Name- getSpecialNucapsValue.c

   Language- C     Type- MAIN

   Version- 1.0    Date- 11/01/2013   Programmer- Mike Pettey (IMSG)

   Function- This program reads values from an NUCAPS file, calculates
             a value and returns it.
*/


#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "ods_capture.h"
#include "stability.h"


#define  RECORD_LENGTH_IDDF       13000
#define  NUM_HEADER_VALUES_IDDF   RECORD_LENGTH_IDDF/4
#define  NUM_DATA_VALUES_IDDF     RECORD_LENGTH_IDDF/2


#define  RELHUM_LIMIT   100.0



float* calculateRelativeHumidities(float *pressures, float *temps, int num_temp_levels, 
				   float *eff_pressures, float *wvaps, int num_wvap_levels, 
				   float surface_pressure);


double getSpecialNUCAPSValue(long parm_number, short buffer[NUM_DATA_VALUES_IDDF],
			   float *retrieved_layer_temps, float *guess_layer_temps,
			   float *mw_layer_temps, float *retrieved_layer_wvaps,
			   float *guess_layer_wvaps, float *mw_layer_wvaps,
			   struct stability_data *stabilities)
  {
  double value = -32768.0;
  short  i2temp;
  int    i, n, parm, layer, ipressure, pass_both, pass_ir, pass_mw, pass;
  int    ret_supersat, guess_supersat;
  float  fval, pressure, max_pressure, fvalue, lat, surface_pressure, max_relhum;
  float  pressures[100], eff_pressures[100], bottom_pressures[100], temps[100], wvaps[100];
  float  guess_temps[100], guess_wvaps[100];
  float  *relhums, *guess_relhums;

  // Latitude

  if (parm_number == -20000)
    {
    value = buffer[3];
    }

  // Longitude

  else if (parm_number == -20001)
    {
    value = buffer[4];
    }

  // Date

  if (parm_number == -20002)
    {
    value = (buffer[5] * 10000) + buffer[6];
    }

  // Time

  else if (parm_number == -20003)
    {
    value = (buffer[7] * 10000) + buffer[8];
    }

  // Orbital node

  else if (parm_number == -20006)
    {
    if (buffer[11] == 0)
      value = 1;
    else
      value = 2;
    }


  // QC flag

  else if (parm_number == -20007)
    {
    pass_both = FALSE;
    pass_mw = FALSE;

    if (buffer[32] == 0)
      {
      pass_both = TRUE;
      }
    else if ((buffer[32] & 8) == 0)
      {
      pass_mw = TRUE;
      }

    if ((pass_both == TRUE) || (pass_mw == TRUE))
      value = 0;
    else
      value = 1;
    }

  // Minute of day

  else if (parm_number == -20008)
    {
    fval = (buffer[7] * 60.0) + (buffer[8] / 100) + ((buffer[8] % 100) / 60.0);
    value = fval * 10;
    }
  
  // TPW
  
  else if (parm_number == -10)
    {
    value = -32768;
    }






  // Supersaturation combination flag
  
  else if (parm_number == -15)
    {
    // Unpack the temperatures and pressures

    for (n=0; n<100; n++)
      {
      pressures[n]   = -32768.0;
      temps[n]       = -32768.0;
      guess_temps[n] = -32768.0;

      if (buffer[n+450] != -32768)
	{
        if (n <= 19)
	  pressures[n] = buffer[n+450] / 1000.0;
	else
	  pressures[n] = buffer[n+450] / 10.0;
	}

      if (buffer[n+650] != -32768)
	temps[n] = buffer[n+650] / 64.0;

      if (buffer[n+850] != -32768)
	guess_temps[n] = buffer[n+850] / 64.0;
      }  // for (n=0...

    // Unpack the moistures and pressures

    for (i=0; i<100; i++)
      {
      eff_pressures[i] = -32768.0;
      wvaps[i]     = -32768.0;

      if (buffer[i+550] != -32768)
	{
        if (i <= 19)
	  eff_pressures[i] = (float)buffer[i+550] / 1000.0;
	else
	  eff_pressures[i] = (float)buffer[i+550] / 10.0;
	}

      if (buffer[i+950] != -32768)
	wvaps[i] = (float)(exp(buffer[i+950]/1024.0));

      if (wvaps[i] < 0.001)
	wvaps[i] = -32768.0;

      if (buffer[i+1150] != -32768)
	guess_wvaps[i] = (float)(exp(buffer[i+1150]/1024.0));

      if (wvaps[i] < 0.001)
	guess_wvaps[i] = -32768.0;
      }  // for (i=0...

    // Unpack the surface pressure

    if (buffer[27] != -32768)
      surface_pressure = buffer[27] / 10.0;

    // Calculate the relative humidities

    relhums = calculateRelativeHumidities(pressures, temps, 100, 
					  eff_pressures, wvaps, 100, surface_pressure);

    guess_relhums = calculateRelativeHumidities(pressures, guess_temps, 100, 
					  eff_pressures, guess_wvaps, 100, surface_pressure);

    // Remove values that are above the upper limit

    for (n=0; n<100; n++)
      {
      if (eff_pressures[n] < 400.0)
	{
	relhums[n] = -32768.0;
	guess_relhums[n] = -32768.0;
	}
      }

    // If the relative humidity at any level is greater than 100, then
    // set the supersaturation flag to true

    ret_supersat   = FALSE;
    guess_supersat = FALSE;

    for (n=0; n<100; n++)
      {
      if ((relhums[n] != -32768.0) && (relhums[n] > RELHUM_LIMIT))
	ret_supersat = TRUE;;

      if ((guess_relhums[n] != -32768.0) && (guess_relhums[n] > RELHUM_LIMIT))
	guess_supersat = TRUE;;
      }

    // Set the flag based on the combination

    if ((ret_supersat == FALSE) && (guess_supersat == FALSE))
      value = 0.0;
    else if ((ret_supersat == TRUE) && (guess_supersat == FALSE))
      value = 1.0;
    else if ((ret_supersat == FALSE) && (guess_supersat == TRUE))
      value = 2.0;
    else if ((ret_supersat == TRUE) && (guess_supersat == TRUE))
      value = 3.0;
    else
      value = -32768.0;

    free(relhums);
    free(guess_relhums);
    }  // if (parm_number == -15  supersat...


  // Retrieval supersaturation
  
  else if (parm_number == -16)
    {

    // Unpack the temperatures and pressures

    for (n=0; n<100; n++)
      {
      pressures[n] = -32768.0;
      temps[n] = -32768.0;

      if (buffer[n+450] != -32768)
	{
        if (n <= 19)
	  pressures[n] = buffer[n+450] / 1000.0;
	else
	  pressures[n] = buffer[n+450] / 10.0;
	}

      if (buffer[n+650] != -32768)
	temps[n] = buffer[n+650] / 64.0;
      }  // for (n=0...

    // Unpack the moistures and pressures

    for (i=0; i<100; i++)
      {
      eff_pressures[i] = -32768.0;
      wvaps[i]     = -32768.0;

      if (buffer[i+550] != -32768)
	{
        if (i <= 19)
	  eff_pressures[i] = (float)buffer[i+550] / 1000.0;
	else
	  eff_pressures[i] = (float)buffer[i+550] / 10.0;
	}

      if (buffer[i+950] != -32768)
	wvaps[i] = (float)(exp(buffer[i+950]/1024.0));

      if (wvaps[i] < 0.001)
	wvaps[i] = -32768.0;
      }  // for (i=0...

    // Unpack the surface pressure

    if (buffer[27] != -32768)
      surface_pressure = buffer[27] / 10.0;

    // Calculate the relative humidities

    relhums = calculateRelativeHumidities(pressures, temps, 100, 
					  eff_pressures, wvaps, 100, surface_pressure);

    // Remove values that are above the upper limit

    for (n=0; n<100; n++)
      {
      if (eff_pressures[n] < 400.0)
	relhums[n] = -32768.0;
      }

    // If the relative humidity at any level is greater than 100, then
    // set the supersaturation flag to true

    value = 0;

    for (n=0; n<100; n++)
      {
      if (relhums[n] != -32768.0)
        {
	if (relhums[n] > RELHUM_LIMIT)
	  value = 1;
	}
      }

    free(relhums);
    }  // if (parm_number == -16  supersat...


  // First guess supersaturation
  
  else if (parm_number == -17)
    {

    // Unpack the temperatures and pressures

    for (n=0; n<100; n++)
      {
      pressures[n] = -32768.0;
      temps[n] = -32768.0;

      if (buffer[n+450] != -32768)
	{
        if (n <= 19)
	  pressures[n] = buffer[n+450] / 1000.0;
	else
	  pressures[n] = buffer[n+450] / 10.0;
	}

      if (buffer[n+850] != -32768)
	temps[n] = buffer[n+850] / 64.0;
      }  // for (n=0...

    // Unpack the moistures and pressures

    for (i=0; i<100; i++)
      {
      eff_pressures[i] = -32768.0;
      wvaps[i]     = -32768.0;

      if (buffer[i+550] != -32768)
	{
        if (i <= 19)
	  eff_pressures[i] = (float)buffer[i+550] / 1000.0;
	else
	  eff_pressures[i] = (float)buffer[i+550] / 10.0;
	}

      if (buffer[i+1150] != -32768)
	wvaps[i] = (float)(exp(buffer[i+1150]/1024.0));

      if (wvaps[i] < 0.001)
	wvaps[i] = -32768.0;
      }  // for (i=0...

    // Unpack the surface pressure

    if (buffer[27] != -32768)
      surface_pressure = buffer[27] / 10.0;

    // Calculate the relative humidities

    relhums = calculateRelativeHumidities(pressures, temps, 100, 
					  eff_pressures, wvaps, 100, surface_pressure);

    // Remove values that are above the upper limit

    for (n=0; n<100; n++)
      {
      if (eff_pressures[n] < 400.0)
	relhums[n] = -32768.0;
      }

    // If the relative humidity at any level is greater than 100, then
    // set the supersaturation flag to true

    value = 0;

    for (n=0; n<100; n++)
      {
      if (relhums[n] != -32768.0)
        {
	if (relhums[n] > RELHUM_LIMIT)
	  value = 1;
	}
      }

    free(relhums);
    }  // if (parm_number == -17  supersat...


  // Maximum relative humidity
  
  else if (parm_number == -18)
    {

    // Unpack the temperatures and pressures

    for (n=0; n<100; n++)
      {
      pressures[n] = -32768.0;
      temps[n] = -32768.0;

      if (buffer[n+450] != -32768)
	{
        if (n <= 19)
	  pressures[n] = buffer[n+450] / 1000.0;
	else
	  pressures[n] = buffer[n+450] / 10.0;
	}

      if (buffer[n+650] != -32768)
	temps[n] = buffer[n+650] / 64.0;
      }  // for (n=0...

    // Unpack the moistures and pressures

    for (i=0; i<100; i++)
      {
      eff_pressures[i] = -32768.0;
      wvaps[i]     = -32768.0;

      if (buffer[i+550] != -32768)
	{
        if (i <= 19)
	  eff_pressures[i] = (float)buffer[i+550] / 1000.0;
	else
	  eff_pressures[i] = (float)buffer[i+550] / 10.0;
	}

      if (buffer[i+950] != -32768)
	wvaps[i] = (float)(exp(buffer[i+950]/1024.0));

      if (wvaps[i] < 0.001)
	wvaps[i] = -32768.0;
      }  // for (i=0...

    // Unpack the surface pressure

    if (buffer[27] != -32768)
      surface_pressure = buffer[27] / 10.0;

    // Calculate the relative humidities

    relhums = calculateRelativeHumidities(pressures, temps, 100, 
					  eff_pressures, wvaps, 100, surface_pressure);

    // Remove values that are above the upper limit

    for (n=0; n<100; n++)
      {
      if (eff_pressures[n] < 400.0)
	relhums[n] = -32768.0;
      }

    // Find the maximum relative humidity between 400 mb and the surface

    max_relhum = -1;

    for (n=0; n<100; n++)
      {
      if (relhums[n] != -32768.0)
        {
	if (relhums[n] > RELHUM_LIMIT)
          {
          if (relhums[n] > max_relhum)
	    max_relhum = relhums[n];
	  }
	}
      }

    // If the maxlmum relative humidity is less than the supersaturation limit,
    // then set the maximum value to -1

    if (max_relhum <= RELHUM_LIMIT)
      value = -1.0;
    else
      value = max_relhum;

    if (value >= 30000.0)
     value = -2.0;

    free(relhums);
    }  // if (parm_number == -18  max supersat...


  // Surface pressure
  
  else if (parm_number == -11)
    {
    value = buffer[27];
    }
  
  // Cloud top pressures
  
  else if (parm_number == -50)
    {
    value = buffer[426];
    }
  else if (parm_number == -51)
    {
    value = buffer[427];
    }
  else if (parm_number == -52)
    {
    value = buffer[428];
    }
  else if (parm_number == -53)
    {
    value = buffer[429];
    }
  else if (parm_number == -54)
    {
    value = buffer[430];
    }
  else if (parm_number == -55)
    {
    value = buffer[431];
    }
  else if (parm_number == -56)
    {
    value = buffer[432];
    }
  else if (parm_number == -57)
    {
    value = buffer[433];
    }
  else if (parm_number == -58)
    {
    value = buffer[434];
    }
  else if (parm_number == -59)
    {
    value = buffer[435];
    }

  // Combined retrieval qc flag
  
  else if (parm_number == -20)
    {
    pass_ir = FALSE;
    pass_mw = FALSE;

    if ((buffer[32] & 1) == 0)
      pass_ir = TRUE;

    if ((buffer[32] & 8) == 0)
      pass_mw = TRUE;

    if ((pass_ir == TRUE) && (pass_mw == TRUE))
      value = 0;
    else if ((pass_ir == TRUE) && (pass_mw == FALSE))
      value = 1;
    else if ((pass_ir == FALSE) && (pass_mw == TRUE))
      value = 2;
    else
      value = 3;
    }


  // IR qc flag
  
  else if (parm_number == -21)
    {
    if ((buffer[32] & 1) == 0)
      pass = TRUE;
    else
      pass = FALSE;

    if (pass == TRUE)
      value = 0;
    else
      value = 1;
    }


  // MW qc flag
  
  else if (parm_number == -22)
    {
    if ((buffer[32] & 8) == 0)
      pass = TRUE;
    else
      pass = FALSE;

    if (pass == TRUE)
      value = 0;
    else
      value = 1;
    }


  // MIT qc flag
  
  else if (parm_number == -23)
    {
    if ((buffer[32] & 2) == 0)
      pass = TRUE;
    else
      pass = FALSE;

    if (pass == TRUE)
      value = 0;
    else
      value = 1;
    }


  // NOAA reg qc flag
  
  else if (parm_number == -24)
    {
    if ((buffer[32] & 4) == 0)
      pass = TRUE;
    else
      pass = FALSE;

    if (pass == TRUE)
      value = 0;
    else
      value = 1;
    }


  // iNOAA qc flag
  
  else if (parm_number == -25)
    {
    if ((buffer[32] & 16) == 0)
      pass = TRUE;
    else
      pass = FALSE;

    if (pass == TRUE)
      value = 0;
    else
      value = 1;
    }

  // Stability Parameters

  else if (parm_number == -100)
    value = stabilities->parp;

  else if (parm_number == -101)
    value = stabilities->part;

  else if (parm_number == -102)
    value = stabilities->pard;

  else if (parm_number == -103)
    value = stabilities->tskn;

  else if (parm_number == -104)
    value = stabilities->tpw;

  else if (parm_number == -105)
    value = stabilities->li;

  else if (parm_number == -106)
    value = stabilities->cape;

  else if (parm_number == -107)
    value = stabilities->ncap;

  else if (parm_number == -108)
    {
    if (stabilities->mxhail != -32768)
      value = stabilities->mxhail;
    else
      value = -32768;
    }

  else if (parm_number == -109)
    value = stabilities->cinh;

  else if (parm_number == -110)
    value = stabilities->ki;

  else if (parm_number == -111)
    value = stabilities->tt;

  else if (parm_number == -112)
    value = stabilities->show;

  else if (parm_number == -113)
    value = stabilities->sweat;

  else if (parm_number == -114)
    value = stabilities->lr85;

  else if (parm_number == -115)
    value = stabilities->cvt;

  else if (parm_number == -116)
    value = stabilities->lcl;

  else if (parm_number == -117)
    value = stabilities->lfc;

  else if (parm_number == -118)
    value = stabilities->el;

  else if (parm_number == -119)
    value = stabilities->elt;

  else if (parm_number == -120)
    value = stabilities->ccl;

  else if (parm_number == -121)
    value = stabilities->mcl;

  else if (parm_number == -122)
    value = stabilities->neg20c;

  else if (parm_number == -123)
    value = stabilities->l15th;

  else if (parm_number == -124)
    value = stabilities->l87th;

  else if (parm_number == -125)
    {
    if (stabilities->frzl != 0)
      value = stabilities->frzl;
    else
      value = -32768.0;
    }

  else if (parm_number == -126)
    {
    if (stabilities->wbfr != 0)
      value = stabilities->wbfr;
    else
      value = -32768.0;
    }

  else if (parm_number == -127)
    value = stabilities->tadv;

  else if (parm_number == -128)
  {
    value = stabilities->pcpt_flag;
//if (stabilities->pcpt_flag > 1)
//printf("Flag: %d\n", stabilities->pcpt_flag);
  }

  else if (parm_number == -129)
    value = stabilities->windex;

  else if (parm_number == -130)
    value = stabilities->hmi;

  else if (parm_number == -131)
    value = stabilities->wmsi;

  else if (parm_number == -132)
    value = stabilities->haines;

  // Level temperatures and water vapors

  else if (parm_number <= -1600000000)
    {
    parm = parm_number / 1000000;
    ipressure = -1 * parm_number % 1000000;
    pressure = ipressure / 100.0;

    // Retrieved temperatures

    if (parm == -1600)
      {
      for (n=0; n<100; n++)
        {
        pressures[n] = -32768.0;
        temps[n] = -32768.0;

        if (buffer[n+450] != -32768)
          {
          if (n <= 19)
            pressures[n] = buffer[n+450] / 1000.0;
          else
            pressures[n] = buffer[n+450] / 10.0;
          }

        if (buffer[n+650] != -32768)
          temps[n] = buffer[n+650] / 64.0;
        }

      interpolateValueAtPressure(pressure, 100, pressures, temps, &fvalue);

      if (buffer[27] != -32768)
	{
	surface_pressure = buffer[27] / 10.0;

	if (pressure > surface_pressure)
	  fvalue = -32768.0;
	}

      if (fvalue != -32768.0)
        value = fvalue * 64;
      }  // retrieved temp


    // First guess temperatures

    else if (parm == -1601)
      {
      for (n=0; n<100; n++)
        {
        pressures[n] = -32768.0;
        temps[n] = -32768.0;

        if (buffer[n+450] != -32768)
          {
          if (n <= 19)
            pressures[n] = buffer[n+450] / 1000.0;
          else
            pressures[n] = buffer[n+450] / 10.0;
          }

        if (buffer[n+850] != -32768)
          temps[n] = buffer[n+850] / 64.0;
        }

      interpolateValueAtPressure(pressure, 100, pressures, temps, &fvalue);

      if (buffer[27] != -32768)
	{
	surface_pressure = buffer[27] / 10.0;

	if (pressure > surface_pressure)
	  fvalue = -32768.0;
	}

      if (fvalue != -32768.0)
        value = fvalue * 64;
      }  // first guess temp


    // MW only temperatures

    else if (parm == -1602)
      {
      for (n=0; n<100; n++)
        {
        pressures[n] = -32768.0;
        temps[n] = -32768.0;

        if (buffer[n+750] != -32768)
          {
          if (n <= 19)
            pressures[n] = buffer[n+450] / 1000.0;
          else
            pressures[n] = buffer[n+450] / 10.0;
          }

        if (buffer[n+1515] != -32768)
          temps[n] = buffer[n+750] / 64.0;
        }

      interpolateValueAtPressure(pressure, 100, pressures, temps, &fvalue);

      if (buffer[27] != -32768)
	{
	surface_pressure = buffer[27] / 10.0;

	if (pressure > surface_pressure)
	  fvalue = -32768.0;
	}

      if (fvalue != -32768.0)
        value = fvalue * 64;
      }  // mw only temp


    // Retrieved water vapor

    else if (parm == -3600)
      {
      for (i=0; i<100; i++)
        {
        pressures[i] = -32768.0;
        wvaps[i]     = -32768.0;

        if (buffer[i+550] != -32768)
          {
          if (i <= 19)
            pressures[i] = (float)buffer[i+550] / 1000.0;
          else
            pressures[i] = (float)buffer[i+550] / 10.0;
          }

        if (buffer[i+950] != -32768)
          wvaps[i] = (float)(exp(buffer[i+950]/1024.0));

        if (wvaps[i] < 0.001)
          wvaps[i] = -32768.0;
        }

      interpolateValueAtPressure(pressure, 100, pressures, wvaps, &fvalue);

      if (buffer[27] != -32768)
	{
	surface_pressure = buffer[27] / 10.0;

	if (pressure > surface_pressure)
	  fvalue = -32768.0;
	}

      if (fvalue != -32768.0)
        value = log(fvalue) * 1024;
      }  // retrieved water vapors


    // First guess water vapor

    else if (parm == -3601)
      {
      for (i=0; i<100; i++)
        {
        pressures[i] = -32768.0;
        wvaps[i]     = -32768.0;

        if (buffer[i+550] != -32768)
          {
          if (i <= 19)
            pressures[i] = (float)buffer[i+550] / 1000.0;
          else
            pressures[i] = (float)buffer[i+550] / 10.0;
          }

        if (buffer[i+1150] != -32768)
          wvaps[i] = (float)(exp(buffer[i+1150]/1024.0));

        if (wvaps[i] < 0.001)
          wvaps[i] = -32768.0;
        }

      interpolateValueAtPressure(pressure, 100, pressures, wvaps, &fvalue);

      if (buffer[27] != -32768)
	{
	surface_pressure = buffer[27] / 10.0;

	if (pressure > surface_pressure)
	  fvalue = -32768.0;
	}

      if (fvalue != -32768.0)
        value = log(fvalue) * 1024;
      }  // first guess water vapors


    // mw only water vapor

    else if (parm == -3602)
      {
      for (i=0; i<100; i++)
        {
        pressures[i] = -32768.0;
        wvaps[i]     = -32768.0;

        if (buffer[i+550] != -32768)
          {
          if (i <= 19)
            pressures[i] = (float)buffer[i+550] / 1000.0;
          else
            pressures[i] = (float)buffer[i+550] / 10.0;
          }

        if (buffer[i+1050] != -32768)
          wvaps[i] = (float)(exp(buffer[i+1050]/1024.0));

        if (wvaps[i] < 0.001)
          wvaps[i] = -32768.0;
        }

      interpolateValueAtPressure(pressure, 100, pressures, wvaps, &fvalue);

      if (buffer[27] != -32768)
	{
	surface_pressure = buffer[27] / 10.0;

	if (pressure > surface_pressure)
	  fvalue = -32768.0;
	}

      if (fvalue != -32768.0)
        value = log(fvalue) * 1024;
      }  // mw only water vapors



//**********************************
//**********************************
//**********************************
//**********************************
//**********************************
//**********************************
//**********************************
//**********************************
    // Retrieved water vapor

    else if (parm == -3700)
      {

      // Unpack the surface pressure

      if (buffer[27] != -32768)
	surface_pressure = buffer[27] / 10.0;

      // Only process this level if it is above the surface pressure

      if (pressure <= surface_pressure)
	{

        // Unpack the temperatures and pressures

        for (n=0; n<100; n++)
          {
          pressures[n] = -32768.0;
	  temps[n] = -32768.0;
	  
	  if (buffer[n+450] != -32768)
	    {
	    if (n <= 19)
	      pressures[n] = buffer[n+450] / 1000.0;
	    else
	      pressures[n] = buffer[n+450] / 10.0;
	    }

	  if (buffer[n+650] != -32768)
	    temps[n] = buffer[n+650] / 64.0;
	  }  // for (n=0...

	// Unpack the moistures and pressures

	for (i=0; i<100; i++)
	  {
          eff_pressures[i] = -32768.0;
	  wvaps[i]     = -32768.0;

	  if (buffer[i+550] != -32768)
	    {
            if (i <= 19)
	      eff_pressures[i] = (float)buffer[i+550] / 1000.0;
	    else
	      eff_pressures[i] = (float)buffer[i+550] / 10.0;
	    }

	  if (buffer[i+950] != -32768)
	    wvaps[i] = (float)(exp(buffer[i+950]/1024.0));

	  if (wvaps[i] < 0.001)
	    wvaps[i] = -32768.0;
	  }  // for (i=0...

	// Calculate the relative humidities

        relhums = calculateRelativeHumidities(pressures, temps, 100, 
  					      eff_pressures, wvaps, 100, surface_pressure);

	// Interpolate to the desired pressure

        interpolateValueAtPressure((double)pressure, 100, eff_pressures, relhums, &fvalue);

        if (fvalue != -32768.0)
	  {
	  if (fvalue >= 30000.0)
	    value = -2.0;
	  else
	    value = fvalue * 10;
	  }

	}  // if (pressure <= surface_pressure...
      }  // retrieved water vapors


    // First guess water vapor

    else if (parm == -3701)
      {
      for (i=0; i<100; i++)
        {
        pressures[i] = -32768.0;
        wvaps[i]     = -32768.0;

        if (buffer[i+550] != -32768)
          {
          if (i <= 19)
            pressures[i] = (float)buffer[i+550] / 1000.0;
          else
            pressures[i] = (float)buffer[i+550] / 10.0;
          }

        if (buffer[i+1150] != -32768)
          wvaps[i] = (float)(exp(buffer[i+1150]/1024.0));

        if (wvaps[i] < 0.001)
          wvaps[i] = -32768.0;
        }

      interpolateValueAtPressure(pressure, 100, pressures, wvaps, &fvalue);

      if (buffer[27] != -32768)
	{
	surface_pressure = buffer[27] / 10.0;

	if (pressure > surface_pressure)
	  fvalue = -32768.0;
	}

      if (fvalue != -32768.0)
        value = log(fvalue) * 1024;
      }  // first guess water vapors


    // mw only water vapor

    else if (parm == -3702)
      {
      for (i=0; i<100; i++)
        {
        pressures[i] = -32768.0;
        wvaps[i]     = -32768.0;

        if (buffer[i+550] != -32768)
          {
          if (i <= 19)
            pressures[i] = (float)buffer[i+550] / 1000.0;
          else
            pressures[i] = (float)buffer[i+550] / 10.0;
          }

        if (buffer[i+1050] != -32768)
          wvaps[i] = (float)(exp(buffer[i+1050]/1024.0));

        if (wvaps[i] < 0.001)
          wvaps[i] = -32768.0;
        }

      interpolateValueAtPressure(pressure, 100, pressures, wvaps, &fvalue);

      if (buffer[27] != -32768)
	{
	surface_pressure = buffer[27] / 10.0;

	if (pressure > surface_pressure)
	  fvalue = -32768.0;
	}

      if (fvalue != -32768.0)
        value = log(fvalue) * 1024;
      }  // mw only water vapors
//**********************************
//**********************************
//**********************************
//**********************************
//**********************************
//**********************************
//**********************************
    }  // else if (parm_number < -500000000


  // Layer temperatures and water vapors

  else if (parm_number <= -150000)
    {
    parm  = parm_number / 100;
    layer = -1 * (parm_number % 100);

    // Retrieved temperatures

    if (parm == -1500)
      {
      if (retrieved_layer_temps[layer-1] != -32768.0)
        value = retrieved_layer_temps[layer-1] * 64;
      }

    // First guess temperatures

    else if (parm == -1501)
      {
      if (guess_layer_temps[layer-1] != -32768.0)
        value = guess_layer_temps[layer-1] * 64;
      }

    // MW only temperatures

    else if (parm == -1502)
      {
      if (mw_layer_temps[layer-1] != -32768.0)
        value = mw_layer_temps[layer-1] * 64;
      }

    // Retrieved water vapor

    else if (parm == -3500)
      {
      if (retrieved_layer_wvaps[layer-1] != -32768.0)
        value = log(retrieved_layer_wvaps[layer-1]) * 1024;
      }

    // First guess water vapor

    else if (parm == -3501)
      {
      if (guess_layer_wvaps[layer-1] != -32768.0)
        value = log(guess_layer_wvaps[layer-1]) * 1024;
      }

    // MW only water vapor

    else if (parm == -3502)
      {
      if (mw_layer_wvaps[layer-1] != -32768.0)
        value = log(mw_layer_wvaps[layer-1]) * 1024;
      }
    }  // else if (parm_number < -150000

  return value;
  }

// end of file
