/*
   Name- processNUCAPS

   Language- C     Type- MAIN

   Version- 1.0    Date-  1/16/2007   Programmer- Mike Pettey (IMSG)
   Version- 2.0    Date- 11/01/2013   Programmer- Mike Pettey (IMSG)
   Version- 3.0    Date-  2/04/2015   Programmer- Mike Pettey (IMSG)

   Function- This program extracts selected data from a IDDF
             and writes the data to an ODS file.  The selected
             data is defined in the run script.  Also in the run
             script are the starting date (YYYYMMDD) and hour and
             the ending date and hour.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "ods7_capture.h"
#include "stability.h"


#define  RECORD_LENGTH_NUCAPS       13000
#define  NUM_HEADER_VALUES_NUCAPS   RECORD_LENGTH_NUCAPS/4
#define  NUM_DATA_VALUES_NUCAPS     RECORD_LENGTH_NUCAPS/2

#define  SWAP_IN   FALSE
#define  SWAP_OUT  TRUE


void getNUCAPSLayerData(struct parameter_info *parm_list, short *buffer, 
		      float *retrieved_temp_layers, float *mw_temp_layers, float *cldyregr_temp_layers,
		      float *retrieved_wvap_layers, float *mw_wvap_layers, float *cldyregr_wvap_layers);

void unpackProfileData(short nucaps_buffer[NUM_DATA_VALUES_NUCAPS], float *pressures, float *temperatures,
                       float *water_vapors, float *surf_pressure, float *surf_temperature, float *surf_water_vapor);

void initializeStabilityParameters(struct stability_data *stabilities);

void calculateStabilities(float *pressures, float *temps, float *wvaps, int num_levels, 
			  float surf_press, float surf_temp, float surf_dwpt,
			  struct stability_data *stabilities);

double getSpecialNUCAPSValue(long parm_number, short buffer[NUM_DATA_VALUES_NUCAPS],
			   float *retrieved_layer_temps, float *mw_layer_temps, 
			   float *cldyregr_layer_temps, float *retrieved_layer_wvaps,
			   float *mw_layer_wvaps, float *cldyregr_layer_wvaps,
			   struct stability_data *stabilities);

void copyParameterInformation(struct parameter_info *parm_list, FILE *ods);

void createProfileInfoFile();

long secondsSinceYear(int year, long retrieval_date, long retrieval_time);



void processNUCAPS(int date_to_process, struct parameter_info *parameter_list)
  {
  int    nucaps_header[NUM_HEADER_VALUES_NUCAPS];
  short  nucaps_buffer[NUM_DATA_VALUES_NUCAPS];
  char   out_buffer_i1[BUFFER_SIZE];
  short  out_buffer_i2[BUFFER_SIZE];
  int    out_buffer_i4[BUFFER_SIZE];
  long   out_buffer_i8[BUFFER_SIZE];
  int    number_of_nucaps_records, last_nucaps_rec_written;
  int    first_record, last_record;
  int    nucaps_record;
  double value;

  long   seconds, earliest_seconds, latest_seconds;
  int    number_of_retrievals;
  int    earliest_date, earliest_time;
  int    latest_date, latest_time;
  int    retrieval_year, retrieval_mmdd;
  int    retrieval_hour, retrieval_mmss;
  long   retrieval_date, retrieval_time;

  char   i1temp;
  long   offset, i8temp;
  int    i4temp, i, j, n, count, status, ival;
  short  i2temp;
  float  lat, lon, pct;

  char   ods_file_name[1000], set_name[100], parmdefs_file_name[1000];
  char   filename_string[1000];
  char   c, char12[12], char36[36];
  char   month_string[10];
  int    buffer_index, block_index, output_index, block_number;
  int    num_values_written;
  int    use_fov, pass_both, pass_cldyregr_only;

  float  surf_pressure, surf_temperature, surf_water_vapor;
  float  pressures[100], temperatures[100], water_vapors[100];

  float  retrieved_layer_temps[30], mw_layer_temps[30], cldyregr_layer_temps[30];
  float  retrieved_layer_wvaps[20], mw_layer_wvaps[20], cldyregr_layer_wvaps[20];

  struct stability_data *stabilities;
  struct parameter_info *ptr;

  int    month, day, year, hour, minute, second, today;

  time_t  date_time;
  struct  tm  *ts;   

  FILE   *nucaps, *ods, *in;


  stabilities = (struct stability_data*)malloc(sizeof(struct stability_data));


  // Create the parameter_info.xml file with the data from each parameter.
  // While doing this, pull out the file name used for each parameter.

  sprintf(ods_file_name, "out.dir/parameter_info.xml");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS parameter_info.xml file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fprintf(ods, "<?xml version=\"1.0\" standalone=\"yes\" ?>\n\n");

  fprintf(ods, "<parameter_info>\n");

  ptr = parameter_list;

  while (ptr != NULL)
    {
    copyParameterInformation(ptr, ods);
    ptr = ptr->next;
    }

  fprintf(ods, "</parameter_info>\n");
  fclose(ods);


  // Create sub-directories for each parameter

  //sprintf(set_name, "%d", date_to_process);
  sprintf(set_name, "set01");

  sprintf(ods_file_name, "out.dir/%s", set_name);
  mkdir(ods_file_name, 0777);

  ptr = parameter_list;

  while (ptr != NULL)
    {
    sprintf(ods_file_name, "out.dir/%s/%s", set_name, ptr->file_name);
    mkdir(ods_file_name, 0777);
    ptr = ptr->next;
    }


  // Open the NUCAPS

  if ((nucaps=fopen("in.file","r")) == NULL)
    {
    //createEmptyFile(capture_info, parameter_list);

    printf("\n\nThe NUCAPS could not be opened for input.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  // Position the read pointer to the start of the first record

  offset = 0;
  fseek(nucaps, offset, SEEK_SET);

  // Read the header record

  fread(&nucaps_header, RECORD_LENGTH_NUCAPS, 1, nucaps);

  if (SWAP_IN == TRUE)
    {
    for (n=0; n<NUM_HEADER_VALUES_NUCAPS; n++)
      nucaps_header[n] = htonl(nucaps_header[n]);
    }


  number_of_nucaps_records = nucaps_header[2];

  printf("%s file:\n", system_description);
  printf("   Number of records in the file:  %d\n",number_of_nucaps_records);

  first_record = 2;
  last_record  = number_of_nucaps_records;


  block_number = 0;

  num_values_written = 0;

  // Initialize the earliest and latest seconds and the lat/lon range

  latest_seconds   = 0;
  earliest_seconds = secondsSinceYear(2000,20291231,235959);

  // Loop through the data records

  buffer_index = 0;
  output_index = 0;

//last_record = 25000;
  for (nucaps_record=first_record; nucaps_record<=last_record; nucaps_record++)
    {
    pct = (float)nucaps_record / (float)last_record * 100.0;

    if ((nucaps_record%1000)==0)
      printf("   %d    %d    %.1f%\n", nucaps_record, last_record, pct);

    // Position the read pointer to the start of the next record

    offset = (nucaps_record - 1) * RECORD_LENGTH_NUCAPS;
    //fseek(nucaps, offset, SEEK_SET);

    // Read the data record

    fread(&nucaps_buffer, RECORD_LENGTH_NUCAPS, 1, nucaps);

    if (SWAP_IN == TRUE)
      {
      for (n=0; n<NUM_DATA_VALUES_NUCAPS; n++)
        nucaps_buffer[n] = htons(nucaps_buffer[n]);
      }


    if (nucaps_buffer[0] == 0)
      {

      // Unpack the date, latitude and longitude

      retrieval_date = (nucaps_buffer[5] * 10000) + nucaps_buffer[6];

      lat = nucaps_buffer[3] / 128.0;
      lon = nucaps_buffer[4] / 128.0;

      // If the retrieval is within both windows, count it

      use_fov = FALSE;

      if ((date_to_process == -32768) || (retrieval_date == date_to_process))
	{
	if ((lat >= bottom_latitude) && (lat <= top_latitude))
  	  {
	  if ((left_longitude <= right_longitude))
	    {
	    if ((lon >= left_longitude) && (lon <= right_longitude))
	      {
	      use_fov = TRUE;
              }
	    }
	  else
	    { 
	    if ((lon <= right_longitude) || (lon >= left_longitude))
	      {
	      use_fov = TRUE;
	      }
	    }
	  }  // if (lats[fov]...
	}  // if (date_to_process...

      if (use_fov == TRUE)
	{

        // Keep track of the earliest and latest time

        retrieval_date = (nucaps_buffer[5] * 10000) + nucaps_buffer[6];
        retrieval_time = (nucaps_buffer[7] * 10000) + nucaps_buffer[8];

        seconds = secondsSinceYear(2000,retrieval_date,retrieval_time);

        if (seconds < earliest_seconds)
          {
          earliest_seconds = seconds;
          earliest_date    = retrieval_date;
          earliest_time    = retrieval_time;
          }

        if (seconds > latest_seconds)
          {
          latest_seconds = seconds;
          latest_date    = retrieval_date;
          latest_time    = retrieval_time;
          }


	// Call the subroutine getNucapsLayerData which will convert some or all
	// of the temperature and moisture data from levels to layers

	getNUCAPSLayerData(parameter_list, nucaps_buffer,
			 retrieved_layer_temps, mw_layer_temps, cldyregr_layer_temps,
			 retrieved_layer_wvaps, mw_layer_wvaps, cldyregr_layer_wvaps);


  // Call the subroutine calculateStabilities to compute the stability
	// parameters

  unpackProfileData(nucaps_buffer, pressures, temperatures, water_vapors,
                    &surf_pressure, &surf_temperature, &surf_water_vapor);


//  for (n=0; n<100; n++)
//    printf("UNPACKED: %d  %f  %f  %f\n", n, pressures[n], temperatures[n], water_vapors[n]);
//  printf("\n%f   %f   %f\n", surf_pressure, surf_temperature, surf_water_vapor);
//if (1==1) exit(0);

	// Calculate the stability parameters

	//stabilities = (struct stability_data*)malloc(sizeof(struct stability_data));

	initializeStabilityParameters(stabilities);

	calculateStabilities(pressures, temperatures, water_vapors, 100,
			     surf_pressure, surf_temperature, surf_water_vapor,
			     stabilities);



        // Fill the nucaps_buffers

        ptr = parameter_list;

        while (ptr != NULL)
	  {
          if (ptr->word_number > 0)
            {
            value = nucaps_buffer[ptr->word_number - 1];
            }  // if (word_number > 0...
          else
            {
	    value = getSpecialNUCAPSValue(ptr->word_number, nucaps_buffer,
					retrieved_layer_temps, mw_layer_temps,
					cldyregr_layer_temps,retrieved_layer_wvaps,
					mw_layer_wvaps, cldyregr_layer_wvaps,
					stabilities);
            }  // else word_number < 0...

	  if (ptr->data_type == BYTE)
	    ptr->byte_buffer[buffer_index] = (char)value;
	  else if (ptr->data_type == SHORT)
	    ptr->short_buffer[buffer_index] = (short)value;
	  else if (ptr->data_type == INT)
	    ptr->int_buffer[buffer_index] = (int)value;
	  else if (ptr->data_type == FLOAT)
	    ptr->short_buffer[buffer_index] = (short)value;
	  else if (ptr->data_type == DOUBLE)
	    ptr->short_buffer[buffer_index] = (short)value;

          ptr = ptr->next;
  	  }

        buffer_index++;


        // If the buffers are full, write them to the file

        if (buffer_index >= BUFFER_SIZE)
          {
          buffer_index = 0;

          block_index = 0;
          ptr = parameter_list;

          while (ptr != NULL)
	    {
            sprintf(ods_file_name, "out.dir/%s/%s/block_%d.dat", set_name, ptr->file_name, block_number);

	    if ((ods=fopen(ods_file_name,"w")) == NULL)
	      {
	      printf("\n\nThe ODS block data file file %s could not be opened for output.\n",
	    	     ods_file_name);
	      printf("Execution ending.\n\n");
	      exit(1);
	      }

	    if (ptr->data_type == BYTE)
	      {
	      for (i=0; i<BUFFER_SIZE; i++)
		{
		out_buffer_i1[i] = ptr->byte_buffer[i];
		}

	      fwrite(out_buffer_i1, 1, BUFFER_SIZE, ods);
	      }

	    else if ((ptr->data_type == SHORT) || (ptr->data_type == FLOAT))
	      {
	      for (i=0; i<BUFFER_SIZE; i++)
		{
                if (SWAP_OUT == FALSE)
		  out_buffer_i2[i] = ptr->short_buffer[i];
		else
		  out_buffer_i2[i] = htons(ptr->short_buffer[i]);
		}

	      fwrite(out_buffer_i2, 2, BUFFER_SIZE, ods);
	      }

	    else if ((ptr->data_type == INT) || (ptr->data_type == DOUBLE))
	      {
	      for (i=0; i<BUFFER_SIZE; i++)
		{
                if (SWAP_OUT == FALSE)
		  out_buffer_i4[i] = ptr->int_buffer[i];
		else
		  out_buffer_i4[i] = htonl(ptr->int_buffer[i]);
		}

	      fwrite(out_buffer_i4, 4, BUFFER_SIZE, ods);
	      }

	    fclose(ods);

	    block_index++;
	    ptr = ptr->next;
	    }  // while (ptr != NULL...

	  num_values_written += BUFFER_SIZE;
          output_index = output_index + BUFFER_SIZE;

	  block_number++;
          }  // if (buffer_index...

	//free(stabilities);
	}  // if (use_fov...
      }  // if (buffer[0]...
    }  // for (nucaps_record=first_record...


  // Write the remainder of the data to the file

  if (buffer_index > 0)
    {
    block_index = 0;
    ptr = parameter_list;

    while (ptr != NULL)
      {
      sprintf(ods_file_name, "out.dir/%s/%s/block_%d.dat", set_name, ptr->file_name, block_number);

      if ((ods=fopen(ods_file_name,"w")) == NULL)
	{
        printf("\n\nThe ODS block data file file %s could not be opened for output.\n",
	       ods_file_name);
	printf("Execution ending.\n\n");
	exit(1);
	}

      if (ptr->data_type == BYTE)
	{
	for (n=0; n<buffer_index; n++)
	  {
	  i1temp = ptr->byte_buffer[n];
	  fwrite(&i1temp, 1, 1, ods);
	  }
	}

      else if ((ptr->data_type == SHORT) || (ptr->data_type == FLOAT))
	{
        for (n=0; n<buffer_index; n++)
	  {
	  i2temp = ptr->short_buffer[n];

	  if (SWAP_OUT == TRUE)
	    i2temp = htons(i2temp);

	  fwrite(&i2temp, 2, 1, ods);
	  }
	}

      else if ((ptr->data_type == INT) || (ptr->data_type == DOUBLE))
	{
	for (n=0; n<buffer_index; n++)
	  {
	  i4temp = ptr->int_buffer[n];

	  if (SWAP_OUT == TRUE)
	    i4temp = htonl(i4temp);

	  fwrite(&i4temp, 4, 1, ods);
	  }
	}

      fclose(ods);

      block_index++;
      ptr = ptr->next;
      }

    num_values_written += block_index;
    block_number++;
    }


  printf("\nNumber of values written to the output file:  %d\n", num_values_written);

  // Print the earliest and latest times found

  year   = earliest_date / 10000;
  month  = (earliest_date % 10000) / 100;
  day    = earliest_date % 100;
  hour   = earliest_time / 10000;
  minute = (earliest_time % 10000) / 100;
  second = earliest_time % 100;

  printf("\nEarliest retrieval found:  %d/%2d/%d %2d:%2d:%2d\n",
          month,day,year,hour,minute,second);

  year   = latest_date / 10000;

  month  = (latest_date % 10000) / 100;
  day    = latest_date % 100;
  hour   = latest_time / 10000;
  minute = (latest_time % 10000) / 100;
  second = latest_time % 100;

  printf("Latest retrieval found:    %d/%2d/%d %2d:%2d:%2d\n\n",
	  month,day,year,hour,minute,second);


  // Get the current date

  time (&date_time);         
  ts = localtime(&date_time);

  month  = ts->tm_mon + 1;    
  day    = ts->tm_mday;       
  year   = ts->tm_year;       

  if (year > 56)
    year = 1900 + year;
  else
    year = 2000 + year;

  today = year*10000 + month*100 + day;


  // Create the ODS header record

  sprintf(ods_file_name, "out.dir/header.xml");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS header.xml file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fprintf(ods, "<?xml version=\"1.0\" standalone=\"yes\" ?>\n\n");

  fprintf(ods, "<header>\n");
  fprintf(ods, "  <file_type>ODS7</file_type>\n\n");

  fprintf(ods, "  <file_creation_date>%d</file_creation_date>\n", today);
  fprintf(ods, "  <file_modification_date>%d</file_modification_date>\n\n", today);

  fprintf(ods, "  <oldest_data>\n");
  fprintf(ods, "    <date>%d</date>\n", earliest_date);
  fprintf(ods, "    <time>%06d</time>\n", earliest_time);
  fprintf(ods, "  </oldest_data>\n\n");

  fprintf(ods, "  <newest_data>\n");
  fprintf(ods, "    <date>%d</date>\n", latest_date);
  fprintf(ods, "    <time>%06d</time>\n", latest_time);
  fprintf(ods, "  </newest_data>\n\n");

  fprintf(ods, "  <data_format>binary</data_format>\n\n");

  fprintf(ods, "  <data_type>sequential</data_type>\n");
  fprintf(ods, "  <footprint_size>40.0</footprint_size>\n\n");

  fprintf(ods, "  <earth_coverage>\n");
  fprintf(ods, "    <left_longitude>%.2f</left_longitude>\n", left_longitude);
  fprintf(ods, "    <right_longitude>%.2f</right_longitude>\n", right_longitude);
  fprintf(ods, "    <top_latitude>%.2f</top_latitude>\n", top_latitude);
  fprintf(ods, "    <bottom_latitude>%.2f</bottom_latitude>\n", bottom_latitude);
  fprintf(ods, "  </earth_coverage>\n\n");

  fprintf(ods, "  <system_name>%s</system_name>\n", system_description);
  fprintf(ods, "</header>\n");

  fclose(ods);


  // Create the set_info.xml file

  sprintf(ods_file_name, "out.dir/set_info.xml");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS set_info.xml file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fprintf(ods, "<?xml version=\"1.0\" standalone=\"yes\" ?>\n\n");

  fprintf(ods, "<set_info>\n");
  fprintf(ods, "  <set>\n");

  fprintf(ods, "    <set_name>%s</set_name>\n", set_name);

  year  = date_to_process / 10000;
  month = (date_to_process % 10000) / 100;
  day   = date_to_process % 100;

  if (month == 1)
    sprintf(month_string, "January");
  else if (month == 2)
    sprintf(month_string, "February");
  else if (month == 3)
    sprintf(month_string, "March");
  else if (month == 4)
    sprintf(month_string, "April");
  else if (month == 5)
    sprintf(month_string, "May");
  else if (month == 6)
    sprintf(month_string, "June");
  else if (month == 7)
    sprintf(month_string, "July");
  else if (month == 8)
    sprintf(month_string, "August");
  else if (month == 9)
    sprintf(month_string, "September");
  else if (month == 10)
    sprintf(month_string, "October");
  else if (month == 11)
    sprintf(month_string, "November");
  else if (month == 12)
    sprintf(month_string, "December");

  fprintf(ods, "    <description>%d/%02d/%4d</description>\n", month, day, year);
  fprintf(ods, "    <date_string>%s %d, %d</date_string>\n", month_string, day, year);

  fprintf(ods, "    <oldest_data>\n");
  fprintf(ods, "      <date>%d</date>\n", earliest_date);
  fprintf(ods, "      <time>%06d</time>\n", earliest_time);
  fprintf(ods, "    </oldest_data>\n\n");

  fprintf(ods, "    <newest_data>\n");
  fprintf(ods, "      <date>%d</date>\n", latest_date);
  fprintf(ods, "      <time>%06d</time>\n", latest_time);
  fprintf(ods, "    </newest_data>\n\n");

  fprintf(ods, "    <system_name>%s</system_name>\n", system_description);
  fprintf(ods, "  </set>\n");
  fprintf(ods, "</set_info>\n");
  fclose(ods);


  // Create the profile_info.xml file
  
  createProfileInfoFile();


  // Create the byte_check.dat file

  sprintf(ods_file_name, "out.dir/byte_check.dat");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS byte_check.dat file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  n = 79688355;

  if (SWAP_OUT == TRUE)
    n = htonl(n);

  fwrite(&n, 4, 1, ods);

  fclose(ods);
  }




//void unpackProfileData(short *nucaps_buffer, float *pressures, float *temperatures, float *water_vapors,
//                       float *surf_pressure, float *surf_temperature, float *surf_water_vapor)
//  {
//  int n;
//
//  for (n=0; n<100; n++)
//    {
//    pressures[n]    = -32768.0;
//    temperatures[n] = -32768.0;
//	  water_vapors[n] = -32768.0;
//
//	  // Unpack the pressures
//
//    if (nucaps_buffer[n+450] != -32768)
//      {
//      if (n <= 19)
//        pressures[n] = nucaps_buffer[n+450] / 1000.0;
//      else
//        pressures[n] = nucaps_buffer[n+450] / 10.0;
//      }
//
//	  // Unpack the temperatures
//
//    if (nucaps_buffer[n+650] != -32768)
//      temperatures[n] = nucaps_buffer[n+650] / 64.0;
//
//	  // Unpack the water vapors
//
//	  if (nucaps_buffer[n+950] != -32768)
//	    water_vapors[n] = (float)(exp(nucaps_buffer[n+950]/1024.0));
//
//	  if (water_vapors[n] < 0.001)
//	    water_vapors[n] = -32768.0;
//    }
//
//  // Unpack the surface data
//
//	if (nucaps_buffer[27] == -32768)
//    *surf_pressure = -32768.0;
//	else
//	  *surf_pressure = nucaps_buffer[27] / 10.0;
//
//	if (nucaps_buffer[28] == -32768)
//	  *surf_temperature = -32768.0;
//	else
//	  *surf_temperature = nucaps_buffer[28] / 64.0;
//
//	*surf_water_vapor = -32768.0;
//  }


void unpackProfileData(short *nucaps_buffer, float *pressures, float *temperatures, float *water_vapors,
                       float *surf_pressure, float *surf_temperature, float *surf_water_vapor)
  {
  int n;

  // Determine whether the sounding passed IR+MW, MW-only or neither
  
  int passed_ir = FALSE;
  int passed_mw = FALSE;
  
  if (nucaps_buffer[16] != -32768)
    {
    if ((nucaps_buffer[16] & 1) == 0)
      passed_ir = TRUE;
    
    if ((nucaps_buffer[16] & 8) == 0)
      passed_mw = TRUE;
    }

  // Pick the appropriate profiles based on which one passed
  
  if (passed_ir == TRUE)
    {
    for (n=0; n<100; n++)
      {
      pressures[n]    = -32768.0;
      temperatures[n] = -32768.0;
      water_vapors[n] = -32768.0;

      // Unpack the pressures

      if (nucaps_buffer[n+450] != -32768)
        {
        if (n <= 19)
          pressures[n] = nucaps_buffer[n+450] / 1000.0;
        else
          pressures[n] = nucaps_buffer[n+450] / 10.0;
        }

      // Unpack the temperatures

      if (nucaps_buffer[n+650] != -32768)
        temperatures[n] = nucaps_buffer[n+650] / 64.0;

      // Unpack the water vapors

      if (nucaps_buffer[n+950] != -32768)
        water_vapors[n] = (float)(exp(nucaps_buffer[n+950]/1024.0));

      if (water_vapors[n] < 0.001)
        water_vapors[n] = -32768.0;
      }

    // Unpack the surface data

    if (nucaps_buffer[27] == -32768)
      *surf_pressure = -32768.0;
    else
      *surf_pressure = nucaps_buffer[27] / 10.0;

    if (nucaps_buffer[28] == -32768)
      *surf_temperature = -32768.0;
    else
      *surf_temperature = nucaps_buffer[28] / 64.0;

    *surf_water_vapor = -32768.0;
    }
  
  else if (passed_mw == TRUE)
    {
    for (n=0; n<100; n++)
      {
      pressures[n]    = -32768.0;
      temperatures[n] = -32768.0;
      water_vapors[n] = -32768.0;

      // Unpack the pressures

      if (nucaps_buffer[n+450] != -32768)
        {
        if (n <= 19)
          pressures[n] = nucaps_buffer[n+450] / 1000.0;
        else
          pressures[n] = nucaps_buffer[n+450] / 10.0;
        }

      // Unpack the temperatures

      if (nucaps_buffer[n+750] != -32768)
        temperatures[n] = nucaps_buffer[n+750] / 64.0;

      // Unpack the water vapors

      if (nucaps_buffer[n+1050] != -32768)
        water_vapors[n] = (float)(exp(nucaps_buffer[n+1050]/1024.0));

      if (water_vapors[n] < 0.001)
        water_vapors[n] = -32768.0;
      }

    // Unpack the surface data

    if (nucaps_buffer[27] == -32768)
      *surf_pressure = -32768.0;
    else
      *surf_pressure = nucaps_buffer[27] / 10.0;

    if (nucaps_buffer[28] == -32768)
      *surf_temperature = -32768.0;
    else
      *surf_temperature = nucaps_buffer[28] / 64.0;

    *surf_water_vapor = -32768.0;
    }
  
  else
    {
    for (n=0; n<100; n++)
      {
      pressures[n]    = -32768.0;
      temperatures[n] = -32768.0;
	    water_vapors[n] = -32768.0;
      }
    
    *surf_pressure    = -32768.0;
    *surf_temperature = -32768.0;
    *surf_water_vapor = -32768.0;
    }
  }



void initializeStabilityParameters(struct stability_data *stabilities)
  {
  stabilities->parp = -32768;
  stabilities->part = -32768;
  stabilities->pard = -32768;
  stabilities->tskn = -32768;
  stabilities->tpw = -32768;
  stabilities->li = -32768;
  stabilities->cape = -32768;
  stabilities->ncap = -32768;
  stabilities->mxhail = -32768;
  stabilities->cinh = -32768;
  stabilities->ki = -32768;
  stabilities->tt = -32768;
  stabilities->show = -32768;
  stabilities->sweat = -32768;
  stabilities->lr85 = -32768;
  stabilities->cvt = -32768;
  stabilities->lcl = -32768;
  stabilities->lfc = -32768;
  stabilities->el = -32768;
  stabilities->elt = -32768;
  stabilities->ccl = -32768;
  stabilities->mcl = -32768;
  stabilities->neg20c = -32768;
  stabilities->l15th = -32768;
  stabilities->l87th = -32768;
  stabilities->frzl = -32768;
  stabilities->wbfr = -32768;
  stabilities->tadv = -32768;

  strcpy(stabilities->pcpt, " \0");

  stabilities->windex = -32768;
  stabilities->hmi = -32768;
  stabilities->wmsi = -32768;
  stabilities->haines = -32768;
  }

// end of file
