/*
   Name- CreateOdsFromNucapsCCR.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  6/01/2014   Programmer- Mike Pettey (IMSG)
   Version- 2.0    Date-  7/09/2015   Programmer- Mike Pettey (IMSG)

   Function- This program extracts selected data from an RRODF
             and writes the data to an EDGE file.  The selected
             data is defined in the run script.  Also in the run
             script are the starting date (YYYYMMDD) and hour and
             the ending date and hour.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>
#include <dirent.h>

#include "ods7_capture.h"

#define  SWAP_IN   FALSE
#define  SWAP_OUT  TRUE


struct parameter_info* processCaptureInfo7();

void copyParameterInformation(struct parameter_info *parm_list, FILE *ods);

int processNucapsGranule(char *file_name, int granule_number, int date_to_process, char set_name[1000],
                         int num_values_written, struct parameter_info *parameter_list,
                         int header_length, int block_length,
                         int* oldest_date, int* oldest_time, int* latest_date, int* latest_time);



main(int argc, char *argv[])
  {
  DIR     *dir;
  FILE    *ods;
  int     i, n, length, i4temp;
  int     date_to_process, granule_number;
  int     num_parameters, num_values_written, num_written_within_granule;
  int     parameter_number, header_length, block_length;
  int     output_header[20], month, day, year, today;
  short   i2temp;
  long    offset;
  char    c, ods_file_name[1000], set_name[1000], month_string[20];
  char    file_template[200], full_file_name[500], system_description[256];
  int     earliest_date, earliest_time, latest_date, latest_time;

  struct parameter_info *parameter_list;
  struct parameter_info *ptr;

  time_t  date_time;
  struct  tm *ts;  

  earliest_date = -32768;
  earliest_time = -32768;
  latest_date   = -32768;
  latest_time   = -32768;

  // Print an initial message

  printf("\nCopying the granule file to an ODS file\n\n");


  // Unpack the date of the data to be processed

  date_to_process = atoi(argv[1]);

  // Unpack the template to use when looking for files to process

  sprintf(file_template, "%s\0", argv[2]);

  printf("Date to process:  %d\n", date_to_process);
  printf("Files:            %s\n\n", file_template);

  // Read the system description from the command line

  if (argc == 4)
    {
    sprintf(system_description, "%s", argv[3]);
    }
  else
    {
    sprintf(system_description, " ");

    printf("No system name was included as a command line argument.\n");
    printf("The file will be captured but the system name will be blank.\n\n");
    }


  // Read the selected parameter info from the parameter file

  parameter_list = (struct parameter_info*)processCaptureInfo7();

  if (parameter_list == NULL)
    {
    printf("processCaptureInfo7() returned NULL. This indicates that there\n");
    printf("was a problem reading in the parameter data from the parameter\n");
    printf("information file. Execution cannot continue.\n\n");
    exit(1);
    }

  // Print the list of parameters and count the number of parameters

  num_parameters = 0;

  printf("The following parameters will be captured:\n\n");
  printf("Word\tDescription\n");
  printf("----\t-----------\n");

  ptr = parameter_list;

  while (ptr != NULL)
    {
    num_parameters++;

    printf("%ld\t%s\n", ptr->word_number, ptr->parmdefs_file_name);
    ptr = ptr->next;
    }

  printf("\n\n");


  // Create the parameter_info.xml file with the data from each parameter.
  // While doing this, pull out the file name used for each parameter.

  sprintf(ods_file_name, "out.dir/parameter_info.xml");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS parameter_info.xml file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fprintf(ods, "<?xml version=\"1.0\" standalone=\"yes\" ?>\n\n");
  fprintf(ods, "<parameter_info>\n");

  ptr = parameter_list;

  while (ptr != NULL)
    {
    copyParameterInformation(ptr, ods);
    ptr = ptr->next;
    }

  fprintf(ods, "</parameter_info>\n");
  fclose(ods);


  // Create sub-directories for each parameter

  //sprintf(set_name, "%d", date_to_process);
  sprintf(set_name, "set01");

  sprintf(ods_file_name, "out.dir/%s", set_name);
  mkdir(ods_file_name, 0777);

  ptr = parameter_list;

  while (ptr != NULL)
    {
    sprintf(ods_file_name, "out.dir/%s/%s", set_name, ptr->file_name);
    mkdir(ods_file_name, 0777);
    ptr = ptr->next;
    }


  // Loop through all of the granules in the input directory and
  // copy the selected parameters to the ODS file

  printf("\nProcessing the granules...\n");

  granule_number = 0;
  num_values_written = 0;

  // Open the current directory

  dir = opendir("in.dir");

  if (! dir)
    {
    printf("\nCannot open the input directory 'in.dir'\n\n");
    exit(1);
    }

  while (1)
    {
    struct dirent * entry;

    entry = readdir(dir);

    if (! entry)
      {
      break;
      }

    if (stringStartsWith(entry->d_name, file_template) == TRUE)
      {
      sprintf(full_file_name, "in.dir/%s", entry->d_name);

      granule_number++;

      num_written_within_granule = processNucapsGranule(full_file_name, granule_number, date_to_process,
							set_name, num_values_written, parameter_list,
							header_length, block_length,
							&earliest_date, &earliest_time,
							&latest_date, &latest_time);

      num_values_written = num_values_written + num_written_within_granule;
      }
    }

  // Close the directory

  if (closedir (dir))
    {
    printf ("\nCould not close the input directory 'in.dir'\n\n");
    exit(1);
    }

  printf ("\nNumber of values written to the ODS file:  %d\n\n", num_values_written);


  // Get today's date

  time (&date_time);         
  ts = localtime(&date_time);

  month  = ts->tm_mon + 1;    
  day    = ts->tm_mday;       
  year   = ts->tm_year;       

  if (year > 56)
    year = 1900 + year;
  else
    year = 2000 + year;

  today = year*10000 + month*100 + day;



  // Create the ODS header record

  sprintf(ods_file_name, "out.dir/header.xml");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS header.xml file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fprintf(ods, "<?xml version=\"1.0\" standalone=\"yes\" ?>\n\n");

  fprintf(ods, "<header>\n");
  fprintf(ods, "  <file_type>ODS7</file_type>\n\n");

  fprintf(ods, "  <file_creation_date>%d</file_creation_date>\n", today);
  fprintf(ods, "  <file_modification_date>%d</file_modification_date>\n\n", today);

  fprintf(ods, "  <oldest_data>\n");
  fprintf(ods, "    <date>%d</date>\n", earliest_date);
  fprintf(ods, "    <time>%06d</time>\n", earliest_time);
  fprintf(ods, "  </oldest_data>\n\n");

  fprintf(ods, "  <newest_data>\n");
  fprintf(ods, "    <date>%d</date>\n", latest_date);
  fprintf(ods, "    <time>%06d</time>\n", latest_time);
  fprintf(ods, "  </newest_data>\n\n");

  fprintf(ods, "  <data_format>binary</data_format>\n\n");

  fprintf(ods, "  <data_type>sequential</data_type>\n");
  fprintf(ods, "  <footprint_size>45.0</footprint_size>\n\n");

  fprintf(ods, "  <earth_coverage>\n");
  fprintf(ods, "    <left_longitude>%.2f</left_longitude>\n", left_longitude);
  fprintf(ods, "    <right_longitude>%.2f</right_longitude>\n", right_longitude);
  fprintf(ods, "    <top_latitude>%.2f</top_latitude>\n", top_latitude);
  fprintf(ods, "    <bottom_latitude>%.2f</bottom_latitude>\n", bottom_latitude);
  fprintf(ods, "  </earth_coverage>\n\n");

  fprintf(ods, "  <system_name>%s</system_name>\n", system_description);
  fprintf(ods, "</header>\n");

  fclose(ods);


  // Create the set_info.xml file

  sprintf(ods_file_name, "out.dir/set_info.xml");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS set_info.xml file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  fprintf(ods, "<?xml version=\"1.0\" standalone=\"yes\" ?>\n\n");

  fprintf(ods, "<set_info>\n");
  fprintf(ods, "  <set>\n");

  fprintf(ods, "    <set_name>%s</set_name>\n", set_name);

  year  = date_to_process / 10000;
  month = (date_to_process % 10000) / 100;
  day   = date_to_process % 100;

  if (month == 1)
    sprintf(month_string, "January");
  else if (month == 2)
    sprintf(month_string, "February");
  else if (month == 3)
    sprintf(month_string, "March");
  else if (month == 4)
    sprintf(month_string, "April");
  else if (month == 5)
    sprintf(month_string, "May");
  else if (month == 6)
    sprintf(month_string, "June");
  else if (month == 7)
    sprintf(month_string, "July");
  else if (month == 8)
    sprintf(month_string, "August");
  else if (month == 9)
    sprintf(month_string, "September");
  else if (month == 10)
    sprintf(month_string, "October");
  else if (month == 11)
    sprintf(month_string, "November");
  else if (month == 12)
    sprintf(month_string, "December");

  fprintf(ods, "    <description>%d/%02d/%4d</description>\n", month, day, year);
  fprintf(ods, "    <date_string>%s %d, %d</date_string>\n", month_string, day, year);

  fprintf(ods, "    <oldest_data>\n");
  fprintf(ods, "      <date>%d</date>\n", earliest_date);
  fprintf(ods, "      <time>%06d</time>\n", earliest_time);
  fprintf(ods, "    </oldest_data>\n\n");

  fprintf(ods, "    <newest_data>\n");
  fprintf(ods, "      <date>%d</date>\n", latest_date);
  fprintf(ods, "      <time>%06d</time>\n", latest_time);
  fprintf(ods, "    </newest_data>\n\n");

  fprintf(ods, "    <system_name>%s</system_name>\n", system_description);
  fprintf(ods, "  </set>\n");
  fprintf(ods, "</set_info>\n");
  fclose(ods);


  // Create the profile_info.xml file
  
  createProfileInfoFile();


  // Create the byte_check.dat file

  sprintf(ods_file_name, "out.dir/byte_check.dat");

  if ((ods=fopen(ods_file_name,"w")) == NULL)
    {
    printf("\n\nThe ODS byte_check.dat file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }

  n = 79688355;

  if (SWAP_OUT == TRUE)
    n = htonl(n);

  fwrite(&n, 4, 1, ods);

  fclose(ods);

  printf("\nProcessing completed.\n\n");
  }

// end of file
