/*
   Name- IASItoIDDF.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  2/18/2007   Programmer- Mike Pettey (IMSG)

   Function- This program extracts selected data from an RRODF
             and writes the data to an EDGE file.  The selected
             data is defined in the run script.  Also in the run
             script are the starting date (YYYYMMDD) and hour and
             the ending date and hour.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>
#include <dirent.h>

#include "ods_capture.h"

#define  SWAP_IN   FALSE
#define  SWAP_OUT  TRUE


main(int argc, char *argv[])
  {
  DIR     *dir;
  FILE    *out;
  int     i, n, length, i4temp;
  int     date_to_process, granule_number, num_footprints, num_within_granule;
  int     num_parameters, num_values_written, num_written_within_granule;
  int     parameter_number, header_length, block_length;
  int     output_header[20], month, day, year, today;
  short   i2temp;
  long    offset;
  char    c;
  char    file_template[200], full_file_name[500], system_description[256];

  struct capture_information *capture_info;
  struct parameter_info *parameter_list;
  struct parameter_info *ptr;

  time_t  date_time;
  struct  tm *ts;


  // Print an initial message

  printf("\nCopying the granule file to an IDDF\n\n");


  /* Unpack the date of the data to be processed */

  date_to_process = atoi(argv[1]);

  /* Unpack the template to use when looking for files to process */

  sprintf(file_template, "%s\0", argv[2]);

  printf("Date to process:  %d\n", date_to_process);
  printf("Files:            %s\n\n", file_template);


  // Read the name of the input info file from the command line

  capture_info = (struct capture_information*)processCaptureInfo();

  if (capture_info == NULL)
    {
    printf("The program encountered a problem while attempting to read\n");
    printf("the input parameter file. The program cannot continue. Please\n");
    printf("check the file and make sure that it is correct.\n\n");
    exit(1);
    }

  // Read the system description from the command line

  if (argc == 4)
    {
    sprintf(system_description, "%s", argv[3]);
    }
  else
    {
    sprintf(system_description, " ");

    printf("No system name was included as a command line argument.\n");
    printf("The file will be captured but the system name will be blank.\n\n");
    }

  // Copy the file names into the capture_info

  strcpy(capture_info->parm_info_file_name, "fileparm.file");


  // Read the selected parameter info from the parameter file

  ptr = (struct parameter_info*)malloc(sizeof(struct parameter_info));

  if (ptr == 0)
    {
    printf("Memory allocation error\n");
    exit(1);
    }

  parameter_list = (struct parameter_info*)readParameterList(ptr,
							     capture_info->word_list,
							     1333,
							     capture_info->parm_info_file_name);

  ptr = parameter_list;
  parameter_list = parameter_list->next;
  
  //dispose(ptr);


  if (parameter_list == NULL)
    {
    printf("readParameterList() returned NULL. This indicates that there\n");
    printf("was a problem reading in the parameter data from the parameter\n");
    printf("information file. Execution cannot continue.\n\n");
    exit(1);
    }

  // Print the list of parameters and count the number of parameters

  num_parameters = 0;

  printf("The following parameters will be captured:\n\n");
  printf("Word\tDescription\n");
  printf("----\t-----------\n");

  ptr = parameter_list;

  while (ptr != NULL)
    {
    num_parameters++;

    printf("%d\t%s\n", ptr->word_number,ptr->description);

    ptr = ptr->next;
    }

  printf("\n\n");


  // Loop through all of the granules in the input directory and count
  // the number of footprints that fall within the date range

  printf("Counting the number of footprints within the time window...\n");

  num_footprints = 0;

  // Open the current directory

  dir = opendir("in.dir");

  if (! dir)
    {
    printf("\nCannot open the input directory 'in.dir'\n\n");
    exit(1);
    }

  while (1)
    {
    struct dirent * entry;
        
    entry = readdir(dir);

    if (! entry)
      {
      break;
      }

    if (stringStartsWith(entry->d_name, file_template) == TRUE)
      {
      sprintf(full_file_name, "in.dir/%s", entry->d_name);

      num_within_granule = countNucapsFootprints(full_file_name, date_to_process);
      num_footprints = num_footprints + num_within_granule;
      }
    }

  // Close the directory

  if (closedir (dir))
    {
    printf ("\nCould not close the input directory 'in.dir'\n\n");
    exit(1);
    }

  printf ("Number of footprints within the time window:  %d\n", num_footprints);


  // Get today's date

  time (&date_time);         
  ts = localtime(&date_time);

  month  = ts->tm_mon + 1;    
  day    = ts->tm_mday;       
  year   = ts->tm_year;       

  if (year > 56)
    year = 1900 + year;
  else
    year = 2000 + year;

  today = year*10000 + month*100 + day;


  // Create the ODS file and initialize the header
    
  if ((out=fopen("out.file", "w")) == NULL)
    {
    printf("Unable to open the ODS file for output.\n");
    printf("Execution ending prematurely.\n\n");
    exit(1);
    }


  output_header[0]  = 69687169;
  output_header[1]  = 10;
  output_header[2]  = 0;  // data type = sequential

  header_length = 116;
  output_header[3]  = header_length;

  output_header[4]  = num_parameters;

  block_length = 1800 + (num_footprints * 2);
  output_header[5]  = block_length;

  output_header[6]  = num_footprints;
  output_header[7]  = today;
  output_header[8]  = today;
  output_header[9]  = date_to_process;
  output_header[10] = 0;
  output_header[11] = date_to_process;
  output_header[12] = 235959;
  output_header[13] = -18000;
  output_header[14] = 18000;
  output_header[15] = 9000;
  output_header[16] = -9000;
  output_header[17] = -32768;
  output_header[18] = -32768;
  output_header[19] = -32768;

  if (SWAP_OUT == TRUE)
    {
    for (i=0; i<20; i++)
      output_header[i] = htonl(output_header[i]);
    }

  fwrite(&output_header, 80, 1, out);

  length = strlen(system_description);
  if (length >= 36) length=36;

  for (i=0; i<length; i++)
    {
    c = system_description[i];
    fwrite(&c, 1, 1, out);
    }

  c = ' ';
  for (i=length; i<36; i++)
    {
    fwrite(&c, 1, 1, out);
    }


  // Write the block headers for each parameter

  parameter_number = 0;

  ptr = parameter_list;

  while (ptr != NULL)
    {
    offset = header_length + (parameter_number * block_length);
    fseek(out, offset, SEEK_SET);

    // Parameter name

    length = strlen(ptr->description);
    if (length >= 36)  length=36;

    for (i=0; i<length; i++)
      {
      c = ptr->description[i];
      fwrite(&c, 1, 1, out);
      }

    c = ' ';
    for (i=length; i<36; i++)
      {
      fwrite(&c, 1, 1, out);
      }

    // Word number

    i2temp = ptr->word_number;

    if (i2temp < 0)
      i2temp = -1 * i2temp;

    if (SWAP_OUT == TRUE)
      i2temp = htons(i2temp);

    fwrite(&i2temp, 2, 1, out);

    // Visibility flag

    if (ptr->word_number >= 0)
      i2temp = 1;
    else
      i2temp = 0;

    if (SWAP_OUT == TRUE)
      i2temp = htons(i2temp);

    fwrite(&i2temp, 2, 1, out);

    // Spot size

    i2temp = ptr->spot_size;

    if (SWAP_OUT == TRUE)
      i2temp = htons(i2temp);

    fwrite(&i2temp, 2, 1, out);

    // Scaling factor

    i2temp = ptr->scaling_factor;

    if (SWAP_OUT == TRUE)
      i2temp = htons(i2temp);

    fwrite(&i2temp, 2, 1, out);

    // Default minimum

    i4temp = ptr->default_min;

    if (SWAP_OUT == TRUE)
      i4temp = htonl(i4temp);

    fwrite(&i4temp, 4, 1, out);

    // Default maximum

    i4temp = ptr->default_max;

    if (SWAP_OUT == TRUE)
      i4temp = htonl(i4temp);

    fwrite(&i4temp, 4, 1, out);

    // Center point

    i4temp = ptr->center_point;

    if (SWAP_OUT == TRUE)
      i4temp = htonl(i4temp);

    fwrite(&i4temp, 4, 1, out);

    // Pressure

    i2temp = ptr->pressure;

    if (SWAP_OUT == TRUE)
      i2temp = htons(i2temp);

    fwrite(&i2temp, 2, 1, out);

    // Alternate label flag

    i2temp = ptr->alt_label_flag;

    if (SWAP_OUT == TRUE)
      i2temp = htons(i2temp);

    fwrite(&i2temp, 2, 1, out);

    // Unit

    length = strlen(ptr->unit);
    if (length >= 12) length=12;

    for (i=0; i<length; i++)
      {
      c = ptr->unit[i];
      fwrite(&c, 1, 1, out);
      }

    c = ' ';
    for (i=length; i<12; i++)
      {
      fwrite(&c, 1, 1, out);
      }

    // Data type

    i2temp = ptr->data_type;

    if (SWAP_OUT == TRUE)
      i2temp = htons(i2temp);

    fwrite(&i2temp, 2, 1, out);

    // Special values

    for (n=0; n<10; n++)
      {
      i4temp = ptr->individual_value[n];

      if (SWAP_OUT == TRUE)
        i4temp = htonl(i4temp);

      fwrite(&i4temp, 4, 1, out);
      }

    // Special value descriptions

    for (n=0; n<10; n++)
      {
      length = strlen(ptr->individual_description[n]);
      if (length >= 12) length=12;

      for (i=0; i<length; i++)
	{
	c = ptr->individual_description[n][i];
	fwrite(&c, 1, 1, out);
	}

      c = ' ';
      for (i=length; i<12; i++)
	{
        fwrite(&c, 1, 1, out);
	}
      }

    // Color scale type

    i2temp = ptr->scale_type;

    if (SWAP_OUT == TRUE)
      i2temp = htons(i2temp);

    fwrite(&i2temp, 2, 1, out);

    // Red, green and blue color scales

    i2temp = 0;

    if (SWAP_OUT == TRUE)
      i2temp = htons(i2temp);

    for (n=0; n<256; n++)
      fwrite(&i2temp, 2, 1, out);

    for (n=0; n<256; n++)
      fwrite(&i2temp, 2, 1, out);

    for (n=0; n<256; n++)
      fwrite(&i2temp, 2, 1, out);

    // Offset within master file (not used)

    fwrite(&i2temp, 2, 1, out);

    // Spares

    for (n=0; n<13; n++)
      fwrite(&i2temp, 2, 1, out);

    parameter_number++;
    ptr = ptr->next;
    }


  // Loop through all of the granules in the input directory and
  // copy the selected parameters to the ODS file

  printf("\nProcessing the granules...\n");

  granule_number = 0;
  num_values_written = 0;

  // Open the current directory

  dir = opendir("in.dir");

  if (! dir)
    {
    printf("\nCannot open the input directory 'in.dir'\n\n");
    exit(1);
    }

  while (1)
    {
    struct dirent * entry;

    entry = readdir(dir);

    if (! entry)
      {
      break;
      }

    if (stringStartsWith(entry->d_name, file_template) == TRUE)
      {
      sprintf(full_file_name, "in.dir/%s", entry->d_name);

      granule_number++;

      num_written_within_granule = processNucapsGranule(full_file_name, granule_number, date_to_process,
							out, num_values_written, parameter_list,
							header_length, block_length);

      num_values_written = num_values_written + num_written_within_granule;
      }
    }

  // Close the directory

  if (closedir (dir))
    {
    printf ("\nCould not close the input directory 'in.dir'\n\n");
    exit(1);
    }

  printf ("\nNumber of values written to the ODS file:  %d\n", num_values_written);

  // Close the ODS file and quit
  
  fclose(out);

  printf("\nProcessing completed.\n\n");
  }




/* end of file */
