#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <netcdf.h>
#include "ods7_capture.h"

/*
   Name- countNucapsFootprints.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  4/21/2014   Programmer- Mike Pettey (IMSG)
   Version- 2.0    Date-  7/09/2015   Programmer- Mike Pettey (IMSG)

   Function- This program accepts a date value in YYYYMMDD format and checks
             to see if the date is valid. A return value of 1 means the date
             is valid while a return value of 0 means it is not valid.
*/

#define  SWAP_IN   FALSE
#define  SWAP_OUT  TRUE


void copyShortBuffer(char *set_name, char *parm_file_name, int block_num, 
		     short *buffer, int num_used_fovs);

void copyIntegerBuffer(char *set_name, char *parm_file_name, int block_num, 
		       int *buffer, int num_used_fovs);

void copyByteBuffer(char *set_name, char *parm_file_name, int block_num, 
		    char *buffer, int num_used_fovs);


int processNucapsGranule(char *file_name, int granule_number, int date_to_process, char set_name[1000],
                         int num_values_written, struct parameter_info *parameter_list,
                         int header_length, int block_length,
                         int* oldest_date, int* oldest_time, int* latest_date, int* latest_time)
  {
  long   offset;
  short  i2temp;
  int    num_values_written_within_granule, parameter_number;
  int    i, n, index, freq_index, status, fov, ncid, dimid;
  int    num_fovs, num_frequencies;
  int    year, mmdd, hour, mmss, yyyymmdd, hhmmss;
  float  min00z;
  int    old_date, old_time, new_date, new_time;
  int    parm_id;
  int    *use_fov;
  short  *buffer, *short_buffer;
  int    *int_buffer;
  float  *float_buffer;
  unsigned char *byte_buffer;
  double *fov_times;

  size_t  dimlen;

  struct parameter_info *ptr;


  old_date = *oldest_date;
  old_time = *oldest_time;
  new_date = *latest_date;
  new_time = *latest_time;

  if (old_date == -32768)
    old_date = 99999999;

  if (old_time == -32768)
    old_time = 999999;

  if (new_date == -32768)
    new_date = -99999999;

  if (new_time == -32768)
    new_time = -999999;

  printf("Granule:  %s\n", file_name);

  num_values_written_within_granule = 0;

  // Open the input file

  status = nc_open(file_name, 0, &ncid);

  if (status == NC_NOERR)
    {

    // Unpack the dimensions

    status = nc_inq_dimid(ncid, "Number_of_CrIS_FORs", &dimid);
    status = nc_inq_dimlen(ncid, dimid, &dimlen);
    num_fovs = (int)dimlen;

    status = nc_inq_dimid(ncid, "Number_of_CrIS_Frequencies", &dimid);
    status = nc_inq_dimlen(ncid, dimid, &dimlen);
    num_frequencies = (int)dimlen;
  
    //printf("Number of FOVs:         %d\n", num_fovs);
    //printf("Number of frequencies:  %d\n", num_frequencies);


    // Allocate memory for the array that will be used to determine
    // which fovs to use

    use_fov = (int*)malloc(num_fovs*sizeof(int));

    for (fov=0; fov<num_fovs; fov++)
      use_fov[fov] = FALSE;

    // Read the times from the granule

    fov_times = (double*)malloc(num_fovs*sizeof(double));

    status = nc_inq_varid(ncid, "Time", &parm_id);
    status = nc_get_var_double(ncid, parm_id, fov_times);

    // Convert the times for each fov to yyyymmdd and determine whether
    // or not the fov is within the time window

    for (fov=0; fov<num_fovs; fov++)
      {
      convertTime(fov_times[fov], &year, &mmdd, &hour, &mmss);

      yyyymmdd = (year*10000) + mmdd;
      hhmmss   = (hour*10000) + mmss;

      // Process the footprint if it is within the time window

      if ((date_to_process == -32768) || (yyyymmdd == date_to_process))
        {
	use_fov[fov] = TRUE;
	num_values_written_within_granule++;

        if (yyyymmdd < old_date)
	  {
	  old_date = yyyymmdd;
	  old_time = hhmmss;
	  }    
        else if ((yyyymmdd == old_date) && (hhmmss < old_time))
	  {
	  old_date = yyyymmdd;
	  old_time = hhmmss;
	  }    

        if (yyyymmdd > new_date)
	  {
	  new_date = yyyymmdd;
	  new_time = hhmmss;
	  }    
        else if ((yyyymmdd == new_date) && (hhmmss > new_time))
	  {
	  new_date = yyyymmdd;
	  new_time = hhmmss;
	  }
	}
      }

    // Set up the buffer that will hold the values prior to writing

    buffer = (short*)malloc(num_values_written_within_granule * sizeof(short));


    // Process each parameter in the parameter list

    parameter_number = 0;

    ptr = parameter_list;

    while (ptr != NULL)
      {

      // Fill the output buffer with missing and reset the index

      for (n=0; n<num_values_written_within_granule; n++)
        buffer[n] = -32768;

      index = 0;

      // Extract the data from the granule based on the parameter id.
      // The data will be stored in the buffer array.

      // Latitude

      if (ptr->parameter_id == -20000)
        {
        float_buffer = (float*)malloc(num_fovs*sizeof(float));
        short_buffer = (short*)malloc(num_values_written_within_granule*sizeof(short));

        status = nc_inq_varid(ncid, "CrIS_Latitude", &parm_id);
        status = nc_get_var_float(ncid, parm_id, float_buffer);

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (float_buffer[fov] == -9999.0)
	      short_buffer[index] = -32768;
	    else
	      short_buffer[index] = float_buffer[fov] * ptr->scaling_factor;
	    }
	  }

	free(float_buffer);

	copyShortBuffer(set_name, ptr->file_name, granule_number, 
			short_buffer, num_values_written_within_granule);

	free(short_buffer);
	}


      // Longitude

      else if (ptr->parameter_id == -20001)
        {
        float_buffer = (float*)malloc(num_fovs*sizeof(float));
	short_buffer = (short*)malloc(num_values_written_within_granule*sizeof(short));

        status = nc_inq_varid(ncid, "CrIS_Longitude", &parm_id);
        status = nc_get_var_float(ncid, parm_id, float_buffer);

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (float_buffer[fov] == -9999.0)
	      short_buffer[index] = -32768;
	    else
	      short_buffer[index] = float_buffer[fov] * ptr->scaling_factor;
	    }
	  }

	free(float_buffer);

	copyShortBuffer(set_name, ptr->file_name, granule_number, 
			short_buffer, num_values_written_within_granule);

	free(short_buffer);
	}


      // Date

      else if (ptr->parameter_id == -20002)
        {
        int_buffer = (int*)malloc(num_values_written_within_granule*sizeof(int));

        index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (fov_times[fov] == -9999)
	      {
	      int_buffer[index] = -32768;
	      }
	    else
	      {
              convertTime(fov_times[fov], &year, &mmdd, &hour, &mmss);
	      int_buffer[index] = (year * 10000) + mmdd;
	      }
	    }
	  }

	copyIntegerBuffer(set_name, ptr->file_name, granule_number, 
			  int_buffer, num_values_written_within_granule);

	free(int_buffer);
	}


      // Time

      else if (ptr->parameter_id == -20003)
        {
        int_buffer = (int*)malloc(num_values_written_within_granule*sizeof(int));

        index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (fov_times[fov] == -9999)
	      {
	      int_buffer[index] = -32768;
	      }
	    else
	      {
              convertTime(fov_times[fov], &year, &mmdd, &hour, &mmss);
	      int_buffer[index] = (hour * 10000) + mmss;
	      }
	    }
	  }

	copyIntegerBuffer(set_name, ptr->file_name, granule_number, 
			  int_buffer, num_values_written_within_granule);

	free(int_buffer);
	}


      // Minutes since 00z

      else if (ptr->parameter_id == -20008)
        {
        short_buffer = (short*)malloc(num_values_written_within_granule*sizeof(short));

        index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (fov_times[fov] == -9999)
	      {
	      int_buffer[index] = -32768;
	      }
	    else
	      {
              convertTime(fov_times[fov], &year, &mmdd, &hour, &mmss);
	      min00z = (float)(hour * 60) + (float)(mmss/100) + ((float)(mmss%100 / 60.0));
	      short_buffer[index] = min00z * ptr->scaling_factor;
	      }
	    }
	  }

	copyShortBuffer(set_name, ptr->file_name, granule_number, 
			short_buffer, num_values_written_within_granule);

	free(short_buffer);
	}


      // Orbital node

      else if (ptr->parameter_id == -20006)
        {
        short_buffer = (short*)malloc(num_fovs*sizeof(short));
	byte_buffer = (char*)malloc(num_values_written_within_granule*sizeof(char));

        status = nc_inq_varid(ncid, "Ascending_Descending", &parm_id);
        status = nc_get_var_short(ncid, parm_id, short_buffer);

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (short_buffer[fov] == -9999)
              byte_buffer[index] = -1;
	    else
	      byte_buffer[index] = short_buffer[fov] + 1;
	    }
	  }

	free(short_buffer);

	copyByteBuffer(set_name, ptr->file_name, granule_number, 
		       byte_buffer, num_values_written_within_granule);

	free(byte_buffer);
	}


      // Quality flag

      else if (ptr->parameter_id == -20007)
        {
	int_buffer = (int*)malloc(num_fovs*sizeof(int));
	byte_buffer = (char*)malloc(num_values_written_within_granule*sizeof(char));

        status = nc_inq_varid(ncid, "Quality_Flag", &parm_id);
        status = nc_get_var_int(ncid, parm_id, int_buffer);

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (int_buffer[fov] == -9999)
	      byte_buffer[index] = -1;
	    else
	      byte_buffer[index] = int_buffer[fov] * ptr->scaling_factor;

	    //if (int_buffer[fov] == -9999)
            //  {
	    //  byte_buffer[index] = -1;
	    //  }
            //else
	    //  {
	    //  if ((int_buffer[fov] == 0) || (int_buffer[fov] == 1))
	    //		byte_buffer[index] = 0;
	    //else
	    //  byte_buffer[index] = 1;
	    //}
	    }
	  }

	free(int_buffer);

	copyByteBuffer(set_name, ptr->file_name, granule_number, 
		       byte_buffer, num_values_written_within_granule);

	free(byte_buffer);
	}


      // Granule number

      else if (ptr->parameter_id == 1)
        {
	short_buffer = (short*)malloc(num_values_written_within_granule*sizeof(short));

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;
	    short_buffer[index] = granule_number;
	    }
	  }

	copyShortBuffer(set_name, ptr->file_name, granule_number, 
			short_buffer, num_values_written_within_granule);

	free(short_buffer);
	}

      // Scan line

      else if (ptr->parameter_id == 2)
        {
	byte_buffer = (char*)malloc(num_values_written_within_granule*sizeof(char));

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;
	    byte_buffer[index] = (fov / 30) + 1;
	    }
	  }

	copyByteBuffer(set_name, ptr->file_name, granule_number, 
		       byte_buffer, num_values_written_within_granule);

	free(byte_buffer);
	}


      // Field of view

      else if (ptr->parameter_id == 3)
        {
	byte_buffer = (char*)malloc(num_values_written_within_granule*sizeof(char));

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;
	    byte_buffer[index] = fov;
	    }
	  }

	copyByteBuffer(set_name, ptr->file_name, granule_number, 
		       byte_buffer, num_values_written_within_granule);

	free(byte_buffer);
	}


      // Beam position

      else if (ptr->parameter_id == 4)
        {
	byte_buffer = (char*)malloc(num_values_written_within_granule*sizeof(char));

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;
	    byte_buffer[index] = (fov % 30) + 1;
	    }
	  }

	copyByteBuffer(set_name, ptr->file_name, granule_number, 
		       byte_buffer, num_values_written_within_granule);

	free(byte_buffer);
	}


      // Retrieval quality flag

      else if (ptr->parameter_id == 13)
        {
        int_buffer = (int*)malloc(num_fovs*sizeof(int));
	byte_buffer = (char*)malloc(num_values_written_within_granule*sizeof(char));

        status = nc_inq_varid(ncid, "Quality_Flag", &parm_id);
        status = nc_get_var_int(ncid, parm_id, int_buffer);

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (int_buffer[fov] == -9999)
	      byte_buffer[index] = -128;
	    else
	      byte_buffer[index] = int_buffer[fov] * ptr->scaling_factor;
	    }
	  }

	free(int_buffer);

	copyByteBuffer(set_name, ptr->file_name, granule_number, 
		       byte_buffer, num_values_written_within_granule);

	free(byte_buffer);
	}


      // View angle

      else if (ptr->parameter_id == 14)
        {
        float_buffer = (float*)malloc(num_fovs*sizeof(float));
	short_buffer = (short*)malloc(num_values_written_within_granule*sizeof(short));

        status = nc_inq_varid(ncid, "CrIS_View_Angle", &parm_id);
        status = nc_get_var_float(ncid, parm_id, float_buffer);

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (float_buffer[fov] == -9999.0)
	      short_buffer[index] = -32768;
	    else
	      short_buffer[index] = float_buffer[fov] * ptr->scaling_factor;
	    }
	  }

	free(float_buffer);

	copyShortBuffer(set_name, ptr->file_name, granule_number, 
			short_buffer, num_values_written_within_granule);

	free(short_buffer);
	}


      // Satellite height

      else if (ptr->parameter_id == 15)
        {
        float_buffer = (float*)malloc(num_fovs*sizeof(float));
	short_buffer = (short*)malloc(num_values_written_within_granule*sizeof(short));

        status = nc_inq_varid(ncid, "Satellite_Height", &parm_id);
        status = nc_get_var_float(ncid, parm_id, float_buffer);

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (float_buffer[fov] == -9999.0)
	      short_buffer[index] = -32768;
	    else
	      short_buffer[index] = float_buffer[fov] * ptr->scaling_factor;
	    }
	  }

	free(float_buffer);

	copyShortBuffer(set_name, ptr->file_name, granule_number, 
			short_buffer, num_values_written_within_granule);

	free(short_buffer);
	}


      // Solar zenith angle

      else if (ptr->parameter_id == 16)
        {
        float_buffer = (float*)malloc(num_fovs*sizeof(float));
	short_buffer = (short*)malloc(num_values_written_within_granule*sizeof(short));

        status = nc_inq_varid(ncid, "Solar_Zenith", &parm_id);
        status = nc_get_var_float(ncid, parm_id, float_buffer);

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    if (float_buffer[fov] == -9999.0)
	      short_buffer[index] = -32768;
	    else
	      short_buffer[index] = float_buffer[fov] * ptr->scaling_factor;
	    }
	  }

	free(float_buffer);

	copyShortBuffer(set_name, ptr->file_name, granule_number, 
			short_buffer, num_values_written_within_granule);

	free(short_buffer);
	}


      // Radiances

      else if (ptr->parameter_id >= 17)
        {
	float_buffer = (float*)malloc(num_fovs*num_frequencies*sizeof(float));
	short_buffer = (short*)malloc(num_values_written_within_granule*sizeof(short));

        status = nc_inq_varid(ncid, "CrIS_Radiances", &parm_id);
        status = nc_get_var_float(ncid, parm_id, float_buffer);

	index = -1;

        for (fov=0; fov<num_fovs; fov++)
	  {
	  if (use_fov[fov])
	    {
	    index++;

	    freq_index = (fov * num_frequencies) + (ptr->parameter_id - 17);

	    if (float_buffer[freq_index] == -9999.0)
	      short_buffer[index] = -32768;
	    else
	      {
	      short_buffer[index] = float_buffer[freq_index] * ptr->scaling_factor;
//	      if (short_buffer[index] < 0) 
//		{
//printf("NEG:  %d   %f    %f\n", short_buffer[index], float_buffer[freq_index], ptr->scaling_factor);
// exit(0);
//		}
	      }
	    }
	  }

        free(float_buffer);

	copyShortBuffer(set_name, ptr->file_name, granule_number, 
			short_buffer, num_values_written_within_granule);

	free(short_buffer);
	}


      parameter_number++;
      ptr = ptr->next;
      }


    free(use_fov);
    free(fov_times);

    // Close the granule

    status = nc_close(ncid);
    }  // if (status==NC_NOERR...


  // Update the earliest and latest dates and times

  if (old_date != 99999999)
    *oldest_date = old_date;

  if (old_time != 999999)
    *oldest_time = old_time;

  if (new_date != 99999999)
    *latest_date = new_date;

  if (new_time != 999999)
    *latest_time = new_time;

  return num_values_written_within_granule;
  }

/* end of file */
