/*
   Name- Cosmic2StarToCDDF.c

   Language- C     Type- MAIN

   Version- 1.0    Date-  4/23/2021   Programmer- Mike Pettey (IMSG)

   Function- This program extracts selected data from an RRODF
             and writes the data to an EDGE file.  The selected
             data is defined in the run script.  Also in the run
             script are the starting date (YYYYMMDD) and hour and
             the ending date and hour.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <dirent.h>
#include <netcdf.h>

#define  TRUE      0
#define  FALSE     1

#define  SWAP_IN   FALSE
#define  SWAP_OUT  FALSE

#define  CDDF_RECORD_LENGTH  15000
#define  CDDF_HEADER_VALUES  CDDF_RECORD_LENGTH / 4
#define  CDDF_DATA_VALUES    CDDF_RECORD_LENGTH / 2

#define  MAX_WET_LEVELS  200
#define  MAX_ATM_LEVELS  300

#define  DATA_FRAME_WIDTH    10.0
#define  DATA_FRAME_HEIGHT   10.0
//#define  DATA_FRAME_WIDTH    30.0
//#define  DATA_FRAME_HEIGHT   30.0


struct data_frame_files
  {
  char wet_file[1000];
  char atm_file[1000];
  char file_stamp[1000];
  struct data_frame_files *next;
  };


struct data_frame
  {
  int   frame_number;
  float left_lon;
  float right_lon;
  float top_lat;
  float bottom_lat;

  struct data_frame_files *files;

  struct data_frame *next;
  };

struct data_frame *data_frames;



int  output_record;
int  wet_levels, atm_levels, atm_par, atm_vec;
int  date_to_process, oldest_date, newest_date;

float sample_size[200], running_sum[200];

int startsWith(const char *a, const char *b);
void convertTime(float fov_time, int *yr, int *md, int *hr, int *ms);
void julianToDate (int julday, int year, int* month, int*day);

float readAttributeFloatInString(int nc_id, char *name);
float readAttributeFloat(int nc_id, char *name);
int readAttributeIntegerInString(int nc_id, char *name);
int readAttributeInteger(int nc_id, char *name);
char* readAttributeString(int nc_id, char *name);
float* readFloatArray(int ncid, char *name, int num_values);
int* readIntegerArray(int ncid, char *name, int num_values);

float* flipFloatProfile(float *before_data, int num_levels);
int* flipIntegerProfile(int *before_data, int num_levels);
float* thinFloatProfile(float *input_array, int num_before_levels, int num_after_levels);
int* thinIntegerProfile(int *input_array, int num_before_levels, int num_after_levels);

float get100mbPressure(int ncid);
float get100mbLatitude(int ncid);
float get100mbLongitude(int ncid);

int processInputFiles(FILE *out, struct data_frame_files *ptr);

void sortFilesIntoDataFrames(char *input_dir_name, int date_to_process);

void processDataFrame(FILE *out, struct data_frame *ptr);

short float2short(float fvalue);



int main(int argc, char *argv[])
  {
  int     n, length;
  int     year, month, day, yyyymmdd;
  int     output_header[CDDF_DATA_VALUES];
  char    *input_dir_name;

  FILE    *out;
  time_t  date_time;
  struct  tm *ts;

  struct data_frame *ptr;
  struct data_frame_files *file_ptr;

  oldest_date = -32768;
  newest_date = -32768;

  // Print an initial message

  printf("\nCopying the COSMIC-2 STAR data to a CDDF\n\n");


  // Unpack the date of the data to be processed and the
  // name of the data directory where the input files are
  // located

  date_to_process = atoi(argv[1]);
  printf("Processing Date: %d\n", date_to_process);

  length = strlen(argv[2]);
  input_dir_name = (char *)malloc(length+1);
  strcpy(input_dir_name, argv[2]);

  printf("Input File Directory: %s\n\n", input_dir_name);


  for (n=0; n<200; n++)
    {
    sample_size[n] = 0.0;
    running_sum[n] = 0.0;
    }


  // Sort all of the files in the input file directory into 
  // data frame bins

  printf("Sorting the COSMIC-2 granules into data frames...\n");
  sortFilesIntoDataFrames(input_dir_name, date_to_process);


  // Get the current date which will be written to the output
  // file header

  time(&date_time);
  ts = localtime(&date_time);

  month  = ts->tm_mon + 1;
  day    = ts->tm_mday;
  year   = ts->tm_year;

  if (year > 56)
    year = 1900 + year;
  else
    year = 2000 + year;

  yyyymmdd = (year * 10000) + (month * 100) + day;


  // Create and initialize a new CDDF file
  
  if ((out=fopen("out.file", "wb")) == NULL)
    {
    printf("\n\nThe output file could not be opened for output.\n");
    printf("Execution ending.\n\n");
    exit(1);
    }


  for (n=0; n<CDDF_HEADER_VALUES; n++)
    output_header[n] = -32768;

  output_header[0] = 67686870;
  output_header[1] = 10;
  output_header[2] = 1;
  output_header[3] = CDDF_RECORD_LENGTH;
  output_header[4] = CDDF_RECORD_LENGTH;
  output_header[5] = yyyymmdd;
  output_header[6] = -32768;

  if (SWAP_OUT == TRUE)
    {
    for (n=0; n<CDDF_HEADER_VALUES; n++)
      output_header[n] = htonl(output_header[n]);
    }

  fseek(out, 0, SEEK_SET);
  fwrite(&output_header, CDDF_RECORD_LENGTH, 1, out);

  output_record = 1;


  // Loop through and process all of the data frames

  ptr = data_frames;

  while (ptr != NULL)
    {
    //printf("\n------------------------------------------------\n");
    //printf("Processing Data Frame %d\n", ptr->frame_number);
    //printf("   Left longitude:   %.1f\n", ptr->left_lon);
    //printf("   Right longitude:  %.1f\n", ptr->right_lon);
    //printf("   Top latitude:     %.1f\n", ptr->top_lat);
    //printf("   Bottom latitude:  %.1f\n\n", ptr->bottom_lat);

    //file_ptr = ptr->files;

    //while (file_ptr !=  NULL)
    //  {
    //  printf("\n");
    //  printf("   Wet File: %s\n", file_ptr->wet_file);
    //  printf("   Atm File: %s\n", file_ptr->atm_file);

    //  file_ptr = file_ptr->next;
    //  }

    if (ptr->files != NULL)
      processDataFrame(out, ptr);

    ptr = ptr->next;
    }

  // Free the memory used for the input directory name string

  free(input_dir_name);

  // Update the number of records in the file header and close the file

  fseek(out, 8, SEEK_SET);
  fwrite(&output_record, 4, 1, out);

  fclose(out);



  for (n=0; n<200; n++)
    {
    if (sample_size[n] > 0.0)
      printf("Bias:  %d    %f\n", n, (running_sum[n] / sample_size[n]));
    }


  printf("\nProcessing completed.\n\n");
  }




// ===============================================================================

void sortFilesIntoDataFrames(char *input_dir_name, int date_to_process)
  {
  int   n, x, y, index, status, length;
  int   num_frames_x, num_frames_y, data_frame_number;
  int   wet_id, atm_id, dimid;
  int   year, month, day, yyyymmdd;
  float frame_top_lat, frame_bottom_lat, frame_left_lon, frame_right_lon;
  float file_lat, file_lon;
  char  *full_file_name, *attribute_string;
  char  *file_stamp;
  DIR   *dir;
  size_t att_length, dimlen;
  struct data_frame *ptr;
  struct data_frame *frame;
  struct data_frame_files *files;
  struct data_frame_files *file_ptr;
  struct dirent *entry;

  // Set up the data frame tree

  num_frames_x = (360 / DATA_FRAME_WIDTH);
  num_frames_y = (180 / DATA_FRAME_HEIGHT);

  if ((360 % (int)DATA_FRAME_WIDTH) != 0)
    num_frames_x++;

  if ((180 % (int)DATA_FRAME_HEIGHT) != 0)
    num_frames_y++;

  data_frame_number = 0;

  frame_top_lat = 90.0 + DATA_FRAME_HEIGHT;

  for (y=0; y<num_frames_y; y++)
    {
    frame_top_lat    = frame_top_lat - DATA_FRAME_HEIGHT;
    frame_bottom_lat = frame_top_lat - DATA_FRAME_HEIGHT;

    if (frame_bottom_lat <= -90.0)
      frame_bottom_lat = -89.999;

    frame_left_lon = -180.0 - DATA_FRAME_WIDTH;

    for (x=0; x<num_frames_x; x++)
      {
      frame_left_lon  = frame_left_lon + DATA_FRAME_WIDTH;
      frame_right_lon = frame_left_lon + DATA_FRAME_WIDTH;

      if (frame_right_lon >= 180.0)
         frame_right_lon = 179.999;

      frame = (struct data_frame*)malloc(sizeof(struct data_frame));

      data_frame_number++;

      frame->frame_number = data_frame_number;
      frame->left_lon     = frame_left_lon;
      frame->right_lon    = frame_right_lon;
      frame->top_lat      = frame_top_lat;
      frame->bottom_lat   = frame_bottom_lat;
      frame->files        = NULL;
      frame->next         = NULL;

      if (data_frames == NULL)
        data_frames = frame;
      else
        ptr->next = frame;

      ptr = frame;
      }  // for (x=0...
    }  // for (y=0...


  // Loop through all of the wetPrf files in the input directory and
  // sort them into data frames

  dir = opendir(input_dir_name);

  if (dir != NULL)
    {
    while ((entry = readdir(dir)) != NULL)
      {
      if (startsWith(entry->d_name, "wetPrf") == TRUE)
        {

        // Build the fully qualified file name

        length = strlen(input_dir_name) + strlen(entry->d_name) + 3;

        full_file_name = (char *)malloc(length);

        for (n=0; n<length; n++)
          full_file_name[n] = '\0';

        index = -1;

        for (n=0; n<strlen(input_dir_name); n++)
          {
          index++;
          full_file_name[index] = input_dir_name[n];
          }

        index++;

        full_file_name[index] = '/';

        for (n=0; n<strlen(entry->d_name); n++)
          {
          index++;
          full_file_name[index] = entry->d_name[n];
          }

        index++;
        full_file_name[index] = '\0';

        files = NULL;


        // Open the wetPrf file

        status = nc_open(full_file_name, 0, &wet_id);

        if (status == NC_NOERR)
          {

          // Unpack the date from the wet file

          year   = readAttributeInteger(wet_id, "year");
          month  = readAttributeInteger(wet_id, "month");
          day    = readAttributeInteger(wet_id, "day");

          if ((year != -32768) && (month != -327678) && (day != -32768))
            yyyymmdd = (year * 10000) + (month * 100) + day;

          // If the date of the file is the same as the date to process,
          // then continue processing it

          if (yyyymmdd == date_to_process)
            {

            // Unpack the file stamp

            file_stamp = readAttributeString(wet_id, "fileStamp");

            // Create a new data_frame_files and add the wetPrf file name

            files = (struct data_frame_files*)malloc(sizeof(struct data_frame_files));

            for (n=0; n<strlen(full_file_name); n++)
              files->wet_file[n] = full_file_name[n];

            files->wet_file[n] = '\0';

            free(full_file_name);

            files->atm_file[0] = '\0';

            for (n=0; n<strlen(file_stamp); n++)
              files->file_stamp[n] = file_stamp[n];

            files->file_stamp[n] = '\0';

            files->next = NULL;


            // Unpack the latitude and longitude

            nc_inq_dimid(wet_id, "MSL_alt", &dimid);
            nc_inq_dimlen(wet_id, dimid, &dimlen);
            wet_levels = (int)dimlen;

            //file_lat = readAttributeFloatInString(wet_id, "lat");
            //file_lon = readAttributeFloatInString(wet_id, "lon");
            file_lat = get100mbLatitude(wet_id);
            file_lon = get100mbLongitude(wet_id);

            // Search the data frame tree and find a frame that the COSMIC
            // file is within

            ptr = data_frames;

            while (ptr != NULL)
              {
              if ((file_lat <= ptr->top_lat) && (file_lat >= ptr->bottom_lat) &&
                  (file_lon >= ptr->left_lon) && (file_lon <= ptr->right_lon))
                {
                if (ptr->files == NULL)
                  {
                  ptr->files = files;
                  }
                else
                  {
                  file_ptr = ptr->files;

                  while (file_ptr->next != NULL)
                    {
                    file_ptr = file_ptr->next;
                    }

                  file_ptr->next = files;
                  }
                }

              ptr = ptr->next;
              }  // while (ptr != NULL...
            }  // if (yyyymmdd == date_to_process...
          }  // if (status == NC_NOERR (nc_open)...

        // Close the wetPRF file

        status = nc_close(wet_id);
        }  // if (starsWith(entry->d_name...
      }  // while (entry...
    }  // if (dir != NULL...


  // Loop through all of the atmPrf files in the input directory and
  // try to match them to the wetPrf files that were just processed

  dir = opendir(input_dir_name);

  if (dir != NULL)
    {
    while ((entry = readdir(dir)) != NULL)
      {
      if (startsWith(entry->d_name, "atmPrf") == TRUE)
        {

        // Build the fully qualified file name

        length = strlen(input_dir_name) + strlen(entry->d_name) + 3;

        full_file_name = (char *)malloc(length);

        for (n=0; n<length; n++)
          full_file_name[n] = '\0';

        index = -1;

        for (n=0; n<strlen(input_dir_name); n++)
          {
          index++;
          full_file_name[index] = input_dir_name[n];
          }

        index++;

        full_file_name[index] = '/';

        for (n=0; n<strlen(entry->d_name); n++)
          {
          index++;
          full_file_name[index] = entry->d_name[n];
          }

        index++;
        full_file_name[index] = '\0';

        // Open the file

        status = nc_open(full_file_name, 0, &atm_id);

        if (status == NC_NOERR)
          {

          // Unpack the file stamp

          file_stamp = readAttributeString(atm_id, "fileStamp");

          // Search the data frames and look for a file with the
          // same file stamp

          ptr = data_frames;

          while (ptr != NULL)
            {
            file_ptr = ptr->files;

            while (file_ptr !=  NULL)
              {
              if (strcmp(file_stamp, file_ptr->file_stamp) == 0)
                {
                for (n=0; n<strlen(full_file_name); n++)
                  file_ptr->atm_file[n] = full_file_name[n];

                file_ptr->atm_file[n] = '\0';
                }

              file_ptr = file_ptr->next;
              }

            ptr = ptr->next;
            }
          }  // if (status == NC_NOERR [nc_open]...

        // Close the atm file

        status = nc_close(atm_id);

        free(full_file_name);
        }  // if (startsWith(entry->d_name...
      }  // while (entry...
    }  // if (dir != NULL...
  }



// ===============================================================================

void processDataFrame(FILE *out, struct data_frame *frame_ptr)
  {
  int   n, index, status, length;
  int   data_frame_record, num_in_data_frame;
  short data_frame_header[CDDF_DATA_VALUES];
  long  offset;
  struct data_frame_files *file_ptr;

//  printf("\n=============================================================================\n");
//  printf("Processing Data Frame %d\n", frame_ptr->frame_number);
//  printf("   Left longitude:   %.1f\n", frame_ptr->left_lon);
//  printf("   Right longitude:  %.1f\n", frame_ptr->right_lon);
//  printf("   Top latitude:     %.1f\n", frame_ptr->top_lat);
//  printf("   Bottom latitude:  %.1f\n\n", frame_ptr->bottom_lat);


  // Set up a new data frame header

  output_record++;
  data_frame_record = output_record;

  num_in_data_frame = 0;


  // Process the wetPrf and atmPrf files that are within the
  // bounds of this data frame

  file_ptr = frame_ptr->files;

  while (file_ptr != NULL)
    {
    status = processInputFiles(out, file_ptr);

    if (status == TRUE)
      num_in_data_frame++;

    file_ptr = file_ptr->next;
    }  // while (file_ptr != NULL...


  // Write the data frame header to the output file

  for (n=0; n<CDDF_DATA_VALUES; n++)
    data_frame_header[n] = -32768;

  data_frame_header[0]  = 3;

  data_frame_header[1]  = frame_ptr->frame_number;;
  data_frame_header[2]  = num_in_data_frame;

  data_frame_header[4]  = frame_ptr->bottom_lat * 128;
  data_frame_header[5]  = frame_ptr->bottom_lat * 128;
  data_frame_header[6]  = frame_ptr->top_lat * 128;
  data_frame_header[7]  = frame_ptr->top_lat * 128;

  data_frame_header[8]  = frame_ptr->left_lon * 128;
  data_frame_header[9]  = frame_ptr->right_lon * 128;
  data_frame_header[10] = frame_ptr->right_lon * 128;
  data_frame_header[11] = frame_ptr->left_lon * 128;

  if (SWAP_OUT == TRUE)
    {
    for (n=0; n<CDDF_DATA_VALUES; n++)
      data_frame_header[n] = ntohs(data_frame_header[n]);
    }
 
  offset = (long)(data_frame_record - 1) * (long)CDDF_RECORD_LENGTH;
  fseek(out, offset, SEEK_SET);
  fwrite(&data_frame_header, CDDF_RECORD_LENGTH, 1, out);
  }



int processInputFiles(FILE *out, struct data_frame_files *ptr)
  {
  int    files_processed_successfully;
  int    n, status, level, dimid;
  int    file_id, num_levels;
  int    *i_profile_data;
  short  buffer[CDDF_DATA_VALUES];
  long   offset;
  float  fval, wvmr, unscaled;
  float  *profile_data, *pressures, *vapor_pressures;
  char   *string;
  size_t dimlen;


  files_processed_successfully = TRUE;

  // Initialize the output data buffer

  for (n=0; n<CDDF_DATA_VALUES; n++)
    buffer[n] = -32768;

  buffer[0] = 0;

  //printf("   Files:   %s\n", ptr->wet_file);
  //printf("            %s\n\n", ptr->atm_file);

  // Open the wet file

  if (strlen(ptr->wet_file) != 0)
    {
    status = nc_open(ptr->wet_file, 0, &file_id);

    if (status == NC_NOERR)
      {

      // Unpack the dimensions

      status = nc_inq_dimid(file_id, "MSL_alt", &dimid);
      status = nc_inq_dimlen(file_id, dimid, &dimlen);
      num_levels = (int)dimlen;

      // Start to fill in the output data buffer
      // Scene type

      buffer[1] = -32768;

      // Latitude and longitude

      //buffer[3] = readAttributeFloatInString(file_id, "lat") * 128;
      //buffer[4] = readAttributeFloatInString(file_id, "lon") * 128;

      buffer[2] = get100mbPressure(file_id) * 10;
      buffer[3] = get100mbLatitude(file_id) * 128;
      buffer[4] = get100mbLongitude(file_id) * 128;

      // Date and time

      buffer[5] = readAttributeInteger(file_id, "year");
      buffer[6] = (readAttributeInteger(file_id, "month") * 100) +
                  readAttributeInteger(file_id, "day");

      buffer[7] = readAttributeInteger(file_id, "hour");
      buffer[8] = (readAttributeInteger(file_id, "minute") * 100) +
                  (int)readAttributeInteger(file_id, "second");

      // Perigee point latitude and longitude

      buffer[10] = readAttributeFloatInString(file_id, "lat") * 128;
      buffer[11] = readAttributeFloatInString(file_id, "lon") * 128;

      // File stamp

      string = readAttributeString(file_id, "fileStamp");

      if (string != NULL)
        {
        for (n=0; n<30; n++)
          {
          if (n < strlen(string))
            buffer[15+n] = string[n];
          else
            buffer[15+n] = ' ';
          }

        free(string);
        }

      // Wet source code version

      string = readAttributeString(file_id, "version");

      if (string != NULL)
        {
        for (n=0; n<10; n++)
          {
          if (n < strlen(string))
            buffer[45+n] = string[n];
          else
            buffer[45+n] = ' ';
          }

        free(string);
        }

      // First guess source string

      string = readAttributeString(file_id, "fgsUsed");

      if (string != NULL)
        {
        for (n=0; n<30; n++)
          buffer[65+n] = string[n];

        free(string);
        }

      // Overall retrieval quality

      buffer[95] = readAttributeInteger(file_id, "Overall_retrieval_quality");

      // Wet file bad flag

      buffer[96] = readAttributeIntegerInString(file_id, "bad");

      // AtmPrf bad flag

      buffer[98] = readAttributeIntegerInString(file_id, "atmPrf_bad");

      // Terrain

      buffer[99] = readAttributeIntegerInString(file_id, "landmask");

      // ATM values

      //buffer[101] = (short)(readAttributeFloat(file_id, "atmPrf_balmax") * 10000.0);
      buffer[101] = float2short(readAttributeFloat(file_id, "atmPrf_balmax") * 10000.0);

      buffer[102] = readAttributeFloat(file_id, "atmPrf_irs");
      buffer[103] = (short)(readAttributeFloat(file_id, "atmPrf_snr1avg") * 1.0);
      buffer[104] = (short)(readAttributeFloat(file_id, "atmPrf_snr2avg") * 1.0);

      fval = readAttributeFloat(file_id, "atmPrf_stdv");
      if (fval != -32768.0)
        buffer[105] = (short)(log(fval) * 1024.0);
//printf("STDV:  %d      %f\n", buffer[105], fval);

      //buffer[105] = (short)(readAttributeFloat(file_id, "atmPrf_stdv") * 10000000.0);
//printf("STDV:   %d    %f\n", buffer[105], readAttributeFloat(file_id, "atmPrf_stdv"));

      buffer[106] = (short)(readAttributeFloat(file_id, "atmPrf_zbalmax") * 10.0);

      // Derived product pressure

      pressures = readFloatArray(file_id, "Pres", num_levels);
      pressures = flipFloatProfile(pressures, num_levels);
      pressures = thinFloatProfile(pressures, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (pressures[n] != -32768.0)
          {
          if (n < 70)
            buffer[200+n] = (short)(pressures[n] * 1000.0);
          else
            buffer[200+n] = (short)(pressures[n] * 10.0);
          }
        }

      // Derived mean sea level altitude

      profile_data = readFloatArray(file_id, "MSL_alt", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[400+n] = (short)(profile_data[n] * 100.0);
        }

      free(profile_data);

      // Derived product latitude

      profile_data = readFloatArray(file_id, "lat", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[600+n] = (short)(profile_data[n] * 128.0);
        }

      free(profile_data);

      // Derived product longitude

      profile_data = readFloatArray(file_id, "lon", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[800+n] = (short)(profile_data[n] * 128.0);
        }

      free(profile_data);

      // Profile retrieval quality

      i_profile_data = readIntegerArray(file_id, "QC_lev", num_levels);
      i_profile_data = flipIntegerProfile(i_profile_data, num_levels);
      i_profile_data = thinIntegerProfile(i_profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (i_profile_data[n] != -32768)
          buffer[1000+n] = (short)i_profile_data[n];
        }

      free(i_profile_data);

      // Retrieved temperature

      profile_data = readFloatArray(file_id, "Temp", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
	  {
          buffer[1200+n] = float2short((profile_data[n]+273.15) * 64.0);



          //unscaled = buffer[1200+n] / 64.0;
          //sample_size[n] = sample_size[n] + 1.0;
          //running_sum[n] = running_sum[n] + ((profile_data[n]+273.15) - unscaled);

	  //if (n == 170)
	  //  {
	  //    printf("Running:  %f      %f   %f      %f\n", (running_sum[n]/sample_size[n]), (profile_data[n]+273.15), unscaled, ((profile_data[n]+273.15) - unscaled));
	  //  }




	  }
        }

      free(profile_data);

      // First guess temperature

      profile_data = readFloatArray(file_id, "Temp_1gs", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[1400+n] = (short)((profile_data[n]+273.15) * 64.0);
        }

      free(profile_data);

      // Retrieved water vapor pressure

      vapor_pressures = readFloatArray(file_id, "Vp", num_levels);
      vapor_pressures = flipFloatProfile(vapor_pressures, num_levels);
      vapor_pressures = thinFloatProfile(vapor_pressures, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (vapor_pressures[n] != -32768.0)
          buffer[1600+n] = (short)(log(vapor_pressures[n]) * 1024.0);
        }

      // Retrieved water vapor mixing ratio

      for (n=0; n<200; n++)
        {
        if ((pressures[n] != -32768.0) && (vapor_pressures[n] != -32768.0))
          {
          wvmr = 622.0 * vapor_pressures[n] / (pressures[n] - vapor_pressures[n]);
          buffer[2000+n] = float2short(log(wvmr) * 1024.0);


          unscaled = exp(buffer[2000+n] / 1024.0);
          sample_size[n] = sample_size[n] + 1.0;
          running_sum[n] = running_sum[n] + (wvmr - unscaled);

	  if (n == 170)
	  {
	    printf("Running:  %f      %f   %f      %f\n", (running_sum[n]/sample_size[n]), wvmr, unscaled, (wvmr - unscaled));
	      	    }


          }
        else
          buffer[2000+n] = -32768.0;
        }

      free(vapor_pressures);

      // First guess water vapor pressure

      vapor_pressures = readFloatArray(file_id, "Vp_1gs", num_levels);
      vapor_pressures = flipFloatProfile(vapor_pressures, num_levels);
      vapor_pressures = thinFloatProfile(vapor_pressures, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (vapor_pressures[n] != -32768.0)
          buffer[1800+n] = (short)(log(vapor_pressures[n]) * 1024.0);
        }

      // First guess water vapor mixing ratio

      for (n=0; n<200; n++)
        {
        if ((pressures[n] != -32768.0) && (vapor_pressures[n] != -32768.0))
          {
          wvmr = 622.0 * vapor_pressures[n] / (pressures[n] - vapor_pressures[n]);
          buffer[2200+n] = (short)(log(wvmr) * 1024.0);
          }
        else
          buffer[2200+n] = -32768.0;
        }

      free(vapor_pressures);

      // Retrieved relative humidity

      profile_data = readFloatArray(file_id, "rh", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[2400+n] = (short)(profile_data[n] * 100.0);
        }

      free(profile_data);

      // Retrieved specific humidity

      profile_data = readFloatArray(file_id, "sph", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[2600+n] = (short)(log(vapor_pressures[n]) * 1024.0);
        }

      free(profile_data);

      // Retrieved refractivity

      profile_data = readFloatArray(file_id, "ref", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
          {
          if (n < 100)
            buffer[2800+n] = (short)(profile_data[n] * 1000.0);
          else
            buffer[2800+n] = (short)(profile_data[n] * 10.0);
          }
        }

      free(profile_data);

      // Retrieved dry pressure

      profile_data = readFloatArray(file_id, "pres_dry", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[3000+n] = (short)(profile_data[n] * 10.0);
        }

      free(profile_data);

      // Retrieved dry temperature

      profile_data = readFloatArray(file_id, "temp_dry", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 200);

      for (n=0; n<200; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[3200+n] = (short)((profile_data[n]+273.15) * 64.0);
        }

      free(profile_data);
      free(pressures);
      }  // if (status == NC_NOERR (nc_open wet file...

    // Close the wet file

    status = nc_close(file_id);
    }  // if (strlen(ptr->atm_file) != 0)


  // Open the atm file

  if (strlen(ptr->atm_file) != 0)
    {
    status = nc_open(ptr->atm_file, 0, &file_id);

    if (status == NC_NOERR)
      {

      // Unpack the dimensions

      status = nc_inq_dimid(file_id, "Pres", &dimid);
      status = nc_inq_dimlen(file_id, dimid, &dimlen);
      num_levels = (int)dimlen;

      // Atm source code version

      string = readAttributeString(file_id, "version");

      if (string != NULL)
        {
        for (n=0; n<10; n++)
          {
          if (n < strlen(string))
            buffer[56+n] = string[n];
          else
            buffer[56+n] = ' ';
          }

        free(string);
        }


      // Atm file bad flag

      buffer[97] = readAttributeInteger(file_id, "bad");

      // Overall retrieval quality

      buffer[100] = (short)(readAttributeFloat(file_id, "radius of curvature (km)") * 1.0);

      // Dry pressure

      pressures = readFloatArray(file_id, "Pres", num_levels);
      pressures = flipFloatProfile(pressures, num_levels);
      pressures = thinFloatProfile(pressures, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (pressures[n] != -32768.0)
          {
//          if (n < 150)
//            buffer[3400+n] = (short)(pressures[n] * 1000.0);
//          else
//            buffer[3400+n] = (short)(pressures[n] * 10.0);
//printf("DRYPRESS:  %d    %f\n", n, pressures[n]);
          buffer[3400+n] = (short)(log(pressures[n]) * 1024.0);
//fval = exp(buffer[3400+n]/1024.0);
//if (pressures[n] > 5000.0)
//printf("DRYPRESS:   %d     %f     %f     %s\n", n, pressures[n], fval, ptr->atm_file);
          }
        }

//if (1==1) exit(0);
      // Altitude

      profile_data = readFloatArray(file_id, "MSL_alt", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[3700+n] = (short)(profile_data[n] * 100.0);
        }

      free(profile_data);

      // Latitude

      profile_data = readFloatArray(file_id, "Lat", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[4000+n] = (short)(profile_data[n] * 128.0);
        }

      free(profile_data);

      // Longitude

      profile_data = readFloatArray(file_id, "Lon", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[4300+n] = (short)(profile_data[n] * 128.0);
        }

      free(profile_data);

      // Dry temperature

      profile_data = readFloatArray(file_id, "Temp", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[4600+n] = (short)((profile_data[n] + 273.15) * 64.0);
//if ((profile_data[n]+273.15) == 0.0)
//printf("BAD TEMP:  %d    %f    %s\n", n, (profile_data[n]+273.15), ptr->atm_file);
        }

      free(profile_data);

      // Dry refractivity

      profile_data = readFloatArray(file_id, "Ref", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          {
          if (n < 150)
            buffer[4900+n] = (short)(profile_data[n] * 1000.0);
          else
            buffer[4900+n] = (short)(profile_data[n] * 10.0);
          }
        }

      free(profile_data);

      // Azimuth

      profile_data = readFloatArray(file_id, "Azim", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[5200+n] = (short)(profile_data[n] * 10.0);
        }

      free(profile_data);

      // Bending angle

      profile_data = readFloatArray(file_id, "Bend_ang", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[5500+n] = (short)(log(profile_data[n]) * 1024.0);
        }

      free(profile_data);

      // L1 bending angle

      profile_data = readFloatArray(file_id, "Bend_ang1", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[5800+n] = (short)(log(profile_data[n]) * 1024.0);
        }

      free(profile_data);

      // L2 bending angle

      profile_data = readFloatArray(file_id, "Bend_ang2", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[6100+n] = (short)(log(profile_data[n]) * 1024.0);
        }

      free(profile_data);

      // Bending angle confidence

      profile_data = readFloatArray(file_id, "Bend_ang_conf", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[6400+n] = (short)(profile_data[n] * 100.0);
        }

      free(profile_data);

      // Bending angle uncertainty

      profile_data = readFloatArray(file_id, "Bend_ang_stdv", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[6700+n] = (short)(log(profile_data[n]) * 1024.0);
        }

      free(profile_data);

      // Impact height

      profile_data = readFloatArray(file_id, "Impact_height", num_levels);
      profile_data = flipFloatProfile(profile_data, num_levels);
      profile_data = thinFloatProfile(profile_data, num_levels, 300);

      for (n=0; n<300; n++)
        {
        if (profile_data[n] != -32768.0)
          buffer[7000+n] = (short)(profile_data[n] * 100.0);
        }

      free(profile_data);
      free(pressures);
      }  // if (status == NC_NOERR (nc_open atm file...

    // Close the atm file

    status = nc_close(file_id);
    }  // if (strlen(ptr->atm_file) != 0)


  // Write the output buffer to the output file

  output_record++;

  offset = (long)(output_record - 1) * (long)CDDF_RECORD_LENGTH;
  fseek(out, offset, SEEK_SET);

  fwrite(&buffer, CDDF_RECORD_LENGTH, 1, out);

  files_processed_successfully = TRUE;

  return files_processed_successfully;
  }



void convertTime(float fov_time, int *yr, int *md, int *hr, int *ms)
  {
  int     year, month, day, mmdd;
  int     hour, minute, second, mmss;
  int     date;
  float   time, second_of_day, second_of_hour;

  // Break the fov_time into the data and time portions

  date = (int)fov_time;
  time = fov_time - date;


  // Convert the date

  year = 2019;

  julianToDate(date, year, &month, &day);


  // Convert the time

  second_of_day = time * 86400.0;

  hour = (int)(second_of_day / 3600.0);
  time = (second_of_day / 3600.0) - hour;

  second_of_hour = time * 3600.0;

  minute = (int)(second_of_hour / 60.0);
  time = (second_of_hour / 60.0) - minute;

  second = (int)(time * 60.0);

  // If the fov_time is 0, fill the date and time with missing values

  //if (fov_time == 0.0)
  //  {
  //  year = -32768;
  //  mmdd = -32768;
  //  hour = -32768;
  //  mmss = -32768;
  //  }

  mmdd = (month * 100) + day;
  mmss = (minute * 100) + second;

  // Return the values

  *yr = year;
  *md = mmdd;
  *hr = hour;
  *ms = mmss;
  }



void julianToDate (int julday, int year, int* month, int*day)
  {
  int imonth, iday;

  int  days_in_month[13] = {0,31,28,31,30,31,30,31,31,30,31,30,31};

  if (((year%400)==0) ||
      (((year%4)==0) && ((year%100)!=0)))
    {
    days_in_month[2] = 29;
    }


  imonth = 0;
  iday   = julday;

  while ((imonth < 12) && (iday > days_in_month[imonth+1]))
    {
    imonth++;
    iday = iday - days_in_month[imonth];
    }

  imonth++;

  *month = imonth;
  *day   = iday;
  }



int startsWith(const char *a, const char *b)
  {
  if (strncmp(a, b, strlen(b)) == 0) return TRUE;
  return FALSE;
  }




// ===============================================================================

float readAttributeFloatInString(int nc_id, char *attribute_name)
  {
  int    status;
  float  attribute_value;
  char   *attribute_string;
  size_t att_length;

  attribute_value = -32768.0;

  status = nc_inq_attlen(nc_id, NC_GLOBAL, attribute_name, &att_length);

  if (status == NC_NOERR)
    {
    attribute_string = (char *)malloc(att_length+1);

    status = nc_get_att_text(nc_id, NC_GLOBAL, attribute_name, attribute_string);

    if (status == NC_NOERR)
      {
      attribute_string[att_length] = '\0';
      attribute_value = atof(attribute_string);

      if (attribute_value == -999.0)
        attribute_value = -32768.0;

      free(attribute_string);
      }
    }

  return attribute_value;
  }



float readAttributeFloat(int nc_id, char *attribute_name)
  {
  int   status;
  float attribute_value;

  attribute_value = -32768.0;

  status = nc_get_att_float(nc_id, NC_GLOBAL, attribute_name, &attribute_value);

  if (status != NC_NOERR)
    attribute_value = -32768.0;

  if (attribute_value == -999.0)
    attribute_value = -32768.0;

  return attribute_value;
  }



int readAttributeIntegerInString(int nc_id, char *attribute_name)
  {
  int    status;
  int    attribute_value;
  char   *attribute_string;
  size_t att_length;

  attribute_value = -32768;

  status = nc_inq_attlen(nc_id, NC_GLOBAL, attribute_name, &att_length);

  if (status == NC_NOERR)
    {
    attribute_string = (char *)malloc((int)att_length+1);

    status = nc_get_att_text(nc_id, NC_GLOBAL, attribute_name, attribute_string);

    if (status == NC_NOERR)
      {
      attribute_string[att_length] = '\0';
      attribute_value = atoi(attribute_string);

      free(attribute_string);
      }
    }

  return attribute_value;
  }



int readAttributeInteger(int nc_id, char *attribute_name)
  {
  int status, attribute_value;

  attribute_value = -32768;

  status = nc_get_att_int(nc_id, NC_GLOBAL, attribute_name, &attribute_value);

  if (status != NC_NOERR)
    attribute_value = -32768;

  return attribute_value;
  }



char* readAttributeString(int nc_id, char *attribute_name)
  {
  int    status;
  char   *attribute_string;
  size_t att_length;

  status = nc_inq_attlen(nc_id, NC_GLOBAL, attribute_name, &att_length);

  if (status == NC_NOERR)
    {
    attribute_string = (char *)malloc(att_length+1);

    status = nc_get_att_text(nc_id, NC_GLOBAL, attribute_name, attribute_string);

    if (status == NC_NOERR)
      {
      attribute_string[att_length] = '\0';
      }
    }

  return attribute_string;
  }



float* readFloatArray(int nc_id, char *variable_name, int num_values)
  {
  float *array;
  int   n, status, var_id, successful_read;

  successful_read = FALSE;

  // Allocate memory for the array to be read in

  array = (float*)malloc(num_values*sizeof(float));

  // Grab the variable id from the netCDF file

  status = nc_inq_varid(nc_id, variable_name, &var_id);

  if (status == NC_NOERR)
    {

    // Read the variable array

    status = nc_get_var_float(nc_id, var_id, array);

    if (status == NC_NOERR)
      successful_read = TRUE;
    }

  // If the data were not read correctly, fill the array with
  // missing values

  if (successful_read == FALSE)
    {
    for (n=0; n<num_values; n++)
      array[n] = -32768.0;
    }

  for (n=0; n<num_values; n++)
    if (array[n] == -999.0)
      array[n] = -32768.0;

  return array;
  }



int* readIntegerArray(int nc_id, char *variable_name, int num_values)
  {
  int  *array;
  int  n, status, var_id, successful_read;

  successful_read = FALSE;

  // Allocate memory for the array to be read in

  array = (int*)malloc(num_values*sizeof(int));

  // Grab the variable id from the netCDF file

  status = nc_inq_varid(nc_id, variable_name, &var_id);

  if (status == NC_NOERR)
    {

    // Read the variable array

    status = nc_get_var_int(nc_id, var_id, array);

    if (status == NC_NOERR)
      successful_read = TRUE;
    }

  // If the data were not read correctly, fill the array with
  // missing values

  if (successful_read == FALSE)
    {
    for (n=0; n<num_values; n++)
      array[n] = -32768;
    }

  for (n=0; n<num_values; n++)
    if (array[n] == -999)
      array[n] = -32768;

  return array;
  }



float get100mbPressure(int ncid)
  {
  int   n, closest_index_from_100mb;
  float closest_dist_from_100mb;
  float *pressures;
  float press = -32768.0;

  pressures = readFloatArray(ncid, "Pres", wet_levels);

  closest_index_from_100mb = -32768;
  closest_dist_from_100mb = 9999.9;

  for (n=0; n<wet_levels; n++)
    {
    if (fabs(pressures[n]-100.0) < closest_dist_from_100mb)
      {
      closest_dist_from_100mb = fabs(pressures[n] - 100.0);
      closest_index_from_100mb = n;
      }
    }

  if (closest_index_from_100mb != -32768)
    press = pressures[closest_index_from_100mb];

  free(pressures);

  return press;
  }



float get100mbLatitude(int ncid)
  {
  int   n, closest_index_from_100mb;
  float closest_dist_from_100mb;
  float *pressures, *latitudes;
  float lat = -32768.0;

  pressures = readFloatArray(ncid, "Pres", wet_levels);
  latitudes = readFloatArray(ncid, "lat", wet_levels);

  closest_index_from_100mb = -32768;
  closest_dist_from_100mb = 9999.9;

  for (n=0; n<wet_levels; n++)
    {
    if (fabs(pressures[n]-100.0) < closest_dist_from_100mb)
      {
      closest_dist_from_100mb = fabs(pressures[n] - 100.0);
      closest_index_from_100mb = n;
      }
    }

  if (closest_index_from_100mb != -32768)
    lat = latitudes[closest_index_from_100mb];

  free(latitudes);
  free(pressures);

  return lat;
  }



float get100mbLongitude(int ncid)
  {
  int   n, closest_index_from_100mb;
  float closest_dist_from_100mb;
  float *pressures, *longitudes;
  float lon = -32768.0;

  pressures  = readFloatArray(ncid, "Pres", wet_levels);
  longitudes = readFloatArray(ncid, "lon", wet_levels);

  closest_index_from_100mb = -32768;
  closest_dist_from_100mb = 9999.9;

  for (n=0; n<wet_levels; n++)
    {
    if (fabs(pressures[n]-100.0) < closest_dist_from_100mb)
      {
      closest_dist_from_100mb = fabs(pressures[n] - 100.0);
      closest_index_from_100mb = n;
      }
    }

  if (closest_index_from_100mb != -32768)
    lon = longitudes[closest_index_from_100mb];

  free(longitudes);
  free(pressures);

  return lon;
  }



float* flipFloatProfile(float *input_array, int num_levels)
  {
  int   n;
  float *array;

  // Allocate memory for the output array

  array = (float*)malloc(num_levels*sizeof(float));

  // Fill the output array with data from the input array in
  // reverse order

  for (n=0; n<num_levels; n++)
    array[n] = input_array[num_levels-n-1];

  // The input array is not needed anymore

  free(input_array);

  return array;
  }



int* flipIntegerProfile(int *input_array, int num_levels)
  {
  int   n, *array;

  // Allocate memory for the output array

  array = (int*)malloc(num_levels*sizeof(int));

  // Fill the output array with data from the input array in
  // reverse order

  for (n=0; n<num_levels; n++)
    array[n] = input_array[num_levels-n-1];

  // The input array is not needed anymore

  free(input_array);

  return array;
  }



float* thinFloatProfile(float *input_array, int num_before_levels, int num_after_levels)
  {
  int    n, index;
  float  skip_amount, float_index;
  float *array;

  skip_amount = (float)num_before_levels / (float)num_after_levels;


  // Allocate memory for the output array

  array = (float*)malloc(num_after_levels*sizeof(float));

  // Starting at the bottom of the profile, add data to the output array
  // while skipping levels in the input array

  float_index = (float)num_before_levels - 1.0 + skip_amount;

  for (n=(num_after_levels-1); n>=0; n--)
    {
    float_index = float_index - skip_amount;
    index = (int)(float_index + 0.5);

    if (index >= 0)
      array[n] = input_array[index];
    else
      array[n] = -32768.0;
    }

  // The input array is not needed anymore

  free(input_array);

  return array;
  }



int* thinIntegerProfile(int *input_array, int num_before_levels, int num_after_levels)
  {
  int    n, index;
  float  skip_amount, float_index;
  int    *array;

  skip_amount = (float)num_before_levels / (float)num_after_levels;


  // Allocate memory for the output array

  array = (int*)malloc(num_after_levels*sizeof(int));

  // Starting at the bottom of the profile, add data to the output array
  // while skipping levels in the input array

  float_index = (float)num_before_levels - 1.0 + skip_amount;

  for (n=(num_after_levels-1); n>=0; n--)
    {
    float_index = float_index - skip_amount;
    index = (int)(float_index + 0.5);

    if (index >= 0)
      array[n] = input_array[index];
    else
      array[n] = -32768;
    }

  // The input array is not needed anymore

  free(input_array);

  return array;
  }

// end of file
