!----------------------------------------------------------------------------
! This program serves the following:
!
!  - It allows reading several types of atmospheric profiles from 
!    different sources (ECMWF/NOAA/etc.) in their respective formats
!  - It outputs a unique-format file that is compatible with the library
!    standard and therefore understood by the forward simulator as well
!    as by the covariance matrix generation tool.
!  - It generates the emissivities using the Emissivity model using 
!    iSfcTyp (inside EMISS_MW, iSfcTYp controls land/water/ice,etc)
!  - It optionally reads the emissivity from an external emissivity databse.
!  - It optionally reads cld/precip parameteres profiles and interpolates/
!    extrapoltes them to the clear-sky profiles pressure grid.
!  - Merges emissivites with atmospheric profiles into one single 
!    geophysical data file to be fed to the forward simulator and to the
!    covariance matrix generation/EOF decomposition tool.
!
!    Note: There should be no hard-wired information about the set or 
!          about the instrument. Everything is controlled externally 
!          through a namelist.
!
!    Author: Sid-Ahmed Boukabara. IMSG Inc @ NOAA/NESDIS/STAR. July 2005. 
!            Kevin Garrett, RTI @ NOAA/NESDIS/STAR. July 2013. Updated to use
!                           updated MIRS with CRTM 2.1.1
! 
!----------------------------------------------------------------------------
program preProcessGeophData
  !---Use library modules
  USE misc
  USE Consts
  USE utils
  USE IO_Scene
  USE IO_InstrConfig
  USE IO_Misc
  USE ErrorHandling
  USE FwdOperator
  !---Use CRTM-provided modules
  USE CRTM_Module
  USE CRTM_FastemX,        ONLY: Compute_FastemX,iVar_type
  USE MWwaterCoeff_Define, ONLY: MWwaterCoeff_type
  USE CRTM_MWwaterCoeff
  !---Everything explicitly declared
  IMPLICIT NONE
  !---Parameters
  INTEGER, PARAMETER                 :: mxEDRs=20,mxG=1000,len=150
  INTEGER, PARAMETER                 :: MxScanPos=100,nScanLines=9999
  !---Scene Information
  INTEGER                            :: nPrf,nPrf0,nLay,nLev,nAbsorb,nG,nLayIn,nLevIn
  INTEGER                            :: SfcType,sfcTypeCRTM,nqc,iProfProcessed,iProfJumped
  INTEGER                            :: AlgSN=1030
  REAL                               :: PresTopQ,PresBotQ,PresTopT,PresBotT
  REAL,    DIMENSION(:), ALLOCATABLE :: temper_In,WVmixratio_In
  REAL,    DIMENSION(:), POINTER     :: level_p_In,layer_p_In,layer_t_In,layer_w_In
  REAL,    DIMENSION(:), POINTER     :: layer_clw_In,layer_rain_In,layer_snow_In
  REAL,    DIMENSION(:), POINTER     :: layer_ice_In,layer_gh_In,layer_o3_In
  REAL,    DIMENSION(:), POINTER     :: level_p,layer_p,layer_t,layer_w
  REAL,    DIMENSION(:), POINTER     :: layer_clw,layer_rain,layer_snow
  REAL,    DIMENSION(:), POINTER     :: layer_ice,layer_gh,layer_o3
  INTEGER, DIMENSION(:), ALLOCATABLE :: AbsorbID,EDR_Length,EDR_Indx,iSpaceMode
  INTEGER, DIMENSION(:), ALLOCATABLE :: qc
  INTEGER(2), DIMENSION(:), ALLOCATABLE :: qcProf
  CHARACTER(LEN=10), DIMENSION(:), ALLOCATABLE :: EDR_Desc
  REAL                               :: SfcWind_s,SfcT,SfcPress,deltaP,tpw
  REAL                               :: clwScene,gwpScene,rwpScene
  TYPE(Scene_type)                   :: Scene
  !---Positioning Information
  REAL                               :: Lat,Lon,scanUTC
  INTEGER                            :: node,scanDay,scanYear,iscanPos,iscanLine
  !---Cld Info 
  REAL                               :: ptop,Iclw,SWP,IWP,RWP,GWP
  !---Instrument Information
  REAL                               :: Ang,evert,ehorz
  INTEGER                            :: nchan
  TYPE(InstrConfig_type)             :: InstrConfig
  REAL,    DIMENSION(:), ALLOCATABLE :: Emiss,Refl
  INTEGER                            :: ipol
  !---GDAS-specific information
  INTEGER, PARAMETER                       :: nvarSfc=14,nvarAtm=73,nlat=181,nlon=360
  REAL,    PARAMETER                       :: minLatNWP=-90,maxLatNWP=90,minLonNWP=0,maxLonNWP=360
  INTEGER, PARAMETER                       :: nLayNWP=25,nLevNWP=nLayNWP+1
  REAL,    DIMENSION(:,:,:),   ALLOCATABLE :: sfcArr,atmArr
  REAL,    DIMENSION(:),       ALLOCATABLE :: latNWP,lonNWP
  REAL,    DIMENSION(4)              :: timeNWP
  REAL                               :: dLatNWP,dLonNWP,SfcAirT,surfRH,surfMixRatio,xalt,xcover
  INTEGER                            :: indx,ilat,ilon,iSfcTypNWP,stype
  REAL,    DIMENSION(nLevNWP)        :: temper,relHum,WVmixratio
  REAL,    DIMENSION(nLevNWP)        :: level_p_NWP=(/1000.,975.,950.,925.,900.,850.,800.,&
                                           750.,700.,650.,600.,550.,500.,450.,400.,350.,&
                                           300.,250.,200.,150.,100.,70.,50.,30.,20.,10./)
  !---Date and Time variables
  INTEGER                            :: ecmYear,ecmDay,ecmTime,ecmMonth
  INTEGER                            :: nyear,nmonth,nday,nhour
  !---Local variables
  INTEGER  :: ichan,iu_In,iuOut,iuInter,iprof,nemis,iLay,i,nEDRs
  !---Variables used to interface with CRTM components
  TYPE(MWwaterCoeff_type)                                  :: MWwaterCoeff 
  TYPE(iVar_type)                                          :: iVar
  TYPE(CRTM_ChannelInfo_type), DIMENSION(:),   POINTER     :: ChannelInfo
  CHARACTER(STRLEN),           DIMENSION(:),   ALLOCATABLE :: CHAR_SensorID
  INTEGER                                    :: nSensors
  INTEGER                                    :: Error_status
  REAL(fp)                                   :: wind,tskin,Angle
  REAL(fp),       DIMENSION(:), ALLOCATABLE  :: Freq,Emissivity,Reflectivity
  REAL(fp),       DIMENSION(4)               :: Emiss4Stokes,Refl4Stokes
  INTEGER,        DIMENSION(:), ALLOCATABLE  :: channel_Index
  REAL,           DIMENSION(:), ALLOCATABLE  :: Xg
  REAL(fp),       PARAMETER                  :: salinity=33.0 !ppt
  REAL(fp),       PARAMETER                  :: azu=-999.0

  !---Variables for Emissivity from MIRS file
  INTEGER                                    :: iprf,nProfsMIRS,cntout
  INTEGER                                    :: cntSfc,iu_mirsem
  INTEGER                                    :: prof2Use
  INTEGER, DIMENSION(:),   ALLOCATABLE       :: idxSfc
  INTEGER, DIMENSION(:),   ALLOCATABLE       :: m_sfctyp
  REAL                                       :: rn
  REAL,    DIMENSION(:),   ALLOCATABLE       :: m_angle,m_chisq
  REAL,    DIMENSION(:,:), ALLOCATABLE       :: m_emiss
  TYPE(Scene_type)                           :: MIRS
  !---Namelist Data
  CHARACTER(LEN=len)                 :: AtmFileInputNOAA,AtmFileInputECMWF,AtmFileInputMM5,AtmFileInputECMWF60
  CHARACTER(LEN=len)                 :: SfcEmissFile1,SfcEmissFile2,mirsFile4Em,FileOutput,AtmFileInput
  CHARACTER(LEN=len)                 :: InstrConfigFile,InterpTestFile,InstrConfPath,OutPath
  CHARACTER(LEN=len)                 :: sfcNWPfileGDAS,atmNWPfileGDAS,Topogr,PressGridFile,Coeff_Path
  INTEGER                            :: nProfs2process,nProfs60,iprintMonitor,InstrSelect,njump
  INTEGER                            :: iFileType,iSfcTyp,iEmisInfoSrc,iTestInterp,cnt
  INTEGER, DIMENSION(2)              :: AngRange
  REAL                               :: minCLW4profSelection,minRWP4profSelection,minGWP4profSelection
  NAMELIST /PreProcessCntrl/nProfs2process,nProfs60,iSfcTyp,iprintMonitor,iTestInterp,njump, &
       iFileType,AtmFileInputNOAA,AtmFileInputECMWF,AtmFileInputECMWF60,AtmFileInputMM5,     &
       sfcNWPfileGDAS,atmNWPfileGDAS,iEmisInfoSrc,SfcEmissFile1,SfcEmissFile2,mirsFile4Em,   &
       AngRange,InstrConfPath,InstrSelect,OutPath,InterpTestFile,Topogr,PressGridFile,       &
       Coeff_Path,minCLW4profSelection,minRWP4profSelection,minGWP4profSelection

  !---------------------------------------------------------------------------
  !    First Part: Preliminary processing/reading
  !---------------------------------------------------------------------------
  !---Read Control parameters & Instrumental configuration file
  READ(*,NML=PreProcessCntrl)
  CALL DeterminInstrAndOutpFiles(FileOutput,InstrConfigFile,InstrConfPath,InstrSelect,&
       OutPath,iSfcTyp,iFileType)
  CALL ReadInstrConfig(InstrConfigFile,InstrConfig)

  CALL GetSensorInfo(InstrSelect,CHAR_SensorID,nSensors)

  ALLOCATE(ChannelInfo(nSensors))
  Error_Status = CRTM_Init(CHAR_SensorID,ChannelInfo,File_Path=Coeff_Path,MWwaterCoeff_File='FASTEM5.MWwater.EmisCoeff.bin')
  IF (Error_Status .ne. 0) CALL ErrHandl(ErrorType,Err_CRTMneCNTRL,'CRTM Init Failed')

  !---Prepare arrays
  nchan = InstrConfig%nchan
  ALLOCATE(Freq(nchan),Emiss(nchan),Refl(nchan),Emissivity(nchan),Reflectivity(nchan))
  Freq(1:nchan) = InstrConfig%CentrFreq(1:nchan)

  !---------------------------------------------------------------------------
  !   Second Part: Read the pressure grid of the output (fixed grid)
  !---------------------------------------------------------------------------
  CALL ReadPressGridFile(PressGridFile,nlev,nlay,level_p,layer_p)
  !---------------------------------------------------------------------------
  !   Third Part: Read header of atmospheric data file
  !---------------------------------------------------------------------------
  IF (iFileType .eq. 1) THEN  !--NOAA-88 Set
     AtmFileInput=AtmFileInputNOAA
     CALL ReadHdrProfData_noaaSet(AtmFileInputNOAA,iu_In,nPrf,nLayIn,nLevIn,level_p_In,nemis)
  ENDIF
  IF (iFileType .eq. 2) THEN  !--MM5
     AtmFileInput=AtmFileInputMM5
     CALL ReadHdrCldInfo(AtmFileInputMM5,iu_In,nPrf,nLayIn)
  ENDIF
  IF (iFileType .eq. 3) THEN  !--ECMWF
     AtmFileInput=AtmFileInputECMWF
     CALL ReadHdrProfData_ecmwfSet(AtmFileInputECMWF,iu_In,nPrf,nLayIn,nLevIn,level_p_In)
  ENDIF
  IF (iFileType .eq. 4) THEN  !--GDAS gridded analysis
     AtmFileInput    = atmNWPFileGDAS
     nlayIn          = nlayNWP
     nlevIn          = nlevNWP
     nprf            = nlat*nlon
     ALLOCATE(sfcArr(nlat,nlon,nvarSfc),atmArr(nlat,nlon,nvarAtm),latNWP(nlat),lonNWP(nlon),&
          level_p_In(nlevIn))
     level_p_In(1:nlevIn) = level_p_NWP(1:nlevIn)
     call readGDASanalys(sfcNWPFileGDAS,sfcArr(:,:,:),nlat,nlon,nvarSfc)
     call readGDASanalys(atmNWPFileGDAS,atmArr(:,:,:),nlat,nlon,nvarAtm)
     indx=index(sfcNWPFileGDAS,'200',back=.true.)
     read(sfcNWPFileGDAS(indx:indx+3),*)     timeNWP(1)
     read(sfcNWPFileGDAS(indx+5:indx+6),*)   timeNWP(2)
     read(sfcNWPFileGDAS(indx+8:indx+9),*)   timeNWP(3)
     read(sfcNWPFileGDAS(indx+12:indx+13),*) timeNWP(4)
     dLatNWP=(maxLatNWP-minLatNWP)/nLat
     dLonNWP=(maxLonNWP-minLonNWP)/nLon
     latNWP(1:nLat) = minLatNWP +(/(i-1,i=1,nLat)/)*dLatNWP + dLatNWP/2.
     lonNWP(1:nLon) = minLonNWP +(/(i-1,i=1,nLon)/)*dLonNWP + dLonNWP/2.
  ENDIF
  IF (iFileType .eq. 5) THEN  !--ECMWF 60L_SD Dataset
     AtmFileInput = AtmFileInputECMWF60
     Nprf=nProfs60
     nLayIn=nlay
     ALLOCATE(level_p_In(nLayIn+1))
  ENDIF 
  !---------------------------------------------------------------------------
  !   Fourth Part: Allocate arrays, determine #profs 2 process, prepare Output
  !---------------------------------------------------------------------------
  ALLOCATE(layer_p_In(nLayIn),layer_t_In(nLayIn),       &
       layer_w_In(nLayIn),layer_clw_In(nLayIn),layer_rain_In(nLayIn),      &
       layer_snow_In(nLayIn),layer_ice_In(nLayIn),layer_gh_In(nLayIn),     &
       layer_o3_In(nLayIn),temper_In(nLayIn),WVmixratio_In(nLayIn))
  ALLOCATE(layer_t(nLay),layer_w(nLay),layer_clw(nLay),layer_rain(nLay),   &
       layer_snow(nLay),layer_ice(nLay),layer_gh(nLay),layer_o3(nLay),     &
       EDR_Desc(mxEDRs),EDR_Indx(mxEDRs),EDR_Length(mxEDRs),Xg(mxG),       &
       iSpaceMode(mxEDRs))
  iSpaceMode = 0
  !---Determine Number of profiles to really process
  nPrf=minval((/nPrf,nProfs2process/))
  !---Write out header in output file
  nAbsorb  = 2
  nqc      = 4
  ALLOCATE(AbsorbID(nAbsorb),qc(nqc),qcProf(nqc))
  AbsorbID = (/1,3/)
  CALL InitHdrScene(nLev,nLay,nChan,InstrConfig%CentrFreq,                 &
       InstrConfig%polarity,level_p,layer_p,Scene,nLay,nLay,nLay,nLay,nLay,&
       nAbsorb,AbsorbID,nqc,0,MxScanPos,nScanLines,AlgSN)
  CALL WriteHdrScene(iuOut,FileOutput,Scene,nProfs2process)
  IF (iTestInterp.eq.1) THEN
     iuInter = 20
     open(iuInter,file=InterpTestFile,form='unformatted')
  ENDIF
  !---------------------------------------------------------------------------
  !    Fifth Part: Loop over profiles
  !---------------------------------------------------------------------------
  print *, 'Data Source File:',trim(AtmFileInput)
  print *, 'Instrument Configuration File:',trim(InstrConfigFile)
  print *, 'Output File:',trim(FileOutput)
  print *, 'Number of profiles to process:',Nprf
  print *, 'Source of Sfc Data (0:Internal, 1->External):',iEmisInfoSrc
  call SetUpIndex(nLay,nchan,EDR_Desc,EDR_Indx,EDR_Length,nEDRs,nG)
  iProfProcessed = 0
  iProfJumped    = 0
   cnt=0
  IF (iEmisInfoSrc.eq.2) THEN
     CALL ReadHdrScene(iu_mirsem,mirsFile4Em,MIRS,nProfsMIRS)
     ALLOCATE(m_sfctyp(nProfsMIRS),m_angle(nProfsMIRS),m_emiss(nProfsMIRS,nchan), &
          m_chisq(nProfsMIRS))
     SceneFileLoop: DO iprof=1,nProfsMIRS
        CALL ReadScene(iu_mirsem,MIRS,Error_Status)
        IF (Error_status /= 0) EXIT SceneFileLoop
        m_sfctyp(iprof)        = MIRS%iTypSfc
        m_angle(iprof)         = MIRS%angle
        m_emiss(iprof,1:nchan) = MIRS%emiss(1:nchan)
        m_chisq(iprof)         = MIRS%ChiSq
     ENDDO SceneFileLoop
  ENDIF
  cntout=0
  ProfLoop: DO iprof=1,Nprf
     qc(:)=0
     qcProf(:)=0
     !---------------------------------------------------------------------------
     !   Read atmopsheric profiles & surface info
     !---------------------------------------------------------------------------
     IF (iFileType .eq. 1) THEN  !---NOAA set
        CALL ReadProfData_noaaSet(iu_In,nLayIn,SfcType,SfcWind_s,SfcT,        &
             SfcPress,level_p_In,layer_p_In,layer_t_In,layer_w_In,         &
             layer_clw_In,layer_rain_In,layer_snow_In,layer_ice_In,        &
             layer_gh_In,layer_o3_In,nemis,lat,lon,nyear,nmonth,nday,nhour)
     ENDIF
     IF (iFileType .eq. 2) THEN  !---MM5
        CALL ReadCldInfo(iu_In,nLayIn,layer_p_In,layer_t_In,layer_w_In,       &
             layer_clw_In,layer_rain_In,layer_ice_In,layer_snow_In,        &
             layer_gh_In,layer_o3_In,SfcPress,ptop,SfcT,SfcAirT,SfcType,   &
             SfcWind_s)
        lat=DEFAULT_VALUE_REAL
        lon=DEFAULT_VALUE_REAL
     ENDIF
     IF (iFileType .eq. 3) THEN  !---ECMWF
        CALL ReadProfData_ecmwfSet(iu_In,nLayIn,SfcType,SfcWind_s,SfcT,       &
             SfcPress,level_p_In,layer_p_In,layer_t_In,layer_w_In,         &
             layer_clw_In,layer_rain_In,layer_snow_In,layer_ice_In,        &
             layer_gh_In,layer_o3_In,lat,lon)
     ENDIF
     IF (iFileType .eq. 4) THEN  !---GDAS
        ilat=int(real(iprof-1)/real(nlon))+1
        ilon=iprof-((ilat-1)*nlon)
        !---Extract NWP info 
        SfcT                    = sfcArr(ilat,ilon,1)
        SfcAirT                 = sfcArr(ilat,ilon,11)
        SurfRH                  = sfcArr(ilat,ilon,13)
        SfcPress                = sfcArr(ilat,ilon,14)/100.
        surfMixRatio            = RelHum_to_mixingratio(SurfRH/100.,SfcT,SfcPress)
        SfcWind_s               = sqrt(sfcArr(ilat,ilon,9)**2+sfcArr(ilat,ilon,10)**2)
        iSfcTypNWP              = sfcArr(ilat,ilon,8)
        temper(1:nLevIn)        = atmArr(ilat,ilon,nLevIn+1:2*nLevIn)
        relHum(1:nLevIn-5)      = atmArr(ilat,ilon,2*nLevIn+1:3*nLevIn-5)
        relHum(nLevIn-4:nLevIn) = relHum(nLevIn-5) 
        SfcType                 = -99
        lat                    = latNWP(ilat)
        lon                    = lonNWP(ilon)
        call Read_topography(Topogr,lat,lon,xalt,SfcType,xcover,sfcTypeCRTM)
        !---Converting the relative himidity into mixing ratio
        do i=1,nLevIn
           WVmixratio(i)    = RelHum_to_mixingratio(RelHum(i)/100.,Temper(i),level_p_In(i))
        enddo
        !---Compute layer values
        do i=1,nLayIn
           layer_p_In(i)     = (level_p_In(i)+level_p_In(i+1))/2.
!           layer_t_In(i)     = (temper(i)+temper(i+1))/2.
!           layer_w_In(i)     = (WVmixratio(i)+WVmixratio(i+1))/2.
           layer_clw_In(i)   = 0.
           layer_rain_In(i)  = 0.
           layer_snow_In(i)  = 0.
           layer_ice_In(i)   = 0.
           layer_gh_In(i)    = 0.
           layer_o3_In(i)    = 0.
        enddo
        !---Reverse order
        layer_p_In(1:nLayIn) = layer_p_In(nLayIn:1:-1)
        temper_In(1:nLayIn)  = temper(nLayIn:1:-1)
        WVmixratio_In(1:nLayIn) = WVmixratio(nLayIn:1:-1)

!        layer_t_In(1:nLayIn) = layer_t_In(nLayIn:1:-1)
!        layer_w_In(1:nLayIn) = layer_w_In(nLayIn:1:-1)
        !---Put longitude on -180 to 180 scale
        IF (lon .gt. 180) lon=lon-360. 
     ENDIF
     IF (iFileType .eq. 5) THEN  !--ECMWF 60L_SD Dataset
        CALL ReadProfData_ecmwf60Set(AtmFileInputECMWF60,iu_In,iProf,nLayIn,SfcType,SfcWind_s,SfcT, &
             SfcPress,level_p,layer_p_In,layer_p_In,layer_t_In,layer_w_In,layer_clw_In,     &
             layer_rain_In,layer_snow_In,layer_ice_In,layer_gh_In,layer_o3_In,lat,lon,      &
             ecmyear,ecmday,ecmmonth,ecmtime,iprof)
         IF (lon .gt. 180) lon=lon-360
     ENDIF
     !---------------------------------------------------------------------------
     !   Jump profiles to sample input file or select right surface type
     !---------------------------------------------------------------------------
     iProfJumped    = iProfJumped+1
     IF (iProfJumped .le. njump) CYCLE profLoop
     IF (iProfJumped .gt. njump) iProfJumped=0
     IF (iSfcTyp .eq. OC_TYP) THEN 
        IF (sfcType .ne. OC_TYP) CYCLE ProfLoop
        IF (SfcT    .le. 272.)   CYCLE ProfLoop
     ENDIF
     IF (iSfcTyp .eq. LD_TYP) THEN
        IF (sfcType .ne. LD_TYP) CYCLE ProfLoop
        IF (SfcT    .le. 272.)   CYCLE ProfLoop
     ENDIF
     IF (iSfcTyp .eq. SEAICE_TYP) THEN
        IF (sfcType .ne. OC_TYP) CYCLE ProfLoop
        IF (SfcT    .gt. 272.)   CYCLE ProfLoop
     ENDIF
     IF (iSfcTyp .eq. SNOW_TYP) THEN
        IF (sfcType .ne. LD_TYP) CYCLE ProfLoop
        IF (SfcT    .gt. 272.)   CYCLE ProfLoop
     ENDIF
     IF (iSfcTyp .eq. COAST_TYP) THEN
        IF (sfcType .ne. COAST_TYP) CYCLE ProfLoop
        IF (SfcT    .lt. 273.)   CYCLE ProfLoop
     ENDIF
     !---------------------------------------------------------------------------
     !   Interpolate/Extrapolate to the 'official' vertical pressure grid 
     !---------------------------------------------------------------------------
     IF (iFileType .le. 3) THEN
        CALL LINT (layer_p_In,layer_t_In,nLayIn,nLayIn,nLay,layer_p,layer_t)
        CALL LINT (layer_p_In,layer_w_In,nLayIn,nLayIn,nLay,layer_p,layer_w)
        CALL LINT (layer_p_In,layer_clw_In,nLayIn,nLayIn,nLay,layer_p,layer_clw)
        CALL LINT (layer_p_In,layer_rain_In,nLayIn,nLayIn,nLay,layer_p,layer_rain)
        CALL LINT (layer_p_In,layer_Snow_In,nLayIn,nLayIn,nLay,layer_p,layer_snow)
        CALL LINT (layer_p_In,layer_Ice_In,nLayIn,nLayIn,nLay,layer_p,layer_ice)
        CALL LINT (layer_p_In,layer_gh_In,nLayIn,nLayIn,nLay,layer_p,layer_gh)     
     ENDIF
     IF (iFIleType .eq. 4) THEN
        CALL intrp2LevelGrid(temper_In,level_p_In(nLevIn:1:-1),nLev,level_p,PresBotT,PresTopT,layer_p,layer_t,qc(1))
        CALL intrp2LevelGrid(WVmixratio_In,level_p_In(nLevIn:1:-1),nLev,level_p,PresBotQ,PresTopQ,layer_p,layer_w,qc(2))
        layer_clw(:)  = 0.
        layer_rain(:) = 0.
        layer_ice(:)  = 0.
        layer_snow(:) = 0.
        layer_gh(:)   = 0.
        IF ((isfcTyp .eq. OC_TYP .or. iSfcTyp .eq. SEAICE_TYP) .and. (PresBotT .lt. 900 .or. PresBotQ .lt. 900)) CYCLE ProfLoop
        IF (isfcTyp .eq. SEAICE_TYP .and. abs(lat) .lt. 50) CYCLE ProfLoop
     ENDIF

     IF (iFileType .eq. 5) THEN
        layer_t    = layer_t_In
        layer_w    = layer_w_In
        layer_clw  = layer_clw_In
        layer_rain = layer_rain_In
        layer_snow = layer_snow_In
        layer_ice  = layer_ice_In
        layer_gh   = layer_gh_In
     ENDIF

     !---------------------------------------------------------------------------
     !   Unrealistic extrapolated high-values of upper-atmopsh. WV are set to 0
     !---------------------------------------------------------------------------
!    do i=1,nLay
!        IF (layer_p(i).lt.maxval((/layer_p_In(1),100./))) layer_w(i)=0.
!     enddo
     !---------------------------------------------------------------------------
     !   Conversion of hydrometeors profiles from g/Kg to kg/m2 (mm) & integrate
     !---------------------------------------------------------------------------
     Do ilay=1,nLay
        deltaP           = level_p(ilay+1)-level_p(ilay)
        layer_clw(ilay)  = deltaP*(layer_clw(ilay))/10.0/9.8
        layer_rain(ilay) = deltaP*(layer_rain(ilay))/10.0/9.8
        layer_snow(ilay) = deltaP*(layer_snow(ilay))/10.0/9.8
        layer_ice(ilay)  = deltaP*(layer_ice(ilay))/10.0/9.8
        layer_gh(ilay)   = deltaP*(layer_gh(ilay))/10.0/9.8
     ENDDO
     !----Exclude those profiles that do not fit within the required cld amounts
     iclw=ColumIntegr(nLay,layer_p(1:nLay),SfcPress,layer_clw(1:nLay))
     gwp=ColumIntegr(nLay,layer_p(1:nLay),SfcPress,layer_gh(1:nLay))
     rwp=ColumIntegr(nLay,layer_p(1:nLay),SfcPress,layer_rain(1:nLay))
     swp=ColumIntegr(nLay,layer_p(1:nLay),SfcPress,layer_snow(1:nLay))
     iwp=ColumIntegr(nLay,layer_p(1:nLay),SfcPress,layer_ice(1:nLay))
     IF (rwp  .lt. minRWP4profSelection) CYCLE ProfLoop
     IF (gwp  .lt. minGWP4profSelection) CYCLE ProfLoop
     IF (iclw .lt. minCLW4profSelection) CYCLE ProfLoop
     iProfProcessed = iProfProcessed+1
     !---------------------------------------------------------------------------
     !   Output for testing the accuracy of the interpolation/on the screen
     !---------------------------------------------------------------------------
     IF (iTestInterp.eq.1) THEN 
        CALL Out4InterpolTest(iuInter,nPrf,iProfProcessed,nLayIn,nLay,layer_p_In,  &
             layer_p,layer_t_In,layer_t,layer_w_In,layer_w,layer_clw_In,layer_clw, &
             layer_rain_In,layer_rain,layer_snow_In,layer_snow,layer_ice_In,       &
             layer_ice,layer_gh_In,layer_gh)
     ENDIF
     !---------------------------------------------------------------------------
     !   Compute the emissivity
     !---------------------------------------------------------------------------
     ang=0

     call random_number(ang)
     !ang   = InstrConfig%minAng + (ang)*(InstrConfig%MaxAng-InstrConfig%MinAng)
     ang   = 0.5
     wind  = SfcWind_s
     IF (iEmisInfoSrc.eq.0 .and. sfcType .eq. OC_TYP) THEN    !sfc emissivity from the atm file
        tskin = sfcT
        Angle = Ang
        !-----FASTEM-1
        !call EMISS_MW(nchan,nchan,channel_Index,InstrConfig%polarity,freq,    &
        !     MAP_TYP_2_CRTM(iSfcTyp),Angle,wind,tskin,Emissivity,Reflectivity)
        !Emiss(1:nchan) = Emissivity(1:nchan)
        !!refl(1:nchan)  = Reflectivity(1:nchan)
        !refl(1:nchan)  = 1.-Emiss(1:nchan)

        !-----FASTEM-5
        !For now, FASTEM5 is seg-faulting, so using fixed value.
        DO ichan=1,InstrConfig%nchan
 !          CALL Fastem3_OCeanEM(freq(ichan),Angle,0._fp_kind,tskin,wind,0._fp_kind,1._fp_kind, & 
 !               3,EmissCRTM4Stokes,ReflCRTM4Stokes)  

!           CALL Compute_FastemX(MWwaterCoeff,freq(ichan),angle,tskin,salinity,wind,ivar,Emiss4Stokes,Refl4Stokes)
!           evert             = Emiss4Stokes(1)
!           ehorz             = Emiss4Stokes(2)
!           ipol              = InstrConfig%polarity(ichan)
!           Emissivity(ichan) = composeEmiss(ipol,evert,ehorz,real(ang)) 
           Emissivity(ichan)=0.45
        ENDDO
        Emiss(1:nchan) = Emissivity(1:nchan)
     ENDIF

     IF (iEmisInfoSrc.eq.1) THEN !sfc emissivity from the emissiv file(s)
        call EMISS_LD_KARBOU_ATLAS(nchan,InstrConfig%polarity,    &
             InstrConfig%CentrFreq,iSfcTyp,Ang,SfcT,Emiss,&
             SfcEmissFile1,SfcEmissFile2)
     ENDIF
     IF (iEmisInfoSrc.eq.2) THEN !sfc emissivity from MIRS scene file
        cntSfc = COUNT(m_sfctyp .eq. iSfcTyp .and. m_angle .ge. AngRange(1) .and. m_angle .le. AngRange(2) &
             .and. m_chisq .le. 5)
        IF (cntSfc .lt. 2) THEN
           print *,'STOP: Not enough scenes for emissivity'
           STOP
        ELSE
           ALLOCATE(idxSfc(cntSfc))
           idxSfc = PACK( (/(i,i=1,SIZE(m_sfctyp))/),&
                (m_sfctyp .eq. iSfcTyp .and. m_angle .ge. AngRange(1) .and. m_angle .le. AngRange(2) &
                .and. m_chisq .le. 5))
           !---Assign Angle, Emiss
           call random_number(rn)
           prof2Use = int(rn*cntSfc)
           Ang            = m_angle(idxSfc(prof2Use))
           Emiss(1:nchan) = m_emiss(idxSfc(prof2Use),1:nchan)
           tskin = sfcT
           DEALLOCATE(idxSfc)
        ENDIF
     ENDIF
     IF (ang .lt. 0) CYCLE ProfLoop
     !---------------------------------------------------------------------------
     !   Set-up and Output Scene
     !---------------------------------------------------------------------------
     call PutTogetherXg(Xg(1:nG),layer_t,layer_w,layer_o3,layer_Clw,       &
          layer_Rain,layer_Snow,layer_Ice,layer_gh,iProf,Emiss,Refl,       &
          SfcWind_s,real(tskin),EDR_Desc(1:nEDRs),EDR_Indx(1:nEDRs),       &
          EDR_Length(1:nEDRs))
     !---Set up the Scene structure
     node      = 0
     scanDay   = DEFAULT_VALUE_INT
     scanYear  = DEFAULT_VALUE_INT
     scanUTC   = DEFAULT_VALUE_INT
     IF (iFileType .eq. 5) THEN
        CALL compJulDay(ecmYear,ecmMonth,ecmDay,scanDay)
        scanYear  = ecmYear
        scanUTC   = ecmTime
     ENDIF
     IF (iFileType .eq. 1) THEN
        nYear=1988
        CALL compJulDay(nYear,nMonth,nDay,scanDay)
        scanYear  = nYear
        scanUTC   = nhour
     ENDIF
     qc(3)     = 0
     qc(4)     = 0
     iscanPos  = 1
     iscanLine = 1
     Scene%ih2o                = 1
     Scene%io3                 = 2
     Scene%lat                 = lat
     Scene%lon                 = lon
     Scene%tskin               = real(tskin)
     Scene%Emiss               = Emiss(1:nchan)
     Scene%Refl                = 1.-Emiss(1:nchan)
     Scene%qc(1) = qc(1)
     Scene%qc(2) = qc(2)
     Scene%qc(3) = qc(3)
     Scene%qc(4) = qc(4)

     call SetUpScene(level_p,layer_p,ang,Scene,iSfcTyp,iProf,     &
          lat,lon,node,scanDAY,scanYear,scanUTC,iscanPos,iscanLine,0.,0.)
     Scene%SfcPress=SfcPress
     call TransfXg2Scene(Xg(1:nG),Scene,EDR_Desc,EDR_Indx,EDR_length,iSpaceMode)

     !---------------------------------------------------------------------------
     !   Screen printing
     !---------------------------------------------------------------------------
     IF (iprintMonitor .eq. 1) THEN
        cntout = cntout+1
        print *,'cntout',cntout
        CALL ComputeTPW(level_p,SfcPress,layer_w,tpw)
        write(*,'(i8,9(a6,f7.2),20f7.2)') iprof,' Iclw:',Iclw, 'RWP:',RWP,' IWP:',IWP,&
             ' GWP:',GWP,' SWP:',SWP,' TPW:',TPW,' SfcP:',SfcPress,' TSK:',SfcT,' EM:',Emiss(1)
     ENDIF

     !do i=1,Scene%nlay
     !   print *,iprof,i,Scene%GRAUPEL(i),Scene%RAIN(i)
     !enddo
     !---Output results

     CALL WriteScene(iuOut,Scene)
  ENDDO ProfLoop
  Print *, '----------------------------------------'
  Print *, 'Profiles processed:',iProfProcessed
  close(iuOut)
  close(iu_In)

CONTAINS

  SUBROUTINE DeterminInstrAndOutpFiles(FileOutput,InstrConfigFile,InstrConfPath,InstrSelect,&
       OutPath,iSfcTyp,iFileType)
    CHARACTER(LEN=*) :: FileOutput,InstrConfigFile,InstrConfPath,OutPath
    INTEGER          :: InstrSelect,iSfcTyp,iFileType
    CHARACTER        :: extSfcTyp*2,extSrc*10

    IF (iSfcTyp .eq. OC_TYP)     extSfcTyp='oc'
    IF (iSfcTyp .eq. LD_TYP)     extSfcTyp='ld'
    IF (iSfcTyp .eq. SEAICE_TYP) extSfcTyp='ic'
    IF (iSfcTyp .eq. SNOW_TYP)   extSfcTyp='sn'
    IF (iSfcTyp .eq. COAST_TYP)  extSfcTyp='cst'
    IF (iSfcTyp .ne. SNOW_TYP .and. iSfcTyp .ne. SEAICE_TYP .and. iSfcTyp .ne. LD_TYP .and. iSfcTyp .ne. OC_TYP .and. &
         iSfcTyp .ne. COAST_TYP) extSfcTyp='xx'

    IF (iFileType .eq. 1) extSrc='noaa88'
    IF (iFileType .eq. 2) extSrc='mm5'
    IF (iFileType .eq. 3) extSrc='ecmwf'
    IF (iFileType .eq. 4) extSrc='gdas'
    IF (iFileType .eq. 5) extSrc='ecmwf60'

    IF (InstrSelect .eq. SENSOR_ID_N18) THEN
       !---AMSU-A/MHS (NOAA-18)
       FileOutput	= trim(OutPath)//'Scene_'//extSfcTyp//'_n18_amsua_mhs_'//trim(extSrc)//'.dat'
       InstrConfigFile  = trim(InstrConfPath)//'InstrConfig_n18_amsua_mhs.dat'
    ENDIF
    IF (InstrSelect .eq. SENSOR_ID_METOPA) THEN
       !---METOp AMSU/MHS
       FileOutput	= trim(OutPath)//'Scene_'//extSfcTyp//'_metopA_amsu_mhs_'//trim(extSrc)//'.dat'
       InstrConfigFile  = trim(InstrConfPath)//'InstrConfig_metopA_amsua_mhs.dat'
    ENDIF
    IF (InstrSelect .eq. SENSOR_ID_F16) THEN
       !---SSMI/S
       FileOutput	= trim(OutPath)//'Scene_'//extSfcTyp//'_f16_ssmis_'//trim(extSrc)//'.dat'
       InstrConfigFile  = trim(InstrConfPath)//'InstrConfig_f16_ssmis.dat'
    ENDIF
    !IF (InstrSelect .eq. SENSOR_ID_WINDSAT) THEN
    !   !---WINDSAT
    !   FileOutput	= trim(OutPath)//'Scene_'//extSfcTyp//'_coriolis_windsat_'//trim(extSrc)//'.dat'
    !   InstrConfigFile  = trim(InstrConfPath)//'InstrConfig_coriolis_windsat.dat'
    !ENDIF
    IF (InstrSelect .eq. SENSOR_ID_AMSRE) THEN
       !---AQUA/AMSR-E
       FileOutput	= trim(OutPath)//'Scene_'//extSfcTyp//'_aqua_amsre_'//trim(extSrc)//'.dat'
       InstrConfigFile  = trim(InstrConfPath)//'InstrConfig_aqua_amsre.dat'
    ENDIF
    IF (InstrSelect .eq. SENSOR_ID_NPP) THEN
       !---ATMS
       FileOutput	= trim(OutPath)//'Scene_'//extSfcTyp//'_npp_atms_'//trim(extSrc)//'.dat'
       InstrConfigFile  = trim(InstrConfPath)//'InstrConfig_npp_atms.dat'
    ENDIF
    IF (InstrSelect .eq. SENSOR_ID_FY3RI) THEN
       !---FY3RI
       FileOutput	= trim(OutPath)//'Scene_'//extSfcTyp//'_fy3ri_mwri_'//trim(extSrc)//'.dat'
       InstrConfigFile  = trim(InstrConfPath)//'InstrConfig_fy3ri_mwri.dat'
    ENDIF
    IF (InstrSelect .eq. SENSOR_ID_MTSA) THEN
       !---MT_SAPHIR
       FileOutput	= trim(OutPath)//'Scene_'//extSfcTyp//'_mtsa_'//trim(extSrc)//'.dat'
       InstrConfigFile  = trim(InstrConfPath)//'InstrConfig_mtsa_saphir.dat'
    ENDIF


  END SUBROUTINE DeterminInstrAndOutpFiles

  SUBROUTINE EMISS_LD_KARBOU_ATLAS(nchan,pol,freq,iSfcTyp,Angle,tskin,Emiss,&
       SfcEmissFile1,SfcEmissFile2)
    !------------------------------------------------------------------------
    !> > 1    Crops, Mixed Farming
    !> > 2    Short Grass
    !> > 3    Evergreen Needleleaf Trees
    !> > 4    Deciduous Needleleaf Tree
    !> > 5    Deciduous Broadleaf Trees
    !> > 6    Evergreen Broadleaf Trees
    !> > 7    Tall Grass
    !> > 8    Desert
    !> > 9    Tundra
    !> > 10    Irrigated Crops
    !> > 11    Semi-desert
    !> > 12    Ice Caps and Glaciers
    !> > 13    Bogs and Marshes
    !> > 14    Inland Water
    !> > 15    ocean
    !> > 16    Evergreen Shrubs
    !> > 17    Deciduous Shrubs
    !> > 18    Mixed Forest
    !> > 19    Interrupted Forest
    !> > 20    Water and Land Mixtures
    !------------------------------------------------------------------------
    INTEGER, PARAMETER            :: ncoef=6,mxchan=4,mxClass=20,mxPts=100000
    INTEGER                       :: nchan,iSfcTyp
    REAL                          :: Angle,tskin
    REAL,            DIMENSION(:) :: freq,Emiss
    INTEGER,         DIMENSION(:) :: pol
    CHARACTER(LEN=*)              :: SfcEmissFile1,SfcEmissFile2
    !---Local variables
    INTEGER, SAVE                                  :: INIT=0
    INTEGER                                        :: iu_coef,iu_map,iSfcClass,Ichan,iprof
    REAL                                           :: rand
    REAL,          DIMENSION(12)                   :: coef
    REAL,    SAVE, DIMENSION(ncoef,mxClass,mxchan) :: coefTot=0.
    REAL,    SAVE, DIMENSION(mxPts,mxClass,mxchan) :: MeanEmisNadir=0.,StdvEmisNadir=0.
    REAL,    SAVE, DIMENSION(mxPts,mxClass)        :: lat=0.,lon=0.
    REAL,    SAVE, DIMENSION(mxchan)               :: AtlasFreq=(/23.8,31.4,50.3,89./)
    INTEGER, SAVE, DIMENSION(mxPts,mxClass)        :: nAvgPts
    INTEGER, SAVE, DIMENSION(mxClass)              :: iPts
    INTEGER, DIMENSION(nchan)                      :: mapAtlas
    
    !---Consistency checks 
    IF (iSfcTyp.eq.OC_TYP) STOP'Choice of SfcTyp incompatible with LdEmss File here' !ocean selected
    !---One-time reading of databases
    IF (INIT.EQ.0) THEN
       !---Open files
       iu_coef=30
       OPEN(iu_coef,file=SfcEmissFile1,status='old',form='formatted')
       iu_map =40       
       OPEN(iu_map,file=SfcEmissFile2,status='old',form='formatted')
       !---Read coeffs file
       Do WHILE (.true.) 
          read(iu_coef,*,end=10) iSfcClass,Ichan,coef(1:6)
          coefTot(1:6,iSfcClass,Ichan) = coef(1:6)
       ENDDO
10     continue
       !---Read atlas file
       iPts=0
       Do WHILE (.true.) 
          read(iu_map,*,end=20) coef(1:12)
          iSfcClass                                  = coef(12)
          iPts(iSfcClass)                            = iPts(iSfcClass)+1
          IF (iPts(iSfcClass) .gt. mxPts) STOP'Error in dimension not sufficient'
          nAvgPts(iPts(iSfcClass),iSfcClass)         = coef(11)
          lon(iPts(iSfcClass),iSfcClass)             = coef(1)
          lat(iPts(iSfcClass),iSfcClass)             = coef(2)
          MeanEmisNadir(iPts(iSfcClass),iSfcClass,1) = coef(3)
          MeanEmisNadir(iPts(iSfcClass),iSfcClass,2) = coef(5)
          MeanEmisNadir(iPts(iSfcClass),iSfcClass,3) = coef(7)
          MeanEmisNadir(iPts(iSfcClass),iSfcClass,4) = coef(9)
          StdvEmisNadir(iPts(iSfcClass),iSfcClass,1) = coef(4)
          StdvEmisNadir(iPts(iSfcClass),iSfcClass,2) = coef(6)
          StdvEmisNadir(iPts(iSfcClass),iSfcClass,3) = coef(8)
          StdvEmisNadir(iPts(iSfcClass),iSfcClass,4) = coef(10)
       ENDDO
20     continue
       close(iu_coef)
       close(iu_map)
       INIT=1
    ENDIF
    !---Pick up a surface class index depending on iSfcTyp
    IF (iSfcTyp.eq.LD_TYP) THEN   !land selected
       DO while (.true.) 
          call random_number(rand)
          isfcClass= 1 + int(rand*19.4)
          IF (iPts(iSfcClass) .eq.0) CYCLE
          IF (isfcClass.ne.12) EXIT   !this is ice
          IF (isfcClass.ne.14) EXIT   !this is inland water
          IF (isfcClass.ne.15) EXIT   !this is ocean
       ENDDO
    ENDIF
    IF (iSfcTyp.eq.SEAICE_TYP) isfcClass=12 !ice selected
    IF (iSfcTyp.eq.SNOW_TYP)   isfcClass=12 !Use ice as there is no snow type in atlas
    !---Choose the mapping vector
    IF (nchan.eq.20) mapAtlas(1:nchan)=(/1,2,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4/) !assumed AMSUA/B-MHS
    IF (nchan.eq.22) mapAtlas(1:nchan)=(/1,2,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4/) !assumed ATMS
    IF (nchan.ne.20.and.nchan.ne.22) STOP'Subroutine valid only for AMSUA/B or AMSUA/MHS or ATMS'
    !---Pick up randomly an emissivty spectrum
    call random_number(rand)
    iprof= 1 + int(rand*iPts(iSfcClass))
    emiss(1:nchan) = MeanEmisNadir(iProf,iSfcClass,mapAtlas(1:nchan))    
    RETURN
  END SUBROUTINE EMISS_LD_KARBOU_ATLAS



  SUBROUTINE ReadHdrProfData_noaaSet(GeophInputFile,iu,nPrf,nLay,nLev,level_p,nemis)
    CHARACTER(LEN=*)                       :: GeophInputFile
    INTEGER                                :: iu,nPrf,nLay,nLev
    REAL,            DIMENSION(:), POINTER :: level_p
    !---Local variables
    CHARACTER(LEN=10)                      :: dummystr
    INTEGER                                :: ncemis, dummy,i,nemis
    INTEGER                                :: Error_Status

    Error_Status = Open_Text_File( TRIM(GeophInputFile),iu)
    read(iu,'(a10, i5, i10, i10, i10, i10 )') dummystr,nprf,nemis,ncemis,nlay,dummy 
    nLev= nLay+1
    ALLOCATE(level_p(nlev))
    read(iu,*)  level_p(2:nlev)
    level_p(1) = 0.008
    DO i=1,10 
       read(iu,'(a)') dummystr
    enddo
    RETURN
  END SUBROUTINE ReadHdrProfData_noaaSet



  SUBROUTINE ReadProfData_noaaSet(FileID,n_Layers,SfcType,SfcWind_s,SfcT,&
       SfcPress,level_p,layer_p,layer_t,layer_w,layer_clw,layer_rain,    &
       layer_snow,layer_ice,layer_gh,layer_o3,nemis,alat,alon,nyear,nmonth,nday,nhour)
    INTEGER               :: SfcType,FileID,n_Layers,nemis
    REAL                  :: SfcWind_s,SfcT,SfcPress
    REAL, DIMENSION(:)    :: level_p,layer_p,layer_t,layer_w,layer_clw
    REAL, DIMENSION(:)    :: layer_rain,layer_snow,layer_ice,layer_gh,layer_o3
    !---Local variables
    CHARACTER(LEN=10)     :: idProf
    REAL                  :: alat,alon,pland,psurf,tsurf,emismw,topog
    REAL                  :: em1,em2,em3,x,tair,h2ocd,ozocd,wlcd,ciw
    REAL                  :: pcldtop,cldfrc,r,q_vap,q_o3
    INTEGER               :: nyear,nmonth,nday,nhour,nminute,nsec,ncld
    INTEGER               :: nspi, nspr,i,j,l

    call random_number(x)
    SfcWind_s = x *30.
    read(FileID,'(a8, 2f9.3, i3, 4i2, i4 )') idprof,alat,alon,nyear,nmonth, &
         nday,nhour,nminute,nsec
    read(FileID,*) pland,SfcPress,SfcT,emismw, topog, ncld
    !---determination of surface type from the pland fraction
    IF (pland.le.0.05) THEN 
       SfcType=OC_TYP !ocean
    ENDIF
    IF (pland.ge.0.95) THEN 
       SfcType=LD_TYP !land
    ENDIF
    IF (pland.ge.0.05 .and. pland .le.0.95) THEN 
       SfcType=COAST_TYP !coastal
    ENDIF
    !---Cloud Info
    DO l=1, ncld 
       read(FileID,*) pcldtop,cldfrc
    ENDDO
    !---Atmospheric Info
    DO i=1, n_Layers 
       read(FileID,*)  tair,h2ocd,ozocd,wlcd,ciw
       q_vap = 100.*MW_H2O*(h2ocd/AVOGADRO_CONSTANT)* &
            STANDARD_GRAVITY / (level_p(i+1) - level_p(i)) !-> g/kg
       q_o3  = 100.*MW_O3*(ozocd/AVOGADRO_CONSTANT)* &
            STANDARD_GRAVITY / (level_p(i+1) - level_p(i)) !-> g/kg
       !---Fill up the state structure
       layer_p(i)    = (level_p(i)+level_p(i+1))/2.
       layer_t(i)    = tair
       layer_w(i)    = q_vap
       layer_clw(i)  = wlcd
       layer_rain(i) = 0.
       layer_snow(i) = 0.
       layer_ice(i)  = ciw
       layer_gh(i)   = 0.
       layer_o3(i)   = q_o3
    ENDDO
    !---Emissivity Info
    DO i=1, nemis 
       read(FileID,*)  em1,em2,em3
    ENDDO
    !---read the cld emittances/reflectances
    DO j=0, 0 
       DO i = 1, ncld 
          read(FileID,*)  em1,em2,em3
       ENDDO
    ENDDO
    !---Read the spare data
    read(FileID,*) nspi, nspr
    if (nspi .gt. 0) then 
       DO j = 1, nspi 
          read(FileID,*) x 
       ENDDO
    endif
    if (nspr .gt. 0) then  
       read(FileID,*) x,x
    endif

    RETURN
  END SUBROUTINE ReadProfData_noaaSet


  SUBROUTINE ReadHdrProfData_ecmwfSet(GeophInputFile,iu,nPrf,nLay,nLev,level_p)
    CHARACTER(LEN=*)                       :: GeophInputFile
    INTEGER                                :: iu,nPrf,nLay,nLev
    REAL,            DIMENSION(:), POINTER :: level_p
    !---Local variables
    CHARACTER(LEN=10)                      :: dummystr
    INTEGER                                :: ncemis, dummy,i,nemis
    INTEGER                                :: Error_Status,iprof0
    REAL                                   :: alat,alon,SfcPress,pland,p,t,q,o3

    !---Open file, read header and read level pressure
    Error_Status = Open_Text_File( TRIM(GeophInputFile),iu)
    read(iu,'(a10 )') dummystr
    read(iu,'(a10 )') dummystr
    read(iu,'(a10 )') dummystr
    nprf = 52
    nlay = 100
    nLev= nLay+1
    ALLOCATE(level_p(nlev))
    read(iu,'(1x,i3,a10,1x,f,1x,f,1x,f,1x,f)') iprof0,dummystr,alat,alon,SfcPress,pland
    read(iu,'(a10 )') dummystr
    DO i=1, nLev 
       read(iu,*)  p,t,q,o3
       level_p(i)  = p
    ENDDO
    CLOSE(iu)
    !----Re-open and read header
    Error_Status = Open_Text_File( TRIM(GeophInputFile),iu)
    read(iu,'(a10 )') dummystr
    read(iu,'(a10 )') dummystr
    read(iu,'(a10 )') dummystr
    RETURN
  END SUBROUTINE ReadHdrProfData_ecmwfSet


  SUBROUTINE ReadProfData_ecmwfSet(FileID,n_Layers,SfcType,SfcWind_s,SfcT,&
       SfcPress,level_p,layer_p,layer_t,layer_w,layer_clw,layer_rain,    &
       layer_snow,layer_ice,layer_gh,layer_o3,alat,alon)
    INTEGER               :: SfcType,FileID,n_Layers
    REAL                  :: SfcWind_s,SfcT,SfcPress
    REAL, DIMENSION(:)    :: level_p,layer_p,layer_t,layer_w,layer_clw
    REAL, DIMENSION(:)    :: layer_rain,layer_snow,layer_ice,layer_gh,layer_o3
    !---Local variables
    REAL                           :: alat,alon,pland,x,t,p,q,o3
    REAL, DIMENSION(SIZE(level_p)) :: level_t,level_q,level_o3
    INTEGER                        :: nLev,i,j,l,iprof0
    INTEGER, SAVE                  :: FirstProfRead=0
    CHARACTER(LEN=10)              :: dummystr

    call random_number(x)
    SfcWind_s = x *30.
    read(FileID,'(1x,i3,a10,1x,f,1x,f,1x,f,1x,f)') iprof0,dummystr,alat,alon,SfcPress,pland
    !---determination of surface type from the pland fraction
    IF (pland.le.0.05) THEN 
       SfcType=OC_TYP !ocean
    ENDIF
    IF (pland.ge.0.95) THEN 
       SfcType=LD_TYP !land
    ENDIF
    IF (pland.ge.0.05 .and. pland .le.0.95) THEN 
       SfcType=COAST_TYP !coastal
    ENDIF
    !---One-time reading 
    IF (FirstProfRead.EQ.0) THEN
       read(FileID,'(a10 )') dummystr
       FirstProfRead=1
    ENDIF
    !---Atmospheric Info
    nLev=n_Layers+1
    DO i=1, nLev 
       read(FileID,*)  p,t,q,o3
       level_p(i)  = p
       level_t(i)  = t
       level_q(i)  = q*1000.  !kg/kg -> g/kg
       level_o3(i) = o3*1000. !kg/kg -> g/kg
    ENDDO
    !---Fill up the layer info
    DO i=1,n_Layers 
       layer_p(i)    = (level_p(i)+level_p(i+1))/2.
       layer_t(i)    = (level_t(i)+level_t(i+1))/2.
       layer_w(i)    = (level_q(i)+level_q(i+1))/2.
       layer_o3(i)   = (level_o3(i)+level_o3(i+1))/2.
       layer_clw(i)  = 0.
       layer_rain(i) = 0.
       layer_snow(i) = 0.
       layer_ice(i)  = 0.
       layer_gh(i)   = 0.
    ENDDO
    !---Dummy value for Tskin
    SfcT=290.
    RETURN
  END SUBROUTINE ReadProfData_ecmwfSet

  SUBROUTINE ReadProfData_ecmwf60Set(GeophInputFile,iu,iProf,nLay_In,SfcType,SfcWind,SfcT, &
             psurf,xpres,level_p,layer_p,layer_t,layer_w,layer_clw,     &
             layer_rain,layer_snow,layer_ice,layer_gh,layer_o3,rlat,rlon,iyear,iday,imonth,itime,theprof)
    implicit none
    !---Program Inputs
    CHARACTER(LEN=*)   :: GeophInputFile

    !---Local variables
    integer, parameter :: klev=60
    integer, parameter :: nslay=4
    integer, parameter :: nlevels=101
    integer, parameter :: xlev=klev+nlevels
    integer            :: nLay_In,theprof
    real               :: randomwind
    real, parameter    :: mdry = 28.964 !molecular weight of dry air (g/mole)
    real, parameter    :: mh2o = 18.    !molecular weight of H2O (g/mole)
    real, parameter    :: mo3  = 48.    !molecular weight of O3  (g/mole)

    !---Local arrays
    real, dimension(:), allocatable       :: arx
    real, dimension(:), allocatable       :: ary
    real       :: aryint(nlevels)
    real       :: pap(klev)
    real       :: paph(klev+1)
    real       :: pres(klev)
    real       :: presfin(xlev)
    real       :: istl(nslay), stl(nslay), swvl(nslay)
    real       :: sistl(nslay), sstl(nslay), sswvl(nslay)
    real       :: stt(klev), swv(klev), so3(klev), ppmvo3(klev)
    real       :: sw(klev), scc(klev), sclw(klev), sciw(klev)
    real       :: tt(klev), o3(klev)
    real(kind=8) ::wv(klev)
    real       :: level_t(nlevels), level_w(nlevels), level_o3(nlevels)
    real       :: level_clw(nlevels), level_ciw(nlevels)
    real       :: w(klev), cc(klev), clw(klev), ciw(klev)
    real       :: wvs(xlev)
    real       :: wvvmfin(xlev)
    real       :: wvmmfin(xlev)
    real       :: wvmmextr(xlev)
    real       :: wvmm(klev+1)
    real       :: wvvmint(nlevels)
    real       :: wvmmint(nlevels)
    real       :: wvint(nlevels)
    real       :: o3vm(klev+1)
    real       :: o3vmfin(xlev)
    real       :: o3mmint(nlevels)
    real       :: o3vmint(nlevels)
    real       :: o3int(nlevels)
    real       :: textr(nlevels)
    real       :: tfin(xlev)
    real       :: clwfin(xlev)
    real       :: ciwfin(xlev)
    real       :: tint(nlevels)
    real       :: clwint(nlevels)
    real       :: ciwint(nlevels)
    real       :: as(5,xlev)
    
    !---Local scalars
    real       :: spres
    real       :: const
    real       :: rh
    real       :: cvl, cvh, tvl, tvh, ci
    real       :: asn, rsn, tsn, sd, sr, al, oro
    real       :: psurf, q2m, st, rlsm, rlon, rlat, u10, v10, t2m
    real       :: gradt, gradq, grado
    real       :: scvl, scvh, stvl, stvh, sci
    real       :: sasn, srsn, stsn, ssd, ssr, sal, soro
    real       :: spsurf, sq2m, sst, srlsm, srlon, srlat, su10, sv10, st2m
    integer    :: jdat
    integer    :: iyyyy, iyyyymm, iyyyymmdd
    integer    :: iyear, imonth, iday, itime
    integer    :: levbot, levtop
    integer    :: iatm,jlev,i,index,iu
    integer    :: nprof,iProf

    !---Outputs
    INTEGER            :: SfcType,LayCnt
    REAL               :: SfcWind,SfcT
    REAL, DIMENSION(:) :: xpres,level_p,layer_p,layer_t,layer_w,layer_clw
    REAL, DIMENSION(:) :: layer_rain,layer_snow,layer_ice,layer_gh,layer_o3

    !-----------------------------------------------------------------------
!    data xpres       /.1,.29,.69,1.42,2.611,4.407,6.95,10.37,14.81, &
!         20.4,27.26,35.51,45.29,56.73,69.97,85.18,102.05,122.04, &
!         143.84,167.95,194.36,222.94,253.71,286.6,321.5,358.28, &
!         396.81,436.95,478.54,521.46,565.54,610.6,656.43,702.73, &
!         749.12,795.09,839.95,882.8,922.46,957.44,985.88,1005.43,1013.25/
    !-----Open input file---------------------------------------------------
    open(iu,file=GeophInputFile,form='unformatted')
    
    tfin(1:nlevels) = 0.
    wvmmfin(1:nlevels) = 0.
    o3vmfin(1:nlevels) = 0.
    !-----Begin loop on profiles
    read(iu) &
         jdat, &                         ! date (yyyymmddhh)
         srlon, &                        ! longitude (deg)
         srlat, &                        ! latitude (deg)
         srlsm, &                        ! land/see mask (1=land, 0=sea)
         sst, &                          ! surface skin temperature (K)
         spsurf, &                       ! surface pressure (hPa)
         su10, &                         ! 10-meter u wind (m/s)
         sv10, &                         ! 10-meter v wind (m/s)
         st2m, &                         ! 2-meter temperature (K)
         sq2m, &                         ! 2-meter specific humidity (kg/kg)
         (stt(jlev),jlev = 1,klev), &    ! temperature (K)
         (swv(jlev),jlev = 1,klev), &    ! specific humidity (kg/kg)
         (so3(jlev),jlev = 1,klev), &    ! specific ozone (kg/kg)
         (scc(jlev),jlev = 1,klev), &    ! cloud cover
         (sclw(jlev),jlev = 1,klev), &   ! cloud liquid water content (kg/kg)
         (sciw(jlev),jlev = 1,klev), &   ! cloud ice water content (kg/kg)
         (sw(jlev),jlev = 1,klev), &     ! vertical velocity (Pa/s)
         scvl, &                         ! low vegetation cover
         scvh, &                         ! high vegetation cover
         stvl, &                         ! type of low vegetation 
         stvh, &                         ! type of high vegetation 
         sci, &                          ! ice cover
         sasn, &                         ! snow albedo (0-1)
         srsn, &                         ! snow density (kg/m3)
         stsn, &                         ! snow temperature (K)
         ssd, &                          ! snow depth (m)
         ssr, &                          ! surface roughness (m)
         sal, &                          ! surface albedo (0-1)
         (sistl(jlev),jlev = 1,nslay), & ! layer ice temp. (K)  (top to bottom)
         (sstl(jlev),jlev = 1,nslay), &  ! layer soil temp. (K) (top to bottom)
         (sswvl(jlev),jlev = 1,nslay),&  ! layer soil volumetric water (m3/m3) (top to bottom)
         soro                            ! surface geometric height (m)  
    !-----Switch to double precision
    rlon = srlon
    rlat = srlat
    rlsm = srlsm
    st = sst
    psurf = spsurf
    u10 = su10
    v10 = sv10
    t2m = st2m
    q2m = sq2m
    tt(1:klev) = stt(1:klev)
    wv(1:klev) = swv(1:klev)
    o3(1:klev) = so3(1:klev)
    cc(1:klev) = scc(1:klev)
    clw(1:klev) = sclw(1:klev)
    ciw(1:klev) = sciw(1:klev)
    w(1:klev) = sw(1:klev)
    cvl = scvl
    cvh = scvh
    tvl = stvl
    tvh = stvh
    ci = sci
    asn = sasn
    rsn = srsn
    tsn = stsn
    sd = ssd
    sr = ssr
    al = sal
    do jlev = 1,nslay
       istl(jlev) = sistl(jlev)
       stl(jlev) = sstl(jlev)
       swvl(jlev) = sswvl(jlev)
    enddo
    oro = soro

    !---Extract date data
    iyyyymmdd = jdat/100
    itime = jdat - iyyyymmdd*100
    iyyyymm = iyyyymmdd/100
    iday = iyyyymmdd - iyyyymm*100
    iyyyy = iyyyymm/100
    imonth = iyyyymm - iyyyy*100
    iyear = iyyyy
    !---Calculate 60 pressure levels for profile
    !---Surface Pressure
    spres = psurf*100.
    call ec_p60l(spres,pap,paph)
    pres(:)=pap(:)/100
    !---Convert h2o and o3
    DO jlev = 1,klev
       !-----Specific humidity(kg/kg) > Mass mixing ratio (g/g)----------------
       wvmm(jlev)= (wv(jlev)/(1-wv(jlev)))
       !-----Specific ozone(kg/kg) > Volume mixing ratio (ppmv)----------------
       o3vm(jlev)= o3(jlev)/( (1-o3(jlev))*mo3/ mdry + o3(jlev) ) * 1e+06
    ENDDO
    wvmm(klev+1)= q2m/(1-q2m)
    !---Fill T,h2o,wv arrays for extrapolation
    presfin(1:klev)=pres(1:klev)
    tfin(1:klev)=tt(1:klev)
    tfin(klev+1)=t2m
    wvmmfin(1:klev+1)=wvmm(1:klev+1)
    o3vmfin(klev+1)=o3vm(klev)
    o3vmfin(1:klev)=o3vm(1:klev)
    clwfin(1:klev) = clw(1:klev)
    ciwfin(1:klev) = ciw(1:klev)

    !-----------------------------------------------------------------------
    levbot=0
    DO i=1,nlevels
       IF (xpres(i) .gt. psurf ) THEN
          levbot=i
          exit
       ENDIF
    ENDDO
    index = klev
    IF (levbot /= 0 ) THEN
       !-----Extrapolates temperature below surface pressure-------------------
       const= 287./1005.
       DO i=levbot,nlevels
          !----- => adiabatic heating
          !    textr(i)=t2m * (xpres(i)/psurf)**const
          !----- => constant profile
          textr(i)=t2m
       ENDDO
       index=nlevels-levbot+1
       index=index+klev+1
       tfin((klev+2):index)= textr(levbot:nlevels)
       presfin((klev+1))=psurf
       presfin((klev+2):index)=xpres(levbot:nlevels)
       
 
       !-----Extrapolates water vapour below surface pressure------------------
       !----- => constant relative humidity
       call supsat (tfin(klev+1),wvs(klev+1),psurf)
       rh=wvmm(klev+1)*1000/wvs(klev+1)
       DO i=levbot,nlevels
          call supsat (textr(i),wvs(i),xpres(i))
          wvmmextr(i)=rh*wvs(i)/ 1000
       ENDDO
       wvmmfin((klev+2):index)=wvmmextr(levbot:nlevels)
       !-----Extrapolates ozone below surface pressure------------------------
       !----- => constant profile
       o3vmfin((klev+2):index)=o3vm(klev)
       !---Set cloud content below surface to zero
       clwfin((klev+2):index)=0.
       ciwfin((klev+2):index)=0.

    ENDIF

    !-----------------------------------------------------------------------
    DO i=1,index
       wvvmfin(i) = wvmmfin(i) / ( mh2o/mdry + wvmmfin(i)) * 1e+06
    ENDDO
    !-----Extrapolates profile above highest declared level-----------------
    !----- => linear extrapolation -----------------------------------------
    levtop = 1
    DO jlev=1,index
       IF (xpres(jlev) >= pres(1) ) exit
       levtop = levtop + 1
    ENDDO
        
    IF (levtop /= 1) THEN
       gradt = (tfin(1) - tfin(2)) / (presfin(1)-presfin(2))
       gradq = (wvvmfin(1) - wvvmfin(2)) / (presfin(1)-presfin(2))
       grado = (o3vmfin(1) - o3vmfin(2)) / (presfin(1)-presfin(2))
       DO jlev=index, 1, -1
          tfin(jlev+levtop-1) = tfin(jlev)
          wvvmfin(jlev+levtop-1) = wvvmfin(jlev)
          o3vmfin(jlev+levtop-1) = o3vmfin(jlev)
          presfin(jlev+levtop-1) = presfin(jlev)
       ENDDO
       index = index + levtop-1
       DO jlev=1,levtop-1
          presfin(jlev) = xpres(jlev)
          tfin(jlev) = tfin(levtop) + gradt * (presfin(jlev) - presfin(levtop))
          wvvmfin(jlev) = wvvmfin(levtop) + gradq * (presfin(jlev) - presfin(levtop))
          o3vmfin(jlev) = o3vmfin(levtop) + grado * (presfin(jlev) - presfin(levtop))
       ENDDO
    ENDIF
    !-----Interpolates to given pressure grid-------------------------------
    ALLOCATE(arx(index),ary(index))
    arx(1:index)=presfin(1:index)
    ary(1:index)= 0
    ary(1:index)=tfin(1:index)
    CALL LINT(arx,ary,index,index,nlevels,xpres,aryint)
    tint(:)=aryint(:)


    ary(1:index)= 0
    ary(1:index)=wvvmfin(1:index)
    CALL LINT(arx,ary,index,index,nlevels,xpres,aryint)
    wvvmint(:)=aryint(:)

    ary(1:index)= 0
    ary(1:index)=o3vmfin(1:index)
    CALL LINT(arx,ary,index,index,nlevels,xpres,aryint)
    o3vmint(:)=aryint(:)

    ary(1:index)= 0
    ary(1:index)=clwfin(1:index)
    CALL LINT(arx,ary,index,index,nlevels,xpres,aryint)
    clwint(:)=aryint(:)

    ary(1:index)= 0
    ary(1:index)=ciwfin(1:index)
    CALL LINT(arx,ary,index,index,nlevels,xpres,aryint)
    ciwint(:)=aryint(:)

    !-----Spline interpolation may exceptionnally give negative values------
    !----- => correction
    DO i=1,nlevels
       wvvmint(i) = max( wvvmint(i), 0.01 )
       o3vmint(i) = max( o3vmint(i), 0.001 )
    ENDDO
    !-----Unit conversions
    DO jlev = 1,nlevels
       !-----Volume mixing ratio (ppmv) > Mass mixing ratio (g/g)
       wvmmint(jlev)= mh2o/mdry * wvvmint(jlev) / (1.e+06-wvvmint(jlev))
       o3mmint(jlev)= mo3 /mdry * o3vmint(jlev) / (1.e+06-o3vmint(jlev)) 
       !-----Mass mixing ratio (g/g)    > Specific humidity (kg/kg)
       wvint(jlev)= wvmmint(jlev) / ( 1+wvmmint(jlev) )
       o3int(jlev)= o3mmint(jlev) / ( 1+o3mmint(jlev) )
    ENDDO

    !---Fill arrays of levels to be used for layer computation
    level_p(1:nlevels)  = xpres(1:nlevels)
    level_t(1:nlevels)  = tint(1:nlevels)
    level_w(1:nlevels)  = wvmmint(1:nlevels)*1000. ! convert to g/kg
    level_o3(1:nlevels) = o3mmint(1:nlevels)*1000. ! convert to g/kg
    level_clw(1:nlevels) = clwint(1:nlevels)*1000.
    level_ciw(1:nlevels) = ciwint(1:nlevels)*1000.

    !---DETERMINE OUTPUTS (CONVERT/COMPUTE IF NECESSARY)
    !---Combine u/v wind vectors
!    SfcWind = sqrt((u10**2)+(v10**2))
    call random_number(randomwind)
    SfcWind = randomwind *30.
    !---Determine surface type
    IF (rlsm .ge. 0 .and. rlsm .le. 0.1) SfcType = 0
    IF (rlsm .le. 1 .and. rlsm .ge. 0.9) SfcType = 2
    IF (rlsm .gt. 0.1 .and. rlsm .lt. 0.9) SfcType = 6
    !---Get Skin Temperature
    SfcT    = st
    !---Fill up the layer info
    DO i=1,nlevels-1
       layer_p(i)    = (level_p(i)+level_p(i+1))/2.
       layer_t(i)    = (level_t(i)+level_t(i+1))/2.
       layer_w(i)    = (level_w(i)+level_w(i+1))/2.
       layer_o3(i)   = (level_o3(i)+level_o3(i+1))/2.
       layer_clw(i)  = (level_clw(i)+level_clw(i+1))/2.
       layer_rain(i) = 0.
       layer_snow(i) = 0.
       layer_ice(i)  = 0.
       layer_gh(i)   = (level_ciw(i)+level_ciw(i+1))/2.
    ENDDO

    DEALLOCATE(arx,ary)
    RETURN
  END SUBROUTINE ReadProfData_ecmwf60Set


  !-----------------------------------------------------------
  !  Subroutine: ec_p601
  !   
  !  Description:
  !     Computes the 60-level vertical pressure grid
  !       associated to the input surface pressure
  !       All pressures are in Pa
  !
  !  Provided by Frederic Chevallier for ECMWF 60L_SD Dataset
  !------------------------------------------------------------
  
  SUBROUTINE ec_p60l(spres,pap,paph)
    implicit none
    integer, parameter    :: nlev=60
    integer               :: jk
    real          :: spres
    real          :: aam(nlev+1), bbm(nlev+1)
    real          :: pap(nlev), paph(nlev+1)
    
    data aam / &
         0.000000,    20.000000,    38.425343, &
         63.647804,    95.636963,   134.483307, &
         180.584351,   234.779053,   298.495789, &
         373.971924,   464.618134,   575.651001, &
         713.218079,   883.660522,  1094.834717, &
         1356.474609,  1680.640259,  2082.273926, &
         2579.888672,  3196.421631,  3960.291504, &
         4906.708496,  6018.019531,  7306.631348, &
         8765.053711, 10376.126953, 12077.446289, &
         13775.325195, 15379.805664, 16819.474609, &
         18045.183594, 19027.695313, 19755.109375, &
         20222.205078, 20429.863281, 20384.480469, &
         20097.402344, 19584.330078, 18864.750000, &
         17961.357422, 16899.468750, 15706.447266, &
         14411.124023, 13043.218750, 11632.758789, &
         10209.500977,  8802.356445,  7438.803223, &
         6144.314941,  4941.778320,  3850.913330, &
         2887.696533,  2063.779785,  1385.912598, &
         855.361755,   467.333588,   210.393890, &
         65.889244,     7.367743,     0.000000, &
         0.000000 &
         /
    
    data bbm / &
         0.0000000000, 0.0000000000, 0.0000000000, &
         0.0000000000, 0.0000000000, 0.0000000000, &
         0.0000000000, 0.0000000000, 0.0000000000, &
         0.0000000000, 0.0000000000, 0.0000000000, &
         0.0000000000, 0.0000000000, 0.0000000000, &
         0.0000000000, 0.0000000000, 0.0000000000, &
         0.0000000000, 0.0000000000, 0.0000000000, &
         0.0000000000, 0.0000000000, 0.0000000000, &
         0.0000758235, 0.0004613950, 0.0018151561, &
         0.0050811190, 0.0111429105, 0.0206778757, &
         0.0341211632, 0.0516904071, 0.0735338330, &
         0.0996746942, 0.1300225109, 0.1643843204, &
         0.2024759352, 0.2439331412, 0.2883229554, &
         0.3351548910, 0.3838921487, 0.4339629412, &
         0.4847715795, 0.5357099175, 0.5861684084, &
         0.6355474591, 0.6832686067, 0.7287858129, &
         0.7715966105, 0.8112534285, 0.8473749161, &
         0.8796569109, 0.9078838825, 0.9319403172, &
         0.9518215060, 0.9676452279, 0.9796627164, &
         0.9882701039, 0.9940194488, 0.9976301193, &
         1.0000000000 &
         /
    
    do jk=1,nlev+1
       paph(jk)=aam(jk)+bbm(jk)*spres
    end do
    do jk=1,nlev
       pap(jk)=0.5*(paph(jk)+paph(jk+1))
    end do
    
    RETURN
  END SUBROUTINE ec_p60l
  
  SUBROUTINE supsat(temp,wv,pres)
    !     Description:
    !     Computes water vapour super-saturation mixing ratio
    !       
    !     Method:
    !       
    !     Owner:
    !     Marco Matricardi
    !       
    !     History:
    !     Version      Date       Comment
    !     1            16-8-1999  Marco Matricardi. ECMWF
    !       
    !     Code description:
    !       Language:              Fortran 90.
    !       Software Standards:    "European Standards for Writing and Documenting
    !                              Exchangeable Fortran 90 code".
    !       
    
    implicit none      
    !     Subroutine arguments:
    
    !       Scalar arguments with intent in:
    real,   intent(in) :: temp
    real,   intent(in) :: pres
    
    !       Scalar arguments with intent out:
    real,   intent(out):: wv
    
    !---Local scalars
    real   :: satlim
    real   :: htol
    
    !-----End of header-----------------------------------------------------
    data satlim /1./
    data htol   /0./
       
    wv = svp(temp)*satlim
    wv = 622.*wv/(pres-wv)+htol
    
    RETURN
  END SUBROUTINE supsat
  
  
  FUNCTION svp(temp)
    
    implicit none
    
    real,   intent(in)    ::  temp
    real                  ::  estab(156)       
    !---Local scalars
    real        ::  tt
    real        ::  e0
    real        ::  e1
    real        ::  svp
    integer     ::  ind
    integer     ::  t0
    
    data  estab(1:60) / &
         9.672e-5,1.160e-4,1.388e-4,1.658e-4,1.977e-4,2.353e-4, &
         2.796e-4,3.316e-4,3.925e-4,4.638e-4,5.472e-4,6.444e-4, &
         7.577e-4,8.894e-4,1.042e-3,1.220e-3,1.425e-3,1.662e-3, &
         1.936e-3,2.252e-3,2.615e-3,3.032e-3,3.511e-3,4.060e-3, &
         4.688e-3,5.406e-3,6.225e-3,7.159e-3,8.223e-3,9.432e-3, &
         1.080e-2,1.236e-2,1.413e-2,1.612e-2,1.838e-2,2.092e-2, &
         2.380e-2,2.703e-2,3.067e-2,3.476e-2,3.935e-2,4.449e-2, &
         5.026e-2,5.671e-2,6.393e-2,7.198e-2,8.097e-2,9.098e-2, &
         1.021e-1,1.145e-1,1.283e-1,1.436e-1,1.606e-1,1.794e-1, &
         2.002e-1,2.233e-1,2.488e-1,2.769e-1,3.079e-1,3.421e-1/ 
    
    data  estab(61:120) / &
         3.798e-1,4.213e-1,4.669e-1,5.170e-1,5.720e-1,6.323e-1, &
         6.985e-1,7.709e-1,8.502e-1,9.370e-1,   1.032,   1.135, &
         1.248,   1.371,   1.506,   1.652,   1.811,   1.984, &
         2.172,   2.376,   2.597,   2.889,   3.097,   3.522, &
         3.8619,  4.2148,  4.5451,  4.8981,  5.2753,   5.678, &
         6.1078,  6.5662,  7.0547,  7.5753,  8.1294,  8.7192, &
         9.3465,  10.013,  10.722,  11.474,  12.272,  13.119, &
         14.017,  14.969,  15.977,  17.044,  18.173,  19.367, &
         20.630,  21.964,  23.373,  24.861,  26.430,  28.086, &
         29.831,  31.671,  33.608,  35.649,  37.796,  40.055/
    
    data  estab(121:156) / &
         42.430,  44.927,  47.551,  50.307,  53.200,  56.236, &
         59.422,  62.762,  66.264,  69.934,  73.777,  77.803, &
         82.015,  86.423,  91.034,  95.855,  100.89,  106.16, &
         111.66,  117.40,  123.40,  129.65,  136.17,  142.98, &
         150.07,  157.46,  165.16,  173.18,  181.53,  190.22, &
         199.26,  208.67,  218.45,  228.61,  239.18,  250.16/
    
    tt=temp-183.15        
    if (tt.le.0.) then
       svp=estab(1)
    else
       ind=int(tt)+1
       ind=min(ind,155)
       t0=ind-1
       e0=estab(ind)
       e1=estab(ind+1)
       svp=e0+(tt-t0)*(e1-e0)
    endif
    
    RETURN
  END FUNCTION svp
     

 SUBROUTINE spline(xi,yi,xo,yo,as,ni,no,ii)
        
!     SPLINE, VERSION OF 17 JUL 81
        
 implicit none
 integer                      :: ni, no, ii
 real                 :: xi(ni)
 real                 :: yi(ni)
 real                 :: xo(no)
 real                 :: yo(no)
 real                 :: as(5,ni)
   
 integer              :: nim1, i, j, k, l, m
 real                 :: xm, xn, xx
 real                 :: c(4)
   
 if(ii.eq.1) call cubist(ni,xi,yi,as)
 nim1=ni-1
 m=1
 do 150 j=1,no
   xx=xo(j)
   do 100 i=m,nim1
     l=i
     xm=xi(i)
     xn=xi(i+1)
     if(xx.eq.xm) go to 120
     if(xx.eq.xn) go to 110
     if(xx.gt.xm.and.xx.lt.xn) go to 130
 100 continue
 110 l=l+1
 120 yo(j)=yi(l)
     m=l
     go to 150
 130 do 140 k=1,4
 140 c(k)=as(k,l)
     yo(j)=c(1)+xx*(c(2)+xx*(c(3)+xx*c(4)))
     m=l
 150 continue
 return
 end subroutine spline
        
 subroutine cubist(n,x,y,as)
        
!     CUBIST, VERSION OF 17 JUL 81
!     CALLED BY 'SPLINE'
!     CUBIC SPLINE GENERATOR BY W.HIBBARD, MOD'D BY H.WOOLF.  SECOND DERIV.
!     NOT CONTINUOUS. THE DEGREES OF FREEDOM GAINED ARE USED IN AN
!     ATTEMPT TO AVOID  OSCILLATIONS.
!     FIT TO THE POINTS (X(I),Y(I)) I=1,...,N .
        
 implicit none
 integer              :: n
 real                 :: x(n),y(n),as(5,n)
        
 integer              :: m,i
 real                 :: t, w, z, s, u, v
 real                 :: zs, zq, ws, wq
 real                 :: aa, ba, ca, da
   
 m=n-1
 do 150 i=1,m
   if(i .eq. 1) go to 110
   t=(y(i+1)-y(i-1))/(x(i+1)-x(i-1))
   go to 120
 110 w=(y(2)+y(3))/2.0
   z=(x(2)+x(3))/2.0
   t=(w-y(1))/(z-x(1))
   t=2.0*(y(2)-y(1))/(x(2)-x(1))-t
 120 if(i .eq. m) go to 130
   s=(y(i+2)-y(i))/(x(i+2)-x(i))
   go to 140
 130 w=(y(n-1)+y(n-2))/2.0
   z=(x(n-1)+x(n-2))/2.0
   s=(y(n)-w)/(x(n)-z)
   s=2.0*(y(n)-y(n-1))/(x(n)-x(n-1))-s
  140 u=y(i+1)
   v=y(i)
   w=(x(i+1)+x(i))/2.0
   z=(x(i+1)-x(i))/2.0
   zs=z*z
   zq=z*zs
   ws=w*w
   wq=w*ws
   aa=.5*(u+v)-.25*z*(s-t)
   ba=.75*(u-v)/z-.25*(s+t)
   ca=.25*(s-t)/z
   da=.25*(s+t)/zs-.25*(u-v)/zq
   as(1,i)=aa-ba*w+ca*ws-da*wq
   as(2,i)=ba-2.0*ca*w+3.0*da*ws
   as(3,i)=ca-3.0*da*w
   as(4,i)=da
 150 as(5,i)=0.
 return
 end subroutine cubist


    SUBROUTINE intrp2LevelGrid(XProfIn,PresLevsIn,nLevOut,PresLevsOut,PresBot,PresTop,PresLayOut,XProfOut,qc)

      IMPLICIT NONE
      !---I/O variables   
      INTEGER,                 INTENT(IN)   :: nLevOut
      REAL,    DIMENSION(:),   INTENT(IN)   :: XProfIn
      REAL,    DIMENSION(:),   INTENT(IN)   :: PresLevsIn
      REAL,    DIMENSION(:),   INTENT(IN)   :: PresLevsOut
      REAL,                    INTENT(OUT)  :: PresBot,PresTop
      REAL,    DIMENSION(:),   INTENT(IN)   :: PresLayOut
      REAL,    DIMENSION(:),   INTENT(OUT)  :: XProfOut
      INTEGER,                 INTENT(OUT)  :: qc
      !---Local variables
      INTEGER                               :: nVal_Levs
      INTEGER                               :: i,j,iLev
      INTEGER, DIMENSION(:),   ALLOCATABLE  :: idx_inLevs
      REAL                                  :: PresDiff
      REAL,    DIMENSION(:),   ALLOCATABLE  :: XProfTrun,PresProfTrun
      REAL,    DIMENSION(nLevOut)          :: XProfLevl
      
      !---Determine number of levels with good data 
      nVal_Levs = COUNT(XProfIn(:) .gt. 0.)
      IF (nVal_Levs .lt. 10) THEN
         XProfOut(:) = -999.
         qc = 1
         RETURN
      ENDIF
      IF (nVal_Levs .ge. 10) THEN
         !---Create index of array elements with good data, these are the elements to interpolate
         ALLOCATE (idx_inLevs(nVal_Levs),XProfTrun(nVal_Levs),PresProfTrun(nVal_Levs))
         idx_inLevs = PACK( (/(i,i=1,SIZE(XProfIn(:)))/), (XProfIn(:) .gt. 0))
         PresTop = PresLevsIn(idx_inLevs(1))
         PresBot = PresLevsIn(idx_inLevs(nVal_Levs))
         XProfTrun(:)    = XProfIn(idx_inLevs)
         PresProfTrun(:) = PresLevsIn(idx_inLevs)
         !---Check for gaps in profile and depth of profile
         DO iLev=1,nVal_levs-1
            PresDiff = PresProfTrun(iLev+1)-PresProfTrun(iLev)
            IF (PresDiff .gt. 100) qc = 1 
         ENDDO
         !---Call linear interpolation subroutine
         CALL LINT(PresProfTrun,XProfTrun,1,nVal_Levs,nLevOut,PresLevsOut,XProfLevl)
         !---Convert to layers
         XProfOut(:)=-999.
         LayLoop: DO iLev=1,nLevOut-1
            IF (PresLayOut(iLev) .lt. PresTop .or. PresLayOut(iLev) .gt. PresBot) CYCLE LayLoop
            xProfOut(iLev) = (xProfLevl(iLev)+xProfLevl(iLev+1))/2
         ENDDO LayLoop

         DEALLOCATE(idx_inLevs,XProfTrun,PresProfTrun)        
      ENDIF
      RETURN
    END SUBROUTINE intrp2levelGrid

end program preProcessGeophData
