#!/bin/bash
#
# --------------------------------------------------------------------------------------------------------
# Script called from "run_HIRES_MESO.bash" to handle the creation of NHC ATCF b-deck file lists for each 
# basin covered by GOES MESO sectors. Updated bdeck file listings are created for each basin (satellite 
# dependent) every time the main driver script is run and the basin files are staged to working directories 
# (by Sat/Sector). The bdeck file listings are then accessed from the winds hi res driver script when comparing 
# Lat/Lon values between the current MESO sector and any active storms or invests. 
# --------------------------------------------------------------------------------------------------------
# 
# example:  ./run_Make_Basin_Storm_Files.bash <sat> <sector> 
# 
#           <sat>       -  Satellite to process
#                          '16' for GOES-16 (as GOES-East)
#                          '18' for GOES-18 (as GOES-West)
#                          '19' for GOES-19 (as GOES-East) 
#           <sector>    -  ABI sector to process
#                          'MESO1' for MESO sector 1 
#                          'MESO2' for MESO sector 2 
#           <startday>  -  day to start search for bdeck files (yyyy-mm-dd format)
#           <starttime> -  time to start search for bdeck files (hh:mm format)
#
# aab 11/13/2024
#

# Set Library paths, etc.
export LD_LIBRARY_PATH=$LD_LIBRARY_PATH:/data/smcd/goesr_winds/routine/mcidas/lib:/data/data311/qzhao/gnu_v5_3_1/local/ecCodes/lib:/data/data311/qzhao/gnu_v4_8_5/local/netcdf/lib:/data/data311/qzhao/gnu_v4_8_5/local/hdf5/lib:/data/data311/qzhao/gnu_v4_8_5/local/szip/lib:/lib:/data/starfs1/utils/mcidas-x/2022.1//tcl/lib:/data/starfs1/utils/mcidas-x/2022.1//tcl/lib/tk8.5:/data/starfs1/utils/mcidas-x/2022.1//tcl/lib/tcl8.4:/data/data311/qzhao/gnu_v4_8_5/local/netcdff/lib
export PATH=$PATH:.:/data/smcd/goesr_winds/routine/mcidas/bin:/data/starfs1/bin:/data/home004/abailey/miniconda3/bin:/data/starfs1/utils/mcidas-x/2022.1/bin:/data/starfs1/utils/mcidas-x/2022.1/tcl/bin:/usr/condabin:/usr/local/bin:/usr/bin:/usr/local/sbin:/usr/sbin:/opt/puppetlabs/bin


echo ' '
echo "============================================================"

# check command line arguments
sat=$1            # possible values: 16, 17, 18, 19
sector=$2         # possible values: MESO1 or MESO2 
startday=$3       # used for scdr call to pull bdeck files ((yyyy-mm-dd format)
starttime=$4      # used for scdr call to pull bdeck files (hh:mm format)

# define working paths
run_ASF_AMV_Main_PATH=/data/smcd8/abailey/run_ASF_GEO_TC_HIRES_MESO/
bdeck_file_working_dir=${run_ASF_AMV_Main_PATH}GOES-${sat}_TC_HIRES/${sector}/b-deck_file_working_dir/

echo Switch to framework directory to run: 
echo ${run_ASF_AMV_Main_PATH}
cd ${run_ASF_AMV_Main_PATH}

echo ' ' 
echo Pull latest NHC ATCF b-deck files  
if [[ "$sat" = "16" || "$sat" = "19" ]] 
then
   echo ' ' 
   echo "Running GOES-East. Pull current list of bdeck files (i.e., files staged within at least the last 6 hours)"
   echo "for both Atlantic and East Pac basins..."
   echo ' ' 

   # Remove old file lists
   rm  ${bdeck_file_working_dir}ATL_bdeckfile_list_current
   rm  ${bdeck_file_working_dir}EPAC_bdeckfile_list_current
   echo "rm ${bdeck_file_working_dir}AL_Basin_current_storms_list"
         rm ${bdeck_file_working_dir}AL_Basin_current_storms_list
   echo "rm ${bdeck_file_working_dir}bdeck_current_reverse"
         rm ${bdeck_file_working_dir}bdeck_current_reverse

   # bdeck files are updated at synoptic times, so look back at least 6 hours for each basin. 
   echo "scdr-files-go -t ATCF_NHC_B_DECK --stime "$(date -d "$starttime $startday -7 hours" '+%Y-%m-%d %H:%M')" --etime "$startday $starttime" | grep bal > ${bdeck_file_working_dir}ATL_bdeckfile_list_current"
         scdr-files-go -t ATCF_NHC_B_DECK --stime "$(date -d "$starttime $startday -7 hours" '+%Y-%m-%d %H:%M')" --etime "$startday $starttime" | grep bal > ${bdeck_file_working_dir}ATL_bdeckfile_list_current


   echo ' '
   echo "scdr-files-go -t ATCF_NHC_B_DECK --stime "$(date -d "$starttime $startday -7 hours" '+%Y-%m-%d %H:%M')" --etime "$startday $starttime" | grep bep > ${bdeck_file_working_dir}EPAC_bdeckfile_list_current"
         scdr-files-go -t ATCF_NHC_B_DECK --stime "$(date -d "$starttime $startday -7 hours" '+%Y-%m-%d %H:%M')" --etime "$startday $starttime" | grep bep > ${bdeck_file_working_dir}EPAC_bdeckfile_list_current

   # Interrogate bdeck file lists for latest file and storm info (storm #, lat/lon, etc.) and compare to MESO sector center
   # Lat/Lon of target (middle) image.
   echo ' '
   echo '==================================================================================='
   echo First check to see if there are any active Atlantic Basin storms or invests from NHC
   echo '==================================================================================='
   bdeck_filecheck=$(ls -l ${bdeck_file_working_dir}ATL_bdeckfile_list_current | awk -F " " '{print $5}')

   if [ "$bdeck_filecheck" = "0" ]
   then 
      echo ' '
      echo "Currently no active storms in Atlantic Basin."
      atl_basin=no

   else
      echo ' ' 
      echo "Check 'current' bdeck file list against master storm file list for any updated storm info..."
    
      storm_match=0
      # Get list of current storms
      filename_current=${bdeck_file_working_dir}ATL_bdeckfile_list_current
      # reverse the order to search through file list from top down (i.e., most recent first)
      tac $filename_current > ${bdeck_file_working_dir}bdeck_current_reverse

      # Get master list of previous/current storms already tracked
      # If no storms have been tracked yet for the year, create a new (empty)
      # basin master storm list file. It will get filled once winds are made. 
      ls -l ${bdeck_file_working_dir}AL_Basin_bdeck_master_storm_list 

      if [ "$?" > "0" ]
      then
	 touch ${bdeck_file_working_dir}AL_Basin_bdeck_master_storm_list
      fi

      filename_master=${bdeck_file_working_dir}AL_Basin_bdeck_master_storm_list
      while read fmaster; do 
        master_list_num=$(echo $fmaster | awk -F ' ' '{print $2$4}')
        #echo "master list storm#:" $master_list_num

        filename_current=${bdeck_file_working_dir}bdeck_current_reverse
        while read fcurrent; do
          current_list_check=$(echo $fcurrent | awk -F '-bal' '{print $2}' | cut -c1-6)

          if [[ "$master_list_num" = "$current_list_check" ]]
          then
             # We have a match...write to basin current storms list. 
             # This is used by driver script to check geolocation of MESO sector
             echo $fcurrent >> ${bdeck_file_working_dir}AL_Basin_current_storms_list
             let storm_match=$storm_match+1
	     break_loop=1  # We only want the first match which will be the most recent file in bdeck list for this storm
	  else
             # No match keep searching through list
	     break_loop=0
          fi

          if [ "$break_loop" -eq "1" ]
          then
             break
          fi

        done < "$filename_current"
      done < "$filename_master"

      echo Current storms being tracked= $storm_match

      # If no current storms currently being tracked, create an empty AL_Basin_current_storms_list file to 
      # accomodate new storm check below 
      if [ "$storm_match" -eq "0" ]
      then
         touch ${bdeck_file_working_dir}AL_Basin_current_storms_list 
      fi

      echo ' ' 
      echo "Now check 'current' bdeck file list for any new storms or invests being tracked:"
      # Loop through current bdeck file listing again and compare to updated Basin current storms list for any new storms 
      # that may have started being tracked since the last update.

      storm_match=0
      # loop through current bdeck list
      filename_current=${bdeck_file_working_dir}bdeck_current_reverse
      while read fcurrent; do
         current_list_check=$(echo $fcurrent | awk -F '-bal' '{print $2}' | cut -c1-6)

         # Look to see if this is a new storm (i.e., not in the current basin file)
         grep -q "bal$current_list_check" ${bdeck_file_working_dir}AL_Basin_current_storms_list
         check_new_storm=$?

         if [ "$check_new_storm" = "1" ]
         then
            # We have a new storm...add this newest bdeck file to basin current storms list.
            # This is used by driver script to check geolocation of MESO sector
            echo $fcurrent >> ${bdeck_file_working_dir}AL_Basin_current_storms_list
            let storm_match=$storm_match+1
         fi

      done < "$filename_current"

      echo New storms/invests being tracked= $storm_match
      echo ' ' 
      echo "The full ATL current bdeck list (via SCDR) will be used to check for geolocation with GOES-$sat $sector sector."
      echo "AL_Basin_current_storms_list:" 
      echo '----------------------------------------------------------------------'
      cat ${bdeck_file_working_dir}AL_Basin_current_storms_list
      echo '----------------------------------------------------------------------'
      echo ' ' 

   fi # ATL bdeck file check
  
   echo '====================================================================================='
   echo Now check to see if there are any active East Pacific Basin storms or invests from NHC
   echo '====================================================================================='

   # Remove old file lists
   rm ${bdeck_file_working_dir}EP_Basin_current_storms_list
   rm ${bdeck_file_working_dir}bdeck_current_reverse

   bdeck_filecheck=$(ls -l ${bdeck_file_working_dir}EPAC_bdeckfile_list_current | awk -F " " '{print $5}')

   if [ "$bdeck_filecheck" = "0" ]
   then
      echo ' '
      echo "Currently no active storms in East Pacific Basin."
      epac_basin=no

   else
      echo ' '
      echo "Check 'current' bdeck file list against master storm file list for any updated storm info..."

      storm_match=0

      # Get list of current storms
      filename_current=${bdeck_file_working_dir}EPAC_bdeckfile_list_current
      # reverse the order to search through latest file list from top down (i.e., most recent first)
      tac $filename_current > ${bdeck_file_working_dir}bdeck_current_reverse

      # Get master list of previous/current storms already tracked
      # If no storms have been tracked yet for the year, create a new (empty)
      # basin master storm list file. It will get filled once winds are made..
      ls -l ${bdeck_file_working_dir}EP_Basin_bdeck_master_storm_list

      if [ "$?" > "0" ]
      then
         touch ${bdeck_file_working_dir}EP_Basin_bdeck_master_storm_list
      fi

      # Get master list of previous/current storms already tracked
      filename_master=${bdeck_file_working_dir}EP_Basin_bdeck_master_storm_list
      while read fmaster; do
        master_list_num=$(echo $fmaster | awk -F ' ' '{print $2$4}')

        filename_current=${bdeck_file_working_dir}bdeck_current_reverse
        while read fcurrent; do
          current_list_check=$(echo $fcurrent | awk -F '-bep' '{print $2}' | cut -c1-6)

          if [[ "$master_list_num" = "$current_list_check" ]]
          then
             # We have a match...write to basin current storms list.
             # This is used by driver script to check geolocation of MESO sector
             echo $fcurrent >> ${bdeck_file_working_dir}EP_Basin_current_storms_list
             let storm_match=$storm_match+1
             break_loop=1  # We only want the first match which will be the most recent file in bdeck list for this storm
          else
             # No match keep searching through list
             break_loop=0
          fi

          if [ "$break_loop" -eq "1" ]
          then
             break
          fi

        done < "$filename_current"
      done < "$filename_master"

      echo Current storms being tracked= $storm_match
      echo ' '

      # If no current storms currently being tracked, create an empty AL_Basin_current_storms_list file to
      # accomodate new storm check below
      if [ "$storm_match" -eq "0" ]
      then
         touch ${bdeck_file_working_dir}EP_Basin_current_storms_list
      fi

      echo "Now check 'current' bdeck file list for any new storms being tracked:"
      # Loop through current bdeck file listing again and compare to updated Basin current storms list for any new storms
      # that may have started being tracked since the last update.

      storm_match=0
      # loop through current bdeck list
      filename_current=${bdeck_file_working_dir}bdeck_current_reverse
      while read fcurrent; do
         current_list_check=$(echo $fcurrent | awk -F '-bep' '{print $2}' | cut -c1-6)

         # Look to see if this is a new storm (i.e., not in the current basin file)
         grep -q "bep$current_list_check" ${bdeck_file_working_dir}EP_Basin_current_storms_list
         check_new_storm=$?

         if [ "$check_new_storm" = "1" ]
         then
            # We have a new storm...add this newest bdeck file to basin current storms list.
            # This is used by driver script to check geolocation of MESO sector
            echo $fcurrent >> ${bdeck_file_working_dir}EP_Basin_current_storms_list
            let storm_match=$storm_match+1
         fi

      done < "$filename_current"

      echo New storms being tracked= $storm_match
      echo ' '
      echo "The full EPAC bdeck list (via SCDR) will be used to check for geolocation with GOES-$sat $sector sector."
      echo "EP_Basin_current_storms_list:"
      echo '----------------------------------------------------------------------'
      cat ${bdeck_file_working_dir}EP_Basin_current_storms_list
      echo '----------------------------------------------------------------------'
      echo ' '

   fi # EPAC bdeck file check

elif [[ "$sat" = "18" ]]
then
   echo ' '
   echo '==========================================================================='
   echo "Running GOES-West. Pull current list of bdeck files (i.e., files staged within at least the last 6 hours)"
   echo "for both East and Central Pacific basins..."
   echo ' '

   # Remove old file lists
   rm  ${bdeck_file_working_dir}EPAC_bdeckfile_list_current
   rm  ${bdeck_file_working_dir}CPAC_bdeckfile_list_current

   # bdeck files are updated at synoptic times, so look back at least 6 hours for each basin.
   echo "scdr-files-go -t ATCF_NHC_B_DECK --stime "$(date -d "$starttime $startday -7 hours" '+%Y-%m-%d %H:%M')" --etime "$startday $starttime" | grep bep > ${bdeck_file_working_dir}EPAC_bdeckfile_list_current"
         scdr-files-go -t ATCF_NHC_B_DECK --stime "$(date -d "$starttime $startday -7 hours" '+%Y-%m-%d %H:%M')" --etime "$startday $starttime" | grep bep > ${bdeck_file_working_dir}EPAC_bdeckfile_list_current

   echo ' '
   echo "scdr-files-go -t ATCF_NHC_B_DECK --stime "$(date -d "$starttime $startday -7 hours" '+%Y-%m-%d %H:%M')" --etime "$startday $starttime" | grep bcp > ${bdeck_file_working_dir}CPAC_bdeckfile_list_current"
         scdr-files-go -t ATCF_NHC_B_DECK --stime "$(date -d "$starttime $startday -7 hours" '+%Y-%m-%d %H:%M')" --etime "$startday $starttime" | grep bcp > ${bdeck_file_working_dir}CPAC_bdeckfile_list_current

   # Interrogate bdeck file lists for latest file and storm info (storm #, lat/lon, etc.) and compare to MESO sector center
   # Lat/Lon of target (middle) image.
   echo '======================================================================================='
   echo First check to see if there are any active East Pacific Basin storms or invests from NHC
   echo '======================================================================================='

   # Remove old file lists
   rm ${bdeck_file_working_dir}EP_Basin_current_storms_list
   rm ${bdeck_file_working_dir}bdeck_current_reverse
   
   bdeck_filecheck=$(ls -l ${bdeck_file_working_dir}EPAC_bdeckfile_list_current | awk -F " " '{print $5}')

   if [ "$bdeck_filecheck" = "0" ]
   then
      echo ' '
      echo "Currently no active storms in East Pacific Basin."
      epac_basin=no

   else
      echo ' '
      echo "Check 'current' bdeck file list against master storm file list for any updated storm info..."

      storm_match=0
      # Get list of current storms
      filename_current=${bdeck_file_working_dir}EPAC_bdeckfile_list_current
      # reverse the order to search through latest file list from top down (i.e., most recent first)
      tac $filename_current > ${bdeck_file_working_dir}bdeck_current_reverse

      # Get master list of previous/current storms already tracked
      # If no storms have been tracked yet for the year, create a new (empty)
      # basin master storm list file. It will get filled once winds are made..
      ls -l ${bdeck_file_working_dir}EP_Basin_bdeck_master_storm_list

      if [ "$?" > "0" ]
      then
         touch ${bdeck_file_working_dir}EP_Basin_bdeck_master_storm_list
      fi

      # Get master list of previous/current storms already tracked
      filename_master=${bdeck_file_working_dir}EP_Basin_bdeck_master_storm_list
      while read fmaster; do
        master_list_num=$(echo $fmaster | awk -F ' ' '{print $2$4}')

        filename_current=${bdeck_file_working_dir}bdeck_current_reverse
        while read fcurrent; do
          current_list_check=$(echo $fcurrent | awk -F '-bep' '{print $2}' | cut -c1-6)

          if [[ "$master_list_num" = "$current_list_check" ]]
          then
             # We have a match...write to basin current storms list.
             # This is used by driver script to check geolocation of MESO sector
             echo $fcurrent >> ${bdeck_file_working_dir}EP_Basin_current_storms_list
             let storm_match=$storm_match+1
             break_loop=1  # We only want the first match which will be the most recent file in bdeck list for this storm
          else
             # No match keep searching through list
             break_loop=0
          fi

          if [ "$break_loop" -eq "1" ]
          then
             break
          fi

        done < "$filename_current"
      done < "$filename_master"

      echo Current storms being tracked= $storm_match
      echo ' '

      # If no current storms currently being tracked, create an empty AL_Basin_current_storms_list file to
      # accomodate new storm check below
      if [ "$storm_match" -eq "0" ]
      then
         touch ${bdeck_file_working_dir}EP_Basin_current_storms_list
      fi

      echo "Now check 'current' bdeck file list for any new storms being tracked:"
      # Loop through current bdeck file listing again and compare to updated Basin current storms list for any new storms
      # that may have started being tracked since the last update.

      storm_match=0
      # loop through current bdeck list
      filename_current=${bdeck_file_working_dir}bdeck_current_reverse
      while read fcurrent; do
         current_list_check=$(echo $fcurrent | awk -F '-bep' '{print $2}' | cut -c1-6)

         # Look to see if this is a new storm (i.e., not in the current basin file)
         grep -q "bep$current_list_check" ${bdeck_file_working_dir}EP_Basin_current_storms_list
         check_new_storm=$?

	 if [ "$check_new_storm" = "1" ]
         then
            # We have a new storm...add this newest bdeck file to basin current storms list.
            # This is used by driver script to check geolocation of MESO sector
            echo $fcurrent >> ${bdeck_file_working_dir}EP_Basin_current_storms_list
            let storm_match=$storm_match+1
         fi

      done < "$filename_current"

      echo New storms being tracked= $storm_match
      echo ' '
      echo "The full EPAC bdeck list (via SCDR) will be used to check for geolocation with GOES-$sat $sector sector."
      echo "EP_Basin_current_storms_list:"
      echo '----------------------------------------------------------------------'
      cat ${bdeck_file_working_dir}EP_Basin_current_storms_list
      echo '----------------------------------------------------------------------'
      echo ' '

   fi # EPAC bdeck file check


   echo '========================================================================================'
   echo Now check to see if there are any active Central Pacific Basin storms or invests from NHC
   echo '========================================================================================'

   # Remove old file lists
   rm ${bdeck_file_working_dir}CP_Basin_current_storms_list
   rm ${bdeck_file_working_dir}bdeck_current_reverse
   
   bdeck_filecheck=$(ls -l ${bdeck_file_working_dir}CPAC_bdeckfile_list_current | awk -F " " '{print $5}')

   if [ "$bdeck_filecheck" = "0" ]
   then
      echo ' '
      echo "Currently no active storms in Central Pacific Basin."
      cpac_basin=no

   else
      echo ' '
      echo "Check 'current' bdeck file list against master storm file list for any updated storm info..."

      storm_match=0
      # Get list of current storms
      filename_current=${bdeck_file_working_dir}CPAC_bdeckfile_list_current
      # reverse the order to search through latest file list from top down (i.e., most recent first)
      tac $filename_current > ${bdeck_file_working_dir}bdeck_current_reverse

      # Get master list of previous/current storms already tracked
      # If no storms have been tracked yet for the year, create a new (empty)
      # basin master storm list file. It will get filled once winds are made..
      ls -l ${bdeck_file_working_dir}CP_Basin_bdeck_master_storm_list

      if [ "$?" > "0" ]
      then
         touch ${bdeck_file_working_dir}CP_Basin_bdeck_master_storm_list
      fi

      # Get master list of previous/current storms already tracked
      filename_master=${bdeck_file_working_dir}CP_Basin_bdeck_master_storm_list
      while read fmaster; do
        master_list_num=$(echo $fmaster | awk -F ' ' '{print $2$4}')

        filename_current=${bdeck_file_working_dir}bdeck_current_reverse
        while read fcurrent; do
          current_list_check=$(echo $fcurrent | awk -F '-bcp' '{print $2}' | cut -c1-6)

          if [[ "$master_list_num" = "$current_list_check" ]]
          then
            # We have a match...write to basin current storms list.
            # This is used by driver script to check geolocation of MESO sector
             echo $fcurrent >> ${bdeck_file_working_dir}CP_Basin_current_storms_list
             let storm_match=$storm_match+1
             break_loop=1  # We only want the first match which will be the most recent file in bdeck list for this storm
          else
             # No match keep searching through list
             break_loop=0
          fi

          if [ "$break_loop" -eq "1" ]
          then
             break
          fi

        done < "$filename_current"
      done < "$filename_master"

      echo Current storms being tracked= $storm_match
      echo ' '

      # If no current storms currently being tracked, create an empty AL_Basin_current_storms_list file to
      # accomodate new storm check below
      if [ "$storm_match" -eq "0" ]
      then
         touch ${bdeck_file_working_dir}CP_Basin_current_storms_list
      fi

      echo "Now check 'current' bdeck file list for any new storms being tracked:"
      # Loop through current bdeck file listing again and compare to updated Basin current storms list for any new storms
      # that may have started being tracked since the last update.

      storm_match=0
      # loop through current bdeck list
      filename_current=${bdeck_file_working_dir}bdeck_current_reverse
      while read fcurrent; do
         current_list_check=$(echo $fcurrent | awk -F '-bcp' '{print $2}' | cut -c1-6)

         # Look to see if this is a new storm (i.e., not in the current basin file)
         grep -q "bcp$current_list_check" ${bdeck_file_working_dir}CP_Basin_current_storms_list
         check_new_storm=$?

         if [ "$check_new_storm" = "1" ]
         then
            # We have a new storm...add this newest bdeck file to basin current storms list.
            # This is used by driver script to check geolocation of MESO sector
            echo $fcurrent >> ${bdeck_file_working_dir}CP_Basin_current_storms_list
            let storm_match=$storm_match+1
         fi

      done < "$filename_current"

      echo New storms being tracked= $storm_match
      echo ' '
      echo "The full CPAC bdeck list (via SCDR) will be used to check for geolocation with GOES-$sat $sector sector."
      echo "CP_Basin_current_storms_list:"
      echo '----------------------------------------------------------------------'
      cat ${bdeck_file_working_dir}CP_Basin_current_storms_list
      echo '----------------------------------------------------------------------'
      echo ' '

   fi # CPAC bdeck file check

fi # sat check

echo "End making of b-deck file listings."
