; Visualization tool used to view the ABI Full-Disk Enterprise AOD product
;
; Modifications:
; 
; Date           Description
; -----------------------------------------------------------------------------
; 06/21/2017     First version
;



;-------------------------------------------------------------------------------
;=============== Procedures Related with Granule Plot ==========================
;-------------------------------------------------------------------------------

FUNCTION IsLeapYear, year
  result = 0B
  IF (year Mod 4) EQ 0 THEN BEGIN
    result = 1B
    IF year Mod 100 EQ 0 AND year Mod 400 NE 0 THEN $
      result = 0B
  ENDIF
  
  RETURN, result
END 
 
;-------------------------------------------------------------------------

FUNCTION DAYOFYEAR, day, month, year
  
  MONDAY = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334]
  
  dayofyear = MONDAY[month-1] + day
  
  IF (isleapyear(year) AND month GT 2) THEN dayofyear = dayofyear + 1
  
  RETURN, dayofyear

END

;-------------------------------------------------------------------------
PRO CALENDAR_DATE, year, dayOfYear, month, dayOfMonth
  
  MONDAY = [31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365]
  IF isleapyear(year) THEN MONDAY[1:11] = MONDAY[1:11] + 1
  
  FOR i = 0, 11 DO BEGIN
     IF (dayOfYear LE MONDAY[i]) THEN BEGIN
        month = i + 1
        IF month EQ 1 THEN dayOfMonth = dayOfYear $
        ELSE dayOfMonth = dayOfYear - MONDAY[i-1]
        RETURN
     ENDIF
  ENDFOR

END

;-------------------------------------------------------------------------

FUNCTION FSC_Normalize, range, Position=position
;
;       This is a utility routine to calculate the scaling vector
;       required to position a graphics primitive of specified range
;       at a specific position in an arbitray coordinate system. The
;       scaling vector is given as a two-element array like this:
;
;          scalingVector = [translationFactor, scalingFactor]
;
;       The scaling vector should be used with the [XYZ]COORD_CONV
;       keywords of a graphics object or model. For example, if you
;       wanted to scale an X axis into the coordinate range of -0.5 to 0.5,
;       you might type something like this:
;
;          xAxis->GetProperty, Range=xRange
;          xScale = FSC_Normalize(xRange, Position=[-0.5, 0.5])
;          xAxis, XCoord_Conv=xScale

   On_Error, 1
   IF N_Params() EQ 0 THEN Message, 'Please pass range vector as argument.'

   IF (N_Elements(position) EQ 0) THEN position = [0.0D, 1.0D] ELSE $
       position=Double(position)
   range = Double(range)

   IF range[1] GE range[0] THEN BEGIN
      scale = [((position[0]*range[1])-(position[1]*range[0])) / $
          ((range[1]-range[0]) > 1e-12), (position[1]-position[0])/((range[1]-range[0]) > 1e-12)]
   ENDIF ELSE BEGIN
      scale = [((position[1]*range[0])-(position[0]*range[1])) / $
          ((range[0]-range[1]) > 1e-12), (position[1]-position[0])/((range[0]-range[1]) > 1e-12)]
      scale[1] = -scale[1]
   ENDELSE
   RETURN, scale
END

;-------------------------------------------------------------------------------

PRO satellite_colorscale, lutR, lutG, lutB
;   create color table.

   ;Create the arrays for the color scale:
   lutR=BYTARR(256)
   lutG=BYTARR(256)
   lutB=BYTARR(256)

   ; RED

   FOR INDEX = 0,50 DO BEGIN
    DYDX = 173.0 - ((173.0*INDEX)/50.0)
    LUTR(INDEX) = BYTE(DYDX)
   END

   LUTR(51:113)=0

   FOR I=114,145 DO BEGIN
    DX=FLOAT(I)-113.
    DYDX=(255.-0.)/(146.-113.)
    LUTR(I)=BYTE((DYDX*DX)+0.)
   END

   LUTR(146:230)=255

   FOR I=231,255 DO BEGIN
    DYDX=255.0 - 75.0*(I-231.0)/(22.0)
    LUTR(I)=BYTE((DYDX))
   END

   ; GREEN

   LUTG(0:36)=0

   FOR I=37,129 DO BEGIN
    DX=FLOAT(I)-36.
    DYDX=(255.-0.)/(130.-36.)
    LUTG(I)=BYTE((DYDX*DX)+0.)
   END

   LUTG(130:154)=255

   FOR I=155,250 DO BEGIN
    DX=FLOAT(I)-154.
    DYDX=(0.-255.)/(251.-154.)
    LUTG(I)=BYTE((DYDX*DX)+255.)
   END

   LUTG(251:255)=0

   ; BLUE

   FOR I=0,60 DO BEGIN
    DYDX = 255.0 - (85.0*(60-I)/60.0)
    LUTB(I)=BYTE(DYDX)
   END

   LUTB(61)=255

   FOR I=62,64 DO BEGIN
    DX=FLOAT(I)-61.
    DYDX=(238.-255.)/(65.-61.)
    LUTB(I)=BYTE((DYDX*DX)+255.)
   END

   LUTB(65:77)=238

   FOR I=78,124 DO BEGIN
    DX=FLOAT(I)-77.
    DYDX=(0.-238.)/(125.-77.)
    LUTB(I)=BYTE((DYDX*DX)+238.)
   END

   LUTB(125:255)=0

END

;-------------------------------------------------------------------------------

FUNCTION Gran_Boundary, lon, lat, limits, center_lon, center_lat
  ; limits = [minlat, minlon, maxlat, maxlon]
   
   vldLocIdx = Where(lon GE -180. AND lon LE 180. AND lat GE -90. AND lat LE 90., nVldLoc)
   IF nVldLoc EQ 0 THEN BEGIN
      limits = FltArr(4)-999.999
      center_lon = -999.999
      center_lat = -999.999
      RETURN, 0B
   ENDIF 
     
   ; collect valid location
   lonv = lon[vldLocIdx]
   latv = lat[vldLocIdx]
   
   ; get lon/lat limits
   minLat = Min(latv, MAX=maxLat)
   minLat = (minLat-1.) > (-90.)
   maxLat = (maxLat+1.) < 90.           
   center_lat = 0.5*(minLat+maxLat)
   dummy = Where(lonv GE  178. AND lonv LE  180., countP) 
   dummy = Where(lonv GE -180. AND lonv LE -178., countN)
   IF (countP GT 0 AND countN GT 0) THEN BEGIN
      ; cross the International Date Line (180/-180 Longtitude)
      eastCell = Where(lonv GE 0. AND lonv LE 180.)
      minLon = (Min(lonv[eastCell]) - 1.) > 0.
      westCell = Where(lonv GE -180. AND lonv LT 0.)
      maxLon = (Max(Lonv[westCell]) + 1.) < 0.      
      center_lon = (minLon+maxLon+360.)*0.5
      IF center_lon GT 180. THEN center_lon = center_lon-360.
   ENDIF ELSE BEGIN
      minLon = Min(lonv, MAX=maxLon)
      minLon = (minLon-1.0) > (-180.)
      maxLon = (maxLon+1.0) < 180.
      center_lon = 0.5*(minLon+maxLon)
   ENDELSE
   
   limits = [minlat, minlon, maxlat, maxlon]

   RETURN, 1B

END 


;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------


PRO fit_green_band, lon, lat, dayofyear, red, blue, green
   @abi.h
   
   ; get land/water mask (1km - > 2km)
   lwmask = BYTARR(CONUS_NX*2, CONUS_NY*2)
   OPENR, lun, /GET_LUN, ABI_DATA_PATH+'Anc'+slash+'lwmask_conus.dat'
   READU, lun, lwmask
   CLOSE, lun
   FREE_LUN, lun
   lwmask = CONGRID(lwmask, CONUS_NX, CONUS_NY)
   
   ; get coefficients
   nxsmp=70L
   nysmp=40L
   nd=46L
   latsmp=fltarr(nxsmp,nysmp)
   lonsmp=fltarr(nxsmp,nysmp)
   corr=fltarr(nxsmp,nysmp,2,nd)
   intparr=fltarr(nxsmp,nysmp,nd)
   slparr1=fltarr(nxsmp,nysmp,nd)
   slparr2=fltarr(nxsmp,nysmp,nd)
   OPENR, lun, /GET_LUN, ABI_DATA_PATH+'Anc'+slash+'rb2grn_lut5.bin'
   READU,lun,latsmp
   READU,lun,lonsmp
   READU,lun,corr
   READU,lun,intparr
   READU,lun,slparr1
   READU,lun,slparr2
   CLOSE,lun
   FREE_LUN,lun
   
   i1 = fix(float(dayofyear-1)/8.)
   slparr1_curr=slparr1(*,*,i1)
   slparr2_curr=slparr2(*,*,i1)
   intparr_curr=intparr(*,*,i1)
   idx=where(slparr1_curr gt 0,ct)
   slp1=mean(slparr1_curr(idx))
   slp2=mean(slparr2_curr(idx))
   intp=mean(intparr_curr(idx))

   ; fitting green
   green = FLTARR(CONUS_NX, CONUS_NY)+!VALUES.F_NAN   
   invldIdx = WHERE(~FINITE(blue) OR ~FINITE(red), ninv)
   
   idx=where(red gt 0.3,ct)
   if ct gt 0 then begin
       green(idx)=0.99*red(idx)+0.02
   endif
   
   idx=where(red gt 0 and red le 0.3 and lwmask ne 0, ct)
   if ct gt 0 then begin
      green(idx)=slp1*red(idx)+slp2*blue(idx)+intp
   endif
  
   bthres=0.1
   idx=where(red gt 0 and red le 0.3 and blue lt bthres and lwmask eq 0, ct)
   if ct gt 0 then begin
      green(idx)=1.07*red(idx)+0.01
   endif

   idx=where(red gt 0 and red le 0.3 and blue ge bthres and lwmask eq 0, ct)
   if ct gt 0 then begin
      green(idx)=slp1*red(idx)+slp2*blue(idx)+intp
   endif
   
   lndIdx = WHERE(lwmask ne 0, nlnd) 
   for i=0L,nlnd-1L do begin
      j = lndIdx[i]
      ixsmp=fix((lon(j)-lonsmp(0,0))+0.5)
      iysmp=fix((lat(j)-latsmp(0,0))+0.5)

      if ixsmp ge 0 and iysmp ge 0 and ixsmp lt nxsmp and $ 
         iysmp lt nysmp then begin
         if  red[j] lt 0.3  and red[j] gt 0. $
             and slparr1(ixsmp,iysmp,i1) gt 0 then begin
            green[j]=slparr1(ixsmp,iysmp,i1)*red[j]+ $
                     slparr2(ixsmp,iysmp,i1)*blue[j]+intparr(ixsmp,iysmp,i1)
         endif   
      endif
   endfor
   
   IF ninv GT 0 THEN green[invldIdx] = !VALUES.F_NAN 
END



;-------------------------------------------------------------------------------

PRO ABI_Aer_Read_AOD, granInfo
   @abi.h
   @abi_data.h

   FOR i=0,NVar[DATA_AOD]-1 DO BEGIN 
      varName = abiAerVar[i].name
      dataType = abiAerVar[i].dataType
      
      data = mg_h5_getdata(granInfo.abiaod_filename, varName)
      IF (dataType EQ 1) THEN BEGIN 
         fillValue = mg_h5_getdata(granInfo.abiaod_filename, varName+'._FillValue')
         add_offset = mg_h5_getdata(granInfo.abiaod_filename, varName+'.add_offset')
         scale_factor = mg_h5_getdata(granInfo.abiaod_filename, varName+'.scale_factor')
         idx = Where(data EQ fillValue[0], nmiss)
         data = UINT(data)*scale_factor[0] + add_offset[0]
         IF nmiss GT 0 THEN data[idx] = !VALUES.F_NAN 
      ENDIF
      granInfo.dataPtrArr[abiAerVar[i].dataSetIdx] = Ptr_New(data,/No_Copy)

   ENDFOR
   
END

;-------------------------------------------------------------------------------

FUNCTION ABI_Aer_Read_ADP, granInfo, filename
   @abi.h
   @abi_data.h
   
   tokens = STRSPLIT(granInfo.strDate, '.', /EXTRACT)
   strDate = tokens[0]+tokens[1]+tokens[2]   ; yyyymmdd
   
   adpFilename = FILE_SEARCH(ABI_DATA_PATH+strDate+'_'+granInfo.strTime+slash+'*-ADPC-*_s'+$
                             granInfo.strDay+granInfo.strTime+'*.nc',COUNT=nf)
   IF nf EQ 0 THEN BEGIN
      dummy = Dialog_Message('ADP CONUS data are not available!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      RETURN, FAIL
   ENDIF
   filename = FILE_BASENAME(adpFilename[0])
   IF Ptr_Valid(granInfo.dataPtrArr[abiAerVar[NVar[DATA_AOD]].dataSetIdx]) THEN RETURN, SUCCEED
   
   prevDataSetId = -1
   FOR i=NVar[DATA_AOD],NVar[DATA_AOD]+NVar[DATA_ADP]-1 DO BEGIN   ; 
      IF abiAerVar[i].dataSetIdx EQ prevDataSetId THEN CONTINUE
      
      varName = abiAerVar[i].name
      dataType = abiAerVar[i].dataType
      
      tokens = STRSPLIT(varName, '@', /EXTRACT)
      varName = tokens[0]
      
      data = mg_h5_getdata(adpFilename[0],varName)
      IF (dataType EQ 1) THEN BEGIN 
         fillValue = mg_h5_getdata(adpFilename[0], varName+'._FillValue')
         add_offset = mg_h5_getdata(adpFilename[0], varName+'.add_offset')
         scale_factor = mg_h5_getdata(adpFilename[0], varName+'.scale_factor')
         idx = Where(data EQ fillValue[0], nmiss)
         data = UINT(data)*scale_factor[0] + add_offset[0]
         IF nmiss GT 0 THEN data[idx] = !VALUES.F_NAN
      ENDIF
      granInfo.dataPtrArr[abiAerVar[i].dataSetIdx] = Ptr_New(data,/No_Copy)
      prevDataSetId = abiAerVar[i].dataSetIdx
   ENDFOR
   
   RETURN, SUCCEED
END

;-------------------------------------------------------------------------------

FUNCTION ABI_Aer_Read_CldMsk, varIdx, granInfo, filename
   @abi.h
   @abi_data.h
   
   tokens = STRSPLIT(granInfo.strDate, '.', /EXTRACT)
   strDate = tokens[0]+tokens[1]+tokens[2]   ; yyyymmdd
   
   cmFilename = FILE_SEARCH(ABI_DATA_PATH+strDate+'_'+granInfo.strTime+slash+'*-ACMC-*_s'+$
                            granInfo.strDay+granInfo.strTime+'*.nc',COUNT=nf)
   IF nf EQ 0 THEN BEGIN
      dummy = Dialog_Message('CONUS cloud mask data is not available!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      RETURN, FAIL
   ENDIF
   filename = FILE_BASENAME(cmFilename[0])
   
   varInfo = abiAerVar[varIdx]
   dataPtrIdx = varInfo.dataSetIdx
   IF Ptr_Valid(granInfo.dataPtrArr[dataPtrIdx]) THEN RETURN, SUCCEED
   
   ; get data
   data = mg_h5_getdata(cmFilename[0],varInfo.name)
   
   ; create the data pointer
   granInfo.dataPtrArr[dataPtrIdx] = Ptr_New(data,/No_Copy)

   RETURN, SUCCEED
END

;-------------------------------------------------------------------------------

FUNCTION ABI_Aer_Read_Geo, varidx, granInfo, filename
   @abi.h
   @abi_data.h
   
   varInfo = abiAerVar[varIdx]
   dataPtrIdx = varInfo.dataSetIdx
   
   IF (varInfo.name EQ 'SatZen2KM') OR (varInfo.name EQ 'SatAzi2KM') THEN BEGIN 
      filename = ''
      IF Ptr_Valid(granInfo.dataPtrArr[dataPtrIdx]) THEN RETURN, SUCCEED
      data = FLTARR(CONUS_NX,CONUS_NY)
      OPENR, lun, /GET_LUN,  ABI_DATA_PATH+'Anc'+slash+varInfo.name+'.dat'
      READU, lun, data
      CLOSE, lun
      FREE_LUN, lun 
   ENDIF ELSE BEGIN
      tokens = STRSPLIT(granInfo.strDate, '.', /EXTRACT)
      strDate = tokens[0]+tokens[1]+tokens[2]   ; yyyymmdd
      auxFilename = FILE_SEARCH(ABI_DATA_PATH+strDate+'_'+granInfo.strTime+slash+'*-AUXC2-*_s'+$
                                granInfo.strDay+granInfo.strTime+'*.nc',COUNT=nf)
      IF nf EQ 0 THEN BEGIN
         dummy = Dialog_Message('CONUS auxiliary data is not available!', $
                                 Dialog_Parent=granInfo.tlb, Title='Error')
         RETURN, FAIL
      ENDIF
      filename = FILE_BASENAME(auxFilename[0])
      IF Ptr_Valid(granInfo.dataPtrArr[dataPtrIdx]) THEN RETURN, SUCCEED
   
      ; get data
      data = mg_h5_getdata(auxFilename[0],varInfo.name)
      fillValue = mg_h5_getdata(auxFilename[0], varInfo.name+'._FillValue')
      add_offset = mg_h5_getdata(auxFilename[0], varInfo.name+'.add_offset')
      scale_factor = mg_h5_getdata(auxFilename[0], varInfo.name+'.scale_factor')
      idx = Where(data EQ fillValue[0], nmiss)
      data = (UINT(data)*scale_factor[0] + add_offset[0])*!RADEG
      IF nmiss GT 0 THEN data[idx] = !VALUES.F_NAN    
   ENDELSE
   
   ; create the data pointer
   granInfo.dataPtrArr[dataPtrIdx] = Ptr_New(data,/No_Copy)
   
   RETURN, SUCCEED
END

;-------------------------------------------------------------------------------

FUNCTION ABI_Aer_Read_SDR, varidx, granInfo, filename
   @abi.h
   @abi_data.h
   
   varInfo = abiAerVar[varIdx]
   varName = varInfo.name
   dataPtrIdx = varInfo.dataSetIdx
   dataType = varInfo.dataType
   channel = varInfo.dataGrp/10
   sdrName = 'C'+String(channel, Format='(I2.2)') ; e.g. 'C03'

   tokens = STRSPLIT(granInfo.strDate, '.', /EXTRACT)
   strDate = tokens[0]+tokens[1]+tokens[2]   ; yyyymmdd
   
   sdrFilename = FILE_SEARCH(ABI_DATA_PATH+strDate+'_'+granInfo.strTime+slash+'*-CMIPC-*'+sdrName+'_G16_s'+$
                             granInfo.strDay+granInfo.strTime+'*.nc',COUNT=nf)
   IF nf EQ 0 THEN BEGIN
      dummy = Dialog_Message('ABI CONUS SDR data is not available!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      RETURN, FAIL
   ENDIF
   filename = FILE_BASENAME(sdrFilename[0])
   IF Ptr_Valid(granInfo.dataPtrArr[dataPtrIdx]) THEN RETURN, SUCCEED
   
   data = mg_h5_getdata(sdrFilename[0], varName)
   IF (dataType EQ 1) THEN BEGIN 
      fillValue = mg_h5_getdata(sdrFilename[0], varName+'._FillValue')
      add_offset = mg_h5_getdata(sdrFilename[0], varName+'.add_offset')
      scale_factor = mg_h5_getdata(sdrFilename[0], varName+'.scale_factor')
      idx = Where(data EQ fillValue[0], nmiss)
      data = UINT(data)*scale_factor[0] + add_offset[0]
      IF nmiss GT 0 THEN data[idx] = !VALUES.F_NAN 
   ENDIF
   granInfo.dataPtrArr[dataPtrIdx] = Ptr_New(data,/No_Copy)

   RETURN, SUCCEED
END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_Initialize, granInfo, succeed
   @abi.h
   @abi_data.h
   
   slash = PATH_SEP()
   succeed = 0B
   
   ; check the existence of output file
   IF STRLEN(granInfo.abiaod_filename) EQ 0 THEN BEGIN
      dummy = Dialog_Message('AOD output file is not available!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      Return
   ENDIF
   
   ; clean up
   ABI_CONUS_CleanUp_Pro, granInfo
   
   ; get CONUS 2km longitude/latitude
   lon = FLTARR(CONUS_NX,CONUS_NY)
   lat = lon
   OPENR, lun, /GET_LUN, ABI_DATA_PATH+'Anc'+slash+'Lon2KM.dat'
   READU, lun, lon
   CLOSE, lun
   OPENR, lun, ABI_DATA_PATH+'Anc'+slash+'Lat2KM.dat'
   READU, lun, lat
   CLOSE, lun
   FREE_LUN, lun
   granInfo.ipLonPtr = Ptr_New(lon, /No_Copy)
   granInfo.ipLatPtr = Ptr_New(lat, /No_Copy)

   ; read AOD outputs
   ABI_Aer_Read_AOD, granInfo
   
   ; get granule location info
   dummy =Gran_Boundary(*granInfo.ipLonPtr, *granInfo.ipLatPtr, $
                        latLongLimits, centerLon, centerLat)  
   granInfo.latLongLimits = latLongLimits
   granInfo.centerLon = centerLon
   granInfo.centerLat = 0
  ; granInfo.centerLat = centerLat
   
   ; create the original image
   granInfo.image_xsize = granInfo.draw_xsize * 0.9 ; =936
   granInfo.image_ysize = granInfo.draw_ysize * 0.7 ; =560
   
   IF granInfo.varIdx LE -1 AND granInfo.varIdx GE -3 THEN BEGIN
      ABI_CONUS_Generate_RGB_Image, granInfo 
   ENDIF ELSE BEGIN
      ABI_CONUS_Generate_Image, granInfo
   ENDELSE
   succeed = 1B
END

;-------------------------------------------------------------------------------
PRO ABI_CONUS_Generate_RGB_Image, granInfo
   @abi.h
   @abi_data.h
   
   ; get Reflectance at B1(blue) and B2(red) at 2km resolution
   
   NV = TOTAL(NVar[0:DATA_GEO])
   ND = TOTAL(NDataset[0:DATA_GEO])
   
   CASE granInfo.varIdx OF
   -1: BEGIN   ; Synthetirc RGB
       
       ; get B1(blue) and B2 (red) reflectance and angles
       status = ABI_Aer_Read_SDR(NV, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Band1 reflectance data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       status = ABI_Aer_Read_SDR(NV+2, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Band2 reflectance data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       NG = TOTAL(NVar[0:DATA_CLDMSK])
       NGD = TOTAL(NDataset[0:DATA_CLDMSK])
       status = ABI_Aer_Read_Geo(NG, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Solar zenith angle data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       status = ABI_Aer_Read_Geo(NG+1, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Satellite zenith angle data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       status = ABI_Aer_Read_Geo(NG+4, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Relative azimuth angle data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
   
       blue = *(granInfo.dataPtrArr[ND])
       red  = *(granInfo.dataPtrArr[ND+2])
       solzen = *(granInfo.dataPtrArr[NGD])
       satzen = *(granInfo.dataPtrArr[NGD+1])
       relazi = *(granInfo.dataPtrArr[NGD+4])
       vldIdx = WHERE(FINITE(blue) AND FINITE(red), nv)
       IF nv EQ 0 THEN BEGIN
          dummy = Dialog_Message('No valid data for Synthetic RGB image!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       
       ; atmospheric correction
       status = atmcor_br(solzen, satzen, relazi, blue, red)
       IF (status EQ 1) THEN BEGIN
          dummy = Dialog_Message('Could perform atmcor for Synthetic RGB image!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       
       ; get synthetic green band reflectance
       fit_green_band, (*granInfo.ipLonPtr), (*granInfo.ipLatPtr), (LONG(granInfo.strDay) MOD 1000), $
                       red, blue, green
 
       ; scale
       imageData = ENHANCE(red, green, blue)
           
       imageTitle = 'RGB Image'+ $
                    ' ('+ granInfo.strDate+' '+STRMID(granInfo.strTime,0,2)+':'+ $
                    STRMID(granInfo.strTime,2,2)+' UTC)'
       chnTitle = 'R=0.64um, G=Synthetic, B=0.47um' 
       END
      
   -2: BEGIN   ; Dust RGB
     
       ; get BT at C11, C14 and C15
       NV = TOTAL(NVar[0:DATA_GEO])
       ND = TOTAL(NDataset[0:DATA_GEO])
       status = ABI_Aer_Read_SDR(NV+12, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Band11 BT data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       status = ABI_Aer_Read_SDR(NV+14, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Band14 BT Data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       status = ABI_Aer_Read_SDR(NV+16, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Band15 BT Data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
   
       bt11Ptr =  granInfo.dataPtrArr[ND+12]
       bt14Ptr =  granInfo.dataPtrArr[ND+14]
       bt15Ptr =  granInfo.dataPtrArr[ND+16]
   
       vldIdx = WHERE(FINITE(*bt11Ptr) AND FINITE(*bt14Ptr) AND FINITE(*bt15Ptr), nv)
       IF nv EQ 0 THEN BEGIN
          dummy = Dialog_Message('No valid data for Dust RGB image!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
   
       red = FLTARR(CONUS_NX, CONUS_NY)
       green = red
       blue = red
       red[vldIdx]   = GmaScl(((*bt15Ptr)[vldIdx]-(*bt14Ptr)[vldIdx]),Gamma=1.0,MIN=-4.,MAX=2.0)
       green[vldIdx] = GmaScl(((*bt14Ptr)[vldIdx]-(*bt11Ptr)[vldIdx]),GAMMA=2.5,MIN=-4.,MAX=5.0)
       blue[vldIdx]  = GmaScl((*bt14Ptr)[vldIdx],GAMMA=1.0,MIN=261.,MAX=289.)
       imageData = BYTARR(3, CONUS_NX, CONUS_NY)
       imageData[0,*,*] = red
       imageData[1,*,*] = green
       imageData[2,*,*] = blue
       
       imageTitle = 'Dust RGB Image'+ $
                    ' ('+ granInfo.strDate+' '+STRMID(granInfo.strTime,0,2)+':'+ $
                    STRMID(granInfo.strTime,2,2)+' UTC)'
       chnTitle = 'R=BT@12um-BT@11um, G=BT@11um-BT@8.5um, B=BT@11um' 
       END
      
   -3: BEGIN   ; Natural RGB
   
       ; get B5(blue) and B3(green) and B2 (red) reflectance
       status = ABI_Aer_Read_SDR(NV+2, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Band2 reflectance data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       status = ABI_Aer_Read_SDR(NV+4, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Band3 reflectance data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       status = ABI_Aer_Read_SDR(NV+8, granInfo, filename)
       IF status NE SUCCEED THEN BEGIN
          dummy = Dialog_Message('Band5 reflectance data are not available!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
   
       rRef  = *(granInfo.dataPtrArr[ND+8])
       gRef = *(granInfo.dataPtrArr[ND+4])
       bRef   = *(granInfo.dataPtrArr[ND+2])
       vldIdx = WHERE(FINITE(bRef) AND FINITE(gRef) AND FINITE(rRef), nv)
       IF nv EQ 0 THEN BEGIN
          dummy = Dialog_Message('No valid data for Natural RGB image!', $
                                  Dialog_Parent=granInfo.tlb, Title='Error')
          RETURN
       ENDIF
       
       red = FLTARR(CONUS_NX, CONUS_NY)
       green = red
       blue = red
  
       scaled = BytScl(rRef[vldIdx],MIN=0.,MAX=1.0)
       red[vldIdx] = scaled
       scaled = BytScl(gRef[vldIdx],MIN=0.,MAX=1.0)
       green[vldIdx] = scaled
       scaled = BytScl(bRef[vldIdx],MIN=0.,MAX=1.0)
       blue[vldIdx] = scaled
       
       imageData = BYTARR(3, CONUS_NX, CONUS_NY)
       imageData[0,*,*] = red
       imageData[1,*,*] = green
       imageData[2,*,*] = blue
       
       imageTitle = 'Natural Color RGB Image'+ $
                    ' ('+ granInfo.strDate+' '+STRMID(granInfo.strTime,0,2)+':'+ $
                    STRMID(granInfo.strTime,2,2)+' UTC)'
       chnTitle = 'R=1.61um, G=0.865um, B=0.64um' 
       END
   ENDCASE
   
   granInfo.imageDataPtr = PTR_NEW(imageData, /No_Copy)
   
   ;--- Generate RGB image --- 
   
   position = [0., 0., 1., 1.]
   Set_Plot,'Z'
   Erase
   Device,Set_Resolution=[granInfo.image_xsize, granInfo.image_ysize]
   Map_Set,granInfo.centerLat,granInfo.centerLon, /CYLIN, $ 
           Position=position, Limit=granInfo.latlonglimits
   
   lon = *granInfo.ipLonPtr
   lat = *granInfo.ipLatPtr
   pix = Convert_Coord(lon, lat, /Data, /To_Normal)
   x = Reform(Round(pix[0,*]*(granInfo.image_xsize-1)))
   y = Reform(Round(pix[1,*]*(granInfo.image_ysize-1)))
   image = BytArr(3,granInfo.image_xsize, granInfo.image_ysize)
   oneChannel = BytArr(granInfo.image_xsize, granInfo.image_ysize)
   oneChannel[x,y] = red
   image[0,*,*] = oneChannel 
   oneChannel[x,y] = green
   image[1,*,*] = oneChannel
   oneChannel[x,y] = blue
   image[2,*,*] = oneChannel
   
   ; add continent border and title
   Map_Continents,COLOR=lineColor, MLINETHICK=1, /HIRES, /CONTINENTS
   Map_Grid, COLOR=lineColor,/label,GLINESTYLE=1, CHARSIZE=0.8, $
             LONLAB=granInfo.latlonglimits[0]+0.5,$
             LATLAB=granInfo.latlonglimits[3]-0.5 
   background=TVRD()
   pixels = Where(Temporary(background) EQ lineColor AND image[0,*,*] EQ 0, nct) 
   IF nct GT 0 THEN BEGIN
      FOR k=0,2 DO BEGIN
         tmp = Reform(image[k,*,*])
         tmp[pixels] = 255
         image[k,*,*] = Temporary(tmp)
      ENDFOR
   ENDIF
   
   pos = [0.05, 0.15, 0.95, 0.85]
   xc = FSC_Normalize([0,granInfo.image_xsize], Position=[pos[0], pos[2]])
   yc = FSC_Normalize([0,granInfo.image_ysize], Position=[pos[1], pos[3]])
   granInfo.theImage = Obj_New('IDLgrImage', image,/No_Copy, $
                          XCoord_Conv=xc, YCoord_Conv=yc)
   
   ;--- Title ---
   
   pos = [0.05, 0.15, 0.95, 0.85]
   helvetica18pt = Obj_New('IDLgrFont', 'Helvetica', Size=18)
   plotTitle = Obj_New('IDLgrText', imageTitle, Color=titleColor, $
                       Location=[0.5, pos(3)+0.045, 0.0], Alignment=0.5,$
                       Font=helvetica18pt, Recompute_Dimensions=1)
   plotTitle2 = Obj_New('IDLgrText', chnTitle, Color=titleColor, $
                       Location=[0.5, pos(3)+0.015, 0.0], Alignment=0.5,$
                       Font=helvetica8pt, Recompute_Dimensions=1)
   
   ;--- Generate image objects ---
   
   IF Obj_Valid(granInfo.theContainer) THEN $
      Obj_Destroy, granInfo.theContainer
   
   granInfo.theModel = Obj_New('IDLgrModel')
   granInfo.theModel->Add, granInfo.theImage
   granInfo.theModel->Add, plotTitle
   granInfo.theModel->Add, plotTitle2
   granInfo.theView = Obj_New('IDLgrView', Viewplane_Rect=[0,0,1,1], $
                          Location=[0,0], Color=backGroundColor)
   granInfo.theView->Add, GranInfo.theModel
   granInfo.theContainer = Obj_New('IDL_Container')
   granInfo.theContainer->Add, granInfo.theModel
   granInfo.theContainer->Add, granInfo.theView
   granInfo.theContainer->Add, granInfo.theImage
   granInfo.theWindow->Draw, granInfo.theView
   
   CASE STRUPCASE(!VERSION.OS_FAMILY) OF
      'WINDOWS': newDevice = 'WIN'
      'UNIX':    newDevice = 'X'
      'MACOS':   newDevice = 'MAC'
      ELSE:      MESSAGE, 'Unknown OS_FAMILY.'
   ENDCASE
   SET_PLOT, newDevice
  
   Widget_Control, granInfo.colorID, SENSITIVE=0
   Widget_Control, granInfo.optionID, SENSITIVE=0
END

;-------------------------------------------------------------------------------
PRO ABI_CONUS_Generate_DstRGB_Image, granInfo
   @abi.h
   @abi_data.h
   
   ; get BT at C11, C14 and C15
   NV = TOTAL(NVar[0:DATA_GEO])
   ND = TOTAL(NDataset[0:DATA_GEO])
   status = ABI_Aer_Read_SDR(NV+12, granInfo, filename)
   IF status NE SUCCEED THEN BEGIN
      dummy = Dialog_Message('Band11 BT data are not available!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      RETURN
   ENDIF
   status = ABI_Aer_Read_SDR(NV+14, granInfo, filename)
   IF status NE SUCCEED THEN BEGIN
      dummy = Dialog_Message('Band14 BT Data are not available!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      RETURN
   ENDIF
   status = ABI_Aer_Read_SDR(NV+16, granInfo, filename)
   IF status NE SUCCEED THEN BEGIN
      dummy = Dialog_Message('Band15 BT Data are not available!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      RETURN
   ENDIF
   
   bt11Ptr =  granInfo.dataPtrArr[ND+12]
   bt14Ptr =  granInfo.dataPtrArr[ND+14]
   bt15Ptr =  granInfo.dataPtrArr[ND+16]
   
   vldIdx = WHERE(FINITE(*bt11Ptr) AND FINITE(*bt14Ptr) AND FINITE(*bt15Ptr), nv)
   IF nv EQ 0 THEN BEGIN
      dummy = Dialog_Message('No valid data for Dust RGB image!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      RETURN
   ENDIF
   
   red = FLTARR(CONUS_NX, CONUS_NY)
   green = red
   blue = red
   red[vldIdx]   = GmaScl(((*bt15Ptr)[vldIdx]-(*bt14Ptr)[vldIdx]),Gamma=1.0,MIN=-4.,MAX=2.0)
   green[vldIdx] = GmaScl(((*bt14Ptr)[vldIdx]-(*bt11Ptr)[vldIdx]),GAMMA=2.5,MIN=-4.,MAX=5.0)
   blue[vldIdx]  = GmaScl((*bt14Ptr)[vldIdx],GAMMA=1.0,MIN=261.,MAX=289.)
   imageData = BYTARR(3, CONUS_NX, CONUS_NY)
   imageData[0,*,*] = red
   imageData[1,*,*] = green
   imageData[2,*,*] = blue
   granInfo.imageDataPtr = PTR_NEW(imageData, /No_Copy)

   
   ;--- Generate RGB image --- 
   
   position = [0., 0., 1., 1.]
   Set_Plot,'Z'
   Erase
   Device,Set_Resolution=[granInfo.image_xsize, granInfo.image_ysize]
   Map_Set,granInfo.centerLat,granInfo.centerLon, /CYLIN, $ 
           Position=position, Limit=granInfo.latlonglimits
   
   lon = *granInfo.ipLonPtr
   lat = *granInfo.ipLatPtr
   pix = Convert_Coord(lon, lat, /Data, /To_Normal)
   x = Reform(Round(pix[0,*]*(granInfo.image_xsize-1)))
   y = Reform(Round(pix[1,*]*(granInfo.image_ysize-1)))
   image = BytArr(3,granInfo.image_xsize, granInfo.image_ysize)
   oneChannel = BytArr(granInfo.image_xsize, granInfo.image_ysize)
   oneChannel[x,y] = red
   image[0,*,*] = oneChannel 
   oneChannel[x,y] = green
   image[1,*,*] = oneChannel
   oneChannel[x,y] = blue
   image[2,*,*] = oneChannel
   
   ; add continent border and title
   Map_Continents,COLOR=lineColor, MLINETHICK=1, /HIRES, /CONTINENTS
   Map_Grid, COLOR=lineColor,/label,GLINESTYLE=1, CHARSIZE=0.8, $
             LONLAB=granInfo.latlonglimits[0]+0.5,$
             LATLAB=granInfo.latlonglimits[3]-0.5 
   background=TVRD()
   pixels = Where(Temporary(background) EQ lineColor AND image[0,*,*] EQ 0, nct) 
   IF nct GT 0 THEN BEGIN
      FOR k=0,2 DO BEGIN
         tmp = Reform(image[k,*,*])
         tmp[pixels] = 255
         image[k,*,*] = Temporary(tmp)
      ENDFOR
   ENDIF
   
   pos = [0.05, 0.15, 0.95, 0.85]
   xc = FSC_Normalize([0,granInfo.image_xsize], Position=[pos[0], pos[2]])
   yc = FSC_Normalize([0,granInfo.image_ysize], Position=[pos[1], pos[3]])
   granInfo.theImage = Obj_New('IDLgrImage', image,/No_Copy, $
                          XCoord_Conv=xc, YCoord_Conv=yc)
   
   ;--- Title ---
   
   pos = [0.05, 0.15, 0.95, 0.85]
   helvetica18pt = Obj_New('IDLgrFont', 'Helvetica', Size=18)
   imageTitle = 'Dust RGB Image'+ $
                ' ('+ granInfo.strDate+' '+STRMID(granInfo.strTime,0,2)+':'+ $
                STRMID(granInfo.strTime,2,2)+' UTC)'
   chnTitle = 'R=BT@12um-BT@11um, G=BT@11um-BT@8.5um, B=BT@11um' 
   plotTitle = Obj_New('IDLgrText', imageTitle, Color=titleColor, $
                       Location=[0.5, pos(3)+0.045, 0.0], Alignment=0.5,$
                       Font=helvetica18pt, Recompute_Dimensions=1)
   plotTitle2 = Obj_New('IDLgrText', chnTitle, Color=titleColor, $
                       Location=[0.5, pos(3)+0.015, 0.0], Alignment=0.5,$
                       Font=helvetica8pt, Recompute_Dimensions=1)
   
   ;--- Generate image objects ---
   
   IF Obj_Valid(granInfo.theContainer) THEN $
      Obj_Destroy, granInfo.theContainer
   
   granInfo.theModel = Obj_New('IDLgrModel')
   granInfo.theModel->Add, granInfo.theImage
   granInfo.theModel->Add, plotTitle
   granInfo.theModel->Add, plotTitle2
   granInfo.theView = Obj_New('IDLgrView', Viewplane_Rect=[0,0,1,1], $
                          Location=[0,0], Color=backGroundColor)
   granInfo.theView->Add, GranInfo.theModel
   granInfo.theContainer = Obj_New('IDL_Container')
   granInfo.theContainer->Add, granInfo.theModel
   granInfo.theContainer->Add, granInfo.theView
   granInfo.theContainer->Add, granInfo.theImage
   granInfo.theWindow->Draw, granInfo.theView
   
   CASE STRUPCASE(!VERSION.OS_FAMILY) OF
      'WINDOWS': newDevice = 'WIN'
      'UNIX':    newDevice = 'X'
      'MACOS':   newDevice = 'MAC'
      ELSE:      MESSAGE, 'Unknown OS_FAMILY.'
   ENDCASE
   SET_PLOT, newDevice
  
   Widget_Control, granInfo.colorID, SENSITIVE=0
   Widget_Control, granInfo.optionID, SENSITIVE=0
   
END

;-------------------------------------------------------------------------------
PRO ABI_CONUS_Generate_NatRGB_Image, granInfo
   @abi.h
   @abi_data.h
   
   ; To be modified
   
END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_Generate_Image, granInfo, DATA_RANGE=data_range, RESTRICTED=restricted
   @abi.h
   @abi_data.h
   
   ;--- Get variable info ---
   varIdx = granInfo.varIdx
   IF varIdx GE 2000 THEN varIdx = varIdx - 2000   ; combination plot
   varInfo = abiAerVar[varIdx]
   dataGrp = varInfo.dataGrp MOD 10 
   dataPtrIdx = varInfo.dataSetIdx
   dataType = varInfo.dataType
   varName = varInfo.name
   imageTitle = varInfo.imageTitle
   astImageTitle = ''
   cbNlevels = varInfo.nLevels
   IF (dataType EQ 1) THEN BEGIN
      nColors = 254
      IF N_ELEMENTS(data_range) GT 0 THEN dataRange=data_range $
      ELSE dataRange = varInfo.dataRange
   ENDIF ELSE BEGIN
      dataRange = [0, cbNlevels]
      nColors = cbNlevels
      cbText = *varInfo.annotation
   ENDELSE
   granInfo.data_range = dataRange
   
   ; get data 
   CASE dataGrp OF
      DATA_AOD: BEGIN  ; AOD 
         filename = FILE_BASENAME(granInfo.abiaod_filename)
         END
         
      DATA_ADP: BEGIN   ; ADP
         success = ABI_Aer_Read_ADP(granInfo, filename)
         IF success EQ FAIL THEN RETURN
         END
         
     DATA_CLDMSK : BEGIN   ; 
         success = ABI_Aer_Read_CldMsk(varIdx, granInfo, filename)
         IF success EQ FAIL THEN RETURN
         END
         
     DATA_GEO : BEGIN   ; 
         success = ABI_Aer_Read_Geo(varIdx, granInfo, filename)
         IF success EQ FAIL THEN RETURN
         END
         
     DATA_SDR : BEGIN   ;  
         success = ABI_Aer_Read_SDR(varIdx, granInfo, filename)
         IF success EQ FAIL THEN RETURN
         END
         
   ENDCASE 
   
   data = *granInfo.dataPtrArr[dataPtrIdx]
   IF (dataType EQ 0) THEN BEGIN  
      tokens = StrSplit(varName,'@',/Extract) 
      IF N_Elements(tokens) GT 1 THEN BEGIN  ; Quality flags
         qfBits = FIX(tokens[1:N_Elements(tokens)-1])
         mask = 0
         FOR k=0, N_Elements(qfBits)-1 DO $
            mask = mask + Ishft(1,qfBits[k])
         data = Ishft(data AND mask, qfBits[0]*(-1))
      ENDIF
   ENDIF
   
   ; get data for combination plot
   IF granInfo.varIdx GE 2000 THEN BEGIN
      
      ; primary data for plot
      
      oriData = data
      IF dataType EQ 0 THEN BEGIN
         ; Integer data
         data = oriData*0 + dataRange[1]+1
         fltMisVal = dataRange[1]+1
      ENDIF ELSE BEGIN
         data = oriData*0. + !VALUES.F_NAN
         fltMisVal = !VALUES.F_NAN
      ENDELSE
      
      ; associated data (quality flags)
      
      astVarIdx = *granInfo.astVarIdx
      astVarVal = *granInfo.astVarVal            
      FOR i=0,N_Elements(astVarIdx)-1 DO BEGIN
         ; get assoicated data (quality flags)
         astVarInfo = abiAerVar[astVarIdx[i]]
         astDataPtrIdx = astVarInfo.dataSetIdx
         CASE astVarInfo.dataGrp OF
            DATA_ADP: BEGIN   ; ADP
               success = ABI_Aer_Read_ADP(granInfo, filename)
               IF success EQ FAIL THEN RETURN
             END
             ELSE:
         ENDCASE 
         astVarData = *granInfo.dataPtrArr[astDataPtrIdx]
         tokens = StrSplit(astVarInfo.name,'@',/Extract) 
         IF N_Elements(tokens) GT 1 THEN BEGIN 
           qfBits = FIX(tokens[1:N_Elements(tokens)-1])
           mask = 0
           FOR k=0, N_Elements(qfBits)-1 DO $
              mask = mask + Ishft(1,qfBits[k])
           astVarData = Ishft(astVarData AND mask, qfBits[0]*(-1))
         ENDIF
                 
         IF astVarVal[i] LT 0 THEN BEGIN   ; OR
            trueAstVarVal = astVarVal[i] + 1000
            idx = Where(astVarData EQ trueAstVarVal, vc)
            IF vc GT 0 THEN data[idx] = oriData[idx]
         ENDIF ELSE BEGIN                 ; AND
            trueAstVarVal = astVarVal[i]
            idx = Where(astVarData NE trueAstVarVal, vc)
            IF vc GT 0 THEN data[idx] = fltMisVal
         ENDELSE
         
         tmpTitle = astVarInfo.imageTitle
         dummy = StrSplit(tmpTitle, '-', /Extract)
         IF N_Elements(dummy) EQ 2 THEN tmpTitle = StrMid(dummy[1],1) 
         IF i EQ 0 THEN BEGIN
            astImageTitle = tmpTitle+'='+(*astVarInfo.annotation)[trueAstVarVal]
         ENDIF ELSE BEGIN
            IF astVarVal[i] LT 0  THEN optStr = ' -OR- ' $
            ELSE optStr = ' -AND- ' 
            astImageTitle = astImageTitle+ optStr + tmpTitle+'='+$
                         (*astVarInfo.annotation)[trueAstVarVal]
         ENDELSE
      ENDFOR
      oriData = 0
   ENDIF
   granInfo.imageDataPtr = PTR_NEW(data)
   
   ;--- Generate Image ---
   
   ; lon/lat
   lonPtr = granInfo.ipLonPtr
   latPtr = granInfo.ipLatPtr
   
   IF ~Ptr_Valid(granInfo.dataPtrArr[dataPtrIdx]) OR $
      ~Ptr_Valid(lonPtr) OR ~Ptr_Valid(latPtr) THEN BEGIN
      dummy = Dialog_Message('Data are not available!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      Return
   ENDIF
   
   lon = *lonPtr
   lat = *latPtr
   vldLocIdx = Where(lon GE -180. AND lon LE 180. AND lat GE -90. AND lat LE 90. AND  $
                     Finite(data), nVldLoc, NCOMPLEMENT=nInvalidLoc)
   
   IF nVldLoc EQ 0 THEN BEGIN
      dummy = Dialog_Message('No valid Data!', $
                              Dialog_Parent=granInfo.tlb, Title='Error')
      Return   
   ENDIF
   
   position = [0., 0., 1., 1.]
   Set_Plot,'Z'
   Erase
   Device,Set_Resolution=[granInfo.image_xsize, granInfo.image_ysize]
   ;Map_Set,granInfo.centerLat,granInfo.centerLon, /Satellite, $
   ;        Position=position, Limit=granInfo.latlonglimits,SAT_P=[6.6,0,0]
   Map_Set,granInfo.centerLat,granInfo.centerLon, /CYLIN, $ 
           Position=position, Limit=granInfo.latlonglimits
   
   
   pix = Convert_Coord(lon, lat, /Data, /To_Normal)
   x = Reform(Round(pix[0,*]*(granInfo.image_xsize-1)))
   y = Reform(Round(pix[1,*]*(granInfo.image_ysize-1)))
   image = BytArr(granInfo.image_xsize, granInfo.image_ysize)+missColor
   
   IF (dataType EQ 0) THEN BEGIN
      ; Integer data
      missIdx = Where(data[vldLocIdx] LT dataRange[0] OR $
                      data[vldLocIdx] GT dataRange[1], nmiss)
      imageData = BYTE(data[vldLocIdx])
   ENDIF ELSE BEGIN 
      ; Floating-point data
      IF KEYWORD_SET(restricted) THEN BEGIN
         missIdx = WHERE(~Finite(data[vldLocIdx]) OR $
                         data[vldLocIdx] LT dataRange[0] OR $
                         data[vldLocIdx] GT dataRange[1], nmiss)
      ENDIF ELSE missIdx = Where(~Finite(data[vldLocIdx]), nmiss)
      imageData = BytScl(data[vldLocIdx],/NaN, min=dataRange[0], $
                         max=dataRange[1], top=numTabCol-1)
   ENDELSE 
   IF nmiss GT 0 THEN imageData[missIdx] = missColor
   image[x[vldLocIdx],y[vldLocIdx]] = Temporary(imageData)
         
   ; create pixel-mapping array
   pixMap = LONARR(granInfo.image_xsize, granInfo.image_ysize)-1
   pixMap[x,y] = LINDGEN(N_ELEMENTS(lon))
   IF ~Ptr_Valid(granInfo.ipLocMapPtr) THEN PTR_FREE, granInfo.ipLocMapPtr
   granInfo.ipLocMapPtr = Ptr_New(pixMap, /No_Copy)
      
   ; add continent border lines
   Map_Continents,COLOR=lineColor, MLINETHICK=1, /HIRES, /CONTINENTS
   Map_Grid, COLOR=lineColor,/label,GLINESTYLE=1, CHARSIZE=0.8, $
             LONLAB=granInfo.latlonglimits[0]+0.5,$
             LATLAB=granInfo.latlonglimits[3]-0.5 
   background=TVRD()
   coastLine_pixels=Where(Temporary(background) EQ lineColor, nct) 
   IF nct GT 0 THEN image[coastLine_pixels] = lineColor
   
   ;--- Generate image objects ---
   
   IF Obj_Valid(granInfo.theContainer) THEN $
      Obj_Destroy, granInfo.theContainer
   
   ; color
   IF dataGrp EQ DATA_SDR THEN BEGIN
      LOADCT, 0, /SILENT
      TVLCT, r,g,b,/GET
   ENDIF ELSE  satellite_colorscale, r, g, b
   IF (dataType EQ 0) THEN BEGIN
      r[0:nColors-1] = discColor[0,0:nColors-1]
      g[0:nColors-1] = discColor[1,0:nColors-1]
      b[0:nColors-1] = discColor[2,0:nColors-1]
   ENDIF
   r[missColor] = colorGray[0]
   g[missColor] = colorGray[1]
   b[missColor] = colorGray[2]
   r[lineColor] = colorBlack[0]
   g[lineColor] = colorBlack[1]
   b[lineColor] = colorBlack[2]
   granInfo.thePalette = Obj_New('IDLgrPalette')
   granInfo.thePalette->SetProperty, RED=r, BLUE=b, GREEN=g 
   
   ; title
   helvetica10pt = Obj_New('IDLgrFont', 'Helvetica', Size=18)
   helvetica8pt = Obj_New('IDLgrFont', 'Helvetica', Size=12)
   imageTitle = imageTitle + $
                ' ('+ granInfo.strDate+' '+STRMID(granInfo.strTime,0,2)+':'+ $
                STRMID(granInfo.strTime,2,2)+' UTC)'
   plotTitle1 = Obj_New('IDLgrText', imageTitle, Color=titleColor, $
                       Location=[0.5, pos(3)+0.045, 0.0], Alignment=0.5,$
                       Font=helvetica10pt, Recompute_Dimensions=1)
   plotTitle2 = Obj_New('IDLgrText', filename, Color=titleColor, $
                       Location=[0.01, pos(3)+0.12, 0.0], Alignment=0,$
                       Font=helvetica8pt, Recompute_Dimensions=1)
   IF StrLen(astImageTitle) GT 0 THEN $
      plotTitle3 = Obj_New('IDLgrText', astImageTitle, Color=titleColor, $
                       Location=[0.5, pos(3)+0.015, 0.0], Alignment=0.5,$
                       Font=helvetica8pt, Recompute_Dimensions=1)
   
   ; Generate image object
   granInfo.theImage = Obj_New('IDLgrImage', image, PALETTE=granInfo.thePalette)
   xc = FSC_Normalize([0,granInfo.image_xsize], Position=[pos[0], pos[2]])
   yc = FSC_Normalize([0,granInfo.image_ysize], Position=[pos[1], pos[3]])
   granInfo.theImage->SetProperty, XCoord_Conv=xc, YCoord_Conv=yc
   
   ; Generate colorbar objects
   cbPosition = [0.05, 0.07, 0.95, 0.1]
   IF N_Elements(cbText) LE 1 THEN BEGIN
      theColorbar = Obj_New('Colorbar', Range=dataRange, NColors=nColors,$
                            Position=cbPosition, Color=titleColor, $
                            Palette=granInfo.thePalette, Major=cbNlevels+1)
   ENDIF ELSE BEGIN
      theColorbar = Obj_New('Colorbar', Range=dataRange, NColors=nColors, $
                             Position=cbPosition, Color=titleColor, $
                             Palette=granInfo.thePalette, Major=cbNlevels+1,$
                             TickName=cbText)
   ENDELSE
   
   granInfo.theModel = Obj_New('IDLgrModel')
   granInfo.theModel->Add, granInfo.theImage
   granInfo.theModel->Add, plotTitle1
   granInfo.theModel->Add, plotTitle2
   IF Obj_Valid(plotTitle3) THEN granInfo.theModel->Add, plotTitle3
   granInfo.theModel->Add, theColorbar
   
   granInfo.theView = Obj_New('IDLgrView', Viewplane_Rect=[0,0,1,1], $
                          Location=[0,0], Color=backGroundColor)
   granInfo.theView->Add, GranInfo.theModel
   granInfo.theContainer = Obj_New('IDL_Container')
   granInfo.theContainer->Add, granInfo.theModel
   granInfo.theContainer->Add, granInfo.theView
   granInfo.theContainer->Add, granInfo.thePalette
   granInfo.theContainer->Add, granInfo.theImage
   
   
   granInfo.theWindow->SetProperty, Palette=granInfo.thePalette 
   granInfo.theWindow->Draw, granInfo.theView
   
   ; block the invalid color options
   
   IF dataType EQ 0 THEN BEGIN
      Widget_Control, granInfo.imgColorID, SENSITIVE=0
      Widget_Control, granInfo.disColorID, SENSITIVE=1
      FOR i=0, nColors-1 DO BEGIN
         Widget_Control, granInfo.fgColorID[i], SENSITIVE=1
         Widget_Control, granInfo.fgColorID[i], $
                 SET_VALUE=cbText[i]
      ENDFOR
      FOR i=nColors,7 DO $
         Widget_Control, granInfo.fgColorID[i], SENSITIVE=0
      Widget_Control, granInfo.rangeID, SENSITIVE=0
   ENDIF ELSE BEGIN
      Widget_Control, granInfo.imgColorID, SENSITIVE=1
      Widget_Control, granInfo.disColorID, SENSITIVE=0
      Widget_Control, granInfo.rangeID, SENSITIVE=1
   ENDELSE
   
   CASE STRUPCASE(!VERSION.OS_FAMILY) of
      'WINDOWS': newDevice = 'WIN'
      'UNIX':    newDevice = 'X'
      'MACOS':   newDevice = 'MAC'
      ELSE:      MESSAGE, 'Unknown OS_FAMILY.'
   ENDCASE
   SET_PLOT, newDevice
   
   Widget_Control, granInfo.colorID, SENSITIVE=1
   Widget_Control, granInfo.optionID, SENSITIVE=1
   
END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_Generate_CombinedImage_Select, event
   @abi_data.h

   Widget_Control, event.top, Get_UValue=cmbInfo
   Widget_Control, event.id, Get_Uvalue=varName   

   Widget_Control, cmbInfo.varListID, Get_UValue=varList
   Widget_Control, cmbInfo.qfListID, Get_UValue=qfList
   
   tokens = StrSplit(varName,'.',/Extract)
   IF N_Elements(tokens) EQ 2 THEN BEGIN    ; QF selection
      IF cmbInfo.varID EQ -999 THEN BEGIN
         dummy = Dialog_Message('Please Select Variable First', $
                                Dialog_Parent=event.top, Title='Error')
         RETURN
      ENDIF
      
      IF tokens[0] EQ 'LOGOPT' THEN BEGIN
         ; select the AND/OR option
         
         IF tokens[1] EQ 'AND' THEN cmbInfo.qfOpt = 0 $
         ELSE cmbInfo.qfOpt = 1B
         ;Widget_Control, cmbInfo.qfOptID, SENSITIVE=0
     
      ENDIF ELSE BEGIN
         ; update the QF list
         
         idx = Where(abiAerVar.imageTitle EQ tokens[0])
         i = idx[0]
         qcVal = FIX(tokens[1])
         tmpTitle = abiAerVar[i].imageTitle
         dummy = StrSplit(tmpTitle, '-', /Extract)
         IF N_Elements(dummy) EQ 2 THEN tmpTitle = StrMid(dummy[1],1) 
         newQC = tmpTitle+' = '+(*abiAerVar[i].annotation)[qcVal]
         
         IF ~Ptr_Valid(cmbInfo.qfID) THEN BEGIN
            ; first selection
            qfList = newQC 
            cmbInfo.qfID = Ptr_New([i])
            qcVal = qcVal - 1000   ; first selection is regarded as OR
            cmbInfo.qfValue = Ptr_New([qcVal])
         ENDIF ELSE BEGIN
            ; multiple selections
                   
            IF cmbInfo.qfOpt EQ -1 THEN BEGIN
               dummy = Dialog_Message('Please Select AND/OR first', $
                                Dialog_Parent=event.top, Title='Error')
               RETURN
            ENDIF
            
            IF cmbInfo.qfOpt EQ 0 THEN BEGIN
               newQC = '-AND- '+newQC    ; positive value for AND
            ENDIF ELSE BEGIN
               newQC = '-OR- '+newQC
               qcVal = qcVal - 1000      ; minus 1000 for OR
            ENDELSE
            
            qfList = [qfList, newQC]
            existQcID = *cmbInfo.qfID
            existQcVal = *cmbInfo.qfValue
            newQcID = [existQcID, i]
            newQcVal = [existQcVal,qcVal]
            Ptr_Free, cmbInfo.qfID 
            cmbInfo.qfID = Ptr_New(newQcID)
            Ptr_Free, cmbInfo.qfValue
            cmbInfo.qfValue = Ptr_New(newQcVal)
            cmbInfo.qfOpt = -1
         ENDELSE
         Widget_Control, cmbInfo.qfOptID, SENSITIVE=1
         Widget_Control, cmbInfo.qfListID, Set_Value=qfList, Set_UValue=qfList
      ENDELSE
         
   ENDIF ELSE BEGIN    ; plot variable selection
      idx = Where(abiAerVar.imageTitle EQ varName)
      i = idx[0]
      varList = abiAerVar[i].imageTitle
      Widget_Control, cmbInfo.varListID, Set_Value=varList, Set_UValue=varList
      cmbInfo.varID = i
   ENDELSE
      
   Widget_Control, event.top, Set_UValue=cmbInfo
END
;-------------------------------------------------------------------------------

PRO ABI_CONUS_Generate_CombinedImage_Event, event
   Widget_Control, event.top, Get_UValue=cmbInfo
   
   createImage = 0B
   CASE event.ID OF
      cmbInfo.varListID: BEGIN
         IF cmbInfo.varID NE -999 THEN BEGIN
            Widget_Control, cmbInfo.varListID, Get_UValue=varList
            cmbInfo.varID = -999
            Ptr_Free, cmbInfo.qfID 
            Ptr_Free, cmbInfo.qfValue
            Widget_Control, cmbInfo.varListID, Set_Value='', Set_UValue=''
            Widget_Control, cmbInfo.qfListID, Set_Value='', Set_UValue=''
            Widget_Control, cmbInfo.qfOptID, SENSITIVE=0 
            Widget_Control, event.top, Set_UValue=cmbInfo  
         ENDIF
         END
      cmbInfo.qfListID: BEGIN
         IF Ptr_Valid(cmbInfo.qfID) THEN BEGIN
            Widget_Control, cmbInfo.qfListID, Get_UValue=qfList
            nQc = N_Elements(qfList)
            existQcID = *cmbInfo.qfID
            existQcVal = *cmbInfo.qfValue
            IF nQc EQ 1 AND event.index EQ 0 THEN BEGIN
               qfList = ''
               Ptr_Free, cmbInfo.qfID 
               Ptr_Free, cmbInfo.qfValue
               Widget_Control, cmbInfo.qfOptID, SENSITIVE=0
            ENDIF ELSE BEGIN
               CASE event.index OF
                  0: BEGIN
                     qfList = qfList[1:nQc-1]
                     newQcID = existQcID[1:nQc-1]
                     newQcVal = existQcVal[1:nQc-1]
                     END
                  nQc-1: BEGIN
                     qfList = qfList[0:nQc-2]
                     newQcID = existQcID[0:nQc-2]
                     newQcVal = existQcVal[0:nQc-2]
                     END
                  ELSE: BEGIN
                     qfList = [qfList[0:event.index-1], qfList[event.index+1:*]]
                     newQcID = [existQcID[0:event.index-1], existQcID[event.index+1:*]]
                     newQcVal = [existQcVal[0:event.index-1], existQcVal[event.index+1:*]]
                     END
               ENDCASE
               Ptr_Free, cmbInfo.qfID 
               cmbInfo.qfID = Ptr_New(newQcID)
               Ptr_Free, cmbInfo.qfValue
               cmbInfo.qfValue = Ptr_New(newQcVal)
               Widget_Control, cmbInfo.qfOptID, SENSITIVE=1
            ENDELSE
            Widget_Control, cmbInfo.qfListID, Set_Value=qfList, Set_UValue=qfList   
            Widget_Control, event.top, Set_UValue=cmbInfo     
         ENDIF
         END
      cmbInfo.acceptID: BEGIN
            IF cmbInfo.varID GE 0 AND Ptr_Valid(cmbInfo.qfID) AND $
               Ptr_Valid(cmbInfo.qfValue) THEN BEGIN
               varID = cmbInfo.varID
               qfID = Ptr_New(*cmbInfo.qfID)
               qfValue = Ptr_New(*cmbInfo.qfValue)
               granInfoPtr = cmbInfo.granInfoPtr
               createImage = 1B
               Widget_Control, event.top, /Destroy
            ENDIF ELSE BEGIN
               dummy = Dialog_Message('Selection is not complete', $
                                      Dialog_Parent=event.top, Title='Error')
               Return            
            ENDELSE
         END
      cmbInfo.cancelID: Widget_Control, event.top, /Destroy
      ELSE: Message, 'Unknown Events'
   ENDCASE
   IF ~createImage THEN Return
   
   ; create image
   
   (*granInfoPtr).varIdx = 2000+varID
   IF Ptr_Valid((*granInfoPtr).astVarIdx) THEN Ptr_Free,(*granInfoPtr).astVarIdx 
   (*granInfoPtr).astVarIdx = qfID
   IF Ptr_Valid((*granInfoPtr).astVarVal) THEN Ptr_Free,(*granInfoPtr).astVarVal 
   (*granInfoPtr).astVarVal = qfValue
   
   Widget_Control, (*granInfoPtr).tlb, HOURGLASS=1
   ABI_CONUS_Generate_Image, (*granInfoPtr)
   Widget_Control, (*granInfoPtr).tlb, HOURGLASS=0
   
END
;-------------------------------------------------------------------------------

PRO  ABI_CONUS_Generate_CombinedImage_CleanUp, id
   Widget_Control, id, GET_UVALUE=cmbInfo
   IF N_Elements(cmbInfo) NE 0 THEN BEGIN
      IF Ptr_Valid(cmbInfo.qfValue) THEN Ptr_Free, cmbInfo.qfValue
      IF Ptr_Valid(cmbInfo.qfID) THEN Ptr_Free, cmbInfo.qfID
   ENDIF
END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_Generate_CombinedImage, granInfo
   @abi.h
   @abi_data.h
   ;font = '-adobe-helvetica-bold-r-normal--14-140-75-75-p-82-iso8859-1'
   font = 'Helvetica'
   
   tlb = Widget_Base(Title='Combination Plot Selection', Row=2, /Modal, $
                     /Base_Align_Center,Group_Leader=granInfo.tlb)
   
   upperBase = Widget_Base(tlb, Column=2)
   
   ; variable selection
   varBase = Widget_Base(upperBase, ROW=2, /Align_Center)
   varMenuID = Widget_Button(varBase, Value='Select Variable', /Menu, $
               Event_Pro='ABI_CONUS_Generate_CombinedImage_Select', FONT=font)
   
   button = WIDGET_BUTTON(varMenuID,VALUE=abiAerVar[0].imageTitle, UVALUE=abiAerVar[0].imageTitle, FONT=font)
   button = WIDGET_BUTTON(varMenuID,VALUE=abiAerVar[NVar[DATA_AOD]].imageTitle, UVALUE=abiAerVar[NVar[DATA_AOD]].imageTitle, FONT=font, /SEPARATOR)
   button = WIDGET_BUTTON(varMenuID,VALUE=abiAerVar[NVar[DATA_AOD]+1].imageTitle, UVALUE=abiAerVar[NVar[DATA_AOD]+1].imageTitle, FONT=font)
   button = WIDGET_BUTTON(varMenuID,VALUE=abiAerVar[NVar[DATA_AOD]+2].imageTitle, UVALUE=abiAerVar[NVar[DATA_AOD]+2].imageTitle, FONT=font)
   varListID = Widget_List(varBase, YSize=5, XSize=30, Value='', Uvalue='')
   
   ; qf selection
   qfBase = Widget_Base(upperBase, ROW=2, /Align_Center)
   qbtnBase = Widget_Base(qfBase, COL=2, /Align_Center)
   qfOptID = Widget_Button(qbtnBase, Value='AND/OR', /Menu, SENSITIVE=0, $
               Event_Pro='ABI_CONUS_Generate_CombinedImage_Select')
   button  = WIDGET_BUTTON(qfOptID,VALUE='AND', UVALUE='LOGOPT.AND',/SEPARATOR)
   button  = WIDGET_BUTTON(qfOptID,VALUE='OR', UVALUE='LOGOPT.OR',/SEPARATOR)
   qfMenuID = Widget_Button(qbtnBase, Value='Select Quality Flag', /Menu, $
               Event_Pro='ABI_CONUS_Generate_CombinedImage_Select')
   
   FOR i=0,NVar[DATA_AOD]+NVar[DATA_ADP]-1 DO BEGIN 
      IF abiAerVar[i].dataType EQ 1 THEN CONTINUE
      IF i EQ NVar[DATA_AOD] THEN $
         button = WIDGET_BUTTON(qfMenuID,VALUE=abiAerVar[i].imageTitle, UVALUE=abiAerVar[i].imageTitle,/MENU,/SEPARATOR) $
      ELSE $
         button = WIDGET_BUTTON(qfMenuID,VALUE=abiAerVar[i].imageTitle, UVALUE=abiAerVar[i].imageTitle,/MENU)
      FOR k=0,abiAerVar[i].nLevels-1 DO $
         val = WIDGET_BUTTON(button,VALUE=(*abiAerVar[i].annotation)[k], $
                             UVALUE=abiAerVar[i].imageTitle+'.'+StrTrim(k,2))
   ENDFOR
   qfListID = Widget_List(qfBase, YSize=5, XSize=60, Value='', Uvalue='')
   
   
   ; action buttons
   buttonBase = Widget_Base(tlb, Row=1, /ALIGN_CENTER)
   acceptID = Widget_Button(buttonBase, Value='Submit', /ALIGN_CENTER)
   cancelID = Widget_Button(buttonBase, Value='Cancel', /ALIGN_CENTER)
   
      ; Position the top-level base.
   Widget_Control, granInfo.tlb, TLB_Get_Offset=offsets
   xpos = offsets[0] + 10
   ypos = offsets[1] + 10
   Widget_Control, tlb, XOffset = xpos, YOffset = ypos
      
      ; Realize the widgets.
   Widget_Control, tlb, /Realize
   
   ; Create the info structure to hold program information.

   granInfoPtr = Ptr_New(granInfo)
   cmbInfo={ varListID:varListID, $    ; The identifier of the variable list widget.
             qfListID: qfListID,  $    ; The identifier of the QF list widget.
             qfOptID:  qfOptID,   $    ; AND/OR option ID
             qfOpt:    -1,        $    ; 0:AND, 1:OR, -1:N/A
             acceptID: acceptID,  $
             cancelID: cancelID,  $
             ; result
             granInfoPtr: granInfoPtr, $
             varID:    -999,      $    ; ID of Selected variable to be ploted (in granInfo.dataPtrArr)
             qfID:     Ptr_New(), $    ; IDs of selected QFs (in granInfo.dataPtrArr)
             qfValue:  Ptr_New()}      ; Values of selected QFs (positive-AND, negative-OR (-1000))
             
   Widget_Control, tlb, Set_UValue=cmbInfo, /No_Copy   

   ; Run the program as a non-blocking widget.
   XManager, 'ABI_CONUS_Generate_CombinedImage', tlb, /No_Block
              Cleanup='ABI_CONUS_Generate_CombinedImage_CleanUp'
   
   ; Update the granInfo
   granInfo = *granInfoPtr
   Ptr_Free, granInfoPtr
END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_Display, event
   @abi_data.h
   
   Widget_Control, event.top, GET_UVALUE=granInfo
   Widget_Control, event.top, HOURGLASS=1
   
   Widget_Control, event.id, Get_Uvalue=varName
   IF varName EQ 'COMBINE' THEN BEGIN
      ABI_CONUS_Generate_CombinedImage, granInfo
   ENDIF ELSE IF varName EQ 'SynRGB' THEN BEGIN
      granInfo.varIdx = -1
      ABI_CONUS_Generate_RGB_Image, granInfo
   ENDIF ELSE IF varName EQ 'DstRGB' THEN BEGIN
      granInfo.varIdx = -2
      ABI_CONUS_Generate_RGB_Image, granInfo
   ENDIF ELSE IF varName EQ 'NatRGB' THEN BEGIN
      granInfo.varIdx = -3
      ABI_CONUS_Generate_RGB_Image, granInfo
   ENDIF ELSE BEGIN
      granInfo.varIdx = WHERE(abiAerVar.imageTitle EQ varName, count)
      
      IF count EQ 0 THEN BEGIN
         dummy = Dialog_Message('Variable '+varName+' is not found', $
                                Dialog_Parent=event.top, Title='Error')
         Widget_Control, event.top, SET_UVALUE=info, /NO_COPY
         Return
      END
      
      ; create image      
      ABI_CONUS_Generate_Image, granInfo
      
   ENDELSE
   
   Widget_Control, event.top, Set_UValue=granInfo
   Widget_Control, event.top, HOURGLASS=0
END
;-------------------------------------------------------------------------------

PRO ABI_CONUS_Select_Event, event
   @abi.h
   Widget_Control, event.top, Get_UValue=pickInfo
   granInfoPtr = pickInfo.granInfoPtr
   Widget_Control, (*granInfoPtr).tlb, HOURGLASS=1
   slash = PATH_SEP()
   
   ; get the input time
   WIDGET_CONTROL, pickInfo.yearID, GET_VALUE=year
   WIDGET_CONTROL, pickInfo.monthID, GET_VALUE=month
   WIDGET_CONTROL, pickInfo.dayID, GET_VALUE=day
   WIDGET_CONTROL, pickInfo.hourID, GET_VALUE=hour
   WIDGET_CONTROL, pickInfo.minuteID, GET_VALUE=minute
   ddd = dayofyear(day, month, year)
   
      ; process the action events
   IF event.ID EQ pickInfo.acceptID THEN BEGIN
      ; check validity of input time
      validTime = 1B
      MONDAY = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31]
      IF IsLeapYear(year) THEN MONDAY[1]=29
      IF (month LT 1 OR month GT 12) THEN validTime = 0B $
      ELSE BEGIN
         IF (year LT 2017 OR year GT 2030) OR $
            (day LT 1 OR day GT MONDAY[month-1]) OR $
            (hour LT 0 OR hour GT 23) OR $
            (minute LT 0 OR minute GT 59) THEN validTime = 0B
      ENDELSE 
    

      IF validTime THEN BEGIN
         strDate = STRTRIM(year*10000L+month*100L+day,2)   ; yyyymmdd
         strDay = STRTRIM(year*1000L+ddd,2)                ; yyyyddd
         strTime = STRING(hour,FORMAT='(I2.2)')+STRING(minute,FORMAT='(I2.2)') ; hhmm
         strCalDate = STRING(year,FORMAT='(I4)')+'.'+ $
                      STRING(month,FORMAT='(I2.2)')+'.'+ $ 
                      STRING(day,FORMAT='(I2.2)')
         
         ; check whether new time inputed
         IF ((*granInfoPtr).strDay EQ strDay) AND $
            ((*granInfoPtr).strTime EQ strTime) THEN BEGIN
            Widget_Control, event.top, /Destroy
            Widget_Control, (*granInfoPtr).tlb, HOURGLASS=0
            RETURN
         ENDIF
         
         ; set info structure
         outFile = FILE_SEARCH(ABI_DATA_PATH+strDate+'_'+strTime+slash+'*-AODC-*_s'+strDay+strTime+'*.nc',COUNT=nf)
         IF nf EQ 1 THEN BEGIN
            (*granInfoPtr).strTime = strTime
            (*granInfoPtr).strDate = strCalDate
            (*granInfoPtr).strDay = strDay
            (*granInfoPtr).abiaod_filename = outFile[0]
         ENDIF ELSE validTime = 0B
      ENDIF
     
      IF ~validTime THEN BEGIN
         dummy = Dialog_Message('No ABI CONUS AOD retrieval found', $
                          Dialog_Parent=event.top, Title='Error')
         Widget_Control, event.top, /Destroy
         Return
      ENDIF 
      
      ; display/block features
      Widget_Control, (*granInfoPtr).fileID, SENSITIVE=1
      Widget_Control, (*granInfoPtr).imgID, SENSITIVE=1
      Widget_Control, (*granInfoPtr).colorID, SENSITIVE=0
      Widget_Control, (*granInfoPtr).optionID, SENSITIVE=0
      Widget_Control, event.top, /Destroy

     
      ; Initialize
      ABI_CONUS_Initialize, *granInfoPtr, succeed
      IF ~succeed THEN BEGIN
         dummy = Dialog_Message('Problem in the initialization', $
                                 Dialog_Parent=event.top, Title='Error')
         ;Widget_Control, granInfo.tlb, /DESTROY
      ENDIF
      
   ENDIF ELSE IF event.ID EQ pickInfo.cancelID THEN BEGIN
      Widget_Control, event.top, /Destroy
   ENDIF
   Widget_Control, (*granInfoPtr).tlb, HOURGLASS=0
   
   
END
   
;-------------------------------------------------------------------------------

PRO ABI_CONUS_Select, event
   @abi.h
   ;font = '-adobe-helvetica-bold-r-normal--14-140-75-75-p-82-iso8859-1'
   font = 'Helvetica'
   
   Widget_Control, event.top, GET_UVALUE=granInfo
   
      ; get the current input time
   tokens = StrSplit(granInfo.strDate, '.', /EXTRACT)
   IF N_Elements(tokens) EQ 3 THEN BEGIN
      year = Fix(tokens[0])
      month = Fix(tokens[1])
      day = Fix(tokens[2])
      hhmm = Fix(StrMid(granInfo.strTime,0,4))
      hour = hhmm/100
      minute = hhmm MOD 100
   ENDIF ELSE BEGIN
      year = 2017
      month = 8
      day = 8
      hour = 18
      minute = 2
   ENDELSE
       
      ; create the widget   
   tlb = Widget_Base(Title='Select Granule By Time', COL=1, /Modal, $
                     /Base_Align_Center,Group_Leader=event.top,$
                     SPACE=5, TAB_MODE=1)
   dBase = Widget_Base(tlb, ROW=1, /ALIGN_CENTER)
   yearID =  CW_Field(dbase, TITLE='Year', Value=year, /INTEGER, XSize=4, $
                             FONT=font, FIELDFONT=font, /ALL_EVENTS)
   monthID = CW_Field(dbase, TITLE='Month', Value=month, /INTEGER, XSize=2, $
                             FONT=font, FIELDFONT=font, /ALL_EVENTS)
   dayID =   CW_Field(dbase, TITLE='Day', Value=day, /INTEGER, XSize=2, $
                             FONT=font, FIELDFONT=font, /ALL_EVENTS)
   hourID =  CW_Field(dbase, TITLE='Hour', Value=hour, /INTEGER, XSize=2, $
                             FONT=font, FIELDFONT=font, /ALL_EVENTS)
   minuteID= CW_Field(dbase, TITLE='Minute', Value=minute, /INTEGER, XSize=2, $
                            FONT=font, FIELDFONT=font, /ALL_EVENTS)
   
   buttonBase = Widget_Base(tlb, Row=1, SPACE=20, FRAME=1)
   acceptID = Widget_Button(buttonBase, Value='Accept', FONT=font)
   cancelID = Widget_Button(buttonBase, Value='Cancel', FONT=font)
   
      ; Position the top-level base.
   Widget_Control, granInfo.tlb, TLB_Get_Offset=offsets
   xpos = offsets[0] + 10
   ypos = offsets[1] + 10
   Widget_Control, tlb, XOffset = xpos, YOffset = ypos
      
      ; Realize the widgets.
   Widget_Control, tlb, /Realize
   
   ; Create the info structure to hold program information.
   granInfoPtr = Ptr_New(granInfo)
   pickInfo={ yearID:       yearID,      $
              monthID:      monthID,     $
              dayID:        dayID,       $
              hourID:       hourID,      $
              minuteID:     minuteID,    $
              acceptID:     acceptID,    $
              cancelID:     cancelID,    $
              granInfoPtr:  granInfoPtr} 
              
   Widget_Control, tlb, Set_UValue=pickInfo, /No_Copy 
   
   ; Run the program as a non-blocking widget.
   XManager, 'ABI_CONUS_Select', tlb, /No_Block
    
   granInfo = *granInfoPtr
   Ptr_Free, granInfoPtr
   Widget_Control, event.top, Set_UValue=granInfo
   
END


;-------------------------------------------------------------------------------

PRO ABI_CONUS_ColorTable, event
   @abi.h
   Widget_Control, event.top, Get_UValue=granInfo, /No_Copy

   thisEvent = Tag_Names(event, /Structure)
   CASE thisEvent OF
      'WIDGET_BUTTON': BEGIN

            ; Load colors so XCOLORS starts up with the correct colors.
         granInfo.thePalette->GetProperty, RED=r, BLUE=b, GREEN=g 
         TVLCT, r, g, b
         XColors, Group=event.top, NotifyID=[event.id, event.top], $
             Title='Image Color Table Selection'

         ENDCASE

      'XCOLORS_LOAD':BEGIN

            ; Store the new color table vectors from XCOLORS.
         r = event.r
         g = event.g
         b = event.b
         r[missColor] = colorGray[0]
         g[missColor] = colorGray[1]
         b[missColor] = colorGray[2]
         r[lineColor] = colorBlack[0]
         g[lineColor] = colorBlack[1]
         b[lineColor] = colorBlack[2]
         
         IF Obj_Valid(granInfo.thePalette) THEN $
             granInfo.thePalette->SetProperty, Red=r, Green=g, Blue=b
         granInfo.theWindow->Draw, granInfo.theView

         ENDCASE

   ENDCASE
   Widget_Control, event.top, Set_UValue=granInfo, /No_Copy

END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_PickColor, event
   @abi.h
   
   Widget_Control, event.top, Get_UValue=granInfo, /No_Copy
   Widget_Control, event.id, Get_Uvalue=colorIndex

   granInfo.thePalette->GetProperty, RED=r, BLUE=b, GREEN=g 
   
   ; Get the new color
   newColor = PickColor(Group_Leader=event.top, Cancel=cancelled)

   IF NOT cancelled THEN BEGIN
      IF colorIndex EQ 'BackgroundColor' THEN BEGIN
         granInfo.theView->SetProperty, Color=Reform(newColor, 3)
      ENDIF ELSE BEGIN
         tokens = StrSplit(colorIndex,'_',/Extract)
         IF N_Elements(tokens) EQ 2 THEN  cid = FIX(tokens[1])-1 $
         ELSE BEGIN
            CASE colorIndex OF
               'MissColor': cid = missColor
               'ContColor': cid = lineColor
            ENDCASE   
         ENDELSE
         r[cid] = newColor[0]
         g[cid] = newColor[1]
         b[cid] = newColor[2]
      ENDELSE
   ENDIF
   
   IF Obj_Valid(granInfo.thePalette) AND (colorIndex NE 'BackgroundColor') THEN $
      granInfo.thePalette->SetProperty, RED=r, BLUE=b, GREEN=g 
   IF Obj_Valid(granInfo.theView) THEN $
      granInfo.theWindow->Draw, granInfo.theView
   Widget_Control, event.top, Set_UValue=granInfo, /No_Copy
END
;-------------------------------------------------------------------------------

PRO SetRange_Event, event

   Widget_Control, event.top, Get_UValue=info
   CASE event.ID OF
      info.cancelID: Widget_Control, event.top, /Destroy
      ELSE: BEGIN
         ; Get minimum and maximum values.

         Widget_Control, info.minID, Get_Value=minVal
         Widget_Control, info.maxID, Get_Value=maxVal
         (*info.ptr).minVal = minVal
         (*info.ptr).maxVal = maxVal
         (*info.ptr).cancel = 0
         if event.ID EQ info.restrictID THEN (*info.ptr).restrict = 1
         Widget_Control, event.top, /Destroy
     ENDCASE
   ENDCASE
END
;-------------------------------------------------------------------------------

PRO ABI_CONUS_ChangeRange, event

   Widget_Control, event.top, GET_UVALUE=granInfo, /NO_COPY
   
   tlb = Widget_Base(Title='Set Data Range', Column=1, /Modal, $
                     /Base_Align_Center,Group_Leader=granInfo.tlb)
   
   labelbase = Widget_Base(tlb, Row=1)
   label = Widget_Label(labelbase, Value='Min:')
   minID = Widget_Text(labelbase, /Editable, Scr_Xsize=150)
   labelbase = Widget_Base(tlb, Row=1)
   label = Widget_Label(labelbase, Value='Max:')
   maxID = Widget_Text(labelbase, /Editable, Scr_Xsize=150)
   buttonBase = Widget_Base(tlb, Row=1)
   cancelID = Widget_Button(buttonBase, Value='Cancel')
   acceptID = Widget_Button(buttonBase, Value='Submit')
   restrictID = Widget_Button(buttonBase, Value='Restrict')
   
      ; Position the top-level base.
   Widget_Control, granInfo.tlb, TLB_Get_Offset=offsets
   xpos = offsets[0] + 50
   ypos = offsets[1] + 10
   Widget_Control, tlb, XOffset = xpos, YOffset = ypos
   Widget_Control, tlb, /Realize
   
      ; store the program information
   ptr = Ptr_New({minVal:0.0, maxVal:0.0, cancel:1, restrict:0})
   info = {ptr: ptr,  $
         minID:minID, $
         maxID:maxID, $
           restrictID: restrictID, $
         cancelID:cancelID}
   Widget_Control, tlb, Set_UValue=info, /No_Copy
   
   XManager, 'SetRange', tlb
   
   ; get the returned values
   minVal = (*ptr).minVal
   maxVal = (*ptr).maxVal
   cancel = (*ptr).cancel
   restrict = (*ptr).restrict
   Ptr_Free, ptr
   IF NOT cancel THEN BEGIN
     IF (minVal GE maxVal) THEN BEGIN
        dummy = Dialog_Message('Invalid data range', $
                                Dialog_Parent=event.top, Title='Error')
        Widget_Control, event.top, SET_UVALUE=granInfo, /NO_COPY
      return
    ENDIF ELSE BEGIN
        data_range = [minVal, maxVal]
        Widget_Control, granInfo.tlb, HOURGLASS=1
        granInfo.data_Range=data_range
        IF restrict THEN $
           ABI_CONUS_Generate_Image, granInfo, DATA_RANGE=data_range, /RESTRICT $
        ELSE ABI_CONUS_Generate_Image, granInfo, DATA_RANGE=data_range
        Widget_Control, granInfo.tlb, HOURGLASS=0
     ENDELSE
   ENDIF
   
   ; Put the info structure back.
   Widget_Control, event.top, SET_UVALUE=granInfo, /NO_COPY
END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_DrawWidget_Event, event
   @abi.h
   @abi_data.h
  
   Widget_Control, event.top, Get_UValue=info;, /No_Copy
   IF ~Obj_Valid(info.theContainer) OR (event.ID NE info.drawID) THEN BEGIN
      Widget_Control, event.top, Set_UValue=info, /No_Copy
      RETURN
   ENDIF
   
   ; Event type
   possibleEventTypes = [ 'DOWN', 'UP', 'MOTION', 'SCROLL', 'EXPOSE' ]
   thisEvent = possibleEventTypes(event.type)
   
   ; boundary of image within draw window (x: [left,right]; y: [bottom,top])
   imageBoundX = ROUND(info.draw_xsize * pos[[0,2]])
   imageBoundY = ROUND(info.draw_ysize * pos[[1,3]])
   
   CASE thisEvent OF
      'EXPOSE': BEGIN  ; Redraw the view.
         IF Obj_Valid(info.theView) THEN $
            info.theWindow->Draw, info.theView
         END
         
      'DOWN': BEGIN   ; Button press
         ; Set the static corners of the box to current cursor location.
         info.xs = imageBoundX[0] > event.x < (imageBoundX[1]-1)
         info.ys = imageBoundY[0] > event.y < (imageBoundY[1]-1)
         
         ; Change the event handler for the draw widget and turn MOTION events ON.
         Widget_Control, event.id, Draw_Motion_Events=1
         ; Initialize the polygon object.
         info.theBox = Obj_New('IDLgrPolyline', Replicate(info.xs, 5) / info.draw_xsize, $
                               Replicate(info.ys, 5) / info.draw_ysize, Color=info.boxColor)

         ; Make an instance of the view
         info.theWindow->Draw, info.theView, CREATE_INSTANCE=2

         ; Create a graphics tree containing the box.
         info.theView->GetProperty, ALL=props
         props.transparent = 1
         info.boxView = Obj_New('IDLgrView', _EXTRA=props)
         
         info.theModel->GetProperty, ALL=props
         info.boxModel = Obj_New('IDLgrModel', _EXTRA=props)
         
         info.boxView->Add, info.boxModel
         info.boxModel->Add, info.theBox
         END
         
      'UP': BEGIN     ; Button release
         ; If this is an UP event, you need to erase the zoombox, turn motion events
         ; OFF, and draw the "zoomed" plot.

         ; Turn motion events off..
         Widget_Control, event.id, Draw_Motion_Events=0
         
         ; Make sure the user didn't just click or draw a line in the window.
         IF (info.xs EQ event.x) OR (info.ys EQ event.y) THEN BEGIN
            Widget_Control, event.top, Set_UValue=info, /No_Copy
            RETURN
         ENDIF
         
         ; Draw the "selected" image. Start by getting the LAST zoom
         ; box outline. These are indices into image array.
         event.x = imageBoundX[0] > event.x < (imageBoundX[1]-1)
         event.y = imageBoundY[0] > event.y < (imageBoundY[1]-1)
         
         x = [info.xs, event.x]
         y = [info.ys, event.y]
         
         ; Make sure the x and y values are ordered as [min, max].
         IF info.xs GT event.x THEN x = [event.x, info.xs]
         IF info.ys GT event.y THEN y = [event.y, info.ys]
         
         ; Convert the image coordinates to the data locations
         hit = info.theWindow->Pickdata(info.theView, info.theImage, [x[0], y[0]], xy)
         idxBtmLeft  = (*info.ipLocMapPtr)[ROUND(xy[0]),ROUND(xy[1])]
         hit = info.theWindow->Pickdata(info.theView, info.theImage, [x[1], y[0]], xy)
         idxBtmRight = (*info.ipLocMapPtr)[ROUND(xy[0]),ROUND(xy[1])]
         hit = info.theWindow->Pickdata(info.theView, info.theImage, [x[0], y[1]], xy)
         idxTopLeft  = (*info.ipLocMapPtr)[ROUND(xy[0]),ROUND(xy[1])]
         hit = info.theWindow->Pickdata(info.theView, info.theImage, [x[1], y[1]], xy)
         idxTopRight = (*info.ipLocMapPtr)[ROUND(xy[0]),ROUND(xy[1])]
         
         dimSize = SIZE(*info.ipLonPtr, /DIMENSION)
         
         xLeft  = (idxTopLeft MOD dimSize[0]) < (idxBtmLeft MOD dimSize[0])
         IF xLeft LT 0 THEN xLeft = 0
         
         xRight = (idxTopRight MOD dimSize[0]) > (idxBtmRight MOD dimSize[0])
         IF xRight LT 0 THEN xRight=dimSize[0]-1
         
         IF idxTopLeft LT 0 THEN BEGIN
            IF idxTopRight LT 0 THEN yTop = 0 $
            ELSE yTop = (idxTopRight/dimSize[0])
         ENDIF ELSE IF idxTopRight LT 0 THEN yTop = (idxTopLeft/dimSize[0]) $
         ELSE yTop   = (idxTopLeft/dimSize[0]) < (idxTopRight/dimSize[0])
         
         IF idxBtmLeft LT 0 THEN BEGIN
            IF idxBtmRight LT 0 THEN yBottm=dimSize[1]-1 $
            ELSE yBottm = (idxBtmRight/dimSize[0])
         ENDIF ELSE IF idxBtmRight LT 0 THEN yBottm=(idxBtmLeft/dimSize[0]) $
         ELSE yBottm = (idxBtmLeft/dimSize[0]) > (idxBtmRight/dimSize[0])
          
         ; sizes of the Zoom Window.
         zoomXSize = xRight - xLeft + 1
         zoomYSize = yBottm - yTop + 1
         info.zoomXSize = zoomXSize
         info.zoomYSize = zoomYSize
         info.zoomXStart = xLeft
         info.zoomYStart = yTop
              
         ; Subset the image (at pixel resolution)
         varIdx = info.varIdx
         IF varIdx GE 2000 THEN varIdx=varIdx-2000
         varInfo = abiAerVar[varIdx]
         dataType = varInfo.dataType
         dataRange = info.data_range ;varInfo.dataRange
         
         IF (varIdx GE 0)  THEN BEGIN  ; 2-D image
            data = (*info.imageDataPtr)[xLeft:xRight, yTop:yBottm]
            data = REVERSE(data, 2)  ; reverse in Y dimension (window display: bottom-up, left-right) 
            ; zoomedImage = BytArr(zoomXSize, zoomYSize)+missColor
            IF (dataType EQ 0) THEN BEGIN
               ; Integer data
               missIdx = Where(data LT dataRange[0] OR $
                               data GT dataRange[1], nmiss)
               zoomedImage = BYTE(data)
            ENDIF ELSE BEGIN 
               missIdx = Where(~Finite(data), nmiss)
               zoomedImage = BytScl(data,/NaN, min=dataRange[0], $
                               max=dataRange[1], top=numTabCol-1)
            ENDELSE 
            IF nmiss GT 0 THEN zoomedImage[missIdx] = missColor
                              
            ; Create or re-populate (if it already exists) the zoomed image object.
            IF Obj_Valid(info.zoomImage) THEN info.zoomImage->SetProperty, Data=zoomedImage, Palette=info.thePalette $
            ELSE info.zoomImage = Obj_New('IDLgrImage', zoomedImage, Palette=info.thePalette, /No_Copy)
         ENDIF ELSE BEGIN  ; RGB 3-D data
            zoomedImage = (*info.imageDataPtr)[*,xLeft:xRight, yTop:yBottm]
            zoomedImage = REVERSE(zoomedImage, 3)  ; reverse in Y dimension (window display: bottom-up, left-right) 
            
            ; Create or re-populate (if it already exists) the zoomed image object.
            IF Obj_Valid(info.zoomImage) THEN info.zoomImage->SetProperty, Data=zoomedImage, Palette=info.thePalette $
            ELSE info.zoomImage = Obj_New('IDLgrImage', zoomedImage, /No_Copy)
         
         ENDELSE
         
         
         ; Scale the image into the view.
         info.zoomImage->SetProperty, XCoord_Conv=FSC_Normalize([0,zoomXSize]), $
                                      YCoord_Conv=FSC_Normalize([0,zoomYSize])

         ; If the Zoom Window exists, make it the proper size and load
         ; the zoomed image into it. If it does not exists, create it.
         IF Obj_Valid(info.zoomWindow) THEN BEGIN
            ; Zoomed window exists. Make it correct size and load image.
            info.zoomWindow->SetProperty, Dimension=[zoomXSize, zoomYSize]
         ENDIF ELSE BEGIN
            ; Get offset positions for the non-existing zoom window.
            Widget_Control, event.top, TLB_Get_Size=sizes, TLB_Get_Offset=offsets
            xpos = sizes[0] + offsets[0] + 20
            ypos = offsets[1] + 40

            ; Create zoomed window.
            zoomtlb = Widget_Base(Title='Zoomed Image', Group=event.top, $
                                  COLUMN=1, /Base_Align_Center, $
                                  Kill_Notify='ABI_CONUS_Zoom_Cleanup', $
                                  XOffset=xpos, YOffset=ypos, MBAR=mbar)
            pixer = WIDGET_BUTTON(mbar, VALUE='Values', FONT=font, /MENU)
            actID = WIDGET_BUTTON(pixer, VALUE='Show values', FONT=font, $
                                  EVENT_PRO='ABI_CONUS_Zoom_PixelValue')
                                  
            zoomdraw = Widget_Draw(zoomtlb, XSize=zoomXSize, YSize=zoomYSize, $
                                   Graphics_Level=2, Retain=0, $
                                   Event_Pro='ZoomWindow_DrawWidget_Event', $
                                   /BUTTON_EVENTS, /EXPOSE_EVENTS)
            
            info.zoomtlb = zoomtlb
            info.zoomDrawID = zoomdraw
            info.pixValID = pixer
            info.pixValActID = actID
            
            Widget_Control, zoomtlb, /Realize
            Widget_Control, zoomdraw, Get_Value=zoomWindow
            zoomWindow->SetCurrentCursor, 'Arrow'
            info.zoomWindow = zoomWindow
            info.zoomView = Obj_New('IDLgrView', Viewplane_Rect=[0,0,1,1])
            info.zoomModel = Obj_New('IDLgrModel')
            info.zoomView->Add, info.zoomModel
            info.zoomModel->Add, info.zoomImage
            info.zoomContainer = Obj_New('IDL_Container')
            info.zoomContainer->Add, info.zoomWindow
            info.zoomContainer->Add, info.zoomModel
            info.zoomContainer->Add, info.zoomView
            info.zoomContainer->Add, info.zoomImage
            
            Widget_Control, zoomtlb, Set_UValue=info
            XMANAGER, 'VIIRS_Aer_VIS_DrawWidget_Event', zoomtlb, /NO_BLOCK
         ENDELSE
         
         ; Update the zoomed image data and draw it.
         info.zoomWindow->Draw, info.zoomView
         
         ; Destroy the graphics tree holding the box object
         Obj_Destroy, info.boxView
         
         ; Redraw the image view to erase the box
         info.theWindow->Draw, info.theView

         ; Clear any motion events that may have occurred.
         Widget_Control, event.id, /Clear_Events
         END

      'MOTION': BEGIN
      
         ; Get the dynamic corner of the box.
         info.xd = imageBoundX[0] > event.x < (imageBoundX[1]-1)
         info.yd = imageBoundY[0] > event.y < (imageBoundY[1]-1)
         
         ; Re-set the box coordinates
         theBoxData = FltArr(2,5)
         theBoxData[0,*] = [info.xs, info.xd, info.xd, info.xs, info.xs] / Float(info.draw_xsize)
         theBoxData[1,*] = [info.ys, info.ys, info.yd, info.yd, info.ys] / Float(info.draw_ysize)
         info.theBox->SetProperty, Data=theBoxData
         
         ; Draw the view containg the box
         info.theWindow->Draw, info.boxView, /DRAW_INSTANCE
         END
         
      ELSE:
   ENDCASE

   Widget_Control, event.top, Set_UValue=info, /No_Copy
END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_Zoom_Cleanup, tlb
   
   ; Clean up the objects created for the zoom window.
      
   Widget_Control, tlb, Get_UValue=info, /NO_COPY 
   Obj_Destroy, info.zoomContainer
   
   Widget_Control, tlb, SET_UVALUE=info, /NO_COPY
   
END
;-------------------------------------------------------------------------------

PRO ZoomWindow_DrawWidget_Event, event

   Widget_Control, event.top, Get_UValue=granInfo;, /No_Copy
   Widget_Control, granInfo.pixValActID, GET_VALUE=pixTask
   IF pixTask EQ 'Show values' THEN BEGIN
      Widget_Control, event.top, Set_UValue=granInfo, /No_Copy
      RETURN
   ENDIF
   
   ; Event type
   possibleEventTypes = [ 'DOWN', 'UP', 'MOTION', 'SCROLL', 'EXPOSE' ]
   thisEvent = possibleEventTypes(event.type)
   
   CASE thisEvent OF
      'DOWN': Widget_Control, event.id, Draw_Motion_Events=1
      'UP': Widget_Control, event.id, Draw_Motion_Events=0
      'EXPOSE': granInfo.zoomWindow->Draw, granInfo.zoomWindow
      ELSE:
   ENDCASE
   
   IF thisEvent NE 'EXPOSE' THEN BEGIN
      ; Calculate the image value at the current location.
      xpt = event.x
      ypt = granInfo.zoomYSize - event.y
      IF xpt LT 0 OR xpt GT (granInfo.zoomXSize-1) OR $
         ypt LT 0 OR ypt GT (granInfo.zoomYSize-1) THEN BEGIN
         xpt = -1
         ypt = -1
      ENDIF 
      ABI_CONUS_Zoom_Get_PixelValue, granInfo, xpt, ypt
      
   ENDIF

   Widget_Control, event.top, Set_UValue=granInfo, /No_Copy
END



;-------------------------------------------------------------------------------
PRO ABI_CONUS_Zoom_PixelValue, event

   Widget_Control, event.top, GET_UVALUE=granInfo, /NO_COPY
   Widget_Control, event.id, GET_VALUE=task
   ;font = '-adobe-helvetica-bold-r-normal--14-140-75-75-p-82-iso8859-1'
   font = 'Helvetica'
   
   IF task EQ 'Show values' THEN BEGIN
      
      ; create new window to show pixel values
      Widget_Control, event.top, TLB_Get_Size=sizes, TLB_Get_Offset=offsets
      xpos = sizes[0] + offsets[0] + 20
      ypos = offsets[1] + 40
      pixvalTlb = Widget_Base(Title='Pixel Retrieval Values', Group=event.top, ROW=2,$
                              /Base_Align_Center, XOffset=xpos, YOffset=ypos, MBAR=mbar)
      smenu = Widget_Button(mbar, VALUE='Save', FONT=font,/MENU)
      saveButton = Widget_Button(smenu, VALUE='To Ascii File', FONT=font,  $
                                Event_Pro='ABI_CONUS_Zoom_PixelValue_Save')
      label = Widget_Label(pixvalTlb, FONT=font, $
                     VALUE='Click and move cursor over zoomed image to show pixel values')
      showPanel = Widget_Base(pixvalTlb, COL=1, FRAME=1)
      pixvalPanel = Widget_List(showPanel, SCR_XSIZE=500, SCR_YSIZE=600, VALUE='')
      granInfo.pixvalPanel = pixvalPanel
      granInfo.pixvalTlb = pixvalTlb
      ABI_CONUS_Zoom_Get_PixelValue, granInfo, -1, -1
     
      pixelInfo = { eventId: event.id}
      Widget_Control, pixvalTlb, Set_Uvalue=pixelInfo, /No_Copy
      Widget_Control, pixvalTlb, /REALIZE
      XMANAGER, 'ABI_CONUS_Zoom_PixelValue', pixvalTlb, /NO_BLOCK, $
                CLEANUP='ABI_CONUS_Zoom_PixelValue_Cleanup'
      
      nextTask = 'Close'
   ENDIF ELSE BEGIN
      Widget_Control, granInfo.pixvalTlb, /DESTROY
      nextTask = 'Show values'
   ENDELSE

   Widget_Control, event.id, SET_VALUE=nextTask

   ; Put the info structure back.
   Widget_Control, event.top, SET_UVALUE=granInfo, /NO_COPY

END
;-------------------------------------------------------------------------------
PRO ABI_CONUS_Zoom_PixelValue_Save, event
   Widget_Control, event.top, GET_UVALUE=info
 
   IF (FILE_TEST('pixval.sav')) THEN BEGIN
     filename = Dialog_Pickfile(/Write, File='ABI_CONUS_value.txt')
     IF filename NE '' THEN BEGIN
       OPENW, lun, /GET_LUN, filename
       RESTORE, 'pixval.sav'
       FOR i=0, N_ELEMENTS(valList)-1 DO BEGIN
          PRINTF, lun, valList[i]
       ENDFOR
       CLOSE, lun
       FREE_LUN, lun
     ENDIF
   ENDIF
END
;-------------------------------------------------------------------------------
PRO ABI_CONUS_Zoom_Get_PixelValue, info, xPos, yPos
   @abi_data.h

   invalid = 0B
   pidx = -1     ; pixel index (1D)
   IF xPos LT 0 OR yPos LT 0 THEN invalid=1B $
   ELSE BEGIN
      dim = SIZE((*info.ipLonPtr), /Dimension)
      xPos = xPos + info.zoomXStart
      yPos = yPos + info.zoomYStart
      pidx = xPos + yPos * dim[0]
   ENDELSE
   
   IF invalid OR pidx LT 0 THEN BEGIN
      valList = '           ABI Baseline Data (N/A)'
   ENDIF ELSE BEGIN   
      valList = StrArr(NVars+10)
      valList[0] = 'ABI Baseline Data'
      valList[1] = 'Longitude: '+String((*info.ipLonPtr)[pidx],FORMAT='(F7.2)')
      valList[2] = 'Latitude:  '+String((*info.ipLatPtr)[pidx],FORMAT='(F7.2)')
      
      IF info.varIdx GE 2000 THEN BEGIN
         ; combination plot
         showVarIdx = info.varIdx - 2000
      ENDIF ELSE  showVarIdx = info.varIdx
     
      valList[4] = ' '
      listIdx = 5
      FOR i=0,NVars-4 DO BEGIN   ; last three are RGBs
         j = abiAerVar[i].dataSetIdx
         IF Ptr_Valid(info.dataPtrArr[j]) THEN BEGIN
            IF abiAerVar[i].dataType EQ 0 THEN BEGIN
               value = (*info.dataPtrArr[j])[pidx]
               tokens = StrSplit(abiAerVar[i].name,'@',/Extract) 
               IF N_Elements(tokens) GT 1 THEN BEGIN 
                  qfBits = FIX(tokens[1:N_Elements(tokens)-1])
                  mask = 0
                  FOR k=0, N_Elements(qfBits)-1 DO $
                     mask = mask + Ishft(1,qfBits[k])
                  value = Ishft(value AND mask, qfBits[0]*(-1))
               ENDIF
               IF value LT abiAerVar[i].dataRange[0] OR $
                  value GT abiAerVar[i].dataRange[1] THEN strValue = 'N/A' $
               ELSE strValue = (*abiAerVar[i].annotation)[value]
            ENDIF ELSE BEGIN
               strValue = StrTrim((*info.dataPtrArr[j])[pidx],2)
            ENDELSE
            IF (i EQ showVarIdx) THEN BEGIN
               valList[3] = abiAerVar[i].imageTitle+': '+strValue 
            ENDIF ELSE BEGIN
               valList[listIdx] = abiAerVar[i].imageTitle+': '+strValue
               listIdx = listIdx + 1
            ENDELSE
         ENDIF
      ENDFOR
   
   ENDELSE
   
   SAVE, valList, FILENAME='pixval.sav'
   WIDGET_CONTROL, info.pixvalPanel, SET_VALUE=valList
    
END
;-------------------------------------------------------------------------------
PRO ABI_CONUS_Zoom_PixelValue_Cleanup, id
   Widget_Control, id, GET_UVALUE=pixelInfo   
   FILE_DELETE, 'pixval.sav', /ALLOW_NONEXISTENT, /QUIET
   valid = Widget_Info(pixelInfo.eventId, /VALID_ID)
   IF valid THEN $
      Widget_Control, pixelInfo.eventId, SET_VALUE='Show values'   
END
;-------------------------------------------------------------------------------
PRO ABI_CONUS_Zoom_PixelValue_Event, event
   ; Do nothing on the pixel value events
END
;-------------------------------------------------------------------------------
PRO ABI_CONUS_CleanUp_Pro, granInfo
   @abi_data.h
   IF Obj_Valid(granInfo.theContainer) THEN Obj_Destroy, granInfo.theContainer
   
   IF Ptr_Valid(granInfo.ipLonPtr) THEN Ptr_Free, granInfo.ipLonPtr
   IF Ptr_Valid(granInfo.ipLatPtr) THEN Ptr_Free, granInfo.ipLatPtr
   IF Ptr_Valid(granInfo.ipLocMapPtr) THEN Ptr_Free, granInfo.ipLocMapPtr
   FOR i=0, NDataSets-1 DO $
      IF Ptr_Valid(granInfo.dataPtrArr[i]) THEN Ptr_Free, granInfo.dataPtrArr[i]
   
   IF Ptr_Valid(granInfo.imageDataPtr) THEN Ptr_Free, granInfo.imageDataPtr
   IF Ptr_Valid(granInfo.astVarIdx) THEN Ptr_Free, granInfo.astVarIdx  
   IF Ptr_Valid(granInfo.astVarVal) THEN Ptr_Free, granInfo.astVarVal 
 
END
;-------------------------------------------------------------------------------

PRO ABI_CONUS_Cleanup, id
   Widget_Control, id, GET_UVALUE=granInfo
   IF N_Elements(granInfo) NE 0 THEN $
      ABI_CONUS_CleanUp_Pro, granInfo
END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_Vis_Output, event

   ; This event handler creates output files.

   Widget_Control, event.top, GET_UVALUE=info, /NO_COPY
      
   ; Get a snapshop of window contents. (TVRD equivalent.)
   WAIT, 0.5 ; To allow menu to disappear.
   info.theWindow->GetProperty, IMAGE_DATA=snapshot
            
   ; What kind of file is wanted?

   Widget_Control, event.id, GET_UVALUE=whichFileType
   CASE whichFileType OF

      'GIF': BEGIN

         ; Because we are using RGB color for this model, we have
         ; a 3-m-n array. Use Color_Quan to create a 2D image and
         ; appropriate color tables for the GIF file.

         image2D = Color_Quan(snapshot, 1, r, g, b)
         filename = Dialog_Pickfile(/Write, File='abi_conus_aer.gif')
         IF filename NE '' THEN Write_GIF, filename, image2d, r, g, b
         END

      'BMP': BEGIN
         image2D = Color_Quan(snapshot, 1, r, g, b)
         filename = Dialog_Pickfile(/Write, File='abi_conus_aer.bmp')
         IF filename NE '' THEN Write_BMP, filename, image2d, r, g, b
         END

      'PNG': BEGIN
         image2D = Color_Quan(snapshot, 1, r, g, b)
         filename = Dialog_Pickfile(/Write, File='abi_conus_aer.png')
         IF filename NE '' THEN Write_PNG, filename, image2d, r, g, b
         END

      'JPEG': BEGIN
         filename = Dialog_Pickfile(/Write, File='abi_conus.jpg')
         IF filename NE '' THEN Write_JPEG, filename, snapshot, True=1
         END

      'TIFF': BEGIN
         filename = Dialog_Pickfile(/Write, File='abi_conus_aer.tif')
         IF filename NE '' THEN BEGIN
            ; TIFF files should have their Y direction reversed for
            ; compatibility with most other software.
            Write_TIFF, filename, Reverse(snapshot,3)
         ENDIF
         END

   ENDCASE
   
   ; Put the info structure back.
   Widget_Control, event.top, SET_UVALUE=info, /NO_COPY

END

;-------------------------------------------------------------------------------

PRO ABI_CONUS_Vis_Exit, event
  
  Widget_Control, event.top, /DESTROY

END



;-------------------------------------------------------------------------------
;========================= M A I N   P R O G R A M =============================
;-------------------------------------------------------------------------------

PRO ABI_CONUS
   @abi.h
   @abi_data.h
      
   ;======== Create widgets for granule plot ==============
   
   ;font = '-adobe-helvetica-bold-r-normal--14-140-75-75-p-82-iso8859-1'
   font = 'Helvetica'
   
   ; Create widget window.
   tlb = Widget_Base(Title='ABI CONUS Aerosol Visualization', COLUMN=1,$
                         /Base_Align_Center, MBAR=mbar)
   
   picker = WIDGET_BUTTON(mbar, VALUE='Time', FONT=font, /MENU)
   button = WIDGET_BUTTON(picker, VALUE='Select', FONT=font, EVENT_PRO='ABI_CONUS_Select')
   
   filer = WIDGET_BUTTON(mbar, VALUE='File', /MENU, FONT=font, SENSITIVE=0)
   output = WIDGET_BUTTON(filer, VALUE='Save As...', FONT=font, /MENU, $
                          EVENT_PRO='ABI_CONUS_Vis_Output')
   button = WIDGET_BUTTON(output, VALUE='PNG File' , FONT=font, UVALUE='PNG')
   button = WIDGET_BUTTON(output, VALUE='JPEG File', FONT=font, UVALUE='JPEG')
   button = WIDGET_BUTTON(output, VALUE='TIFF File', FONT=font, UVALUE='TIFF')
   button = WIDGET_BUTTON(output, VALUE='BMP File',  FONT=font, UVALUE='BMP')
   IF (Float(!Version.Release) LT 5.4) THEN $
      button = WIDGET_BUTTON(output, VALUE='GIF File', FONT=font, UVALUE='GIF')
   quitter = WIDGET_BUTTON(filer, VALUE='Quit', FONT=font,/SEPARATOR, $
                           EVENT_PRO='ABI_CONUS_Vis_Exit')
   
   ; Create IMAGE menu buttons for selecting variable to be plotted.
   imager = WIDGET_BUTTON(mbar, VALUE='Display', FONT=font, /MENU, $
            EVENT_PRO='ABI_CONUS_Display', SENSITIVE=0) 
   
   ; AOD Outputs
   out_aod = WIDGET_BUTTON(imager,VALUE='AodOutputs', FONT=font, /MENU)
   NV=0
   FOR i=NV,NV+NVar[DATA_AOD]-1 DO $
      button = WIDGET_BUTTON(out_aod,VALUE=abiAerVar[i].imageTitle, UVALUE=abiAerVar[i].imageTitle, FONT=font)
   
   ; ADP Outputs
   out_adp = WIDGET_BUTTON(imager,VALUE='AdpOutputs', FONT=font, /MENU, /SEPARATOR)
   NV = NVar[DATA_AOD]
   FOR i=NV,NV+NVar[DATA_ADP]-1 DO $
      button = WIDGET_BUTTON(out_adp,VALUE=abiAerVar[i].imageTitle, UVALUE=abiAerVar[i].imageTitle, FONT=font)
   
   ; RGB Images
   rgb =  WIDGET_BUTTON(imager,VALUE='RGB Images', FONT=font,  /MENU, /SEPARATOR)
   button = WIDGET_BUTTON(rgb,VALUE='Syntheric RGB', UVALUE='SynRGB', FONT=font)
   button = WIDGET_BUTTON(rgb,VALUE='Dust RGB', UVALUE='DstRGB', FONT=font)
   button = WIDGET_BUTTON(rgb,VALUE='Natural RGB', UVALUE='NatRGB', FONT=font)
   
   ; Inputs
   input = WIDGET_BUTTON(imager,VALUE='Inputs', FONT=font, /MENU, /SEPARATOR)
   
      ; cloud mask 
   in_cm = WIDGET_BUTTON(input,VALUE='CloudMask', FONT=font, /MENU)
   NV = TOTAL(NVar[0:DATA_ADP])
   FOR i=NV,NV+NVar[DATA_CLDMSK]-1 DO $
     button = WIDGET_BUTTON(in_cm,VALUE=abiAerVar[i].imageTitle, UVALUE=abiAerVar[i].imageTitle, FONT=font)
      
      ; geometry
   in_geo = WIDGET_BUTTON(input,VALUE='Geometry', FONT=font, /MENU, /SEPARATOR)
   NV = TOTAL(NVar[0:DATA_CLDMSK])
   FOR i=NV,NV+NVar[DATA_GEO]-1 DO $
     button = WIDGET_BUTTON(in_geo,VALUE=abiAerVar[i].imageTitle, UVALUE=abiAerVar[i].imageTitle, FONT=font)
   
   in_sdr = WIDGET_BUTTON(input,VALUE='ABI Channels', FONT=font, /MENU, /SEPARATOR) 
   NV = TOTAL(NVar[0:DATA_GEO])
   channel = 'Band'+['1 (0.47um)','2 (0.64um)','3 (0.865um)','4 (1.378um)',$
                  '5 (1.61um)','6 (2.25um)','11 (8.5um)','14 (11.2um)',  '15 (12.3um)']
   FOR j=0,8 DO BEGIN
      cher = WIDGET_BUTTON(in_sdr,VALUE=channel[j], FONT=font,  /MENU, /SEPARATOR)
      k = j*2+NV
      FOR i=k,k+1 DO BEGIN
         button = WIDGET_BUTTON(cher,VALUE=abiAerVar[i].imageTitle, UVALUE=abiAerVar[i].imageTitle, FONT=font)
      ENDFOR
   ENDFOR
   
   ; Combination Plot
   cmber = WIDGET_BUTTON(imager,VALUE='Combination Plot', UVALUE='COMBINE', FONT=font, /SEPARATOR)
   
   ; Create COLOR menu buttons 
   optioner = WIDGET_BUTTON(mbar, VALUE='Options', FONT=font, SENSITIVE=0,/MENU)
   colorer = WIDGET_BUTTON(optioner, VALUE='Colors', FONT=font, /MENU, SENSITIVE=0)
   imgColorID = WIDGET_BUTTON(colorer, Value='Image Color Table', FONT=font, $
                              Event_Pro='ABI_CONUS_ColorTable')
   backColorID = WIDGET_BUTTON(colorer, Value='Color for Background', FONT=font, $
                              UValue='BackgroundColor', Event_Pro='ABI_CONUS_PickColor')
   missColorID = WIDGET_BUTTON(colorer, Value='Color for Missing Value', FONT=font, $
                              UValue='MissColor', Event_Pro='ABI_CONUS_PickColor')
   contColorID = WIDGET_BUTTON(colorer, Value='Color for Continent Boundary', FONT=font, $
                              UValue='ContColor', Event_Pro='ABI_CONUS_PickColor')
   disColorID = WIDGET_BUTTON(colorer, Value='Color for Flags', FONT=font,/MENU, $
                            Event_Pro='ABI_CONUS_PickColor')
   color1ID = WIDGET_BUTTON(disColorID, Value='1st Color', FONT=font, UValue='Color_1')
   color2ID = WIDGET_BUTTON(disColorID, Value='2nd Color', FONT=font, UValue='Color_2')
   color3ID = WIDGET_BUTTON(disColorID, Value='3rd Color', FONT=font, UValue='Color_3')
   color4ID = WIDGET_BUTTON(disColorID, Value='4th Color', FONT=font, UValue='Color_4')
   color5ID = WIDGET_BUTTON(disColorID, Value='5th Color', FONT=font, UValue='Color_5')
   color6ID = WIDGET_BUTTON(disColorID, Value='6th Color', FONT=font, UValue='Color_6')
   color7ID = WIDGET_BUTTON(disColorID, Value='7th Color', FONT=font, UValue='Color_7')
   color8ID = WIDGET_BUTTON(disColorID, Value='8th Color', FONT=font, UValue='Color_8')
   
   ; Displayed value range
   rangeID = WIDGET_BUTTON(optioner, VALUE='Data Range', FONT=font, /SEPARATOR, $
                           EVENT_PRO='ABI_CONUS_ChangeRange')
   
   ; layout base                 
   layout_widget = WIDGET_BASE(tlb, ROW=1)   
     
   ; Create the draw widget
   xsize = 1200
   ysize = 700 
   conusDrawID = WIDGET_DRAW(layout_widget, XSIZE=xsize, YSIZE=ysize, $
                        GRAPHICS_LEVEL=2, RETAIN=0, FRAME=5, $
                        EVENT_PRO='ABI_CONUS_DrawWidget_Event',$
                        /BUTTON_EVENTS, /EXPOSE_EVENTS, RENDERER=1)
   
   ; Create widget on the screen
   Widget_Control, tlb, /REALIZE
   Widget_Control, conusDrawID, GET_VALUE=theWindow
   theWindow->SetCurrentCursor, 'Arrow'
   

   ; Create an info structure to hold program information.
   granInfo = {tlb:tlb, $   
               
               ; time selection  
               strDay: '',        $  ; YYYYDDD
               strDate: '',       $  ; YYYY.MM.DD 
               strTime: '',       $  ; HHMM
               
               abiaod_filename: '', $
               
               fileID:  filer,     $
               imgID:   imager,    $
               optionID: optioner, $
               colorID: colorer,   $
               rangeID: rangeID,   $
               imgColorID:imgColorID,     $ ; The widget identifier of image color table menu     
               disColorID:disColorID,     $ ; The widget identifier of discrete color menu
               fgColorID:[color1ID, color2ID, color3ID,color4ID, color5ID, color6ID, color7ID, color8ID], $      
                                            ; The widget identifier of flag color buttons
               image_xsize:0,             $ ; the x size of the image within the draw widget window    
               image_ysize:0,             $ ; the y size of the image within the draw widget window
               centerLat:0.,              $ ; central latitude of granule image  
               centerLon:0.,              $ ; central longitude of granule image
               latlonglimits:Fltarr(4),   $ ; plot region [minlat, minlon, maxlat, maxlon]
               imageDataPtr:PTR_NEW(),    $ ; data used to plot image
               
               drawID:conusDrawID,        $ ; draw widget ID
               theBox:Obj_New(),          $ ; The rubberband box object.
               boxColor:[200,0,0],        $ ; The box color. dark red to start.
               boxModel:Obj_New(),        $ ; The box model
               boxView:Obj_New(),         $ ; The box view
               xs:0,                      $ ; X static corner of the zoom box. (start) (image coordinates)
               ys:0,                      $ ; Y static corner of the zoom box.
               xd:0,                      $ ; X dynamic corner of the zoom box. (end)
               yd:0,                      $ ; Y dynamic corner of the zoom box.
               
               zoomtlb: 0L,               $ ;
               zoomDrawID: 0L,            $ ;
               zoomContainer:Obj_New(),   $ ; 
               zoomImage:Obj_New(),       $ ; The scaled and resized subimage.
               zoomWindow:Obj_New(),      $ ; The zoom window object.
               zoomView:Obj_New(),        $ ; The zoom window view object.
               zoomModel:Obj_New(),       $ ; The zoom model.
               zoomXSize:0L,              $ ; Horizontal size of zoom window 
               zoomYSize:0L,              $ ; Vertical size of zoom window
               zoomXStart:0L,             $ ; index of the zoom window's left boundary within data dimension
               zoomYStart:0L,             $ ; index of the zoom window's top boundary within data dimension
               data_range:FLTARR(2),      $ ; 
               
               pixValID: 0L,              $ ; 
               pixValActID:  0L,          $ ;
               pixvalTlb:   -1L,          $
               pixvalPanel: -1L,          $
               
               theWindow:theWindow,       $ ; The main image window object.
               thePalette:Obj_New(),      $ ; The palette object.
               theImage:Obj_New(),        $ ; The original image.
               theView:Obj_New(),         $ ; The view that will be rendered.
               theModel:Obj_New(),        $ ; The image model.
               theContainer:Obj_New(),    $ ; The main container object.
               draw_xsize:xsize,          $ ; The x size of the draw widget window.
               draw_ysize:ysize,          $ ; The y size of the draw widget window.
               
               ipLonPtr:   Ptr_New(),    $ ; (pixel) longitude
               ipLatPtr:   Ptr_New(),    $ ; (pixel) latitude
               ipLocMapPtr:  Ptr_New(),  $ ; 
               dataPtrArr:PtrArr(NDataSets), $ ; data pointers 
               
               varIdx: 0,                $ ; index of variable selected for display (initially AOD550)
               astVarIdx: Ptr_New(),     $ ; associated vairable index (indices) used for the combination plot 
               astVarVal: Ptr_New()}       ;                     value(s)
          
   ; Store state information
   Widget_Control, tlb, Set_Uvalue=granInfo, /No_Copy
   
   ; Start event manager
   XMANAGER, 'ABI_CONUS', tlb, /NO_BLOCK, $
             CLEANUP='ABI_CONUS_Cleanup'
             
END
