;
; Atmospheric correction of ABI blue (B1) and red (B2) reflectance
;
; Inputs:
;   solzen: solar zenith angle  [degree]
;   satzen: satellite zenith angle  [degree] 
;   relazi: relative azimuth angle  [degree] 
;   blue:   ABI blue band (B1) reflectance
;   red:    ABI red band (B2) reflectance
;
; Outputs:
;   blue: atmospheric-corrected blue band reflectance (solzen<=85, satzen<=80)
;   red:  atmospheric-corrected red band reflectance  (solzen<=85, satzen<=80)
;   return 0 if succeed, 1 if fail
;

;------------------------------------------------------------------------
PRO rayleigh_sphalb, MolOD, RaySphAlb
      
      SPH_ALB_COEFF = [-.57721566, 0.99999193,-0.24991055,0.05519968,-0.00976004, 0.00107857]
      
      ExpInt = SPH_ALB_COEFF[0] - alog(MolOD);
      Trans = exp(-MolOD);
      
      ;/* Compute the Exponential integral of order 1 using 
      ;   5th-order polynomial formula                      */
      FOR IndexOrder=1, 5 DO $
         ExpInt += (MolOD ^ IndexOrder) * SPH_ALB_COEFF[IndexOrder];
      
      ;/* Using recuurence relation of calculate the Exponential integral 
      ;   of order 3 and 4  */
      FOR IndexOrder=2, 4 DO BEGIN
         ExpInt = (Trans - MolOD * ExpInt) / (IndexOrder-1);
         if (IndexOrder EQ 3) THEN ExpInt3 = ExpInt;
         if (IndexOrder EQ 4) THEN ExpInt4 = ExpInt;
      ENDFOR   
      
      RaySphAlb = (3 * MolOD - 4.* ExpInt3 + 6.* ExpInt4) /  $
                   (4.+ 3 * MolOD);

END

;------------------------------------------------------------------------

PRO rayleigh_tr, MolOD, CosSza, CosVza, RelAzi, rhoray, trup, trdown
   ; calcualte Rayleigh reflectance and transmittance   
   
   np = N_ELEMENTS(CosSza)
   RAYCOEFF0 = [0.33243832, 0.16285370, -0.30924818, -0.10324388,0.11493334, $
                -6.777104e-02, 1.577425e-03, -1.240906e-02,3.241678e-02, -3.503695e-02]
   RAYCOEFF1 = [0.19666292, -5.439061e-02]
   RAYCOEFF2 = [0.14545937, -2.910845e-02]
   
   Phi = !PI - RelAzi * !DTOR ;    /* PI - (SolAzi - SatAzi)        */
   CosPhi0 = 1.;                   /* Cosine of N*Phi  (N=0,1,2)    */
   CosPhi1 = cos(Phi);
   CosPhi2 = cos(2.*Phi);
   
   
   ;/* Fourier expansion of Rayleigh Scattering phase function (N=0,1,2)     */
   
   DepFac = 0.0279;                    /* Depolarization factor   */
   DepCorr = DepFac / (2.0 - DepFac);  /* Depolarization correction */
   DepCorr = (1.0 - DepCorr)/(1.0 + 2.* DepCorr);
   PhaseFun0 = 1.0 + (3.0 * CosSza * CosSza - 1.0) *  $
                     (3.0 * CosVza * CosVza - 1.0) * DepCorr / 8.0;
   PhaseFun1 = -1.0 * CosSza * CosVza * sqrt(1.0 - CosSza * CosSza) * $
               sqrt(1.0 - CosVza * CosVza) * DepCorr * 0.5 * 1.5;
   PhaseFun2 = (1.0 - CosSza * CosSza) * (1.0 - CosVza * CosVza) * $
               DepCorr * 0.5 * 0.375;
   
   
   ; /* Fourier expansion of single scattering contribution (N=0,1,2) */
   
   Tmp = (1.0 - exp(-MolOD * (1.0 / CosSza + 1.0 / CosVza))) / $
         (4.0 * (CosSza + CosVza));
   SingleScat0 = PhaseFun0 * Tmp;
   SingleScat1 = PhaseFun1 * Tmp;
   SingleScat2 = PhaseFun2 * Tmp;
   
   
   ;/* Calculate DeltaMolOD   (N=0,1,2) */
   
   DeltaMolODCoef = PTRARR(10);   /* Coefficients used for estimating DeltaMolOD */
   DeltaMolODCoef[0] = PTR_NEW(1.0);
   DeltaMolODCoef[1] = PTR_NEW(alog(MolOD));
   DeltaMolODCoef[2] = PTR_NEW(CosSza + CosVza);
   DeltaMolODCoef[3] = PTR_NEW((*DeltaMolODCoef[1]) * (*DeltaMolODCoef[2]));
   DeltaMolODCoef[4] = PTR_NEW(CosSza * CosVza);
   DeltaMolODCoef[5] = PTR_NEW((*DeltaMolODCoef[1]) * (*DeltaMolODCoef[4]));
   DeltaMolODCoef[6] = PTR_NEW(CosSza * CosSza + CosVza * CosVza);
   DeltaMolODCoef[7] = PTR_NEW((*DeltaMolODCoef[1]) * (*DeltaMolODCoef[6]));
   DeltaMolODCoef[8] = PTR_NEW(CosSza * CosSza * CosVza * CosVza);
   DeltaMolODCoef[9] = PTR_NEW((*DeltaMolODCoef[1]) * (*DeltaMolODCoef[8]));
   
   DeltaMolOD0 = FLTARR(np);
   FOR i = 0, 9 DO $
      DeltaMolOD0 += *(DeltaMolODCoef[i]) * RAYCOEFF0[i];    
   
   DeltaMolOD1 = (*DeltaMolODCoef[0]) * RAYCOEFF1[0] +  $
                 (*DeltaMolODCoef[1]) * RAYCOEFF1[1];
   DeltaMolOD2 = (*DeltaMolODCoef[0]) * RAYCOEFF2[0] +  $
                 (*DeltaMolODCoef[1]) * RAYCOEFF2[1];
   
   
   ;/* Calculate Rayleigh Reflection 
   ;    Summation of first 3 (N=0,1,2) Fourier expansion terms               */
   
   Tmp = (1.0 - exp(-MolOD / CosSza)) * (1.0 - exp(-MolOD/ CosVza));
   
   RayRef0 = (SingleScat0 + PhaseFun0 * DeltaMolOD0 * Tmp) * CosPhi0;
   RayRef1 = (SingleScat1 + PhaseFun1 * DeltaMolOD1 * Tmp) * CosPhi1 * 2.0;
   RayRef2 = (SingleScat2 + PhaseFun2 * DeltaMolOD2 * Tmp) * CosPhi2 * 2.0;
   
   rhoray = (RayRef0 + RayRef1 + RayRef2);
   trup = exp(-MolOD/CosSza);
   trdown =  exp(-MolOD/CosVza);
   
   ; free memory
   FOR i=0,9 DO PTR_FREE, DeltaMolODCoef[i]
END

;------------------------------------------------------------------------


FUNCTION atmcor_br, solzen, satzen, relazi, blue, red
   
   MAXSZA = 85.
   MAZVZA = 80.
   vldIdx = WHERE(solzen LE MAXSZA AND satzen LE MAZVZA AND $
                  FINITE(red) AND FINITE(blue), nvld)
   IF nvld EQ 0 THEN RETURN, 1 
   
   mus = cos(solzen[vldIDx]*!DTOR);
   muv = cos(satzen[vldIDx]*!DTOR);
   airMass = 1./mus + 1./muv
   
   ; No water vapor absorption in ABI B1
   aH2O = [0.0, -5.60723]  ; water vapor absorption coefficients
   bH2O = [0.0, 0.820175]
   aO3  = [0.00743232, 0.0715289]  ; Ozone absorption coefficients
   
   taur = [0.18484,  0.052383]   ; Rayleigh optical depth
   
   ; Surface elevation correction (!FIXME: N/A now)
   ;psurfratio = expf(-height / 8000.)   ; height in [m]
   ;FOR ib=0,1 DO taur[ib] = taur[ib] * psurfratio
   
   FOR ib=0,1 DO BEGIN
      ; Calculate Rayleigh reflectance and transmittance
      rayleigh_tr,taur[ib], mus, muv, relazi, rhoray, trup, trdown
      Ttotrayu = ((2 / 3. + muv) + (2 / 3. - muv) * trup)   / (4 / 3. + taur[ib]);
      Ttotrayd = ((2 / 3. + mus) + (2 / 3. - mus) * trdown) / (4 / 3. + taur[ib]);
      
      ; Get Rayleigh spherical albedo
      rayleigh_sphalb, taur[ib], sphalb
      
      ; Get ozon tranmittance
      tO3 = exp(-1. * airMass * 0.319 * aO3[ib]);
      
      ; Get water vapor tranmittance 
      IF (bH2O[ib] NE 0.0) THEN tH2O = exp(-exp(aH2O[ib] + bH2O[ib] * alog(airMass * 2.93))) $
      ELSE tH2O = 1.0
      TtotraytH2O = Ttotrayu * Ttotrayd * tH2O
      
      ; correction
      CASE ib OF
         0: refl = blue[vldIdx]   
         1: refl = red[vldIdx] 
      ENDCASE
      
      corr_refl = (refl / tO3 - rhoray) / TtotraytH2O;
      corr_refl /= (1.0 + corr_refl * sphalb);
      idx = WHERE(corr_refl LT 0., n)
      IF n GT 0 THEN corr_refl[idx] = 0.0
      
      CASE ib OF
         0: blue[vldIdx] =  corr_refl  
         1: red[vldIdx]  = corr_refl
      ENDCASE
         
   ENDFOR
   
   RETURN, 0
END
