;******************************************************************************
; PROGRAM NAME: 
;	viirs_edr_convert
;
; AUTHOR:
; Stephen Superczynski, NOAA-STAR
; Last modified on 9/16/2015
;
; PURPOSE:
; This program takes a VIIRS Aerosol EDR with associated geolocation data and
; converts it into a 'MODIS-like' EOS-HDF format. Individual granules with 
; matching geolocation granule as well as aggregated Aerosol EDR "tiles" 
; packaged with geolocation data from NOAA-CLASS can be converted. Using the 
; supplied input data path it searches the directory for .h5 files to convert. 
; The output is a matching .hdf file for each input file, and located in the 
; same input directory.
;
; SYNTAX
;		viirs_edr_convert, data_path
;
; PARAMTERS:
; 	data_path, required,
;			The local path used to search for VIIRS .h5 files.
; 
; OUTPUT:
; Returns a .hdf file with all SDS formatted as EOS-HDF. File names are the 
; same as the input file, with '_EOS.hdf' ammended to the end.
;
; VERSION HISTORY:
; 09/04/2014 - Inital version
; 12/30/2014 - Added ability to process individual granules
; 01/07/2015 - Improved error handling and replaced directory management with
;              new file searching paradigm.
; 02/11/2015 - Revised methodology for setting SDS attributes.
; 09/16/2015 - Fixed bug that prevented Aerosol EDR datasets from being displayed
;              properly by some HDF readers.
;******************************************************************************

; This procedure places the dataset into the new EOS swatch.
PRO create_EOS, FID, SID, mem_name, data
	;attach to swath
  SID = EOS_SW_attach(FID, 'VIIRS_AERO')
	if SID eq -1 then begin
		print, 'Problem attaching to swath'
		RETURN
	endif
	;write SDS to new HDF4 file
	status = EOS_SW_writefield(SID, mem_name, data)
;	STOP
	if status eq -1 then begin
		print, 'Problem writing '+mem_name+' to swath'
		RETURN
	endif
	;detach from swath
	status = EOS_SW_detach(SID) 
;	STOP
	if status eq -1 then begin 
		print, 'Problem finally detaching from swath'
	 RETURN
	endif
;	STOP
print, mem_name, ' - written'
END

;------------------------------------------------------------------------------
;Main Program
;------------------------------------------------------------------------------

PRO viirs_edr_convert, data_path

@viirs_to_mod_convert.h

;remove trailing '/' if included in data_path
check1 = n_elements(data_path)
if check1 gt 0 then begin
	cd, current=original
	cd, data_path
	cd, current=current
	print, 'Current Directory: ', current
endif

;search for files in specified path or current directory if no path is given.
h5_file = file_search('*VAOOO*.h5', count=count)

;define variables
aero_path = 'All_Data/VIIRS-Aeros-EDR_All'
geo_path = 'All_Data/VIIRS-Aeros-EDR-GEO_All'

;check input path for presence of hdf5 files
if count eq 0 then begin
	MESSAGE, 'ERROR - No HDF5 files found!'
endif

for f=0,count-1 do begin
	;check to make sure file is actually an HDF5 file with both AERO and GEO
	;information from CLASS.
	valid_hdf = h5f_is_hdf5(h5_file[f])
	if valid_hdf eq 0 then begin
		MESSAGE, 'ERROR - HDF5 file is not valid.'
	endif
	
  ;check to be sure file is VIIRS Aerosol EDR. Also determine if supplied data is
	;a single granule or aggregated dataset. 
	str = file_basename(h5_file[f], '.h5')
  check2 = strcmp('GAERO-', str[0], 6)
  check3 = strcmp('VAOOO_', str[0], 6)
  if check2 eq 0 and check3 eq 0 then begin
    MESSAGE, 'ERROR - File must be VIIRS aerosol dataset.'
  endif

  ;check file then open HDF5 file for reading
  if check2 eq 1 then begin
    geo_id = H5F_OPEN(h5_file[f])
    aer_id = geo_id
  endif
  if check3 eq 1 then begin
    base = strmid(h5_file[f], strpos(h5_file[f], 'npp'))
		base2 = strmid(base, 0, strpos(base, 'c'))
		geo_file = file_search('GAERO_'+base2+'*.h5')
		if strmatch('', geo_file) eq 1 then begin
      MESSAGE, 'ERROR - Matching geolocation file required for tool to run'
    endif
    geo_id = H5F_OPEN(geo_file)
    aer_id = H5F_OPEN(h5_file[f])
  endif

	;set new HDF4 filename
	hdf4_filename = str[0]+'_EOS.hdf'
	FID = EOS_SW_open(hdf4_filename, /create)
	SID = EOS_SW_create(FID, 'VIIRS_AERO')
	print, ''
	print, hdf4_filename+' open for writing...'	
		
	;extract dimensions from HDF5 file
	temp_id = h5d_open(geo_id, geo_path+'/Latitude')						
	space_id = h5d_get_space(temp_id)
	array_dims = h5s_get_simple_extent_dims(space_id)					;data array dimensions
	h5s_close, space_id
	h5d_close, temp_id

	temp_id = h5d_open(geo_id, geo_path+'/QF1_SCAN_VIIRSAEROGEO')
	space_id = h5d_get_space(temp_id)
	scan_num = h5s_get_simple_extent_dims(space_id)						;number of scans
	h5s_close, space_id
  h5d_close, temp_id

	;define datset dimensions
  dim1 = EOS_SW_defdim(SID, 'AcrossTrack', array_dims[0])
  dim2 = EOS_SW_defdim(SID, 'AlongTrack', array_dims[1])
  dim3 = EOS_SW_defdim(SID, 'Scan', scan_num )
  dim4 = EOS_SW_defdim(SID, 'Direction', 3)
;WRITE GEOLOCATION DATASETS

	;get Geolocation group members
	num_geo_mem = H5G_GET_NMEMBERS(geo_id, geo_path)
	print, ''
	print, 'GEOLOCATION DATASETS:'

	for i=0, num_geo_mem-1 do begin
  	mem_name = H5G_GET_MEMBER_NAME(geo_id, geo_path, i)

  	;define geolocation datasets
  	str_match1 = strmatch(mem_name, 'QF1*VIIRSAEROGEO')
  	str_match2 = strmatch(mem_name, 'QF2_VIIRSAEROGEO')
  	str_match3 = strmatch(mem_name, '*Time')
  	str_match4 = strmatch(mem_name, 'SC*')

  	CASE 1 of
    	str_match1 eq 1: BEGIN
      	type = 3
      	dim_names = 'Scan'
      	status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
  			if status eq -1 then begin
    			print, 'Problem defining '+mem_name+' field'
    			BREAK
				endif
      	END
    	str_match2 eq 1: BEGIN
      	dim_names ='AcrossTrack,AlongTrack'
      	type = 3
      	status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
      	if status eq -1 then begin
        	print, 'Problem defining '+mem_name+' field'
        	BREAK
      	endif
      	END
    	str_match3 eq 1: BEGIN
      	type = 5
      	dim_names = 'Scan'
      	status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
      	if status eq -1 then begin
        	print, 'Problem defining '+mem_name+' field'
        	BREAK
      	endif
      	END
    	str_match4 eq 1: BEGIN
      	type = 5
      	dim_names = 'Direction,Scan'
      	status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
      	if status eq -1 then begin
        	print, 'Problem defining '+mem_name+' field'
        	BREAK
      	endif
      	END
    	ELSE: BEGIN
      	type = 5
      	dim_names = 'AcrossTrack,AlongTrack'
      	if mem_name eq 'Latitude' or mem_name eq 'Longitude' then begin
    			status = EOS_SW_defgeofield(SID, mem_name, dim_names, type)
  			endif else begin
    			status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
  			endelse
  			if status eq -1 then begin
    			print, 'Problem defining '+mem_name+' field'
    			BREAK
  			endif
      	END
  	ENDCASE
	
	endfor

	;detach from swath	
	status = EOS_SW_detach(SID)
	if status eq -1 then begin
		print, 'Problem detaching from swath'
	endif

	for i=0, num_geo_mem-1 do begin
  	mem_name = H5G_GET_MEMBER_NAME(geo_id, geo_path, i)

		;read VIIRS Geolocation data from HDF5 file
  	data_id = H5D_OPEN(geo_id, geo_path+'/'+mem_name)
  	data = H5D_READ(data_id)

		;write geolocation datasets
		str_match1 = strmatch(mem_name, 'QF1*VIIRSAEROGEO')
		str_match2 = strmatch(mem_name, 'QF2_VIIRSAEROGEO')
		str_match3 = strmatch(mem_name, '*Time')
		str_match4 = strmatch(mem_name, 'SC*')

  	CASE 1 of
			str_match1 eq 1: BEGIN
				type = 3
				CREATE_EOS, FID, SID, mem_name, data
				END
    	str_match2 eq 1: BEGIN
				type = 3
      	CREATE_EOS, FID, SID, mem_name, data
      	END
			str_match3 eq 1: BEGIN
				type = 5
				CREATE_EOS, FID, SID, mem_name, data
				END
			str_match4 eq 1: BEGIN
				type = 5
				CREATE_EOS, FID, SID, mem_name, data
      	END
			ELSE: BEGIN
				type = 5
      	CREATE_EOS, FID, SID, mem_name, data
      	END
		ENDCASE
	endfor

;WRITE AEROSOL DATASETS

	;get Aerosol group members
	num_aero_mem = H5G_GET_NMEMBERS(aer_id, aero_path)
	print, ''
	print, 'AEROSOL DATASETS:'

	;attach to swath
	SID = EOS_SW_attach(FID, 'VIIRS_AERO')
	if SID eq -1 then begin
  	print, 'Problem attaching to swath'
	endif

	;define aerosol fields
	for j=0, num_aero_mem-1 do begin
  	mem_name = H5G_GET_MEMBER_NAME(aer_id, aero_path, j)

	  str_match5 = strmatch(mem_name, 'Aerosol*nm')
  	str_match6 = strmatch(mem_name, 'AngstromExponent')
	  str_match7 = strmatch(mem_name, 'QF*EDR')
	  str_match8 = strmatch(mem_name, 'SmallModeFraction')

  	dim_names = 'AcrossTrack,AlongTrack'

  	CASE 1 OF
    	str_match5 eq 1: BEGIN
      	type = 22
	      status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
  	    if status eq -1 then begin
    	    print, 'Problem defining '+mem_name+' field'
      	  BREAK
	      endif
  	    END
    	str_match6 eq 1: BEGIN
      	type = 22
	      status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
  	    if status eq -1 then begin
    	    print, 'Problem defining '+mem_name+' field'
      	  BREAK
	      endif
  	    END
    	str_match7 eq 1: BEGIN
      	type = 3
	      status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
  	    if status eq -1 then begin
    	    print, 'Problem defining '+mem_name+' field'
      	  BREAK
	      endif 
  	    END
    	str_match8 eq 1: BEGIN
      	type = 3
	      status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
  	    if status eq -1 then begin
    	    print, 'Problem defining '+mem_name+' field'
      	 	BREAK
	      endif
  	    END
    	ELSE: BREAK
	  ENDCASE
	endfor

	;detach from swath
	status = EOS_SW_detach(SID)
	if status eq -1 then begin
  	print, 'Problem detaching from swath'
	endif

	;get aerosol group members
	for j=0, num_aero_mem-1 do begin
  	mem_name = H5G_GET_MEMBER_NAME(aer_id, aero_path, j)
  
		;read VIIRS Aerosol data from HDF5 file
  	data_id = H5D_OPEN(aer_id, aero_path+'/'+mem_name)
	  data = H5D_READ(data_id)

		str_match5 = strmatch(mem_name, 'Aerosol*nm')
		str_match6 = strmatch(mem_name, 'AngstromExponent')
  	str_match7 = strmatch(mem_name, 'QF*EDR')
	  str_match8 = strmatch(mem_name, 'SmallModeFraction')

  	CASE 1 OF
    	str_match5 eq 1: BEGIN
				good = where(data lt 65000, complement=fill)
				data_tmp = float(data)
				data_tmp[good] = (data_tmp[good]*.00016787) - 1
				new_aod = long(data_tmp*1000)
				new_aod[fill] = -9999
				CREATE_EOS, FID, SID, mem_name, new_aod
  	    END
    	str_match6 eq 1: BEGIN
				good = where(data lt 65000, complement=fill)
				data_temp = float(data)
				data_temp[good] = (data_temp[good]*.000183131) - 2
				new_ae = long(data_temp*1000)
				new_ae[fill] = -9999
    	  CREATE_EOS, FID, SID, mem_name, new_ae
      	END
	    str_match7 eq 1: BEGIN
				CREATE_EOS, FID, SID, mem_name, data
    	  END
	    str_match8 eq 1: BEGIN
  	    CREATE_EOS, FID, SID, mem_name, data
    	  END
			ELSE: BREAK	
  	ENDCASE
	endfor

	;close swath file access
	result = eos_sw_close(FID)
	
	;start SD Interface for setting SDS attributes
	SDid = HDF_SD_start(hdf4_filename, /rdwr)
	for i=0, num_geo_mem-1 do begin
	
		mem_name = H5G_GET_MEMBER_NAME(geo_id, geo_path, i)
		
		if mem_name eq 'MidTime' or mem_name eq 'StartTime' or mem_name eq $
			'QF1_SCAN_VIIRSAEROGEO' then CONTINUE
		idx = where(viirsEDRgeo.name eq mem_name, num)
		index = HDF_SD_nametoindex(SDid, mem_name)
		SDSid = HDF_SD_select(SDid, index)
		
		if num gt 0 then begin
			HDF_SD_attrset ,SDSid, 'valid_range', viirsEDRgeo[idx].valid_range
			HDF_SD_attrset, SDSid, '_FillValue', viirsEDRgeo[idx]._FillValue
			HDF_SD_attrset, SDSid, 'long_name', viirsEDRgeo[idx].long_name
    	HDF_SD_attrset, SDSid, 'units', viirsEDRgeo[idx].units
  	  HDF_SD_attrset, SDSid, 'scale_factor', viirsEDRgeo[idx].scale_factor
	    HDF_SD_attrset, SDSid, 'add_offset', viirsEDRgeo[idx].add_offset
			HDF_SD_endaccess, SDSid
		endif

  endfor
	
	for j=0, num_aero_mem-1 do begin

		mem_name = H5G_GET_MEMBER_NAME(aer_id, aero_path, j)
		
		if mem_name eq 'AerosolOpticalDepthFactors' or mem_name eq 'AngstromExponentFactors' $
			then CONTINUE
		idx = where(viirsEDRaero.name eq mem_name, num)
		index = HDF_SD_nametoindex(SDid, mem_name)
		SDSid = HDF_SD_select(SDid, index)

		if num gt 0 then begin
			HDF_SD_attrset, SDSid, 'valid_range', viirsEDRaero[idx].valid_range
			HDF_SD_attrset, SDSid, '_FillValue', viirsEDRaero[idx]._FillValue, /INT
			HDF_SD_attrset, SDSid, 'long_name', viirsEDRaero[idx].long_name
			HDF_SD_attrset, SDSid, 'units', viirsEDRaero[idx].units
			HDF_SD_attrset, SDSid, 'scale_factor', viirsEDRaero[idx].scale_factor
			HDF_SD_attrset, SDSid, 'add_offset', viirsEDRaero[idx].add_offset
			HDF_SD_endaccess, SDSid
  	endif
	endfor

	;close HDF file access
	h5f_close, aer_id
	HDF_SD_end, SDid

endfor
;change back to original directory
if check1 gt 0 then cd, original

END
