;******************************************************************************
; PROGRAM NAME: 
;		viirs_ip_convert
;
; AUTHOR:
; Stephen Superczynski, NOAA-STAR
; Last modified on 2/11/2015
;
; PURPOSE:
; This program takes VIIRS Aerosol IP with associated geolocation data and
; converts it into a 'MODIS-like' EOS-HDF format. An IP granule along with 
; the matching geolocation granule is required for use with this program. 
; Using the supplied input data path it searches the directory for .h5 files
; to convert. The output is a matching .hdf file for each input file, and 
; located in the same input directory.
;
; SYNTAX
;		viirs_ip_convert, data_path
;
; PARAMTERS:
; 	data_path, required,
;			The local path used to search for VIIRS .h5 files.
; 
; OUTPUT:
; Returns a .hdf file with all SDS formatted as EOS-HDF. File names are the 
; same as the input file, with '_EOS.hdf' ammended to the end.
;
; VERSION HISTORY:
; 12/30/2104 - Inital version
; 01/07/2015 - Improved error handling and replaced directory management with 
;							 new file searching paradigm.
; 02/11/2015 - Revised methodology for setting SDS attributes.
;******************************************************************************

; This procedure places the dataset into the new EOS swatch.
PRO create_EOS, FID, SID, mem_name, data
	;attach to swath
  SID = EOS_SW_attach(FID, 'VIIRS_AERO')
	if SID eq -1 then begin
		print, 'Problem attaching to swath'
		RETURN
	endif
	;write SDS to new HDF4 file
	status = EOS_SW_writefield(SID, mem_name, data)
	if status eq -1 then begin
		print, 'Problem writing '+mem_name+' to swath'
		RETURN
	endif
	;detach from swath
	status = EOS_SW_detach(SID) 
	if status eq -1 then begin 
		print, 'Problem finally detaching from swath'
	 RETURN
	endif
print, mem_name, ' - written'
END

;------------------------------------------------------------------------------
;Main Program
;------------------------------------------------------------------------------

PRO viirs_ip_convert, data_path

@viirs_to_mod_convert.h

;remove trailing '/' if included in data_path
check1 = n_elements(data_path)
if check1 eq 1 then begin
	cd, current=original
	cd, data_path
	cd, current=current
	print, 'Current Directory: ', current
endif

;search for IP AOT files.
h5_file = file_search('IVAOT*.h5', count=count)

;define variables
aero_path = 'All_Data/VIIRS-Aeros-Opt-Thick-IP_All'
geo_path = 'All_Data/VIIRS-MOD-GEO-TC_All'

;check input path for presence of hdf5 files
if count eq 0 then begin
	MESSAGE, 'ERROR - No HDF5 files found!'
endif

for f=0,count-1 do begin
	;print name of file being processed.
	print, 'Processing: ', h5_file[f]

	;check to make sure file is actually an HDF5 file with both AERO and GEO
	;information from CLASS.
	valid_hdf = h5f_is_hdf5(h5_file[f])
	if valid_hdf eq 0 then begin
		MESSAGE, 'ERROR - HDF5 file is not valid.'
	endif
	
  ;check to be sure file is VIIRS Aerosol IP file
	str = file_basename(h5_file[f], '.h5')
  check2 = strcmp('IVAOT_', str[0], 6)
  if check2 eq 0 then begin
    MESSAGE, 'ERROR - File must be VIIRS aerosol dataset.'
  endif

  ;check file then open HDF5 file for reading
  if check2 eq 1 then begin
    base = strmid(h5_file[f], strpos(h5_file[f], 'npp'))
		base2 = strmid(base, 0, strpos(base, 'c'))
		geo_file = file_search('GMTCO_'+base2+'*.h5')
		if strmatch('', geo_file) eq 1 then begin
      MESSAGE, 'ERROR - Matching geolocation file required for tool to run'
    endif
    geo_id = H5F_OPEN(geo_file)
    aer_id = H5F_OPEN(h5_file[f])
  endif

	;set new HDF4 filename
	hdf4_filename = str[0]+'_EOS.hdf'
	FID = EOS_SW_open(hdf4_filename, /create)
	SID = EOS_SW_create(FID, 'VIIRS_AERO')
	print, ''
	print, hdf4_filename+' open for writing...'	
		
	;extract dimensions from HDF5 file
	temp_id = h5d_open(geo_id, geo_path+'/Latitude')						
	space_id = h5d_get_space(temp_id)
	array_dims = h5s_get_simple_extent_dims(space_id)					;data array dimensions
	h5s_close, space_id
	h5d_close, temp_id

	temp_id = h5d_open(geo_id, geo_path+'/NumberOfScans')
	scan_num = h5d_read(temp_id)					;number of scans
  h5d_close, temp_id

	;define datset dimensions
  dim1 = EOS_SW_defdim(SID, 'AcrossTrack', array_dims[0])
  dim2 = EOS_SW_defdim(SID, 'AlongTrack', array_dims[1])
  dim3 = EOS_SW_defdim(SID, 'Scan', scan_num )
  dim4 = EOS_SW_defdim(SID, 'Direction', 3)
	dim5 = EOS_SW_defdim(SID, 'Data', 0)

;WRITE GEOLOCATION DATASETS

	;get Geolocation group members
	num_geo_mem = H5G_GET_NMEMBERS(geo_id, geo_path)
	print, ''
	print, 'GEOLOCATION DATASETS:'

	for i=0, num_geo_mem-1 do begin
  	mem_name = H5G_GET_MEMBER_NAME(geo_id, geo_path, i)

  	;define geolocation datasets
		str_match1 = strmatch(mem_name, '*Time')
  	str_match2 = strmatch(mem_name, 'QF*_S*')
		str_match3 = strmatch(mem_name, 'SCSolar*Angle')
  	str_match4 = strmatch(mem_name, 'SC[!S]*')
  	str_match5 = strmatch(mem_name, 'Mode*')
  	str_match6 = strmatch(mem_name, 'Num*')
		str_match7 = strmatch(mem_name, 'QF2_V*')
		str_match8 = strmatch(mem_name, 'Pad*')

  	CASE 1 of
    	str_match1 eq 1: BEGIN
      	type = 5
      	dim_names = 'Scan'
      	status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
  			if status eq -1 then begin
    			print, 'Problem defining '+mem_name+' field'
    			BREAK
				endif
      	END
    	str_match2 eq 1: BEGIN
      	dim_names ='Scan'
      	type = 21
      	status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
      	if status eq -1 then begin
        	print, 'Problem defining '+mem_name+' field'
        	BREAK
      	endif
      	END
    	str_match3 eq 1: BEGIN
      	type = 5
      	dim_names = 'Scan'
      	status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
      	if status eq -1 then begin
        	print, 'Problem defining '+mem_name+' field'
        	BREAK
      	endif
      	END
    	str_match4 eq 1: BEGIN
      	type = 5
      	dim_names = 'Direction,Scan'
      	status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
      	if status eq -1 then begin
        	print, 'Problem defining '+mem_name+' field'
        	BREAK
      	endif
      	END
			str_match5 eq 1: BEGIN
        type = 21
        dim_names = 'Scan'
        status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
        if status eq -1 then begin
          print, 'Problem defining '+mem_name+' field'
          BREAK
        endif
        END
			str_match6 eq 1: BEGIN
        type = 22
        dim_names = 'Data'
        status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
        if status eq -1 then begin
          print, 'Problem defining '+mem_name+' field'
          BREAK
        endif
        END
			str_match7 eq 1: BEGIN
        type = 21
        dim_names = 'AcrossTrack,AlongTrack'
        status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
        if status eq -1 then begin
          print, 'Problem defining '+mem_name+' field'
          BREAK
        endif
        END
			str_match8 eq 1: BEGIN
        type = 21
        dim_names = 'Data'
        status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
        if status eq -1 then begin
          print, 'Problem defining '+mem_name+' field'
          BREAK
        endif
        END
    	ELSE: BEGIN
      	type = 5
      	dim_names = 'AcrossTrack,AlongTrack'
      	if mem_name eq 'Latitude' or mem_name eq 'Longitude' then begin
    			status = EOS_SW_defgeofield(SID, mem_name, dim_names, type)
  			endif else begin
    			status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
  			endelse
  			if status eq -1 then begin
    			print, 'Problem defining '+mem_name+' field'
    			BREAK
  			endif
      	END
  	ENDCASE
	
	endfor

	;detach from swath	
	status = EOS_SW_detach(SID)
	if status eq -1 then begin
		print, 'Problem detaching from swath'
	endif

	for i=0, num_geo_mem-1 do begin
  	mem_name = H5G_GET_MEMBER_NAME(geo_id, geo_path, i)

		;read VIIRS Geolocation data from HDF5 file
  	data_id = H5D_OPEN(geo_id, geo_path+'/'+mem_name)
  	data = H5D_READ(data_id)

		;write geolocation datasets
		str_match1 = strmatch(mem_name, '*Time')
    str_match2 = strmatch(mem_name, 'QF*_S*')
    str_match3 = strmatch(mem_name, 'SCSolar*Angle')
    str_match4 = strmatch(mem_name, 'SC[!S]*')
    str_match5 = strmatch(mem_name, 'Mode*')
    str_match6 = strmatch(mem_name, 'Num*')
    str_match7 = strmatch(mem_name, 'QF2_V*')
    str_match8 = strmatch(mem_name, 'Pad*')

  	CASE 1 of
			str_match1 eq 1: BEGIN
				type = 5
				CREATE_EOS, FID, SID, mem_name, data
				END
    	str_match2 eq 1: BEGIN
				type = 21
      	CREATE_EOS, FID, SID, mem_name, data
      	END
			str_match3 eq 1: BEGIN
				type = 5
				CREATE_EOS, FID, SID, mem_name, data
				END
			str_match4 eq 1: BEGIN
				type = 5
				CREATE_EOS, FID, SID, mem_name, data
      	END
			str_match5 eq 1: BEGIN
        type = 21
        CREATE_EOS, FID, SID, mem_name, data
				END
			str_match6 eq 1: BEGIN
        type = 24
        CREATE_EOS, FID, SID, mem_name, data
				END
			str_match7 eq 1: BEGIN
        type = 21
        CREATE_EOS, FID, SID, mem_name, data
				END
			str_match8 eq 1: BEGIN
        type = 21
        CREATE_EOS, FID, SID, mem_name, data
				END
			ELSE: BEGIN
				type = 5
      	CREATE_EOS, FID, SID, mem_name, data
      	END
		ENDCASE
	endfor

;WRITE AEROSOL DATASETS

	;get Aerosol group members
	num_aero_mem = H5G_GET_NMEMBERS(aer_id, aero_path)
	print, ''
	print, 'AEROSOL DATASETS:'

	;attach to swath
	SID = EOS_SW_attach(FID, 'VIIRS_AERO')
	if SID eq -1 then begin
  	print, 'Problem attaching to swath'
	endif

	;define aerosol fields
	for j=0, num_aero_mem-1 do begin
  	mem_name = H5G_GET_MEMBER_NAME(aer_id, aero_path, j)

	  str_match9 = strmatch(mem_name, 'QF*')

  	dim_names = 'AcrossTrack,AlongTrack'

  	CASE 1 OF
    	str_match9 eq 1: BEGIN
      	type = 21
	      status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
  	    if status eq -1 then begin
    	    print, 'Problem defining '+mem_name+' field'
      	 	BREAK
	      endif
  	    END
    	ELSE: BEGIN
				type = 5
        status = EOS_SW_defdatafield(SID, mem_name, dim_names, type)
        if status eq -1 then begin
          print, 'Problem defining '+mem_name+' field'
          BREAK
        endif
        END
	  ENDCASE
	endfor

	;detach from swath
	status = EOS_SW_detach(SID)
	if status eq -1 then begin
  	print, 'Problem detaching from swath'
	endif

	;get aerosol group members
	for j=0, num_aero_mem-1 do begin
  	mem_name = H5G_GET_MEMBER_NAME(aer_id, aero_path, j)
  
		;read VIIRS Aerosol data from HDF5 file
  	data_id = H5D_OPEN(aer_id, aero_path+'/'+mem_name)
	  data = H5D_READ(data_id)

	 	CREATE_EOS, FID, SID, mem_name, data
	endfor

	;close swath file access
	result = eos_sw_close(FID)

	;start SD Interface for setting SDS attributes
  SDid = HDF_SD_start(hdf4_filename, /rdwr)

	for i=0, num_geo_mem-1 do begin

    mem_name = H5G_GET_MEMBER_NAME(geo_id, geo_path, i)
		
		if mem_name eq 'MidTime' or mem_name eq 'StartTime' or mem_name eq 'ModeGran' or $
			mem_name eq 'ModeScan' or mem_name eq 'NumberOfScans' or mem_name eq 'PadByte1' or $
			mem_name eq 'QF1_SCAN_VIIRSSDRGEO' or mem_name eq 'QF2_SCAN_VIIRSSDRGEO' or $
			mem_name eq 'SCSolarAzimuthAngle' or mem_name eq 'SCSolarZenithAngle' then CONTINUE
    idx = where(viirsMODgeo.name eq mem_name, num)
    index = HDF_SD_nametoindex(SDid, mem_name)
    SDSid = HDF_SD_select(SDid, index)

    if num gt 0 then begin
      HDF_SD_attrset ,SDSid, 'valid_range', viirsMODgeo[idx].valid_range
      HDF_SD_attrset, SDSid, '_FillValue', viirsMODgeo[idx]._FillValue
      HDF_SD_attrset, SDSid, 'long_name', viirsMODgeo[idx].long_name
      HDF_SD_attrset, SDSid, 'units', viirsMODgeo[idx].units
      HDF_SD_attrset, SDSid, 'scale_factor', viirsMODgeo[idx].scale_factor
      HDF_SD_attrset, SDSid, 'add_offset', viirsMODgeo[idx].add_offset
      HDF_SD_endaccess, SDSid
    endif

  endfor
	
  for j=0, num_aero_mem-1 do begin

    mem_name = H5G_GET_MEMBER_NAME(aer_id, aero_path, j)

		idx = where(viirsIPaero.name eq mem_name, num)
    index = HDF_SD_nametoindex(SDid, mem_name)
    SDSid = HDF_SD_select(SDid, index)

    if num gt 0 then begin
      HDF_SD_attrset ,SDSid, 'valid_range', viirsIPaero[idx].valid_range
      HDF_SD_attrset, SDSid, '_FillValue', viirsIPaero[idx]._FillValue
      HDF_SD_attrset, SDSid, 'long_name', viirsIPaero[idx].long_name
      HDF_SD_attrset, SDSid, 'units', viirsIPaero[idx].units
      HDF_SD_attrset, SDSid, 'scale_factor', viirsIPaero[idx].scale_factor
      HDF_SD_attrset, SDSid, 'add_offset', viirsIPaero[idx].add_offset
      HDF_SD_endaccess, SDSid
    endif

  endfor
	

	;close HDF file access
	h5f_close, aer_id
	HDF_SD_end, SDid
endfor

;change back to original directory
if check1 eq 1 then cd, original

END
